/*
 * ckey.c
 *
 * implements channel key list (so you don't have to remember them!)
 * 
 * written by Joshua J. Drake
 */

#include "ckey.h"
#include "channels.h"
#include "ninja.h"
#include "dma.h"
#include "list.h"
#include "ircaux.h"
#include "output.h"

/* global variables */
CKey *ckey_list = (CKey *) NULL;	/* channel key list */

/*
 * gets the key to a channel from the ckey list.
 *
 * returns a pointer to the actual key string.
 */
u_char *
get_ckey(u_char *chan)
{
   CKey *tmp;
   
   /* no ckeys, no business */
   if (!ckey_list)
     return empty_string;

   /* if we can't find a ckey for the specified channel, return "" */
   if ((tmp = (CKey *) find_in_list((List **)&ckey_list, chan, 0)) == NULL)
     return empty_string;
   return tmp->key;
}


/*
 * free all data associated with a ckey list entry
 */
void
free_ckey(tmp)
    CKey **tmp;
{
   CKey *ok = *tmp;
   
   dma_Free(&(ok->key));
   dma_Free(&(ok->channel));
   dma_Free(&ok);
}

/*
 * adds a ckey to the ckey list, unless of course one already exists.
 * in that case, it replaces it with the key specified.
 */
void
add_ckey(chan, key)
   u_char *chan, *key;
{
   CKey *tmp, *new;
   
   /* if either argument is empty, never mind */
   if (key == NULL || chan == NULL)
     return;
   
   /* try to remove the ckey from the list. (fails if it doesn't exist) */
   if ((tmp = (CKey *) remove_from_list((List **)&ckey_list, chan)) != NULL)
     free_ckey(&tmp);

   /* okay, so now there's no ckey for that channel even if there was one.. so we make a new entry */
   new = (CKey *) dma_Malloc(sizeof(CKey));
   dma_strcpy(&new->key, key);
   dma_strcpy(&new->channel, chan);
   
   /* add the new entry to the list */
   add_to_list((List **)&ckey_list, (List *)new);
   
   /* save the list every time one is added.. (good for when you core dump) */
   if (!qflag)
     (void)save_ckeys(1);
}

/*
 * saves the ckey list to NINJA_CKEY_FILE for later retreival (when you start ninja again)
 */
int
save_ckeys(quiet)
    int quiet;
{
   u_char *fullname = NULL;
   FILE *fdesc;
   CKey *tmp;
   int count = 0;

   fullname = expand_twiddle(NINJA_CKEY_FILE);
   fdesc = fopen(fullname, "w");
   if (fdesc == (FILE *)NULL)
     {
        put_error("Cannot write channel key list: %s", strerror(errno));
	dma_Free(&fullname);
        return 0;
     }
   dma_Free(&fullname);
   
   /* run through the list, printing a line for each, format: <chan> <key>\n */
   for (tmp = ckey_list; tmp; tmp = tmp->next)
     {
        fprintf(fdesc, "%s %s\n", tmp->channel, tmp->key);
        count++;
     }
   fclose(fdesc);
   
   /* if we're not being quiet, and there were keys saved, then tell the user about it */
   if (!quiet && count)
     put_info("Saved %d channel key%s...", count, count != 1 ? UP("s") : empty_string);
   return count;
}

/*
 * load the ckey list back into memory from disk.
 */
void
load_ckeys(void)
{
   u_char *fullname = NULL, tmpinbuf[1024], *tibptr, *chan, *key;
   FILE *fdesc;

   fullname = expand_twiddle(NINJA_CKEY_FILE);
   fdesc = fopen(fullname, "r");
   if (fdesc == (FILE *) NULL)
     {
        /* we're not gonna print this anymore
	put_info("Cannot read Channel Key list..");
	 */
	dma_Free(&fullname);
        return;
     }
   dma_Free(&fullname);

   /* read the list. */
   memset(tmpinbuf, 0, sizeof(tmpinbuf));
   while (fgets(tmpinbuf, sizeof(tmpinbuf)-1, fdesc))
     {
	/* strip the new line/carriage return */
	tibptr = my_rindex(tmpinbuf, '\n');
	if (tibptr)
	  *tibptr = '\0';
	tibptr = my_rindex(tmpinbuf, '\r');
	if (tibptr)
	  *tibptr = '\0';
	
	/* extract the channel/key */
        tibptr = tmpinbuf;
        chan = next_arg(tibptr, &tibptr);
        key = next_arg(tibptr, &tibptr);

	/* add it to the list */
        add_ckey(chan, key);
	memset(tmpinbuf, 0, sizeof(tmpinbuf));
     }
   fclose(fdesc);
}


void
free_ckeylist(void)
{
   CKey *tmp, *next;
   
   for (tmp = ckey_list; tmp; tmp = next)
     {
	next = tmp->next;
	free_ckey(&tmp);
     }
}
