
#include "gnutella.h"

#include <fcntl.h>
#include <sys/types.h>
#include <string.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <assert.h>

#include "interface.h"

GSList *sl_nodes = (GSList *) NULL;

gint nodes_in_list = 0;
gint nodes_active = 0;

guint32 global_messages = 0;
guint32 global_searches = 0;
guint32 routing_errors = 0;
guint32 dropped_messages = 0;
gint nodes_errno;

GHookList node_added_hook_list;
struct gnutella_node *node_added; /* For use by node_added_hook_list hooks, since we can't add a parameter at list invoke time. */


/* Network init ----------------------------------------------------------------------------------- */

void network_init(void)
{
	g_hook_list_init(&node_added_hook_list, sizeof(GHook));
	node_added_hook_list.seq_id = 1;
	node_added = NULL; 
}

/* Nodes ------------------------------------------------------------------------------------------ */

gboolean on_the_net(void)
{
	GSList *l = (GSList *) NULL;

	for (l = sl_nodes; l; l = l->next)
		if (((struct gnutella_node *) l->data)->status == GTA_NODE_CONNECTED) return TRUE;

	return FALSE;
}

void node_real_remove(struct gnutella_node *n)
{
	gint row;

	g_return_if_fail(n);

	row = gtk_clist_find_row_from_data(GTK_CLIST(clist_nodes), (gpointer) n);
	gtk_clist_remove(GTK_CLIST(clist_nodes), row);

	sl_nodes = g_slist_remove(sl_nodes, n);
	if (--nodes_in_list < 0) nodes_in_list = 0;

	if (n->tryagain == 1) host_add(NULL, n->ip, n->port, 0); // put in host cache at top
	if (n->tryagain == 2) host_add(NULL, n->ip, n->port, 1); // put in host cache at last

	g_free(n);
	n = NULL;
}


// remove a node, display reason, tryagain - 1= right away, 2= later
void node_remove(struct gnutella_node *n, const gchar *reason, gint tryagain)
{

	g_return_if_fail(n);

	if (n->status == GTA_NODE_REMOVING) return;

	if (n->status == GTA_NODE_CONNECTED) {
		routing_node_remove(n);
		if (--nodes_active < 0) nodes_active = 0;
		}

	if (n->socket)
	{
		n->socket->resource.node = (struct gnutella_node *) NULL;
		socket_destroy(n->socket);
		n->socket = NULL;
	}

	if (n->gdk_tag) gdk_input_remove(n->gdk_tag);

	while (n->send_packet_list) { // free any unsent packets
		if (n->send_packet_list->data) g_free(n->send_packet_list->data); // free the packet memory
		n->send_packet_list = g_slist_remove(n->send_packet_list, n->send_packet_list->data);
		}

	n->status = GTA_NODE_REMOVING;
	n->last_update = time((time_t *) NULL);
	n->tryagain = tryagain;

	if (reason) strncpy(n->remove_msg, reason, 255);
	else n->remove_msg[0] = '\0';
	n->remove_msg[255] = '\0'; // always terminate the string, strncpy doesn't always do it

	gui_update_c_gnutellanet();
	gui_update_node(n, TRUE);

}


gboolean have_node(guint32 ip)
{
	GSList *l = (GSList *) NULL;

	if (!stop_host_get) { // we may want to connect to our own IP on another port for testing
		if (ip == local_ip || (force_local_ip && ip == forced_local_ip)) return TRUE;
		for (l = sl_nodes; l; l = l->next) if (((struct gnutella_node *) l->data)->ip == ip) return TRUE;
		}
	return FALSE;
}



// if socket is NULL, create outgoing connection
struct gnutella_node *node_add(struct gnutella_socket *s, guint32 ip, guint16 port)
{
	struct gnutella_node *n;
	struct gnutella_socket *s2;
	gchar *titles[4];
	gint row;
	gboolean incoming = FALSE, already_connected = FALSE;

#ifdef NO_RFC1918
		/* This needs to be a runtime option.  I could see a need for someone
		* to want to run gnutella behind a firewall over on a private network. */
	if (is_private_ip(ip)) {
		if (s) socket_destroy(s);
		return (struct gnutella_node *) NULL;
		}
#endif

	/* Too many gnutellaNet connections */
	if (nodes_active >= max_connections) {
		if (s) socket_destroy(s);
		return (struct gnutella_node *) NULL;
		}

	n = (struct gnutella_node *) g_malloc0(sizeof(struct gnutella_node));

	n->ip = ip;
	n->port = port;
	n->tryagain = 0;
	n->remove_msg[0] = '\0';
	strcpy (n->gui_info, "Generic");
	n->send_packet_list = NULL;
	n->sq_size = 0;

	if (s) /* This is an incoming control connection */
	{
		n->socket        = s;
		s->type          = GTA_TYPE_CONTROL;
		n->status        = GTA_NODE_PRE_WELCOME;
		s->resource.node = n;

		incoming = TRUE;
		titles[1] = (gchar *) "Incoming";
	}
	else /* We have to create an outgoing control connection for the node */
	{
		s2 = socket_connect(ip, port, GTA_TYPE_CONTROL); // s2 is new socket

		if (s2)
		{
			n->status        = GTA_NODE_CONNECTING;
			s2->resource.node = n;
			n->socket        = s2;
		}
		else
		{
			n->status     = GTA_NODE_REMOVING; // will be removed shortly
			n->socket     = NULL;
			if (errno == 101) strncpy(n->remove_msg, "Network unreachable (retrying)", 255);
			else strncpy(n->remove_msg, "Connection failed", 255);
		}

		titles[1] = (gchar *) "Outgoing";
	}

	nodes_errno = errno;
	titles[0] = ip_port_to_gchar(n->ip, n->port);
	titles[2] = (gchar *) "";

	row = gtk_clist_append(GTK_CLIST(clist_nodes), titles);
	gtk_clist_set_row_data(GTK_CLIST(clist_nodes), row, (gpointer) n);

	/* Check if we already have a connection to this node before adding the node to the list */

	already_connected = have_node(n->ip);

	sl_nodes = g_slist_prepend(sl_nodes, n);
	nodes_in_list++;

	if (incoming) /* Welcome the incoming node */
	{
		if (already_connected)
		{
			node_remove(n, "Already connected", 0);
			return (struct gnutella_node *) NULL;
		}
		else
		{
			gdk_input_remove(s->gdk_tag); // any more socket bytes go to node_read
			s->gdk_tag = 0;
			s->gdk_tag = gdk_input_add(s->file_desc, (GdkInputCondition) GDK_INPUT_READ | GDK_INPUT_EXCEPTION,
							node_read, (gpointer) n);
		}
	}

	gui_update_node(n, TRUE);
	gui_update_c_gnutellanet();

	return n;
}

/* Reading of messages ---------------------------------------------------------------------------- */

void node_parse(struct gnutella_node *node)
{
	static struct gnutella_node *n;
	gboolean drop = FALSE;

	g_return_if_fail(node);

	n = node;

	/* First some simple checks */

	switch (n->header.function)
	{
		case GTA_MSG_INIT:
		  if (*n->header.size) drop = TRUE;
			break;
		case GTA_MSG_INIT_RESPONSE:
			if (*n->header.size != sizeof(struct gnutella_init_response)) drop = TRUE;
			break;
		case GTA_MSG_PUSH_REQUEST:
			if (*n->header.size != sizeof(struct gnutella_push_request)) drop = TRUE;
			break;
		case GTA_MSG_SEARCH:
			if (!*n->header.size) drop = TRUE;
			else if (*n->header.size > search_queries_forward_size) drop = TRUE;
			break;
		case GTA_MSG_SEARCH_RESULTS:
			if (*n->header.size > search_answers_forward_size) drop = TRUE;
			break;

		default: /* Unknown message type - we drop it */
			drop = TRUE;
	}

	/* Route (forward) then handle the message if required */

	if (drop)
	{
		n->dropped++;
		dropped_messages++;
	}
	else if (route_message(&n)) /* We have to handle the message */
	{
		switch(n->header.function)
		{
			case GTA_MSG_INIT:           { reply_init(n); break; }
			case GTA_MSG_INIT_RESPONSE:  { host_add(n, 0, 0, 0); break; }
			case GTA_MSG_PUSH_REQUEST:   { handle_push_request(n); break; }
			case GTA_MSG_SEARCH:         { search_request(n); break; }
			case GTA_MSG_SEARCH_RESULTS: { search_results(n); break; }
			default:                     { message_dump(n); }
		}
	}

	if (!n) return;	/* The node has been removed during processing */

	n->have_header = FALSE;
	n->pos = 0;

}

// called from sockets.c
void node_init_outgoing(struct gnutella_node *n)
{
	if (write(n->socket->file_desc, gnutella_hello, strlen(gnutella_hello)) < 0)
	{
		node_remove(n, "Write error", 2); // 2 = we can retry this one later
	}
	else
	{
		n->status = GTA_NODE_HELLO_SENT;
		gui_update_node(n, TRUE);
	}
}

// we always queue all outgoing packets, then if the node gets slow we can
// easily start dropping packets, but priority packets (all the ones
// from us) get first in line, aren't we lucky?

gboolean node_enqueue(struct gnutella_node *n, struct node_send_packet *data, gboolean priority)
{
	/* Enqueue data for a node */

	g_return_val_if_fail(n, FALSE);

	assert(n->status != GTA_NODE_REMOVING);

	if(data == NULL) {
	  g_warning("node_enqueue:called with data == NULL\n");
	  return TRUE;
	}

	if(data->size == 0) {
	  if (data) g_free(data); // free the packet buffer memory
	  g_warning("node_enqueue:called with size == 0\n");
	  return TRUE;
	}

	// try to keep node connections, we need them to get search returns and
	// handle push requests. we can't just drop the connection because it's
	// overloaded, it may clear up in a few seconds.
	// if sendqueue is "full" because the node is slow, start dropping non
	// priority packets, we will timeout if the node is completely
	// unresponsive, in main.c
	// if the node needs more than a node_sendqueue_max_size buffer, we kick it off.

	// this kicks a node if it does not seem to let us ever write, it's maxed out and screwed up
	if (n->sq_size > node_sendqueue_max_size) {
		if (data) g_free(data); // free the packet buffer memory
		node_remove(n, "Can't write to this node", 0); // this will free buffers too
		return FALSE;
		}

	// if a priority packet, put this packet at the start of the list to be sent out
	if (priority) n->send_packet_list = g_slist_prepend(n->send_packet_list, (gpointer) data);
	else {
		if (data->size + n->sq_size > node_sendqueue_size) { // drop this packet?
			if (data) g_free(data); // free the packet buffer memory
			return FALSE;
			}
		// add this packet to the end of the list to be sent out
		n->send_packet_list = g_slist_append(n->send_packet_list, (gpointer) data);
		}

	n->sq_size += data->size;

	if (!n->gdk_tag) { // turn on the node_write call if it isn't on already
		n->gdk_tag = gdk_input_add(n->socket->file_desc, GDK_INPUT_WRITE | GDK_INPUT_EXCEPTION, node_write, (gpointer) n);

		/* We assume that if this is valid, it is non-zero */
		assert(n->gdk_tag);
	}

	return TRUE;
}


#include <sys/time.h>
#include <unistd.h>

void node_write(gpointer data, gint source, GdkInputCondition cond)
{
	struct gnutella_node *n = (struct gnutella_node *) data;
	gint r;
	guint32 size;
	guchar *packet;

	g_return_if_fail(n);

	if (cond & GDK_INPUT_EXCEPTION)	{ node_remove(n, "Write failed (Input Exception)", 2); return;}

	/* We can write again on the node's socket */

	if (n->send_packet_list) {

		packet = (guchar *) ((struct node_send_packet *) n->send_packet_list->data)->packet; // less CPU cycles this way
		size = (guint32) ((struct node_send_packet *) n->send_packet_list->data)->size;

		r = write(n->socket->file_desc, packet, size); // write out up to one packet

		if (r > 0) {

			if (r == size) { // we sent one complete packet
				if (n->send_packet_list->data) g_free(n->send_packet_list->data); // free the packet memory
				if (n->send_packet_list->data)
					n->send_packet_list = g_slist_remove(n->send_packet_list, n->send_packet_list->data);
				n->sent++;
				gui_update_node(n, FALSE);
				}
			else { // move the remaining data to the beginning of the buffer
				memmove(packet, packet + r, size - r);
				size -= r; // adjust the size for next time around
				}
			n->sq_size -= r;

		}
		else if(r == 0) { // try to handle all the possible errors, yuck!
				g_error("node_write:write returned 0?\n");
			}
			else if (errno == EAGAIN || errno == EINTR) {
		  			return;
				} else if (errno == EPIPE || errno == ENOSPC || errno == EIO || errno == ECONNRESET || errno == ETIMEDOUT) {
		  				node_remove(n, "Write to node failed", 2);
		  				return;
					}
					else {
		  				int terr = errno;
		  				time_t t = time(NULL);
		  				g_error("%s:node_write: write failed with unexpected errno: %d (%s)\n", ctime(&t), terr, g_strerror(terr));
					}
	}

	else // nothing to send out, turn off socket write gdk_tag call
	{
		gdk_input_remove(n->gdk_tag);
		n->gdk_tag = 0;
	}
}



// very fast loop to check for \r\n or \r\n\r\n, returns pointer to first \r
gchar *node_check_eol(gchar *buffer, gint len, gint mode)
{
	gchar *mybuffer = buffer; // make a local copy of buffer pointer
	gint i = len;

	if (i < 1) return NULL;
	do {
		if (*mybuffer == '\r') { // this makes it very fast
			if (*(mybuffer + 1) == '\n') {
				if (mode == 0) return mybuffer; // mode 0 is for \r\n check only
				if (*(mybuffer + 2) == '\r') {
					if (*(mybuffer + 3) == '\n') return mybuffer;
					}
				}
			}
		mybuffer++;
		} while (--i);
	return NULL; // no match
}


// every time we get something from a connected node, gdk will call this
void node_read(gpointer data, gint source, GdkInputCondition cond)
{
	gint r;
	struct gnutella_node *n = (struct gnutella_node *) data;
	struct gnutella_socket *s;
	guchar *w;

	gchar gnut_status_string[4]="   ", tmpbuf[256], *endlines;
	gint gnut_status, i, spaces;
	guint32 offs;
	gboolean kick = FALSE, ultrapeer;

	g_return_if_fail(n);
	s = n->socket;
	g_return_if_fail(s);

	w = (guchar *) &n->header;

	if (cond & GDK_INPUT_EXCEPTION) { node_remove(n, "Failed (Input Exception)", 2); return; }


	// extract connection data first, once per connection
	// at this point there is still data in s->buffer
	if (n->status == GTA_NODE_PRE_WELCOME) {

			// we are looking at something like
			// GNUTELLA CONNECT/0.6<cr><lf>
			// User-Agent: LimeWire 2.0.0<cr><lf>
			// Query-Routing: 0.2<cr><lf>
			// <cr><lf>

		// NOTE: n->buffer is NOT null terminated! use n->pos

		if (s->pos) { // one time per new connection, move socket buffer to our little node buffer
			memcpy(n->buffer, s->buffer, s->pos);
			n->pos = s->pos;
			s->pos = 0;
			}

		if (node_check_eol(n->buffer, n->pos, 1) == NULL) { // check for \r\n\r\n

			if (n->pos > 1000) { // we have read too many bytes, should be connected by now
				node_remove(n, "Failed (connect header too big)", 2);
				return;
				}

			r = read(s->file_desc, n->buffer + n->pos, 512); // get more

			if (!r) { node_remove(n, "Failed (no connect msg)", 2); return; }
			else if (r < 0 && errno == EAGAIN) return;
			else if (r < 0) { node_remove(n, "Failed (Disconnected hsk1)", 2); return; }

			n->pos += r;

			// old Gnutella client got to here, it could happen
			if (n->pos > 20 && n->pos < 25) if (g_strncasecmp(s->buffer, "GNUTELLA CONNECT/0.4", 20) == 0) {
				node_remove(n, "Failed (old 0.4 client)", 0);
				return;
				}

			// check this after the read too
			if (node_check_eol(n->buffer, n->pos, 1) == NULL) return; // still no \r\n\r\n in buffer yet
			}


		if (g_strncasecmp(n->buffer, "GNUTELLA CONNECT/", 17) != 0) { // mostly HTTP busy messages here
			node_remove(n, "Busy", 2); // 2 = try again later
			return;
			}


		offs=0;
		ultrapeer = FALSE;
		do { // move through each line
			do {
				offs++;
				} while (n->buffer[offs] != '\n' && offs < n->pos);
			if (offs >= n->pos) break; // go till end
			offs++;
			if (!g_strncasecmp(n->buffer+offs, "User-Agent:", 11)) {
				offs = offs + 12; // skip past the space
				i = 0;
				spaces = 2; // 1 if you want only the agent name displayed. 2 for version, etc....
				do {
					if (n->buffer[offs] == ' ') {
						if (--spaces <= 0) break;
						}
					n->gui_info[i] = n->buffer[offs];
					offs++;
					i++;
					} while (n->buffer[offs] != '\r' && offs < n->pos && i < 31);
				n->gui_info[i] = '\0';
				continue;
				}
			if (!g_strncasecmp(n->buffer+offs, "X-Ultrapeer:", 12)) {
				offs = offs + 13; // skip past the space
				if (!g_strncasecmp(n->buffer+offs, "true", 4)) ultrapeer = TRUE;
				continue;
				}
			} while(1);

		// show we have a ultrapeer
		if (strlen(n->gui_info) < 32 && ultrapeer) strcat(n->gui_info, " U");

		if (write(s->file_desc, gnutella_servant, strlen(gnutella_servant)) < 0) { // send welcome
			node_remove(n, "Write welcome failed", 2);
			return;
			}

		memset(n->buffer, 0, n->pos); // clear the space we used
		n->pos = 0; // start with new buffer again
		n->status = GTA_NODE_WELCOME_SENT;
		return; // go get more data
		}


	// we sent welcome msg, now wait for 200 OK and other headers
	if (n->status == GTA_NODE_WELCOME_SENT) {

			// we are looking for something like
			// GNUTELLA/0.6 200 OK<cr><lf>
			// <cr><lf>

		// read some more if no \r\n\r\n in buffer yet up to a limited size
		if ((endlines = node_check_eol(n->buffer, n->pos, 1)) == NULL) {

			if (n->pos > 512) { // we have read too many bytes, should be at end of all "OK" headers by now
				node_remove(n, "Failed (connect header too big)", 2);
				return;
				}

			// this time we only read in chunks that are gnutella header size
			// because there could be header info directly after the OK
			r = read(s->file_desc, n->buffer + n->pos, sizeof(struct gnutella_header));

			if (!r) {
				// they didn't respond with 200 OK for some reason!
				strcpy (tmpbuf, n->gui_info);
				strcat (tmpbuf, " Timeout, no OK msg");
				node_remove(n, tmpbuf, 2);
				return;
				}
			else if (r < 0 && errno == EAGAIN) return;
			else if (r < 0) {
						// Hey! We were dropped! WTF ???
						strcpy (tmpbuf, n->gui_info);
						strcat (tmpbuf, " Disconnected");
						node_remove(n, tmpbuf, 2);
						return;
						}

			n->pos += r;

			// check this after the read too
			if ((endlines = node_check_eol(n->buffer, n->pos, 1)) == NULL) return; // still no \r\n\r\n in buffer yet
			}

		if (g_strncasecmp(n->buffer, "GNUTELLA/", 9) != 0) { // strange message? HTTP busy message?
			node_remove(n, "Busy", 2); // 2 = try again later
			return;
			}

		offs = 9; // the following isn't really needed, we should always get 200 OK
		do {
			offs++; // move past the spaces
			} while (g_strncasecmp(n->buffer+offs, " ", 1) && offs < n->pos);
		offs++;
		strncpy(gnut_status_string, n->buffer+offs, 3); // get the status number
		gnut_status=atoi(gnut_status_string);
		if ((gnut_status > 299) || (gnut_status < 200)) { // 503 is typical "busy"
			if (gnut_status >= 500 && gnut_status <= 599) {
				strcpy (tmpbuf, n->gui_info);
				strcat (tmpbuf, " Busy");
				node_remove(n, tmpbuf, 2); // 2 = try again later
				}
			else {
				strcpy (tmpbuf, n->gui_info);
				strcat (tmpbuf, " Bad Gnutella status");
				node_remove(n, tmpbuf, 2); // 2 = try again later
				}
			return;
			}

		endlines += 4; // move past the last endline, then move it all over
		i = endlines - n->buffer;
		n->pos -= i;

		// restrict size just in case
		if (n->pos > sizeof(struct gnutella_header)) n->pos = sizeof(struct gnutella_header);

		// if we have a few header bytes left over, place them in the header struct properly
		if (n->pos) memmove(w, endlines, n->pos);

		if (n->pos == sizeof(struct gnutella_header)) n->have_header = TRUE; // we have it all

		n->status = GTA_NODE_CONNECTED;
		nodes_active++;

		gtk_widget_set_sensitive(button_host_catcher_get_more, TRUE);

		node_added = n;
		g_hook_list_invoke(&node_added_hook_list, TRUE);
		node_added = NULL;

	}


	if (!n->have_header)	/* We haven't got the header yet */
	{

			r = read(n->socket->file_desc, w + n->pos, sizeof(struct gnutella_header) - n->pos);

			if (!r) {
				strcpy (tmpbuf, n->gui_info);
				strcat (tmpbuf, " Disconnected or Busy");
				node_remove(n, tmpbuf, 2); // 2 = try again later
				return;
				}
			else if (r < 0 && errno == EAGAIN) return;
			else if (r < 0) {
						strcpy (tmpbuf, n->gui_info);
						strcat (tmpbuf, " Disconnected");
						node_remove(n, tmpbuf, 2); // 2 = try again later
						return;
						}

			n->pos += r;

			if (n->pos < sizeof(struct gnutella_header)) return;


		/* Okay, we have read the full header */

		n->have_header = TRUE;

		n->received++;
		global_messages++;

		gui_update_node(n, FALSE);

		READ_GUINT32_LE(n->header.size, n->size);

		n->pos = 0;

		/* If the message hasn't got any data, we process it now */

		if (!n->size) { node_parse(n); return; }

		/* is the message too big? */

		switch (n->header.function)
		{
			case GTA_MSG_SEARCH:
				if (n->size > search_queries_kick_size) kick = TRUE; break;

			case GTA_MSG_SEARCH_RESULTS:
				if (n->size > search_answers_kick_size) kick = TRUE; break;

			default:
				if (n->size > other_messages_kick_size) kick = TRUE;
				if (n->size > sizeof(n->buffer)) kick = TRUE; // the ultimate limit
				break;
		}

		if (kick) {
			strcpy (tmpbuf, n->gui_info);
			strcat (tmpbuf, " Kicked (message too big)");
			node_remove(n, tmpbuf, 1); // 1 = try again right away
			return;
			}

		/* Okay, we have the header, now go back and get the data */

		return;
	}

	/* Reading of the message data */

		r = read(n->socket->file_desc, n->buffer + n->pos, n->size - n->pos);

			if (!r) {
				strcpy (tmpbuf, n->gui_info);
				strcat (tmpbuf, " Timeout, no msg");
				node_remove(n, tmpbuf, 2); // 2 = try again later
				return;
				}
			else if (r < 0 && errno == EAGAIN) return;
			else if (r < 0) {
						strcpy (tmpbuf, n->gui_info);
						strcat (tmpbuf, " Disconnected");
						node_remove(n, tmpbuf, 2); // 2 = try again later
						return;
						}

		n->pos += r;

	if (n->pos >= n->size) node_parse(n);

}

/* Reads an outgoing connecting CONTROL node */

void node_read_connecting(gpointer data, gint source, GdkInputCondition cond)
{
	static struct gnutella_socket *s;
	static gint r;
	gchar gnut_status_string[4]="   ", tmpip[40], *tmp;
	gint gnut_status, i, spaces, cntip;
	guint32 offs;
	guint16 port;
	gboolean ultrapeer;

	s = (struct gnutella_socket *) data;

	if (cond & GDK_INPUT_EXCEPTION) { node_remove(s->resource.node, "Failed (Input Exception)", 2); return; }

	r = read(s->file_desc, s->buffer + s->pos, 1024 - s->pos);

	if (!r) { node_remove(s->resource.node, "Disconnected", 2); return; }
	else if (r < 0 && errno == EAGAIN) return;
	else if (r < 0) { node_remove(s->resource.node, "Disconnected", 2); return; }

	s->pos += r;
	s->buffer[s->pos] = '\0';

	if (s->pos > 11 && s->pos < 15) if (g_strncasecmp(s->buffer, "GNUTELLA OK", 11) == 0) { // old Gnutella client
		node_remove(s->resource.node, "Failed (old 0.4 client)", 0);
		return;
		}

	if (s->pos > 1000) { // we have read too many bytes, should be connected by now
		node_remove(s->resource.node, "Failed (connect header too big)", 2); // 2 = try again later
		return;
		}

	if (node_check_eol(s->buffer, s->pos, 1) == NULL) return; // no \r\n\r\n, go read some more

	if (g_strncasecmp(s->buffer, "GNUTELLA/", 9) != 0) { // mostly HTTP busy messages here
		node_remove(s->resource.node, "Busy", 2); // 2 = try again later
		return;
		}

	offs=0;
	do {
		offs++; // move past the spaces
		} while (g_strncasecmp(s->buffer+offs, " ", 1) && offs < s->pos);
	offs++;
	strncpy(gnut_status_string, s->buffer+offs, 3); // get the status number
	gnut_status=atoi(gnut_status_string);
	if ((gnut_status > 299) || (gnut_status < 200)) { // 503 is typical "busy"
		if (gnut_status >= 500 && gnut_status <= 599) node_remove(s->resource.node, "Busy", 2); // 2 = try again later
		else node_remove(s->resource.node, "Bad Gnutella status", 2); // 2 = try again later
		return;
		}

		offs=0;
		ultrapeer = FALSE;
		do { // move through each line
			do {
				offs++;
				} while (s->buffer[offs] != '\n' && offs < s->pos);
			if (offs >= s->pos) break; // go till end
			offs++;
			if (!g_strncasecmp(s->buffer+offs, "User-Agent:", 11)) {
				offs = offs + 12; // skip past the space
				i = 0;
				spaces = 2; // 1 if you want only the agent name displayed. 2 for version, etc....
				do {
					if (s->buffer[offs] == ' ') {
						if (--spaces <= 0) break;
						}
					s->resource.node->gui_info[i] = s->buffer[offs];
					offs++;
					i++;
					} while (s->buffer[offs] != '\r' && offs < s->pos && i < 31);
				s->resource.node->gui_info[i] = '\0';
				continue;
				}
			if (!g_strncasecmp(s->buffer+offs, "X-Ultrapeer:", 12)) {
				offs = offs + 13; // skip past the space
				if (!g_strncasecmp(s->buffer+offs, "true", 4)) ultrapeer = TRUE;
				continue;
				}

			// X-Try: and X-Try-Ultrapeers: happen on outgoing connect, and mostly LimeWire for now
			if (!g_strncasecmp(s->buffer+offs, "X-Try:", 6)) {
				offs = offs + 7; // skip past the space
				cntip = 0;
				while (s->buffer[offs] != '\r' && offs < s->pos && cntip <= 15) {
					i = 0;
					while (s->buffer[offs] != '\r' && s->buffer[offs] != ',' && offs < s->pos && i < 31) {
						tmpip[i] = s->buffer[offs];
						offs++;
						i++;
						};
					if (offs >= s->pos) break; // not sure what we have here, skip it
					tmpip[i] = '\0';
					tmp = tmpip;
					while (*tmp && *tmp != ':') tmp++;
					port = (*tmp)? atoi(tmp + 1) : 6346;
					*tmp++ = 0;
					if (strlen(tmpip) > 7) host_add(NULL, gchar_to_ip(tmpip), port, 0); // add to top of list
					if (s->buffer[offs] == '\r') break;
					offs++;
					cntip++;
					};
				continue;
				}

			if (!g_strncasecmp(s->buffer+offs, "X-Try-Ultrapeers:", 17)) { // heck, why not?
				offs = offs + 18; // skip past the space
				cntip = 0;
				while (s->buffer[offs] != '\r' && offs < s->pos && cntip <= 15) {
					i = 0;
					while (s->buffer[offs] != '\r' && s->buffer[offs] != ',' && offs < s->pos && i < 31) {
						tmpip[i] = s->buffer[offs];
						offs++;
						i++;
						};
					if (offs >= s->pos) break; // not sure what we have here, skip it
					tmpip[i] = '\0';
					tmp = tmpip;
					while (*tmp && *tmp != ':') tmp++;
					port = (*tmp)? atoi(tmp + 1) : 6346;
					*tmp++ = 0;
					if (strlen(tmpip) > 7) host_add(NULL, gchar_to_ip(tmpip), port, 0); // add to top of list
					if (s->buffer[offs] == '\r') break;
					offs++;
					cntip++;
					};
				continue;
				}
			if (!g_strncasecmp(s->buffer+offs, "Remote-IP:", 10)) { // pick up our reflected IP
				offs = offs + 11; // skip past the space
				i = 0;
				while (s->buffer[offs] != '\r' && offs < s->pos && i < 17) {
					tmpip[i] = s->buffer[offs];
					offs++;
					i++;
					};
				tmpip[i] = '\0';
				gtk_label_set(GTK_LABEL(label_reflected_ip), tmpip); // show it every time
				continue;
				}
			} while(1);


	if (strlen(s->resource.node->gui_info) < 32 && ultrapeer) strcat(s->resource.node->gui_info, " U");

	// connected, now send welcome message

	if (write(s->file_desc, gnutella_welcome, strlen(gnutella_welcome)) < 0) {
		node_remove(s->resource.node, "Write failed (sending welcome msg)", 2); // 2 = try again later
		return;
		}

	/* Okay, we are now really connected to a gnutella node */

	gdk_input_remove(s->gdk_tag);

	s->resource.node->have_header = FALSE;
	s->resource.node->pos = 0;
	s->pos = 0;

	s->gdk_tag = gdk_input_add(s->file_desc, GDK_INPUT_READ | GDK_INPUT_EXCEPTION, node_read, (gpointer) s->resource.node);

	/* We assume that if this is valid, it is non-zero */
	assert(s->gdk_tag);

	s->resource.node->status = GTA_NODE_CONNECTED;
	nodes_active++;

	gui_update_node(s->resource.node, TRUE);

	gtk_widget_set_sensitive(button_host_catcher_get_more, TRUE);

	send_init(s->resource.node);

	node_added = s->resource.node;
	g_hook_list_invoke(&node_added_hook_list, TRUE);
	node_added = NULL;
}



/* vi: set ts=3: */

