
#include "gnutella.h"

#include <fcntl.h>
#include <sys/types.h>
#include <string.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include "interface.h"

GSList *sl_catched_hosts = (GSList *) NULL;
GSList *sl_catched_urls = (GSList *) NULL;

GSList *ping_reqs = (GSList *) NULL;
guint32 n_ping_reqs = 0;
struct ping_req *pr_ref = (struct ping_req *) NULL;

gchar h_tmp[4096];
gchar *urlcache[WEBCACHEURLS]; // global, we use it in main.c also
gchar url1[2048],url2[2048];

gint hosts_idle_func = 0;
gint hosts_full_countdown = 100;

static FILE *hosts_r_file = (FILE *) NULL;

static struct gnutella_socket *cachesocket = (struct gnutella_socket *) NULL;	/* Socket for urlcache read */


#define MAX_PING_REQS 64	/* How many ping requests do we have to remember */

/* Hosts ------------------------------------------------------------------------------------------ */

gboolean find_host(guint32 ip, guint16 port)
{
	GSList *l = (GSList *) NULL;

	/* Check our local ip */

	if (ip == local_ip || (force_local_ip && ip == forced_local_ip)) return TRUE;

	/* Check the nodes */

	for (l = sl_nodes; l; l = l->next) {
		if (l->data) {
			if (((struct gnutella_node *) l->data)->ip == ip) {
				if (((struct gnutella_node *) l->data)->socket) {
					if (((struct gnutella_node *) l->data)->socket->direction == GTA_CONNECTION_INCOMING) return TRUE;
					else if (((struct gnutella_node *) l->data)->port == port) return TRUE;
					}
				}
			}
		}

	/* Check the hosts */

	for (l = sl_catched_hosts; l; l = l->next) {
		if (l->data) {
			if (((struct gnutella_host *) l->data)->ip == ip
				&& ((struct gnutella_host *) l->data)->port == port) return TRUE;
			}
		}

	return FALSE;
}

void host_remove(struct gnutella_host *h, gboolean from_clist_too)
{
	gint row;

	if (from_clist_too)
	{
		row = gtk_clist_find_row_from_data(GTK_CLIST(clist_host_catcher), (gpointer) h);
		gtk_clist_remove(GTK_CLIST(clist_host_catcher), row);
	}

	sl_catched_hosts = g_slist_remove(sl_catched_hosts, h);

	if (!sl_catched_hosts) gtk_widget_set_sensitive(button_host_catcher_clear, FALSE);

	g_free(h);
}

gboolean check_valid_host(guint32 ip, guint16 port)
{
	if (!ip || !port) return FALSE;									/* IP == 0 || Port == 0					*/

	if (ip == (guint32) 0x01020304 || ip == (guint32) 0x01010101) return FALSE;	/* IP == 1.2.3.4 || IP == 1.1.1.1 	*/
	if ((ip & (guint32) 0xFF000000) == (guint32) 0x00000000) return FALSE;			/* IP == 0.0.0.0 / 8					 	*/
	if ((ip & (guint32) 0xFF000000) == (guint32) 0x7F000000) return FALSE;			/* IP == 127.0.0.0 / 8					*/
	if ((ip & (guint32) 0xFF000000) == (guint32) 0x0A000000) return FALSE;			/* IP == 10.0.0.0 / 8					*/
	if ((ip & (guint32) 0xFFF00000) == (guint32) 0xAC100000) return FALSE;			/* IP == 172.16.0.0 / 12				*/
	if ((ip & (guint32) 0xFFFF0000) == (guint32) 0xC0A80000) return FALSE;			/* IP == 192.168.0.0 / 16				*/
	if (ip  == (guint32) 0xFFFFFFFF) return FALSE;											/* IP == 255.255.255.255				*/

	return TRUE;
}


// add a host to the cache list, comes from a pong response or our file, n tells you which one
// set last if you want it to be put as last on list

void host_add(struct gnutella_node *n, guint32 t_ip, guint16 t_port, gboolean last)
{
	struct gnutella_host *host;
	gchar *titles[2];
	gint row;
	guint32 ip;
	guint16 port;

	if (n) { // read it from the returned gnutella packet
		READ_GUINT32_BE(n->buffer + 2, ip);
		READ_GUINT16_LE(n->buffer, port);
		}
	else {
		ip = t_ip;
		port = t_port;
		}

	if (!check_valid_host(ip, port)) return;	/* Check wether the host is a valid one */

	if (find_host(ip, port)) return;			 /* Check if we don't already have this host */

	/* Okay, we got a new host, so create a cache list record for it & add to list */

	host = (struct gnutella_host *) g_malloc0(sizeof(struct gnutella_host));

	host->port         = port;
	host->ip           = ip;
	host->files_count  = 0;
	host->kbytes_count = 0;

	if (n) { // save this info from the gnutella packet
		READ_GUINT32_LE(n->buffer + 6,  host->files_count);
		READ_GUINT32_LE(n->buffer + 10, host->kbytes_count);
		}

	titles[0] = ip_port_to_gchar(ip, port);

	row = gtk_clist_append(GTK_CLIST(clist_host_catcher), titles);
	gtk_clist_set_row_data(GTK_CLIST(clist_host_catcher), row, (gpointer) host);

	if (!sl_catched_hosts) gtk_widget_set_sensitive(button_host_catcher_clear, TRUE);

	if (--hosts_full_countdown == 0) { // add 100 then check to see if we have too many
		hosts_full_countdown = 100; // keeps us from updating the gui too often
		while(g_slist_length(sl_catched_hosts) > 3000) {
			host_remove(g_slist_last(sl_catched_hosts)->data, TRUE); // remove the oldest entries
			}
		}

	if (last) sl_catched_hosts = g_slist_append(sl_catched_hosts, host); // adds to bottom (last)
	else sl_catched_hosts = g_slist_prepend(sl_catched_hosts, host); // adds to top of list
}


/* Hosts text files ------------------------------------------------------------------------------- */

gint hosts_reading_func(gpointer data)
{
	gchar *s, *e;
	guint16 port;
	gint i;

	// each entry is about 21 bytes, file system reads a block into ram so it
	// won't take too long to read a bunch all at once, but not too many
	for (i = 0; i < 100; i++) { // 100 is about 2K
		if ((e = fgets(h_tmp, sizeof(h_tmp), hosts_r_file))) {
			s = h_tmp;
			while (*s && *s != ':') s++;
			port = (*s)? atoi(s + 1) : 6346;
			*s++ = 0;
			// add to last part of list, newest entries were first in file
			host_add(NULL, gchar_to_ip(h_tmp), port, 1); // 1 = add to last
			}
		else break;
		}

	if (e) return TRUE; // end of file? (NULL = EOF)

	fclose(hosts_r_file);

	hosts_r_file = (FILE *) NULL;

	hosts_idle_func = 0;

	gtk_clist_thaw(GTK_CLIST(clist_host_catcher));

	gui_set_status(NULL);

	return FALSE;
}

void hosts_read_from_file(gchar *path, gboolean quiet)
{
	/* Loads 'catched' hosts from a text file */

	hosts_r_file = fopen(path, "r");

	if (!hosts_r_file)
	{
		if (!quiet) g_warning("Unable to open file %s (%s)\n", path, g_strerror(errno));
		return;
	}

	gtk_clist_freeze(GTK_CLIST(clist_host_catcher));

	hosts_idle_func = gtk_idle_add(hosts_reading_func, (gpointer) NULL);

	gui_set_status("Reading catched hosts file...");
}

void hosts_write_to_file(gchar *path)
{
	/* Saves the currently catched hosts to a file, called at exit */

	FILE *f;
	GSList *l = (GSList *) NULL;
	struct gnutella_socket *s;

	f = fopen(path, "w");

	if (!f)
	{
		g_warning("Unable to open output file %s (%s)\n", path, g_strerror(errno));
		return;
	}

	for (l = sl_nodes; l; l = l->next) {
		if (l->data) {
			if (((struct gnutella_node *) l->data)->status == GTA_NODE_CONNECTED) { // save connected nodes
				s = ((struct gnutella_node *) l->data)->socket;
				if (s) if (s->direction != GTA_CONNECTION_INCOMING) { // don't save incoming
					fprintf(f, "%s\n", ip_port_to_gchar(((struct gnutella_node *) l->data)->ip,
							((struct gnutella_node *) l->data)->port));
					}
				}
			}
		}

	for (l = sl_catched_hosts; l; l = l->next)
		fprintf(f, "%s\n", ip_port_to_gchar(((struct gnutella_host *) l->data)->ip, ((struct gnutella_host *) l->data)->port));

	fclose(f);
}



/* Remote Host Cache URL access ------------------------------------------------------------------------------- */


// read in the user supplied list of external host caches
void hosts_urlcache_list_read(gchar *path)
{
	/* Loads 'Catche URL List' hosts from a text file */

	FILE *cachef;
	gchar *s;
	guint32 skip, n = 0;
	urlcache[0] = NULL; // clear the first pointer

	if ((cachef = fopen("NAPS-urlcache.txt", "r")) == NULL) { // try to open file in local directory
		if (is_directory(path)) {
			g_snprintf(h_tmp, sizeof(h_tmp), "%s/NAPS-urlcache.txt", path);
			cachef = fopen(h_tmp, "r"); // open the normal file
			}
		}

	if (!cachef) {
		g_warning("Unable to open 'NAPS-urlcache.txt' in '%s' NO CACHE URLS AVAILABLE! (%s)\n", path, g_strerror(errno));
		return;
		}

	while (fgets(h_tmp, sizeof(h_tmp), cachef))
	{
		if (strlen(h_tmp) < 10) continue; // something's wrong with this URL, not enough chars
		if (h_tmp[0] == '#') continue; // comment line
		s = h_tmp;
		skip = 0;
		while (*s) {
			if (*s == ' ') skip = 1; // skip any URLs with spaces in them
			if (*s == '\n') {
					*s = 0; // get rid of end character
					break;
					}
				s++;
			}
		if (skip) continue;
		urlcache[n] = g_strdup(h_tmp); // allocate memory and move the string into it
		n++;
		urlcache[n] = NULL; // mark the end
		if (n >= (WEBCACHEURLS - 1)) break; // we only hold up to WEBCACHEURLS of these
		}

	fclose(cachef); // main.c will access the urlcache strings next

}



// try to open a socket to a web server, port 80
// we only access one http site at a time, if the last one was slow we stop it now
void hosts_urlcache_open_socket(gchar *url)
{

	guint32 ip = 0;

	if (cachesocket) socket_destroy(cachesocket); // close any open connections

	url2[0] = '/';
	url2[1] = 0;
	// extract the server name and calling line
	if(!sscanf(url, "http://%2000[^/]%2000s", url1, url2)) return;

	ip = host_to_ip(url1, FALSE); // try DNS lookup, don't report errors
	if (ip == 0) return; // no error if not found, just try next in list

	cachesocket = socket_connect(ip, 80, GTA_TYPE_HTTP);

	// that's all we need to do, once connected hosts_send_request() will be called

}


// called by socket_destroy
void hosts_http_close(struct gnutella_socket *s)
{

	g_return_if_fail(s);

	if (s->gdk_tag)   gdk_input_remove(s->gdk_tag);
	if (s->file_desc != -1) close(s->file_desc);

	g_free(s);
	s = NULL;
	cachesocket = (struct gnutella_socket *) NULL;

}


// called after a connection is established
// the URL calling line is designed for the GWebCache system, we should
// receive a list of newline separated ip:port numbers
void hosts_send_request(struct gnutella_socket *s)
{

	gchar hosts_tmp1[4096], *tmp2;

	g_return_if_fail(s);

	tmp2 = ip_port_to_gchar((force_local_ip)? forced_local_ip : local_ip, listen_port); // ip:port in str
	g_snprintf(hosts_tmp1, sizeof(hosts_tmp1),
		"GET %s?client=%s&version=%s&ip=%s&hostfile=1 HTTP/1.0\r\nConnection: Keep-Alive\r\nHost: %s\r\nUser-Agent: %s\r\n\r\n",
		url2, vendor_code, vendor_version, tmp2, url1, name_version);

	if (write(s->file_desc, hosts_tmp1, strlen(hosts_tmp1)) < 0)
	{
		socket_destroy(s); // bye bye
		return;
	}

}


// attempt to read ip:port lines from a list off the web, page is in buffer
// called once we have entire page or buffer is full, socket closed after this
// is called from socket_get_http()
void hosts_parse_http(struct gnutella_socket *s)
{
	gchar *h = s->buffer;
	gchar line[45], ip1[5], ip2[5], ip3[5], ip4[5], port[15];
	guint32 items, count1;
	guint16 porti;

	g_return_if_fail(s);

	count1 = 0;
	while(1) {
		items = sscanf(h, "%40[^\n\r]", line);
			if (items == 1) {
				items = sscanf(line, "%3[^.].%3[^.].%3[^.].%3[^:]:%12[^\n\r]", ip1, ip2, ip3, ip4, port);
				if (items == 5) { // we have a valid line here, must have 5 items
					g_snprintf(line, sizeof(line), "%s.%s.%s.%s",ip1, ip2, ip3, ip4);
					porti= atoi(port);
					if (porti > 0 ) { // don't accept port 0
						host_add(NULL, gchar_to_ip(line), porti, 1); // 1 = add to last
						if (count1++ > 20) break; // only allow 20 from a host cache
						}
					}
				}
		while (*h) { // move to next line
			if ((*h == 0x0D) || (*h == 0x0A)) break;
			h++;
			}
		if (*h) { // it could be 00
			h++; // get past end of line
			if ((*h == 0x0D) || (*h == 0x0A)) h++; // go past any other end of line chars
			}
		else break;
		}

}


/* gnutellaNet stats ------------------------------------------------------------------------------ */

/* Registers a new ping request */

void register_ping_req(guchar *muid)
{
	struct ping_req *p;

	if (n_ping_reqs >= MAX_PING_REQS)
	{
		GSList *l = g_slist_last(ping_reqs);
		p = (struct ping_req *) l->data;
		ping_reqs = g_slist_remove_link(ping_reqs, l);
		g_slist_free_1(l);
	}
	else
	{
		p = (struct ping_req *) g_malloc(sizeof(struct ping_req));
	}

	memcpy(p->muid, muid, 16);

	gettimeofday(&(p->tv), (struct timezone *) NULL);

	p->delay = 0;
	p->hosts = 0;
	p->files = 0;
	p->kbytes = 0;

	ping_reqs = g_slist_prepend(ping_reqs, p);
}

/* Adds a reply to the stats, called from routing.c route_message(...) n is the sending node */

void ping_stats_add(struct gnutella_node *n)
{
	GSList *l = (GSList *) NULL;
	struct gnutella_init_response *r;
	struct ping_req *p;
	struct timeval tv;
	guint32 v;

	/* First look for a matching req in the ping reqs list */

	for (l = ping_reqs; l; l = l->next)
		if (!memcmp(((struct ping_req *) l->data)->muid, n->header.muid, 16)) break;

	if (!l) return;	/* Found no request for this reply */

	r = (struct gnutella_init_response *) n->buffer;
	p = (struct ping_req *) l->data;

	p->hosts++;

	READ_GUINT32_LE(r->files_count, v); p->files += v;
	READ_GUINT32_LE(r->kbytes_count, v); p->kbytes += v;

	gettimeofday(&tv, (struct timezone *) NULL);

	p->delay = (tv.tv_sec - p->tv.tv_sec) * 1000 + (tv.tv_usec / 1000 - p->tv.tv_usec / 1000);
	if (!pr_ref || (p->hosts > pr_ref->hosts)) pr_ref = p;
}

/* Update the stats when user presses the button to do so */

void ping_stats_update(void)
{
	GSList *l = ping_reqs;

	while (l) { g_free(l->data); l = l->next; }

	g_slist_free(ping_reqs);

	ping_reqs = NULL;
	n_ping_reqs = 0;

	pr_ref = NULL;

	gui_update_stats();

	send_init(NULL);
}

/* Messages --------------------------------------------------------------------------------------- */

/* Sends an init request */

void send_init(struct gnutella_node *n)
{
	static struct gnutella_msg_init m;

	message_set_muid(&(m.header));

	m.header.function = GTA_MSG_INIT;
	m.header.ttl      = my_ttl;
	m.header.hops     = 0;

	WRITE_GUINT32_LE(0, m.header.size);

	message_add(m.header.muid, GTA_MSG_INIT, NULL);

	if (n) sendto_one(n, (guchar *) &m, NULL, sizeof(struct gnutella_msg_init), TRUE);
	else   sendto_all((guchar *) &m, NULL, sizeof(struct gnutella_msg_init), TRUE);

	register_ping_req(m.header.muid);
}

/* Replies to an init request */

void reply_init(struct gnutella_node *n)
{	
	static struct gnutella_msg_init_response r;

	if (!force_local_ip && !local_ip) return; /* If we don't know yet your local IP, we can't reply */

	WRITE_GUINT16_LE(listen_port,   r.response.host_port);
	WRITE_GUINT32_BE((force_local_ip)? forced_local_ip : local_ip, r.response.host_ip);
	WRITE_GUINT32_LE(files_scanned, r.response.files_count);
	WRITE_GUINT32_LE(kbytes_scanned, r.response.kbytes_count);

	r.header.function = GTA_MSG_INIT_RESPONSE;
	r.header.ttl      = my_ttl;
	r.header.hops     = 0;

	memcpy(&r.header.muid, n->header.muid, 16);

	WRITE_GUINT32_LE(sizeof(struct gnutella_init_response), r.header.size);

	sendto_one(n, (guchar *) &r, NULL, sizeof(struct gnutella_msg_init_response), TRUE);
}

/* vi: set ts=3: */

