
/* napshare configuration */

#include "gnutella.h"
#include "interface.h"
#include "search.h"


#include <sys/stat.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <netdb.h>

gboolean clear_uploads					= FALSE;
gboolean clear_downloads				= FALSE;
gboolean monitor_enabled				= FALSE;
gboolean force_local_ip					= FALSE;

guint8 max_ttl								= 7;
guint8 my_ttl								= 5;

guint16 listen_port	     				= 6346;

guint32 up_connections					= 3;
guint32 max_connections             = 4;
guint32 max_downloads					= 4;
guint32 max_host_downloads				= 1;
guint32 max_uploads						= 4;
guint32 minimum_speed					= 0;
guint32 monitor_max_items				= 25;
guint32 connection_speed				= 56;
gint32 search_max_items					= 64; /* For now, this is limited to 255 anyway */
guint32 forced_local_ip					= 0;
guint32 download_connecting_timeout	= 30;
guint32 download_push_sent_timeout	= 60;
guint32 download_connected_timeout	= 60;
guint32 download_retry_timeout_min	= 20;
guint32 download_retry_timeout_max	= 90;
guint32 download_max_retries			= 5;
guint32 node_connected_timeout		= 25;
guint32 node_connecting_timeout		= 7;
guint32 node_sendqueue_size			= 131072; /* for holding non-priority messages to other slow nodes */
guint32 node_sendqueue_max_size		= 1048576; /* buffer absolute max to disconnect node */
guint32 search_queries_forward_size	= 256;
guint32 search_queries_kick_size		= 5120;
guint32 search_answers_forward_size	= 32768;
guint32 search_answers_kick_size		= 40960;
guint32 other_messages_kick_size		= 40960;
guint32 hops_random_factor				= 0;
gint dbg										= 0; // debug level, for development use
gint stop_host_get						= 0; // stop get new hosts, non activity ok, for debug use
gint enable_err_log						= 0; // enable writing to a log file for Gnutella errors & further improvement
gint search_strict_and					= 0; // enable search filter for strict AND of results
gboolean search_select_all				= FALSE; // enable selecting all like files in search
gboolean search_select_name			= FALSE; // enable select them by name, else by exact size
gboolean jump_to_downloads				= TRUE; // on search screen
gboolean throttle_uploads				= FALSE;
gint max_uploads_ip						= 2; // maximum uploads per IP

time_t tab_update_time = 5;

gchar *scan_extensions					= NULL;
gchar *save_file_path					= NULL;
gchar *move_file_path					= NULL;
gchar *shared_dirs_paths				= NULL;
gchar *completed_file_path				= NULL;
gchar *home_dir		  					= NULL;
gchar *config_dir							= NULL;
gchar *vendor_code						= NULL;
gchar *vendor_name						= NULL;
gchar *vendor_version					= NULL;
gchar *name_version						= NULL;
gchar *gnutella_hello					= NULL;
gchar *gnutella_servant					= NULL;
gchar *gnutella_welcome					= NULL;
gchar *protocol_name						= NULL;
gchar *lan_name							= NULL;

guint32 nodes_col_widths[]          = { 140, 80, 80 };
guint32 uploads_col_widths[]        = { 200, 140, 80 };
guint32 dl_active_col_widths[]      = { 240, 80, 80 };
guint32 dl_queued_col_widths[]      = { 320, 80 };
guint32 search_results_col_widths[] = { 210, 80, 50, 140, 140 };
guint32 auto_col_widths[]           = { 138, 168, 118, 50 };


gint w_x = 0, w_y = 0, w_w = 0, w_h = 0;

guint32 search_reissue_timeout = 600; /* 10 minutes */

gboolean proxy_connections = FALSE;
gint socks_protocol = 4;
gchar *proxy_ip = "0.0.0.0";
gint proxy_port = 1080;

static guint32 gnutella_welcome_length = 0;

gchar *socksv5_user = "proxyuser";
gchar *socksv5_pass = "proxypass";

enum
{
	k_up_connections = 0, k_clear_uploads, k_max_downloads, k_max_host_downloads, k_max_uploads, k_clear_downloads, 
	k_minimum_speed, k_monitor_enabled, k_monitor_max_items, k_old_save_file_path, k_scan_extensions, 
	k_listen_port, k_max_ttl, k_my_ttl, k_shared_dirs, k_forced_local_ip, k_connection_speed, 
	k_search_max_items, k_force_local_ip, k_hosts_catched, k_download_connecting_timeout,
	k_download_push_sent_timeout, k_download_connected_timeout, k_node_connected_timeout,
	k_node_connecting_timeout, k_node_sendqueue_size, k_node_sendqueue_max_size, k_search_queries_forward_size,
	k_search_queries_kick_size, k_search_answers_forward_size, k_search_answers_kick_size,
	k_other_messages_kick_size, k_save_file_path, k_move_file_path,
	k_win_x, k_win_y, k_win_w, k_win_h, k_win_coords, k_widths_nodes, k_widths_uploads,
	k_widths_dl_active, k_widths_dl_queued, k_widths_search_results, k_show_results_tabs,
	k_hops_random_factor, k_send_pushes, k_jump_to_downloads, k_max_connections,  k_proxy_connections,
	k_socks_protocol, k_proxy_ip, k_proxy_port, k_socksv5_user, k_socksv5_pass, k_search_reissue_timeout,
	k_dbg, k_stop_host_get, k_enable_err_log, k_max_uploads_ip, k_search_strict_and, k_search_select_all,
	k_search_select_name, k_throttle_uploads, k_widths_auto, k_vendor_code, k_vendor_name, k_vendor_version,
	k_protocol_name, k_lan_name, k_end
};


gchar *keywords[] = 
{
	"up_connections",							/* k_up_connections					*/
	"auto_clear_completed_uploads",		/* k_clear_uploads					*/
	"max_simultaneous_downloads",			/* k_max_downloads					*/
	"max_simultaneous_host_downloads",	/* k_max_host_downloads				*/
	"max_simultaneous_uploads",			/* k_max_uploads					*/
	"auto_clear_completed_downloads",	/* k_clear_downloads					*/
	"search_minimum_speed",					/* k_minimum_speed					*/
	"monitor_enabled",						/* k_monitor_enabled					*/
	"monitor_max_items",						/* k_monitor_max_items				*/
	"save_downloaded_files_to",			/* k_old_save_file_path				*/
	"shared_files_extensions",				/* k_scan_extensions					*/
	"listen_port",								/* k_listen_port						*/
	"max_ttl",									/* k_max_ttl							*/
	"my_ttl",									/* k_my_ttl								*/
	"shared_dirs",								/* k_shared_dirs						*/
	"forced_local_ip",						/* k_forced_local_ip					*/
	"connection_speed",						/* k_connection_speed				*/
	"limit_search_results",					/* k_search_max_items				*/
	"force_local_ip",							/* k_force_local_ip					*/
	"hc",											/* k_hosts_catched					*/
	"download_connecting_timeout",		/* k_download_connecting_timeout	*/
	"download_push_sent_timeout",			/* k_download_push_sent_timeout	*/
	"download_connected_timeout",			/* k_download_connected_timeout	*/
	"node_connected_timeout",				/* k_node_connected_timeout		*/
	"node_connecting_timeout",				/* k_node_connecting_timeout		*/
	"node_sendqueue_size",					/* k_node_sendqueue_size			*/
	"node_sendqueue_max_size",				/* k_node_sendqueue_max_size			*/
	"search_queries_forward_size",		/* k_search_queries_forward_size	*/
	"search_queries_kick_size",			/* k_search_queries_kick_size		*/
	"search_answers_forward_size",		/* k_search_answers_forward_size	*/
	"search_answers_kick_size",			/* k_search_answers_kick_size		*/
	"other_messages_kick_size",			/* k_other_messages_kick_size		*/
	"store_downloading_files_to",			/* k_save_file_path					*/
	"move_downloaded_files_to",			/* k_move_file_path					*/
	"window_x",									/* k_win_x								*/
	"window_y",									/* k_win_y								*/
	"window_w",									/* k_win_w								*/
	"window_h",									/* k_win_h								*/
	"window_coords",							/* k_win_coords						*/
	"widths_nodes",							/* k_width_nodes						*/
	"widths_uploads",							/* k_width_uploads					*/
	"widths_dl_active",						/* k_width_dl_active					*/
	"widths_dl_queued",						/* k_width_dl_queued					*/
	"widths_search_results",				/* k_width_search_results			*/
	"show_results_tabs",						/* k_show_results_tabs				*/
	"hops_random_factor",					/* k_hops_random_factor 			*/
	"send_pushes",								/* k_send_pushes						*/
	"jump_to_downloads",						/* k_jump_to_downloads				*/
	"max_connections",
	"proxy_connections",
	"socks_protocol",
	"proxy_ip",
	"proxy_port",
	"socksv5_user",
	"socksv5_pass",
	"search_reissue_timeout",
	"dbg",
	"stop_host_get",
	"enable_err_log",
	"max_uploads_ip",
	"search_strict_and",
	"search_select_all",
	"search_select_name",
	"throttle_uploads",
	"widths_auto",								/* k_widths_auto						*/
	"vendor_code",								/* k_vendor_code						*/
	"vendor_name",								/* k_vendor_name						*/
	"vendor_version",							/* k_vendor_version					*/
	"protocol_name",							/* k_protocol_name					*/
	"lan_name",									/* k_lan_name							*/
	NULL
};

gchar cfg_tmp[4096];

guint cfg_use_local_file = 0; // true if we are using config file in same directory

void config_read(void);

/* ------------------------------------------------------------------------------------------------ */

void config_init(void)
{
	gint i;
	struct passwd *pwd = NULL;
	gchar tmp[80];

	config_dir = g_strdup(getenv("NAPSHARE_DIR"));

	pwd = getpwuid(getuid());

	if (pwd && pwd->pw_dir) home_dir = g_strdup(pwd->pw_dir);
	else home_dir = g_strdup(getenv("HOME"));

	if (!home_dir) fprintf(stderr, "\nWARNING - Can't find your home directory !\n\n");

	if (config_dir && !is_directory(config_dir))
	{
		fprintf(stderr, "\nWARNING - '%s' does not exist or is not a directory !\n\n", config_dir);
		g_free(config_dir);
		config_dir = NULL;
	}

	if (!config_dir)
	{
		if (home_dir)
		{
			g_snprintf(cfg_tmp, sizeof(cfg_tmp), "%s/.napshare", home_dir);
			config_dir = g_strdup(cfg_tmp);
		}
		else fprintf(stderr, "\nWARNING - No configuration directory: Prefs will not be saved !\n\n");
	}

	if (config_dir)
	{
		/* Parse the configuration */

		config_read();

		/* Loads the catched hosts */
		if (cfg_use_local_file) hosts_read_from_file("NAPS-hosts", TRUE);
		else {
			g_snprintf(cfg_tmp, sizeof(cfg_tmp), "%s/NAPS-hosts", config_dir);
			hosts_read_from_file(cfg_tmp, TRUE);
			}

		/* Loads the automation settings */
		if (cfg_use_local_file) {
			gtk_label_set(GTK_LABEL(auto_label_left), "Auto Config File: ./NAPS-auto.txt"); // show user where the file is
			auto_read_from_file("NAPS-auto.txt", TRUE);
			}
		else {
			g_snprintf(tmp, 75, "%s/NAPS-auto.txt", config_dir); // limit to 75 chars
			g_snprintf(cfg_tmp, sizeof(cfg_tmp), "Auto Config File: %s", tmp);
			gtk_label_set(GTK_LABEL(auto_label_left), cfg_tmp); // show user where the file is
			g_snprintf(cfg_tmp, sizeof(cfg_tmp), "%s/NAPS-auto.txt", config_dir);
			auto_read_from_file(cfg_tmp, TRUE);
			}

		hosts_urlcache_list_read(config_dir); // go get the url list of host caches

	}

	if (!save_file_path || !is_directory(save_file_path))
		save_file_path = (home_dir)? g_strdup(home_dir) : g_strdup("/tmp");

	if (!move_file_path || !is_directory(move_file_path))
		move_file_path = g_strdup(save_file_path);

	if (!forced_local_ip) force_local_ip = FALSE;

	if (!shared_dirs_paths) shared_dirs_paths = g_strdup("");

	if (!extensions)
		parse_extensions("mp3;mp2;mp1;vqf;avi;mpg;mpeg;wav;mod;voc;it;xm;s3m;stm;wma;mov;asf;zip;rar;txt;jpg;pdf");

	// now we construct the connect strings
	if (!vendor_code) vendor_code = g_strdup("NAPS");
	if (!vendor_name) vendor_name = g_strdup("NapShare");
	if (!vendor_version) vendor_version = g_strdup(VERSION);
	if (!protocol_name) protocol_name = g_strdup("GNUTELLA");
	if (!lan_name) lan_name = g_strdup("");

	g_snprintf(cfg_tmp, sizeof(cfg_tmp), "%s %s", vendor_name, vendor_version);
	name_version = g_strdup(cfg_tmp);
	if (strlen(lan_name) > 0) {
		g_snprintf(cfg_tmp, sizeof(cfg_tmp),"%s CONNECT/0.6\r\nUser-Agent: %s\r\nLAN: %s\r\n\r\n", protocol_name, name_version, lan_name);
		gnutella_hello = g_strdup(cfg_tmp);
		g_snprintf(cfg_tmp, sizeof(cfg_tmp),"%s/0.6 200 OK\r\nUser-Agent: %s\r\nLAN: %s\r\n\r\n", protocol_name, name_version, lan_name);
		gnutella_servant = g_strdup(cfg_tmp);
		}
	else {
		g_snprintf(cfg_tmp, sizeof(cfg_tmp),"%s CONNECT/0.6\r\nUser-Agent: %s\r\n\r\n", protocol_name, name_version);
		gnutella_hello = g_strdup(cfg_tmp);
		g_snprintf(cfg_tmp, sizeof(cfg_tmp),"%s/0.6 200 OK\r\nUser-Agent: %s\r\n\r\n", protocol_name, name_version);
		gnutella_servant = g_strdup(cfg_tmp);
		}
	g_snprintf(cfg_tmp, sizeof(cfg_tmp),"%s/0.6 200 OK\r\n\r\n", protocol_name);
	gnutella_welcome = g_strdup(cfg_tmp);
	gnutella_welcome_length = strlen(gnutella_welcome);

        /* watch for filter_file defaults */
     
	if (0 && !local_ip)	/* We need our local address */
	  {
	    char hostname[255];
	    struct hostent* hostinfo;
	    gethostname(hostname, 255);
	    hostinfo = gethostbyname(hostname);
	    local_ip = g_ntohl(((struct in_addr*)(hostinfo->h_addr))->s_addr);
	  }


	/* Okay, update the GUI with values loaded */

	gui_update_count_downloads();
	gui_update_count_uploads();

	gui_update_minimum_speed(minimum_speed);
	gui_update_up_connections();
	gui_update_max_connections();
	gui_update_config_port();
	gui_update_config_force_ip();

	gui_update_save_file_path();
	gui_update_move_file_path();

	gui_update_monitor_max_items();

	gui_update_max_ttl();
	gui_update_my_ttl();

	gui_update_max_downloads();
	gui_update_max_host_downloads();
	gui_update_max_uploads();
	gui_update_files_scanned();

	gui_update_connection_speed();

	gui_update_search_max_items();

	gui_update_search_reissue_timeout();

	gui_update_scan_extensions();
	gui_update_shared_dirs();

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_monitor), monitor_enabled);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_clear_uploads), clear_uploads);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_clear_downloads), clear_downloads);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_config_force_ip), force_local_ip);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_never_push), !send_pushes);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_jump_to_downloads), jump_to_downloads);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_auto_run_at_startup), auto_on_at_start);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_search_select_all), search_select_all);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_search_select_name), search_select_name);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_config_throttle), throttle_uploads);

        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton_proxy_connections), proxy_connections);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_socksv4), (socks_protocol == 4)? TRUE:FALSE);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio_socksv5), (socks_protocol == 5)? TRUE:FALSE);  
 
        gui_update_socks_host();
        gui_update_socks_port();
        gui_update_socks_user();
        gui_update_socks_pass();

	if (w_w && w_h)
	{
		gtk_widget_set_uposition(main_window, w_x, w_y);
		gtk_window_set_default_size(GTK_WINDOW(main_window), w_w, w_h);
	}

	for (i = 0; i < 3; i++) gtk_clist_set_column_width(GTK_CLIST(clist_nodes), i, nodes_col_widths[i]);
	for (i = 0; i < 3; i++) gtk_clist_set_column_width(GTK_CLIST(clist_uploads), i, uploads_col_widths[i]);
	for (i = 0; i < 3; i++) gtk_clist_set_column_width(GTK_CLIST(clist_downloads), i, dl_active_col_widths[i]);
	for (i = 0; i < 2; i++) gtk_clist_set_column_width(GTK_CLIST(clist_download_queue), i, dl_queued_col_widths[i]);
	for (i = 0; i < 4; i++) gtk_clist_set_column_width(GTK_CLIST(clist_auto), i, auto_col_widths[i]);

	// as soon as this is corrected in Glade, you can do this, check the variable names and take out the
	// stuff in search.c that sets this up
	//	for (i = 0; i < 5; i++) gtk_clist_set_column_width(GTK_CLIST(clist_search_results), i, search_results_col_widths[i]);

	/* Transition : HOME/.napshare is now a directory */

	if (config_dir && !is_directory(config_dir))
	{
		if (unlink(config_dir) == -1)
		{
			if (errno != ENOENT)
			{
				g_warning("unlink(%s) failed (%s) !\n", config_dir, g_strerror(errno));
				g_free(config_dir); config_dir = NULL;
				return;
			}
		}
		else
		{
			/* We are replacing the old config file by a directory. */
			fprintf(stdout, "Creating configuration directory '%s'\n", config_dir);
		}

		if (mkdir(config_dir, 0755) == -1)
		{
			g_warning("mkdir(%s) failed (%s) !\n\n", config_dir, g_strerror(errno));
			g_free(config_dir); config_dir = NULL;
			return;
		}
	}
}

// For the keyword "hc", add hosts by including them in HEX format
// 12AB13AC18CD would be IP 18.171.19.172:6349 (leave last 4 chars out for default port)
// This isn't used anymore
void config_hosts_catched(gchar *str)
{
	gchar **h = g_strsplit(str, ",", 0);
	gint i = 0;
	gchar p[16];

	while (h[i] && *h[i])
	{
		if (strlen(h[i]) == 8) /* This is a host with default port */
		{
			host_add(NULL, strtoul(h[i], NULL, 16), 6346, 0);
		}
		else if (strlen(h[i]) == 12) /* This is a host with a port */
		{
			strncpy(p, h[i] + 8, 4);
			h[i][8] = 0;
			host_add(NULL, strtoul(h[i], NULL, 16), strtoul(p, NULL, 16), 0);
		}

		i++;
	}

	g_strfreev(h);
}

// parse comma delimited settings

guint32 *config_parse_array(gchar *str, guint32 n) // this can handle up to 10 items
{
	static guint32 array[10];
	gchar **h  = g_strsplit(str, ",", n + 1);
	guint32 *r = array;
	gint i;

	for (i = 0; i < n; i++)
	{
		if (!h[i]) { r = NULL; break; }
		array[i] = atol(h[i]);
	}

	g_strfreev(h);
	return r;
}

void config_set_param(guint32 keyword, gchar *value)
{
	gint32 i = atol(value);
	guint32 *a;

	switch (keyword)
	{
		case k_monitor_enabled: { monitor_enabled = (gboolean) !g_strcasecmp(value, "true"); return; }
		case k_monitor_max_items: { if (i > 0 && i < 512) monitor_max_items = i; return; }
		case k_clear_uploads: { clear_uploads = (gboolean) !g_strcasecmp(value, "true"); return; }
		case k_clear_downloads: { clear_downloads = (gboolean) !g_strcasecmp(value, "true"); return; }
		case k_up_connections: { if (i >= 0 && i < 512) up_connections = i; return; }
		case k_max_downloads: { if (i > 0 && i < 512) max_downloads = i; return; }
		case k_max_host_downloads: { if (i > 0 && i < 512) max_host_downloads = i; return; }
		case k_max_uploads: { if (i >= 0 && i < 512) max_uploads = i; return; }
		case k_minimum_speed: { minimum_speed = atol(value); return; }
		case k_listen_port: { listen_port = atoi(value); return; }
		case k_max_ttl: { if (i > 0 && i < 255) max_ttl = i; return; }
		case k_my_ttl:  { if (i > 0 && i < 255) my_ttl = i; return; }
		case k_search_max_items: { if (i >= -1 && i < 256) search_max_items = i; return; }
		case k_connection_speed: {
					if (i > 0 && i < 65535) {
						connection_speed = i;
						global_Bps = (i * 1000) / 11;
						return;
						}
					}
		case k_force_local_ip: { force_local_ip = (gboolean) !g_strcasecmp(value, "true"); return; }
		case k_scan_extensions: { parse_extensions(value); return; }
		case k_old_save_file_path:
		case k_save_file_path: { save_file_path = g_strdup(value); return; }
		case k_move_file_path: { move_file_path = g_strdup(value); return; }
		case k_shared_dirs: { shared_dirs_parse(value); return; }
		case k_hosts_catched: { config_hosts_catched(value); return; }
		case k_node_sendqueue_size: { if (i > 4096) node_sendqueue_size = i; return; }
		case k_node_sendqueue_max_size: { if (i > 4096) node_sendqueue_max_size = i; return; }
		case k_node_connecting_timeout: { if (i > 1 && i < 3600) node_connecting_timeout = i; return; }
		case k_node_connected_timeout: { if (i > 1 && i < 3600) node_connected_timeout = i; return; }
		case k_download_connecting_timeout: { if (i > 1 && i < 3600) download_connecting_timeout = i; return; }
		case k_download_push_sent_timeout: { if (i > 1 && i < 3600) download_push_sent_timeout = i; return; }
		case k_download_connected_timeout: { if (i > 1 && i < 3600) download_connected_timeout = i; return; }
		case k_search_queries_forward_size: { if (i > 512 && i < 65535) search_queries_forward_size = i; return; }
		case k_search_queries_kick_size: { if (i > 512 && i < 65535) search_queries_kick_size = i; return; }
		case k_search_answers_forward_size: { if (i > 512 && i < 1048576) search_answers_forward_size = i; return; }
		case k_search_answers_kick_size: { if (i > 512 && i < 1048576) search_answers_kick_size = i; return; }
		case k_other_messages_kick_size: { if (i > 0 && i < 1048576) other_messages_kick_size = i; return; }
		case k_win_x: { w_x = i; return; }
		case k_win_y: { w_y = i; return; }
		case k_win_w: { w_w = i; return; }
		case k_win_h: { w_h = i; return; }
		case k_win_coords: { if ((a = config_parse_array(value, 4))) { w_x = a[0]; w_y = a[1]; w_w = a[2]; w_h = a[3]; } return; }
		case k_widths_nodes: { if ((a = config_parse_array(value, 3))) for (i=0; i < 3; i++) nodes_col_widths[i] = a[i]; return; }
		case k_widths_uploads: { if ((a = config_parse_array(value, 3))) for (i=0; i < 3; i++) uploads_col_widths[i] = a[i]; return; }
		case k_widths_dl_active: { if ((a = config_parse_array(value, 3))) for (i=0; i < 3; i++) dl_active_col_widths[i] = a[i]; return; }
		case k_widths_dl_queued: { if ((a = config_parse_array(value, 2))) for (i=0; i < 2; i++) dl_queued_col_widths[i] = a[i]; return; }
		case k_widths_search_results: { if ((a = config_parse_array(value, 5))) for (i=0; i < 5; i++) search_results_col_widths[i] = a[i]; return; }
		case k_show_results_tabs: { search_results_show_tabs = (gboolean) !g_strcasecmp(value, "true"); return; }
		case k_forced_local_ip: { forced_local_ip = gchar_to_ip(value); return; }
		case k_hops_random_factor: { if (i >= 0 && i <= 3) hops_random_factor = i; return; }
		case k_send_pushes: { send_pushes = i ? 1 : 0; return; }
		case k_jump_to_downloads: { jump_to_downloads = i ? TRUE : FALSE; return; }
		case k_proxy_connections: { proxy_connections = i ? TRUE : FALSE; return; }
		case k_socks_protocol: { socks_protocol = i; return;}
		case k_proxy_ip: { proxy_ip = g_strdup(value); return;}
		case k_proxy_port: { proxy_port = i; return; }
		case k_socksv5_user: { socksv5_user = g_strdup(value); return; }
		case k_socksv5_pass: { socksv5_pass = g_strdup(value); return; }
		case k_max_connections: { if (i >= 0 && i < 512) max_connections = i; return; }
		case k_search_reissue_timeout: { if (i >= 300 || i == 0) search_reissue_timeout = i; return; }
		case k_dbg: { dbg = i; return; }
		case k_stop_host_get: { stop_host_get = i; return; }
		case k_enable_err_log: { enable_err_log = i; return; }
		case k_max_uploads_ip: { if (i >= 1 && i < 512) max_uploads_ip = i; return; }
		case k_search_strict_and: { search_strict_and = i; return; }
		case k_search_select_all: { search_select_all = i ? TRUE : FALSE; return; }
		case k_search_select_name: { search_select_name = i ? TRUE : FALSE; return; }
		case k_throttle_uploads: { throttle_uploads = i ? TRUE : FALSE; return; }
		case k_widths_auto: { if ((a = config_parse_array(value, 4))) for (i=0; i < 4; i++) auto_col_widths[i] = a[i]; return; }
		case k_vendor_code: { vendor_code = g_strdup(value); return; }
		case k_vendor_name: { vendor_name = g_strdup(value); return; }
		case k_vendor_version: { vendor_version = g_strdup(value); return; }
		case k_protocol_name: { protocol_name = g_strdup(value); return; }
		case k_lan_name: { lan_name = g_strdup(value); return; }
	}
}


void config_read(void)
{
	FILE *config;
	gchar *s, *k, *v;
	guint32 i, n = 0;

	static gchar *err = "Bad line %u in config file, ignored\n";

	if (is_directory(config_dir))
		g_snprintf(cfg_tmp, sizeof(cfg_tmp), "%s/NAPS-settings.txt", config_dir);
	else
		strncpy(cfg_tmp, config_dir, sizeof(cfg_tmp));

	if ((config = fopen("NAPS-settings.txt", "r")) == NULL) { // try to open settings file in local directory
		config = fopen(cfg_tmp, "r"); // open the normal file
		}
   else cfg_use_local_file = 1; // show we are using a local file

	if (!config) return;

	gtk_clist_freeze(GTK_CLIST(clist_host_catcher));

	while (fgets(cfg_tmp, sizeof(cfg_tmp), config))
	{
		n++;
		s = cfg_tmp;
		while (*s && (*s == ' ' || *s == '\t')) s++;
		if (!((*s >= 'A' && *s <= 'Z') || (*s >= 'a' && *s <= 'z'))) continue;
		k = s;
		while (*s =='_' || (*s >= 'A' && *s <= 'Z') || (*s >= 'a' && *s <= 'z') || (*s >= '0' && *s <= '9')) s++;
		if (*s != '=' && *s != ' ' && *s != '\t') { fprintf(stderr, err, n); continue; }
		v = s;
		while (*s == ' ' || *s == '\t') s++;
		if (*s != '=') { fprintf(stderr, err, n); continue; }
		*v = 0; s++;
		while (*s == ' ' || *s == '\t') s++;
		if (*s == '"')
		{
			v = ++s;
			while (*s && *s != '\n' && *s != '"') s++;
			if (!*s || *s == '\n') { fprintf(stderr, err, n); continue; }
		}
		else { v = s; while (*s && *s != '\n' && *s != ' ' && *s != '\t') s++; }
		*s = 0;

		for (i = 0; i < k_end; i++) if (!g_strcasecmp(k, keywords[i])) { config_set_param(i, v); break; }

		if (i >= k_end) fprintf(stderr, "config file, line %u: unknown keyword '%s', ignored\n", n, k);
	}

	gtk_clist_thaw(GTK_CLIST(clist_host_catcher));

	fclose(config);
}

gchar *config_boolean(gboolean b)
{
	static gchar *b_true  = "TRUE";
	static gchar *b_false = "FALSE";
	return (b)? b_true : b_false;
}

void config_save(void)
{
	FILE *config;
	gint win_x, win_y, win_w, win_h;

	if (!config_dir)
	{
		fprintf(stderr, "\nNo configuration directory !\n\nPreferences have not been saved.\n\n");
		return;
	}

	g_snprintf(cfg_tmp, sizeof(cfg_tmp), "%s/NAPS-settings.txt", config_dir);

	if (cfg_use_local_file) config = fopen("NAPS-settings.txt", "w"); // write settings file in local directory
	else config = fopen(cfg_tmp, "w"); // write to the normal file

	if (!config)
	{
		fprintf(stderr, "\nfopen(): %s\n\nUnable to write your configuration in %s\nPreferences have not been saved.\n\n", g_strerror(errno), cfg_tmp);
		return;
	}

	gdk_window_get_root_origin(main_window->window, &win_x, &win_y);
	gdk_window_get_size(main_window->window, &win_w, &win_h);

	fprintf(config, "\n# NapShare %s - %s\n\n", VERSION, GTA_WEBSITE);
	fprintf(config, "# This is the NapShare configuration file - you may edit it if you're careful.\n");
	fprintf(config, "# Edit it when the program is not running, it is written over after quit.\n\n");
	fprintf(config, "%s = %u\n", keywords[k_up_connections], up_connections);
	fprintf(config, "\n");
	fprintf(config, "%s = %u\n", keywords[k_max_connections], max_connections);
	fprintf(config, "\n");
	fprintf(config, "%s = %s\n", keywords[k_clear_uploads], config_boolean(clear_uploads));
	fprintf(config, "\n");
	fprintf(config, "%s = %u\n", keywords[k_max_downloads], max_downloads);
	fprintf(config, "\n");
	fprintf(config, "%s = %u\n", keywords[k_max_host_downloads], max_host_downloads);
	fprintf(config, "\n");
	fprintf(config, "%s = %u\n", keywords[k_max_uploads], max_uploads);
	fprintf(config, "\n");
	fprintf(config, "%s = %s\n", keywords[k_clear_downloads], config_boolean(clear_downloads));
	fprintf(config, "\n");
	fprintf(config, "%s = %u\n", keywords[k_minimum_speed], minimum_speed);
	fprintf(config, "\n");
	fprintf(config, "%s = %s\n", keywords[k_monitor_enabled], config_boolean(monitor_enabled));
	fprintf(config, "%s = %u\n", keywords[k_monitor_max_items], monitor_max_items);
	fprintf(config, "\n");
	fprintf(config, "%s = \"%s\"\n", keywords[k_save_file_path], save_file_path);
	fprintf(config, "%s = \"%s\"\n", keywords[k_move_file_path], move_file_path);
	fprintf(config, "\n");
	fprintf(config, "%s = \"%s\"\n", keywords[k_shared_dirs], (shared_dirs_paths)? shared_dirs_paths : "");
	fprintf(config, "%s = \"%s\"\n", keywords[k_scan_extensions], (scan_extensions)? scan_extensions : "");
	fprintf(config, "\n");
	fprintf(config, "%s = %s\n", keywords[k_force_local_ip], config_boolean(force_local_ip));
	fprintf(config, "%s = \"%s\"\n", keywords[k_forced_local_ip], ip_to_gchar(forced_local_ip));
	fprintf(config, "%s = %u\n", keywords[k_listen_port], listen_port);
	fprintf(config, "\n");
	fprintf(config, "%s = %u\n", keywords[k_connection_speed], connection_speed);
	fprintf(config, "\n");
	fprintf(config, "%s = %d\n", keywords[k_search_max_items], search_max_items);
	fprintf(config, "\n");
	fprintf(config, "%s = %u\n", keywords[k_max_ttl], max_ttl);
	fprintf(config, "%s = %u\n\n", keywords[k_my_ttl], my_ttl);
	fprintf(config, "%s = %u\n\n", keywords[k_search_reissue_timeout], search_reissue_timeout);

	fprintf(config, "%s = %s\n\n", keywords[k_show_results_tabs], config_boolean(search_results_show_tabs));

	fprintf(config, "%s = %u\n\n", keywords[k_jump_to_downloads], jump_to_downloads);
	fprintf(config, "%s = %u\n\n", keywords[k_max_uploads_ip], max_uploads_ip);
	fprintf(config, "%s = %u\n\n", keywords[k_search_select_all], search_select_all);
	fprintf(config, "%s = %u\n\n", keywords[k_search_select_name], search_select_name);
	fprintf(config, "%s = %u\n\n", keywords[k_throttle_uploads], throttle_uploads);

	fprintf(config, "\n# GUI values\n\n");

	fprintf(config, "%s = %u,%u,%u,%u\n\n", keywords[k_win_coords], win_x, win_y, win_w, win_h);

	fprintf(config, "%s = %u,%u,%u\n", keywords[k_widths_nodes], nodes_col_widths[0], nodes_col_widths[1], nodes_col_widths[2]);
	fprintf(config, "%s = %u,%u,%u\n", keywords[k_widths_uploads], uploads_col_widths[0], uploads_col_widths[1], uploads_col_widths[2]);
	fprintf(config, "%s = %u,%u,%u\n", keywords[k_widths_dl_active], dl_active_col_widths[0], dl_active_col_widths[1], dl_active_col_widths[2]);
	fprintf(config, "%s = %u,%u\n", keywords[k_widths_dl_queued], dl_queued_col_widths[0], dl_queued_col_widths[1]);
	fprintf(config, "%s = %u,%u,%u,%u,%u\n", keywords[k_widths_search_results],
					search_results_col_widths[0], search_results_col_widths[1], search_results_col_widths[2],
					search_results_col_widths[3], search_results_col_widths[4]);
	fprintf(config, "%s = %u,%u,%u,%u\n\n", keywords[k_widths_auto], auto_col_widths[0], auto_col_widths[1],
					auto_col_widths[2], auto_col_widths[3]);

	fprintf(config, "\n# The following variables cannot yet be configured with the GUI.\n\n");

	fprintf(config, "# Number of seconds before timeout for a connecting download\n%s = %u\n\n", keywords[k_download_connecting_timeout], download_connecting_timeout);
	fprintf(config, "# Number of seconds before timeout for a 'push sent' download\n%s = %u\n\n", keywords[k_download_push_sent_timeout], download_push_sent_timeout);
	fprintf(config, "# Number of seconds before timeout for a connected download\n%s = %u\n\n", keywords[k_download_connected_timeout], download_connected_timeout);
	fprintf(config, "# Number of seconds before timeout for a connecting node\n%s = %u\n\n", keywords[k_node_connecting_timeout], node_connecting_timeout);
	fprintf(config, "# Number of seconds before timeout for a connected node\n%s = %u\n\n", keywords[k_node_connected_timeout], node_connected_timeout);

	fprintf(config, "# Sendqueue size limit for non priority packets, for slow nodes (in bytes)\n%s = %u\n\n", keywords[k_node_sendqueue_size], node_sendqueue_size);

	fprintf(config, "# Sendqueue absolute max to disconnect node (in bytes)\n%s = %u\n\n", keywords[k_node_sendqueue_max_size], node_sendqueue_max_size);

	fprintf(config, "# Random factor for the hops field in search packets we send (between 0 and 3 inclusive)\n%s = %u\n\n", keywords[k_hops_random_factor], hops_random_factor);

	fprintf(config, "# Whether or not to send pushes.\n%s = %u\n\n", keywords[k_send_pushes], send_pushes);

	fprintf(config, "# Set to 1 to filter search results with a strict AND\n"
			"%s = %u\n\n", keywords[k_search_strict_and], search_strict_and);

	fprintf(config, "\n# Proxy Info\n\n");

        fprintf(config, "%s = %u\n\n", keywords[k_proxy_connections], proxy_connections);

        fprintf(config, "%s = %u\n\n", keywords[k_socks_protocol], socks_protocol);
 
        fprintf(config, "%s = \"%s\"\n\n", keywords[k_proxy_ip], proxy_ip);
 
        fprintf(config, "%s = %u\n\n", keywords[k_proxy_port], proxy_port);
 
        fprintf(config, "%s = \"%s\"\n\n", keywords[k_socksv5_user], socksv5_user);
 
        fprintf(config, "%s = \"%s\"\n\n", keywords[k_socksv5_pass], socksv5_pass);

	fprintf(config, "\n# Vendor info, sent to other clients, code = \"NAPS\", name = \"NapShare\", etc... no spaces!\n\n");

        fprintf(config, "%s = \"%s\"\n\n", keywords[k_vendor_code], vendor_code);
        fprintf(config, "%s = \"%s\"\n\n", keywords[k_vendor_name], vendor_name);
        fprintf(config, "%s = \"%s\"\n\n", keywords[k_vendor_version], vendor_version);

	fprintf(config, "\n# Private LAN special settings, lan_name adds a \"LAN: lan_name\" header to connect messages, some private\n");
	fprintf(config, "# networks also change the protocol name \"GNUTELLA\", watch network packets using tcpdump on your LAN.\n\n");

        fprintf(config, "%s = \"%s\"\n\n", keywords[k_protocol_name], protocol_name);
        fprintf(config, "%s = \"%s\"\n\n", keywords[k_lan_name], lan_name);

        fprintf(config, "\n# For developers only, debug stuff\n\n");

	fprintf(config, "# Debug level, each one prints more detail (between 0 and 5)\n"
			"%s = %u\n\n", keywords[k_dbg], dbg);
	fprintf(config, "# Set to 1 to stop getting new hosts and stop timeout, manual connect only\n"
			"%s = %u\n\n", keywords[k_stop_host_get], stop_host_get);
	fprintf(config, "# Set to 1 to log network errors for later inspection, for developer improvements\n"
			"%s = %u\n\n", keywords[k_enable_err_log], enable_err_log);

	/* I'm not sure yet that the following variables are really useful...

	fprintf(config, "# WARNING: *PLEASE* DO NOT MODIFY THE FOLLOWING VALUES IF YOU DON'T KNOW WHAT YOU'RE DOING\n\n"); 

	fprintf(config, "# [NOT IMPLEMENTED] Maximum size of search queries messages we forward to others (in bytes)\n%s = %u\n\n", keywords[k_search_queries_forward_size], search_queries_forward_size);
	fprintf(config, "# [NOT IMPLEMENTED] Maximum size of search queries messages we allow before closing the connection (in bytes)\n%s = %u\n\n", keywords[k_search_queries_kick_size], search_queries_kick_size);
	fprintf(config, "# [NOT IMPLEMENTED] Maximum size of search answers messages we forward to others (in bytes)\n%s = %u\n\n", keywords[k_search_answers_forward_size], search_answers_forward_size);
	fprintf(config, "# [NOT IMPLEMENTED] Maximum size of search answers messages we allow before closing the connection (in bytes)\n%s = %u\n\n", keywords[k_search_answers_kick_size], search_answers_kick_size);
	fprintf(config, "# [NOT IMPLEMENTED] Maximum size of unknown messages we allow before closing the connection (in bytes)\n%s = %u\n\n", keywords[k_other_messages_kick_size], other_messages_kick_size);

	*/
	
	fprintf(config, "\n\n");

	fclose(config);

	/* Save the catched hosts */

	if (hosts_idle_func)
	{
		g_warning("exit() while still reading the hosts file, catched hosts not saved !\n");
	}
	else
	{
		if (cfg_use_local_file) hosts_write_to_file("NAPS-hosts");
		else {
			g_snprintf(cfg_tmp, sizeof(cfg_tmp), "%s/NAPS-hosts", config_dir);
			hosts_write_to_file(cfg_tmp);
			}
	}

}

/* vi: set ts=3: */

