# -*- Mode: perl; perl-indent-level: 8; coding: utf-8 -*-
#
# Net::MsnMessenger::Message
#
# Copyright (C) 2003 <incoming@tiscali.cz>  All rights reserved.
# This module is free software; You can redistribute and/or modify it under
# the same terms as Perl itself.
#
# $Id: Message.pm,v 1.8 2003/07/15 21:34:05 incoming Exp $

package Net::MsnMessenger::Message;

use Net::MsnMessenger::Data;
use strict qw(subs vars);
use vars   qw($AUTOLOAD);

sub AUTOLOAD
{
	my $self = shift;
	my $name = $AUTOLOAD;
	$name =~ s/.*:://;
	return if $name =~ /DESTROY$/;

	if (!exists $self->{$name})
	{
		Carp::croak("AUTOLOAD: $name is not a valid method\n");
	}
	$self->{$name} = shift if @_;
	$self->{$name};
}

# Net::MsnMessenger::Message->new
sub new 
{
	my ($this, %args) = @_;
	my $self = {};

	$self->{message} = undef;
	$self->{just_sent} = undef;

	if (defined $args{message})
	{
		$self->{message} = $args{message};
		$self->{message} =~ s/\r?\n/\r\n/g;
	}
	$self->{msn} = $args{msn};

	bless $self, $this;
	return $self;
}

# Net::MsnMessenger::Message->add_header_file
sub add_header_file
{
	my $self = shift;

	$self->{header}->{application_name} = 'File Transfer';
	$self->{header}->{application_guid} = $Net::MsnMessenger::Data::Guid{'File Transfer'};
	1;
}

# Net::MsnMessenger::Message->add_header_netmeeting
sub add_header_netmeeting
{
	my $self = shift;

	$self->{header}->{application_name} = 'NetMeeting';
	$self->{header}->{application_guid} = $Net::MsnMessenger::Data::Guid{NetMeeting};
	1;
}

# Net::MsnMessenger::Message->add_header_remote_assistance
sub add_header_remote_assistance
{
	my $self = shift;

	$self->{header}->{application_name} = 'Remote Assistance';
	$self->{header}->{application_guid} = $Net::MsnMessenger::Data::Guid{'Remote Assistance'};
	1;
}

# Net::MsnMessenger::Message->add_header_voice
sub add_header_voice
{
	my $self = shift;

	$self->{header}->{application_name} = 'Voice Conversation';
	$self->{header}->{application_guid} = $Net::MsnMessenger::Data::Guid{'Voice Conversation'};
	1;
}

# Net::MsnMessenger::Message->generate_session_id
sub generate_session_id
{
	my $self = shift;
	my $s_id;

	# Generate a session ID and return it
	for (8,4,4,4,12)
	{
		$s_id .= uc sprintf "%x", rand(16) for 1..$_;
		$s_id .= '-' unless $_ == 12;
	}
	return '{' . $s_id . '}';
}

# Net::MsnMessenger::Message->create_clientcaps
sub create_clientcaps
{
	my ($self, %attributes) = @_;
	my $r_message;

	# Create the text/x-clientcaps message

	$attributes{mime_version} ||= '1.0';
	$attributes{charset} ||= 'UTF-8';

	$r_message  = "MIME-Version: $attributes{mime_version}\r\n";
	$r_message .= "Content-Type: text/x-clientcaps; charset=$attributes{charset}\r\n\r\n";

	$self->_add_field(\$r_message, "Client-Name",  $attributes{client_name});
	$self->_add_field(\$r_message, "Chat-Logging", $attributes{chat_logging});
	$self->_add_field(\$r_message, "Buddy-Icons",  $attributes{buddy_icons});

	return $self->_finalize_message('U', $r_message . "\r\n");
}

# Net::MsnMessenger::Message->create_control
sub create_control
{
	my ($self, %attributes) = @_;
	my $r_message;

	# Create the text/x-msmsgscontrol message

	$attributes{mime_version} ||= '1.0';
	$attributes{charset} ||= 'UTF-8';

	$r_message  = "MIME-Version: $attributes{mime_version}\r\n";
	$r_message .= "Content-Type: text/x-msmsgscontrol; charset=$attributes{charset}\r\n";

	$self->_add_field(\$r_message, "TypingUser", $attributes{typinguser});
	return $self->_finalize_message('U', $r_message . "\r\n\r\n");
}    

# Net::MsnMessenger::Message->create_invite
sub create_invite
{
	my ($self, %attributes) = @_;
	my $r_message;

	# Create an invitation text/x-msmsgsinvite message

	$attributes{mime_version} ||= '1.0';
	$attributes{charset} ||= 'UTF-8';

	if (defined $self->{header})
	{
		$attributes{$_} = $self->{header}->{$_} for keys %{$self->{header}};
	}

	$r_message  = "MIME-Version: $attributes{mime_version}\r\n";
	$r_message .= "Content-Type: text/x-msmsgsinvite; charset=$attributes{charset}\r\n\r\n";

	$self->_add_field(\$r_message, "Application-Name",     $attributes{application_name});
	$self->_add_field(\$r_message, "Application-GUID",     $attributes{application_guid});
	$self->_add_field(\$r_message, "Application-File",     $attributes{application_file});
	$self->_add_field(\$r_message, "Application-FileSize", $attributes{application_filesize});
	$self->_add_field(\$r_message, "Application-URL",      $attributes{application_url});

	$self->_add_field(\$r_message, "Invitation-Command",   $attributes{invitation_command});
	$self->_add_field(\$r_message, "Invitation-Cookie",    $attributes{invitation_cookie});

	$self->_add_field(\$r_message, "Session-ID",           $attributes{session_id});
	$self->_add_field(\$r_message, "Session-Protocol",     $attributes{session_protocol});
	$self->_add_field(\$r_message, "Conn-Type",            $attributes{conn_type});
	$self->_add_field(\$r_message, "Context-Data",         $attributes{context_data});
	$self->_add_field(\$r_message, "Cancel-Code",          $attributes{cancel_code});
	$self->_add_field(\$r_message, "IP-Address",           $attributes{ip_address});
	$self->_add_field(\$r_message, "Port",                 $attributes{port});
	$self->_add_field(\$r_message, "AuthCookie",           $attributes{authcookie});
	$self->_add_field(\$r_message, "Launch-Application",   $attributes{launch_application});
	$self->_add_field(\$r_message, "Connectivity",         $attributes{connectivity});
	$self->_add_field(\$r_message, "Sender-Connect",       $attributes{sender_connect});
	$self->_add_field(\$r_message, "IP-Address-Internal",  $attributes{ip_address_internal});
	$self->_add_field(\$r_message, "PortX",                $attributes{portx});
	$self->_add_field(\$r_message, "PortX-Internal",       $attributes{portx_internal});
	$self->_add_field(\$r_message, "Private-IP",           $attributes{private_ip});
	$self->_add_field(\$r_message, "Public-IP",            $attributes{public_ip});
	$self->_add_field(\$r_message, "Request-Data",         $attributes{request_data});

	return $self->_finalize_message('N', $r_message . "\r\n");
}

# Net::MsnMessenger::Message->create_pager
sub create_pager
{
	my ($self, %attributes) = @_;
	my $r_message = "";

	# Create a pager message

	$self->{message} = ' ' if !defined $self->message;

	if (defined $attributes{number_type} && defined $attributes{number})
	{
		$r_message  = "<PHONE pri=\"1\"><LOC>".uc($attributes{number_type})."</LOC>";
		$r_message .= "<NUM>$attributes{number}</NUM></PHONE>";
	}
	$r_message .= "<TEXT xml:space=\"preserve\" enc=\"utf-8\">$self->{message}</TEXT>\r\n";

	if (length($self->message) > $Net::MsnMessenger::Data::Pager_len)
	{
		# The message is longer than the maximal length, return the first part with the
		# maximal length of a single message and keep the rest.

		my $diff  = length($self->message) - $Net::MsnMessenger::Data::Pager_len;
		my $first = substr $self->message, 0, -$diff;
		my $rest  = substr $self->message, -$diff, $diff;

		$self->message($first);
		my $new_msg = $self->create_pager(%attributes);

		$self->message($rest);
		$self->just_sent($first);
		return $new_msg;
	}
	else
	{
		# The message is not longer than the maximal length

		$self->just_sent($self->message);
		$self->{message} = undef;
		return $self->_finalize_message('', $r_message);
	}
}

# Net::MsnMessenger::Message->create_switchboard
sub create_switchboard
{
	my ($self, %attributes) = @_;
	my $r_message;

	# Create a switchboard message

	$self->{message} = ' ' if !defined $self->{message};

	$attributes{mime_version} ||= '1.0';
	$attributes{charset} ||= 'UTF-8';

	# Font attributes defaults
	$attributes{EF} ||= '';                         # Effect
	$attributes{FN} ||= 'Arial';                    # Name
	$attributes{CO} ||= '0';                        # Color
	$attributes{CS} ||= '0';                        # Charset
	$attributes{PF} ||= '0';                        # Pitch & Family
	$attributes{RL} ||= '0';                        # Right-Left

	$r_message  = "MIME-Version: $attributes{mime_version}\r\n";
	$r_message .= "Content-Type: text/plain; charset=$attributes{charset}\r\n";

	$self->_add_field(\$r_message, "User-Agent", $attributes{user_agent});

	# Add the attributes

	$r_message .= "X-MMS-IM-Format: FN=$attributes{FN}; EF=$attributes{EF}; CO=$attributes{CO}; ";
	$r_message .= "CS=$attributes{CS}; PF=$attributes{PF}; RL=$attributes{RL}\r\n\r\n$self->{message}";

	# Check the maximal message length

	my $f_message = $self->_finalize_message('A', $r_message);
	my ($f_length) = $f_message =~ /^\S\s+(\d+)\r\n/;

	if ($f_length > $Net::MsnMessenger::Data::Message_len)
	{
		# The message is longer than the maximal length, return the first part with the
		# maximal length of a single message and keep the rest.

		my $diff  = $f_length - $Net::MsnMessenger::Data::Message_len;
		my $first = substr $self->message, 0, -$diff;
		my $rest  = substr $self->message, -$diff, $diff;

		$self->message($first);
		my $new_msg = $self->create_switchboard(%attributes);

		$self->message($rest);
		$self->just_sent($first);
		return $new_msg;
	}
	else
	{
		# The message is not longer than the maximal length

		$self->just_sent($self->{message});
		$self->{message} = undef;
		return $f_message;
	}
}

# -------------------- Private methods -------------------- #

# Net::MsnMessenger::Message->_add_field
sub _add_field
{
	my ($self, $msg_ref, $mime_field, $param) = @_;

	if (defined $msg_ref && defined $mime_field && defined $param)
	{
		$$msg_ref .= $mime_field . ": $param\r\n";
	}
	1;
}

# Net::MsnMessenger::Message->_finalize_message
sub _finalize_message
{
	my ($self, $verif, $r_message) = @_;
	my $message = '';

	$r_message =~ s/\r?\n/\r\n/g;

	if (length($verif))
	{
		$message  = $verif . ' ';
	}
	$message .= length($r_message) . "\r\n$r_message";
	return $message;
}

"Net::MsnMessenger::Message";
__END__

