# -*- Mode: perl; perl-indent-level: 8; coding: utf-8 -*-
#
# Net::MsnMessenger::Group
#
# Copyright (C) 2003 <incoming@tiscali.cz>  All rights reserved.
# This module is free software; You can redistribute and/or modify it under
# the same terms as Perl itself.
#
# $Id: Group.pm,v 1.5 2003/07/15 21:34:05 incoming Exp $

package Net::MsnMessenger::Group;

=head1 NAME

Net::MsnMessenger::Group - Object representing a group in the forward list

=head1 SYNOPSIS

  my @groups = $msn->get_all_groups;

  do_something($groups[0]->ID);
  do_something($groups[0]->name);
  do_something($groups[0]->users);

=head1 DESCRIPTION

Every group in Net::MsnMessenger is represented by an object. This manual page describes
what can you do with an object representing a group.

=cut

use Carp;
use strict qw(subs vars);
use vars   qw($AUTOLOAD);

sub AUTOLOAD
{
	my $self = shift;
	my $name = $AUTOLOAD;
	$name =~ s/.*:://;
	return if $name =~ /DESTROY$/;

	if (!exists $self->{$name})
	{
		Carp::croak("AUTOLOAD: $name is not a valid method\n");
	}
	$self->{$name} = shift if @_;
	$self->{$name};
}

=head1 METHODS

In this section are documented all the public methods of the Net::MsnMessenger::Group class.

=head2 Net::MsnMessenger::Group->new

Create a new Net::MsnMessenger::Group object. It can be called with the following arguments:

 ID   => group server ID
 name => URL encoded group name

=cut

# Net::MsnMessenger::Group->new
sub new
{
	my $this = shift;
	my %args = @_;
	my $self = {};

	$self->{ID} = $args{ID} if defined $args{ID};
	$self->{name} = $args{name} if defined $args{name};
	$self->{users} = 0;

	$self->{_users_list} = ();

	bless $self, $this;
	return $self;
}

=head2 $group->add_user ( PASSPORT )

Add an user to the group. The parameter is the passport.

=cut

# Net::MsnMessenger::Group->add_user
sub add_user
{
	my ($self, $passport) = @_;

	return if grep {$_ eq $passport} @{$self->{_users_list}};

	push @{$self->{_users_list}}, $passport;
	$self->{users}++;
	1;
}

=head2 $group->is_user_in_group ( PASSPORT )

Return 1 if the specified user is present in this group.

=cut

# Net::MsnMessenger::Group->is_user_in_group
sub is_user_in_group
{
	my ($self, $passport) = @_;

	if (defined $passport && $self->users)
	{
		return 1 if grep {$_ eq $passport} @{$self->{_users_list}};
	}
	undef;
}

=head2 $group->remove_user ( PASSPORT )

Remove a user from the group. The parameter is the passport.

=cut

# Net::MsnMessenger::Group->remove_user
sub remove_user
{
	my ($self, $passport) = @_;

	for (my $i = 0; $i < @{$self->{_users_list}}; $i++)
	{
		if ($self->{_users_list}->[$i] eq $passport)
		{
			splice @{$self->{_users_list}}, $i, 1;
			$self->{users}--;
		}
	}
	1;
}

"Net::MsnMessenger::Group";
__END__

=head1 AUTOLOADED METHODS

An object attributes that are accessible my methods.

=head2 $group->ID

The server's ID of the group.

=head2 $group->name

The URL and UTF-8 encoded name of the group.

=head2 $group->users

Number of users currently in the group.

=head1 SEE ALSO

Net::MsnMessenger(3), Net::MsnMessenger::Event(3), Net::MsnMessenger::Contact(3)

=head1 AUTHOR

E-mail: B<incoming@tiscali.cz>

=head1 COPYRIGHT

This module is free software; you can redistribute it and/or modify it under
the same terms as Perl itself.

=cut

