# -*- Mode: perl; perl-indent-level: 8; coding: utf-8 -*-
#
# Net::MsnMessenger::File
#
# Copyright (C) 2003 <incoming@tiscali.cz>  All rights reserved.
# This module is free software; You can redistribute and/or modify it under
# the same terms as Perl itself.
#
# $Id: File.pm,v 1.7 2003/07/17 08:12:48 incoming Exp $

package Net::MsnMessenger::File;

use strict qw(subs vars);
use vars   qw($AUTOLOAD);

sub AUTOLOAD
{
	my $self = shift;
	my $name = $AUTOLOAD;
	$name =~ s/.*:://;
	return if $name =~ /DESTROY$/;

	if (exists $self->{$name})
	{
		$self->{$name} = shift if @_;
		return $self->{$name};
	}

	if (defined $self->{c_type})
	{
		return $self->{"_".$self->{c_type}}->$name(@_) if $self->{"_".$self->{c_type}}->can($name);

		if (exists $self->{"_".$self->{c_type}}->{$name})
		{
			$self->{"_".$self->{c_type}}->{$name} = shift if @_;
			return $self->{"_".$self->{c_type}}->{$name};
		}
	}
	Carp::croak("AUTOLOAD: $name is not a valid method\n");
}

# Net::MsnMessenger::File->new
sub new
{
	my ($this, %args) = @_;
	my $self = {};

	$self->{file}              = $args{file} || undef;
	$self->{file_size}         = $args{file_size} || undef;
	$self->{invitation_cookie} = $args{invitation_cookie} || undef;
	$self->{file_dest_dir}     = $args{destination} || undef;

	$self->{file_session} = $args{file_session};
	$self->{swb_session}  = $args{swb_session};
	$self->{msn}          = $args{msn};
	$self->{authcookie}   = undef;
	$self->{c_type}       = undef;
	$self->{file_dest}    = undef;
	$self->{file_handle}  = undef;
	$self->{sending_to}   = undef;
	$self->{transfered}   = 0;

	$self->{type}         = 'File Transfer';
	$self->{incoming}     = $args{incoming} || undef;
	$self->{outgoing}     = $args{outgoing} || undef;

	$self->{_client} = undef;
	$self->{_server} = undef;

	$self->{_buffer}      = undef;
	$self->{_buffer_left} = undef;

	$self->{ip_address} = undef;
	$self->{port}       = undef;

	bless $self, $this;

	if (defined $self->{file_dest_dir})      # Destination directory for the file
	{
		if (!-d $self->{file_dest_dir})
		{
			if (-e $self->{file_dest_dir})
			{
				$self->msn->error("$self->{file_dest_dir} already exists and it's not a directory");
				return undef;
			}

			# Create the directory if it doesn't exist
			require File::Path;
			File::Path::mkpath($self->{file_dest_dir});
		}

		require File::Spec;
		$self->{file_dest} = File::Spec->catfile($self->{file_dest_dir}, $self->{file});
	}

	$self->{swb} = $self->msn->{_swb}->[$self->{swb_session}];
	return $self;
}

# Net::MsnMessenger::File->connected
sub connected
{
	my $self = shift;

	return 0 if !defined $self->{_client} && !defined $self->{_server};
	return (defined $self->{_client}) ? $self->{_client}->connected : $self->{_server}->connected;
}

# Net::MsnMessenger::File->invite
sub invite
{
	my $self = shift;
	my $message = Net::MsnMessenger::Message->new;
	my %attributes;

	$self->invitation_cookie(int(rand(9999999)+1));

	require File::Basename;
	$attributes{invitation_command} = "INVITE";
	$attributes{invitation_cookie} = $self->invitation_cookie;
	$attributes{application_file} = File::Basename::basename($self->{file});
	$attributes{application_filesize} = $self->{file_size};

	$message->add_header_file;
	$self->swb->send_packet('message', $message->create_invite(%attributes));
	1;
}

# Net::MsnMessenger::File->invite_accept
sub invite_accept
{
	my $self = shift;
	my $message = Net::MsnMessenger::Message->new;
	my %attributes;

	$attributes{invitation_command} = "ACCEPT";
	$attributes{invitation_cookie} = $self->invitation_cookie;
	$attributes{launch_application} = "FALSE";
	$attributes{request_data} = "IP-Address:";

	$self->swb->send_packet('message', $message->create_invite(%attributes));
	1;
}

# Net::MsnMessenger::File->invite_accept_confirm
sub invite_accept_confirm
{
	my $self = shift;
	my $message = Net::MsnMessenger::Message->new;
	my %attributes;

	$self->authcookie(int(rand(9999999)+1));
	$self->ip_address($self->msn->ip_address);
	$self->port(6891);

	# The server port could get changed. This would be done in Connection class so create
	# the server before sending the ACCEPT message.
	$self->create_server;

	$attributes{invitation_command} = "ACCEPT";
	$attributes{invitation_cookie} = $self->{invitation_cookie};
	$attributes{ip_address} = $self->ip_address;
	$attributes{port} = $self->port;
	$attributes{authcookie} = $self->authcookie;
	$attributes{launch_application} = "FALSE";
	$attributes{request_data} = "IP-Address:";

	$self->swb->send_packet('message', $message->create_invite(%attributes));
	1;
}

# Net::MsnMessenger::File->invite_cancel
sub invite_cancel
{
	my $self = shift;
	my $message = Net::MsnMessenger::Message->new;
	my %attributes;

	$attributes{invitation_command} = "CANCEL";
	$attributes{invitation_cookie} = $self->invitation_cookie;
	$attributes{cancel_code} = "TIMEOUT";

	$self->swb->send_packet('message', $message->create_invite(%attributes));
	1;
}

# Net::MsnMessenger::File->invite_reject
sub invite_reject
{
	my $self = shift;
	my $message = Net::MsnMessenger::Message->new;
	my %attributes;

	$attributes{invitation_command} = "CANCEL";
	$attributes{invitation_cookie} = $self->{invitation_cookie};
	$attributes{cancel_code} = "REJECT";

	$self->swb->send_packet('message', $message->create_invite(%attributes));
	1;
}

# Net::MsnMessenger::File->invite_fttimeout
sub invite_fttimeout
{
	my $self = shift;
	my $message = Net::MsnMessenger::Message->new;
	my %attributes;

	$attributes{invitation_cookie} = $self->invitation_cookie;
	$attributes{invitation_command} = "CANCEL";
	$attributes{cancel_code} = "FTTIMEOUT";

	$self->swb->send_packet('message', $message->create_invite(%attributes));
	1;
}

# Net::MsnMessegner::File->create_client
sub create_client
{
	my $self = shift;

	if (!defined $self->{file_dest_dir})   # Destination directory
	{
		require Cwd;
		$self->{file_dest_dir} = Cwd::getcwd();
	}
	if (!defined $self->{file_dest})       # File destination
	{
		require File::Spec;
		$self->{file_dest} = File::Spec->catfile($self->{file_dest_dir}, $self->{file});
	}

	# The File handle we are writing the file to
	require FileHandle;
	$self->{file_handle} = new FileHandle "> $self->{file_dest}";

	if (!defined $self->{file_handle})
	{
		$self->msn->_callback('FILE_RECEIVE_CANCEL', $self->{swb_session}, $self->{file_session},
				      "Couldn't write to $self->{file_dest}: $!");
		return undef;
	}
	$self->{file_handle}->autoflush(1);

	# Create the client
	$self->{_client} = Net::MsnMessenger::Connection->new(
		address         => $self->ip_address,
		port            => $self->port,
		connection_type => 'client',
		server_type     => 'FTP',
		protocol        => 'tcp',
		swb_session     => $self->swb_session,
		file_session    => $self->file_session,
		msn             => $self->msn,
	);

	$self->c_type('client');

	if (!$self->create_connection)
	{
		$self->{file_handle}->close;  unlink $self->{file_dest};
		return undef;
	}
	
	$self->send_packet('version', "MSNFTP", "\r\n");
	1;
}

# Net::MsnMessenger::File->create_server
sub create_server
{
	my $self = shift;

	# The File handle we are reading the file from
	$self->{file_handle} = new FileHandle "< $self->{file}";

	if (!defined $self->{file_handle})
	{
		$self->msn->_callback('FILE_SEND_CANCEL', $self->{swb_session}, $self->{file_session},
				      "Couldn't read $self->{file}: $!");
		return undef;
	}

	# Create the server
	$self->{_server} = Net::MsnMessenger::Connection->new(
		address         => $self->ip_address,
		port            => $self->port,
		connection_type => 'server',
		server_type     => 'FTP',
		protocol        => 'tcp',
		swb_session     => $self->swb_session,
		file_session    => $self->file_session,
		msn             => $self->msn,
	);

	$self->c_type('server');
	$self->create_connection or return undef;
	1;
}

# Net::MsnMessenger::File->_add_to_file
sub _add_to_file
{
	my ($self, $file_part) = @_;

	$self->file_handle->print($file_part);
	$self->{transfered} += length($file_part);

	$self->msn->_callback('FILE_RECEIVE_PROGRESS', $self->{swb_session}, $self->{file_session},
			      $self->transfered, $self->file_size);

	$self->_debug_inf("Currently transfered " . $self->transfered . " out of " . $self->file_size);

	# The whole file transfered
	if ($self->transfered >= $self->file_size)
	{
		$self->send_packet('leave', '16777989', "\r\n");
		$self->file_transfering(0);

		$self->msn->disconnect_file($self->{swb_session}, $self->{file_session});
		$self->msn->_callback('FILE_RECEIVE_SUCCESS', $self->{swb_session}, $self->{file_session});

		return 30;
	}
	1;
}

# Net::MsnMessenger::File->_handle_file
sub _handle_file
{
	my ($self, $buffer) = @_;

	if ($self->{_buffer_left})  # Continue receiving for the current file packet
	{
		if (length($buffer) > $self->{_buffer_left})
		{
			my $left = substr $buffer, 0, $self->{_buffer_left};
			my $next = substr $buffer, $self->{_buffer_left};

			my $old_buffer = $self->{_buffer};

			$self->{_buffer_left} = undef;
			$self->{_buffer} = undef;

			$self->_debug_inf("Going to handle ". (length($old_buffer) + length($left)) . " bytes. " .
					  "Left in the buffer " . length($next));

			$self->_handle_file($old_buffer . $left);
			$buffer = $next;
		}
		else
		{
			$self->{_buffer} .= $buffer;
			$self->{_buffer_left} -= length($buffer);

			$self->_debug_inf("Got another ".length($buffer)." bytes. Waiting for $self->{_buffer_left}");

			return 1;
		}
	}

	my $t_header = substr $buffer, 0, 3, '';
	my @header   = map {ord} split //, $t_header;

	$self->_debug_inf("Packet header: \"$header[0];$header[1];$header[2]\"");

	if ($header[0] eq 0)
	{
		# The length of the file part
		my $packet_len = $header[1] + (256 * $header[2]);
		$self->_debug_inf("File part length: $packet_len; Total buffer size: " . length($buffer));

		if (length($buffer) < $packet_len)
		{
			# Wait for the next packet
			$self->{_buffer} = $t_header . $buffer;
			$self->{_buffer_left} = $packet_len - length($buffer);

			$self->_debug_inf("Waiting for next $self->{_buffer_left} bytes for the current packet");
			return 1;
		}

		if (length($buffer) >= $packet_len)
		{
			$self->_add_to_file(substr $buffer, 0, $packet_len, '');
			$self->_debug_inf("Added $packet_len to the file. Left in the buffer: " . length($buffer));

			# 2048 constant should be safe here. The last packet will probably be smaller, but it
			# will be handled in a new _handle_file run

			while (length($buffer) >= 2048)
			{
				$self->_handle_file(substr $buffer, 0, 2048, '');
			}
			if (length($buffer))
			{
				return $self->_handle_file($buffer);  # A new packet
			}
		}
		return 1 if !length($buffer);
	}
	else
	{
		my $c_reason = ($header[0] eq 1)
		    ? "The sender cancelled the file transfer"
		    : "Invalid packet header. Cannot continue receiving the file";

		if ($header[0] ne 1)
		{
			# Mostly for debugging purposes, this mostly indicates a bug on the other side
			# -----
			# The file transfer wasn't cancelled by the server. We have to do it
			$self->send_packet('file_cancel');
		}

		$self->file_transfering(0);

		$self->msn->disconnect_file($self->{swb_session}, $self->{file_session});
		$self->msn->_callback('FILE_RECEIVE_CANCEL', $self->{swb_session},
				      $self->{file_session}, $c_reason);
		return undef;
	}
	1;
}

"Net::MsnMessenger::File";
__END__

