# -*- Mode: perl; perl-indent-level: 8; coding: utf-8 -*-
#
# Net::MsnMessenger::Event
#
# Copyright (C) 2003 <incoming@tiscali.cz>  All rights reserved.
# This module is free software; You can redistribute and/or modify it under
# the same terms as Perl itself.
#
# $Id: Event.pm,v 1.15 2003/07/15 21:34:05 incoming Exp $

package Net::MsnMessenger::Event;

=head1 NAME

Net::MsnMessenger::Event - A part of Net::MsnMessenger to send events

=head1 SYNPOSIS

  use Net::MsnMessenger;

  my $msn = Net::MsnMessenger->new;
  $msn->login;

  $msn->add_group("My new group");

=head1 DESCRIPTION

This class is responsible for sending events to the server/clients. It is not
to be used directly, but through a Net::MsnMessenger object as explained in the
SYNOPSIS.

=cut

use Net::MsnMessenger::Data;
use strict qw(subs vars);
use vars   qw($AUTOLOAD);

sub AUTOLOAD
{
	my $self = shift;
	my $name = $AUTOLOAD;
	$name =~ s/.*:://;
	return if $name =~ /DESTROY$/;

	if (exists $self->{$name})
	{
		$self->{$name} = shift if @_;
		return $self->{$name};
	}
	return $self->msn->$name(@_) if defined $self->msn && $self->msn->can($name);
	Carp::confess("AUTOLOAD: $name is not a valid method\n");
}

=head1 METHODS

This section documents the methods of Net::MsnMessenger::Event. Do not use this class directly,
instead call this methods using Net::MsnMessenger object as described in the SYNPOSIS.

=cut

# Net::MsnMessenger::Event->new
sub new
{
	my ($this, %args) = @_;
	my $self = {};

	$self->{msn} = $args{msn};

	bless $self, $this;
	return $self;
}

=head2 $msn->add_user ( PASSPORT, CONTACT_LIST [, GROUP_ID ] )

Add an user to a list. If CONTACTS_LIST is forward_list, you can optionally add the group ID.
Available contact lists are:

 * forward_list
 * allow_list
 * block_list

However if you want to add someone to the block list, consider using the block_user method.
Callbacks ADD_USER (for forward list) and ALLOW_USER (for allow list) confirm the success.

=cut

# Net::MsnMessenger::Event->add_user
sub add_user
{
	my ($self, $passport, $c_list, $group_id) = @_;
	$self->_check_connection || return undef;

	if (defined $passport)
	{
		$c_list ||= 'forward_list' if !exists $Contact_List{$c_list};  # Use forward list as a default

		$group_id = undef if $c_list ne 'forward_list';

		# The group of 0 is the default one, so use it if the specified one is invalid
		$group_id = 0 if $c_list eq 'forward_list' && !defined $self->get_group($group_id);

		my $r_user = $self->get_user($passport);
		my $fname = (defined $r_user && defined $r_user->fname && $r_user->fname ne $r_user->passport)
		    ? $r_user->fname : $passport;

		$self->msn->send_packet('add_user', $Contact_List{$c_list}, $passport, $fname, $group_id, "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->add_group ( GROUP_NAME )

Add a group to the forward list. The name of the group should be UTF-8 encoded.
Callback ADD_GROUP confirms the success.

=cut

# Net::MsnMessenger::Event->add_group
sub add_group
{
	my ($self, $group_name) = @_;
	$self->_check_connection || return undef;

	if (defined $group_name)
	{
		$self->msn->send_packet('add_group', $self->msn->url_encode($group_name), "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->block_user ( PASSPORT )

Block an user. Remove him from the allow list (if he is there) and add him to the block list.
Callback BLOCK_USER confirms the success.

=cut

# Net::MsnMessenger::Event->block_user
sub block_user
{
	my ($self, $passport) = @_;
	$self->_check_connection || return undef;

	if (defined $passport)
	{
		return undef if exists $self->msn->{_contact}->{$passport} &&
		    $self->msn->{_contact}->{$passport}->is_user_in_list('block_list');

		# Remove from the allow list
		$self->remove_user($passport, 'allow_list');

		# Add to the block list
		$self->add_user($passport, 'block_list');
		return 1;
	}
	undef;
}

=head2 $msn->can_use_hotmail ()

Return 1 if hotmail services (get_hotmail()) can be used.

=cut

# Net::MsnMessenger::Event->can_use_hotmail
sub can_use_hotmail
{
	my $self = shift;
	return ($self->msn->{hotmail}->{enabled}) ? 1 : 0;
}

=head2 $msn->change_friendly_name ( PASSPORT, NEW_NAME, URL_ENCODED )

Change a friendly name. As a passport can be also specified your own. The friendly name
should be encoded in UTF-8. URL_ENCODED flag specifies that the new friendly name is already
URL encoded. If you use your own passport, CHANGE_MY_FRIENDLY_NAME callback confirms the
success.

=cut

# Net::MsnMessenger::Event->change_friendly_name
sub change_friendly_name
{
	my ($self, $passport, $new_name, $encoded) = @_;
	$self->_check_connection || return undef;

	if (defined $new_name)
	{
		$new_name = $self->msn->url_encode($new_name) if !$encoded;

		$self->msn->send_packet('friendly_name', $passport, $new_name, "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->change_phone_number ( NUMBER_TYPE, NEW_NUMBER )

Change an own phone number, the number type can be (home, mobile, work). The new phone number
should be encoded in UTF-8.

=cut

# Net::MsnMessenger::Event->change_phone_number
sub change_phone_number
{
	my ($self, $number_type, $new_number) = @_;
	$self->_check_connection || return undef;

	$new_number = '' if !defined $new_number;

	if (defined $number_type && $number_type =~ /^(home|mobile|work)$/i)
	{
		$self->msn->send_packet('phone_list_own', $Command->{Contact_List}->{"phone_".lc($number_type)},
					$self->msn->url_encode($new_number), "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->change_status ( NEW_STATUS )

Change your own status. Status has to be one of the followings:
 * online
 * offline (disconnects you)
 * idle
 * away
 * busy
 * be_right_back
 * on_the_phone
 * out_to_lunch
 * appear_offline

The successful status change will be confirmed with CHANGE_MY_STATUS callback.

=cut

# Net::MsnMessenger::Event->change_status
sub change_status
{
	my ($self, $status) = (shift, lc shift);
	$self->_check_connection || return undef;

	if (defined $status && exists $Status{$status})
	{
		if ($status eq 'offline')
		{
			$self->msn->disconnect;
			$self->msn->_callback('DISCONNECT', "Status changed to offline");
		}
		else
		{
			$self->msn->send_packet('change_my_status', $Status{$status}, "\r\n");
		}
		return 1;
	}
	undef;
}

=head2 $msn->file_accept ( SWB_SESSION, FILE_SESSION, FILE_DESTINATION )

Accept a file transfer. The parameters are the switchboard and file session numbers and path
where to save the received file (directory).

=cut

# Net::MsnMessenger::Event->file_accept
sub file_accept
{
	my ($self, $swb_session, $file_session, $destination) = @_;

	if (defined $swb_session && defined $file_session && defined $destination)
	{
		$self->_check_connection($swb_session) || return undef;

		# The current file session
		my $this_file = $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session] if defined
		    $self->msn->{_swb}->[$swb_session] && defined
		    $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session];

		if (defined $this_file)
		{
			$this_file->file_dest_dir($destination);  # Set the destination directory
			$this_file->invite_accept;
			return 1;
		}
	}
	undef;
}

=head2 $msn->file_cancel ( SWB_SESSION, FILE_SESSION )

Cancel a file transfer. The parameters are the switchboard and file session numbers.

=cut

# Net::MsnMessenger::Event->file_cancel
sub file_cancel
{
	my ($self, $swb_session, $file_session) = @_;

	if (defined $swb_session && defined $file_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_file = $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session] if defined
		    $self->msn->{_swb}->[$swb_session] && defined
		    $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session];

		if (defined $this_file)
		{
			eval { $this_file->connected };

			if (!$@ && $this_file->connected && defined $this_file->c_type)
			{
				($this_file->c_type eq 'client')
				    ? $this_file->send_packet('file_cancel', "\r\n")  # Receiving
				    : $this_file->send_packet_file_cancel;            # Sending
			}

			else { $this_file->invite_cancel }
			return 1;
		}
	}
	undef;
}

=head2 $msn->file_invite ( SWB_SESSION, FILE_PATH )

Invite an user to the file transfer (send someone a file). You have to send a file
through a switchboard session. The parameters are the switchboard session number and
the path to the file to be sent.

=cut

# Net::MsnMessenger::Event->file_invite
sub file_invite
{
	my ($self, $swb_session, $file_path) = @_;

	if (defined $swb_session && defined $file_path && -f $file_path)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_swb = $self->msn->{_swb}->[$swb_session] if defined $self->msn->{_swb}->[$swb_session];

		if (defined $this_swb)
		{
			$this_swb->{_file}->[$this_swb->{_file_sessions}] = Net::MsnMessenger::File->new(
				file         => $file_path,
				file_size    => -s $file_path,
				swb_session  => $swb_session,
				file_session => $this_swb->{_file_sessions},
				outgoing     => 1,
				msn          => $self->msn,
			);

			$this_swb->{_file}->[$this_swb->{_file_sessions}]->invite;  # Send the initial invitation
			return $this_swb->{_file_sessions}++;
		}
	}
	undef;
}

=head2 $msn->file_leave_timeout ( SWB_SESSION, FILE_SESSION )

If you are sending a file to an user, and the whole file is already transfered but you
did not receive a confirmation from the client (FILE_SEND_SUCCESS), you can send this message
indicating a timeout. The parameters are the switchboard and file session numbers.

=cut

# Net::MsnMessenger::Event->file_leave_timeout
sub file_leave_timeout
{
	my ($self, $swb_session, $file_session) = @_;

	if (defined $swb_session && defined $file_session)
	{
		my $this_file = $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session] if defined
		    $self->msn->{_swb}->[$swb_session] && defined
		    $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session];

		if (defined $this_file)
		{
			$this_file->invite_fttimeout;
			return 1;
		}
	}
	undef;
}

=head2 $msn->file_reject ( SWB_SESSION, FILE_SESSION )

Reject a file transfer. The parameters are the switchboard and file session numbers.

=cut

# Net::MsnMessenger::Event->file_reject
sub file_reject
{
	my ($self, $swb_session, $file_session) = @_;

	if (defined $swb_session && defined $file_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_file = $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session] if defined
		    $self->msn->{_swb}->[$swb_session] && defined
		    $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session];

		if (defined $this_file)
		{
			$this_file->invite_reject;
			return 1;
		}
	}
	undef;
}

=head2 $msn->find_member ( F_NAME, L_NAME [, COUNTRY, STATE, CITY ] )

Find a member in the hotmail member directory. The parameters are the first name, the
last name (these two are required), optionally the country, state and city.
After this you should receive FIND_RESULTS callback with the results.

=cut

# Net::MsnMessenger::Event->find_member
sub find_member
{
	my ($self, $fname, $lname, $country, $state, $city) = @_;

	if (defined $fname && defined $lname)
	{
		$country = '*' if !defined $country;
		$state   = '*' if !defined $state;
		$city    = '*' if !defined $city;

		$state = $self->msn->url_decode($state);
		$city = $self->msn->url_decode($city);

		$self->msn->send_packet('find', "fname=$fname", "lname=$lname", "city=$city", "state=$state",
					"country=" . uc($country), "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->find_member_invite ( FIND_MATCH, MESSAGE )

When you find a person using the member search, you can send an e-mail to this person to
add you to his/her contact list using this method. The first parameter is the number of the
match (counted from 1) from the FIND_RESULTS callback, and the second parameter is the message
that you want to have included with the e-mail. The FIND_INVITE_SUCCESS callback confirms the
success.

=cut

# Net::MsnMessenger::Event->find_member_invite
sub find_member_invite
{
	my ($self, $find_match, $message) = @_;

	if (defined defined $find_match)
	{
		$message = ' ' if !defined $message;
		my $LCID = (defined $self->msn->LCID) ? sprintf "0x%x", $self->msn->LCID : '0x409';
		my $length = length $message;

		# This command's syntax *is* stupid
		$self->msn->send_packet('find_invite', $find_match, $LCID, 'MSMSGS', 'WindowsMessenger', 'X',
					'X', $self->msn->fname, $length, "\r\n", $message);
		return 1;
	}
	undef;
}

=head2 $msn->get_connections ()

Return an array of hash references with all the connected sockets. A hash refrence is in the
following format:

 * $conn[$number]->{connected}        - 1 if connected (always 1 here)
 * $conn[$number]->{connection_type}  - server or client
 * $conn[$number]->{server_type}      - DS/NS/SS/FTP/Voice
 * $conn[$number]->{protocol}         - TCP/UDP
 * $conn[$number]->{host}             - Hostname or IP Address
 * $conn[$number]->{port}             - Host port

=cut

# Net::MsnMessenger::Event->get_connections
sub get_connections
{
	my $self = shift;
	my @connections = ();
	$self->_check_connection || return undef;

	if ($self->msn->{_handle})
	{
		for (@{$self->msn->{_handle}})
		{
			push @connections,
			{ connected => $_->connected, connection_type => $_->{_connhash}->{con_type},
			  server_type => $_->{_connhash}->{srv_type}, protocol => $_->{_connhash}->{proto},
			  host => $_->{_connhash}->{address}, port => $_->{_connhash}->{port},
		        };
		}
	}
	@connections;
}

=head2 $msn->get_file_send_user ( SWB_SESSION, FILE_SESSION )

This method returns the passport of the user you are sending a file to. The file session has
to be established for this to work. The parameters are the switchboard and file session
numbers.

=cut

# Net::MsnMessenger::Event->get_file_send_user
sub get_file_send_user
{
	my ($self, $swb_session, $file_session) = @_;

	if (defined $swb_session && defined $file_session)
	{
		my $this_file = $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session];

		return $this_file->sending_to if defined $this_file && defined $this_file->sending_to;
	}
	undef;
}

=head2 $msn->get_last_packet_file ( SWB_SESSION, FILE_SESSION )

Return the unix time of the last transfered packet in a file session. The parameters are
the switchboard and file session numbers.

=cut

# Net::MsnMessenger::Event->get_last_packet_file
sub get_last_packet_file
{
	my ($self, $swb_session, $file_session) = @_;

	if (defined $swb_session && defined $file_session)
	{
		my $this_file = $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session];

		if (defined $this_file)
		{
			# If it was not yet trying to connect we would get an error from AUTOLOAD, so eval here
			eval { $this_file->connected };
			return -1 if $@ || (!$@ && !$this_file->connected);

			eval { $this_file->last_packet };
			return
			    (!$@ && defined $this_file->last_packet) ? $this_file->last_packet : undef;
		}
		return -1;
	}
	undef;
}

=head2 $msn->get_last_packet_swb ( SWB_SESSION )

Return the unix time of the last transfered packet in a switchboard session. The parameter
is the switchboard session number.

=cut

# Net::MsnMessenger::Event->get_last_packet_swb
sub get_last_packet_swb
{
	my ($self, $swb_session) = @_;

	if (defined $swb_session)
	{
		my $this_swb = $self->msn->{_swb}->[$swb_session] if defined $self->msn->{_swb}->[$swb_session];

		if (defined $this_swb)
		{
			eval { $this_swb->connected };
			return -1 if $@ || (!$@ && !$this_swb->connected);

			eval { $this_swb->last_packet };
			return (!$@ && defined $this_swb->last_packet) ? $this_swb->last_packet : undef;
		}
		return -1;
	}
	undef;
}

# Net::MsnMessenger::Event->get_last_packet_voice
sub get_last_packet_voice
{
	my ($self, $swb_session, $voice_session) = @_;

	if (defined $swb_session && defined $voice_session)
	{
		my $this_voice = $self->msn->{_swb}->[$swb_session]->{_voice}->[$voice_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_voice}->[$voice_session];

		if (defined $this_voice)
		{
			eval { $this_voice->connected };
			return -1 if $@ || (!$@ && !$this_voice->connected);

			eval { $this_voice->last_packet };
			return (!$@ && defined $this_voice->last_packet) ? $this_voice->last_packet : undef;
		}
	}
	undef;
}

=head2 $msn->get_hotmail ( URL_TYPE [, PARAMETERS ] )

Request a hotmail URL (form a page). The type of the URL can be one of the followings:

 * ADDRBOOK  - Hotmail address book
 * CHAT      - MSN Chat
 * CHGMOB    - Mobile settings
 * COMPOSE   - E-mail composition (can have a parameter)
 * FOLDERS   - Hotmail folders
 * PERSON    - Member services
 * PROFILE   - Edit your profile

The parameter can be an e-mail address if you are using COMPOSE.
You will receive URL callback as a reply to this.

=cut

# Net::MsnMessenger::Event->get_hotmail
sub get_hotmail
{
	my ($self, $url_type, @params) = (shift, uc(shift), @_);

	return undef if !defined $self->msn->passport || $self->msn->passport !~ /\@(hotmail|msn)\.com$/i;
	$self->_check_connection || return undef;

	if (!$self->can_use_hotmail)
	{
		$self->msn->error("The e-mail support is not available for this account");
		return undef;
	}

	if (defined $url_type && $url_type =~ /^(ADDRBOOK|CHGMOB|PROFILE|PERSON|CHAT|INBOX|COMPOSE|FOLDERS)$/)
	{
		if ($url_type eq 'PROFILE' || $url_type eq 'PERSON' || $url_type eq 'CHAT')
		{
			@params = (defined $self->msn->LCID) ? sprintf "0x%x", $self->msn->LCID : '0x409';
		}

		$self->msn->send_packet('url', $url_type, @params, "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->get_swb_users ( SWB_SESSION )

Return an array of contacts (every contacts is a Net::MsnMessenger::Contact object) in
the specified switchboard session (the only parameter).

=cut

# Net::MsnMessenger::Event->get_swb_users
sub get_swb_users
{
	my ($self, $swb_session) = @_;

	if (defined $swb_session)
	{
		return $self->msn->{_swb}->[$swb_session]->get_users if defined $self->msn->{_swb}->[$swb_session];
	}
	return ();
}

=head2 $msn->invite_switchboard ( PASSPORT [, SWB_SESSION ] )

Invite an user a the switchboard (chat) session. The first parameter is the passport and the
second (optional) is the switchboard session number (use this if inviting someone to already
established session - multiuser chat).

=cut

# Net::MsnMessenger::Event->invite_switchboard
sub invite_switchboard
{
	my ($self, $passport, $swb_session) = @_;

	(defined $swb_session)
	    ? $self->_check_connection($swb_session) || return undef
	    : $self->_check_connection || return undef;

	if (defined $passport)
	{
		if (!defined $swb_session)
		{
			push @{$self->msn->{_inviting}}, $passport;
			$self->msn->send_packet('redirect', 'SB', "\r\n");
			return 1;
		}

		$self->msn->{_swb}->[$swb_session]->send_packet('invite', $passport, "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->is_connected_file ( SWB_SESSION, FILE_SESSION )

Return 1 if a file session is established. The parameters are the switchboard and file
session numbers.

=cut

# Net::MsnMessenger::Event->is_connected_file
sub is_connected_file
{
	my ($self, $swb_session, $file_session) = @_;

	if (defined $swb_session && defined $file_session && defined $self->msn->{_swb}->[$swb_session] &&
	    defined $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session])
	{
		eval { $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session]->connected };
		return
		    ($@) ? undef : $self->msn->{_swb}->[$swb_session]->{_file}->[$file_session]->connected;
	}
	undef;
}

=head2 $msn->is_connected_swb ( SWB_SESSION )

Return 1 if a switchboard session (the only parameter) is established.

=cut

# Net::MsnMessenger::Event->is_connected_swb
sub is_connected_swb
{
	my ($self, $swb_session) = @_;

	if (defined $swb_session && defined $self->msn->{_swb}->[$swb_session])
	{
		eval { $self->msn->{_swb}->[$swb_session]->connected };
		return
		    ($@) ? undef : $self->msn->{_swb}->[$swb_session]->connected;
	}
	undef;
}

# Net::MsnMessenger::Event->is_connected_voice
sub is_connected_voice
{
	my ($self, $swb_session, $voice_session) = @_;

	if (defined $swb_session && defined $voice_session && defined $self->msn->{_swb}->[$swb_session] &&
	    defined $self->msn->{_swb}->[$swb_session]->{_voice}->[$voice_session])
	{
		eval { $self->msn->{_swb}->[$swb_session]->{_voice}->[$voice_session]->connected };
		return
		    ($@) ? undef : $self->msn->{_swb}->[$swb_session]->{_voice}->[$voice_session]->connected;
	}
	undef;
}

=head2 $msn->leave_switchboard ( SWB_SESSION )

Leave (disconnect) a switchboard session (the only parameter).

=cut

# Net::MsnMessenger::Event->leave_switchboard
sub leave_switchboard
{
	my ($self, $swb_session) = @_;

	if (defined $swb_session)
	{
		$self->_check_connection($swb_session) || return undef;
		$self->msn->disconnect_swb($swb_session);
		$self->msn->_callback('CLOSE_CHAT', $swb_session);
		return 1;
	}
	undef;
}

=head2 $msn->move_to_group ( PASSPORT, SRC_GROUP_ID, DEST_GROUP_ID )

Move an user in the forward list to a different group. The parameters are the passport,
identification number of the source and destination group.

=cut

# Net::MsnMessenger::Event->move_to_group
sub move_to_group
{
	my ($self, $passport, $g_from, $g_to) = @_;
	$self->_check_connection || return undef;

	if (defined $passport && defined $g_from && defined $g_to)
	{
		my $r_user = $self->get_user($passport);

		return undef if !defined $r_user || !$r_user->is_user_in_list('forward_list');
		return undef if !defined $self->get_group($g_from) || !defined $self->get_group($g_to);

		$self->add_user($passport, 'forward_list', $g_to) || return undef;
		$self->remove_user($passport, 'forward_list', $g_from) || return undef;
		return 1;
	}
	undef;
}

=head2 $msn->netmeeting_accept ( SWB_SESSION, NETMEETING_SESSION )

Accept a netmeeting invitation. The parameters are the switchboard and netmeeting session
numbers.

=cut

# Net::MsnMessenger::Event->netmeeting_accept
sub netmeeting_accept
{
	my ($self, $swb_session, $netmeeting_session) = @_;

	if (defined $swb_session && defined $netmeeting_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_netmeeting = $self->msn->{_swb}->[$swb_session]->{_netmeeting}->[$netmeeting_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_netmeeting}->[$netmeeting_session];

		if (defined $this_netmeeting)
		{
			$this_netmeeting->invite_accept;
			return 1;
		}
	}
	undef;
}

=head2 $msn->netmeeting_cancel ( SWB_SESSION, NETMEETING_SESSION )

Cancel a netmeeting session (invitation). The parameters are the switchboard and netmeeting
session numbers.

=cut

# Net::MsnMessenger::Event->netmeeting_cancel
sub netmeeting_cancel
{
	my ($self, $swb_session, $netmeeting_session) = @_;

	if (defined $swb_session && defined $netmeeting_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_netmeeting = $self->msn->{_swb}->[$swb_session]->{_netmeeting}->[$netmeeting_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_netmeeting}->[$netmeeting_session];

		if (defined $this_netmeeting)
		{
			$this_netmeeting->invite_cancel;
			return 1;
		}
	}
	undef;
}

=head2 $msn->netmeeting_invite ( SWB_SESSION )

Invite an user to use netmeeting or a similar application (gnomemeeting). The parameter is the
switchboard session number.

=cut

# Net::MsnMessenger::Event->netmeeting_invite
sub netmeeting_invite
{
	my ($self, $swb_session) = @_;

	if (defined $swb_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_swb = $self->msn->{_swb}->[$swb_session];

		$this_swb->{_netmeeting}->[$this_swb->{_netmeeting_sessions}] = Net::MsnMessenger::NetMeeting->new(
			swb_session        => $swb_session,
			netmeeting_session => $this_swb->{_netmeeting_sessions},
			outgoing           => 1,
			msn                => $self->msn,
		);

		$this_swb->{_netmeeting}->[$this_swb->{_netmeeting_sessions}]->invite; # Send the initial invitation
		return $this_swb->{_netmeeting_sessions}++;
	}
	undef;
}

=head2 $msn->netmeeting_reject ( SWB_SESSION, NETMEETING_SESSION )

Reject a netmeeting invitation. The parameters are the switchboard an netmeeting session
numbers.

=cut

# Net::MsnMessenger::Event->netmeeting_reject
sub netmeeting_reject
{
	my ($self, $swb_session, $netmeeting_session) = @_;

	if (defined $swb_session && defined $netmeeting_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_netmeeting = $self->msn->{_swb}->[$swb_session]->{_netmeeting}->[$netmeeting_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_netmeeting}->[$netmeeting_session];

		if (defined $this_netmeeting)
		{
			$this_netmeeting->invite_reject;
			return 1;
		}
	}
	undef;
}

=head2 $msn->ping ()

Ping the server. As a reply you will receive PONG (callback).

=cut

# Net::MsnMessenger::Event->ping
sub ping
{
	my $self = shift;
	$self->_check_connection || return undef;

	$self->msn->send_packet('ping', "\r\n");
	1;
}

# Net::MsnMessenger::Event->ra_accept
sub ra_accept
{
	my ($self, $swb_session, $ra_session) = @_;

	if (defined $swb_session && defined $ra_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_ra = $self->msn->{_swb}->[$swb_session]->{_rassistance}->[$ra_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_rassistance}->[$ra_session];


		if (defined $this_ra)
		{
			$this_ra->invite_accept;
			return 1;
		}
	}
	undef;
}

# Net::MsnMessenger::Event->ra_cancel
sub ra_cancel
{
	my ($self, $swb_session, $ra_session) = @_;

	if (defined $swb_session && defined $ra_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_ra = $self->msn->{_swb}->[$swb_session]->{_rassistance}->[$ra_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_rassistance}->[$ra_session];

		if (defined $this_ra)
		{
			$this_ra->invite_cancel;
			return 1;
		}
	}
	undef;
}

# Net::MsnMessenger::Event->ra_invite
sub ra_invite
{
	my ($self, $swb_session) = @_;

	if (defined $swb_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_swb = $self->msn->{_swb}->[$swb_session];

		$this_swb->{_rassistance}->[$this_swb->{_rassistance_sessions}] =
		    Net::MsnMessenger::RemoteAssistance->new(
			swb_session         => $swb_session,
			rassistance_session => $this_swb->{_rassistance_sessions},
			outgoing            => 1,
			msn                 => $self->msn,
		);

		$this_swb->{_rassistance}->[$this_swb->{_rassistance_sessions}++]->invite;
		return $this_swb->{_rassistance_sessions}++;
	}
	undef;
}

# Net::MsnMessenger::Event->ra_reject
sub ra_reject
{
	my ($self, $swb_session, $ra_session) = @_;

	if (defined $swb_session && defined $ra_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_ra = $self->msn->{_swb}->[$swb_session]->{_rassistance}->[$ra_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_rassistance}->[$ra_session];

		if (defined $this_ra)
		{
			$this_ra->invite_reject;
			return 1;
		}
	}
	undef;
}

=head2 $msn->reject_not_installed ( SWB_SESSION, TYPE, TYPE_SESSION )

Reject an invitation as it is not installed. All unknown invitations are rejected this way but
you can also explicitelly reject a known invitation. The paramters are the switchboard session,
invitation type (file, netmeeting, voice, rassistance), and the session of the invitation type.

=cut

# Net::MsnMessenger::Event->reject_not_installed
sub reject_not_installed
{
	my ($self, $swb_session, $type, $type_session) = @_;
	my $h_name;

	if (defined $swb_session && defined $type && defined $type_session)
	{
		$h_name = '_file' if lc($type) eq 'file';
		$h_name = '_voice' if lc($type) eq 'voice';
		$h_name = '_netmeeting' if lc($type) eq 'netmeeting';
		$h_name = '_rassistance' if lc($type) eq 'remote_assistance';

		if (!defined $h_name)
		{
			$self->msn->error("Unknown invitation type: $type");
			return undef;
		}

		my $obj_ref = $self->msn->{_swb}->[$swb_session]->{$h_name}->[$type_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{$h_name}->[$type_session];

		if (defined $obj_ref)
		{
			(defined $obj_ref->session_id)
			    ? $self->_reject_ni_cookie($swb_session, $obj_ref->invitation_cookie,
						       $obj_ref->session_id)
			    : $self->_reject_ni_cookie($swb_session, $obj_ref->invitation_cookie);
			return 1;
		}
	}
	undef;
}

=head2 $msn->remove_group ( GROUP_ID )

Remove a group from the contact list. Notice that if you remove a group you will also
remove all the users inside the group.

=cut

# Net::MsnMessenger::Event->remove_group
sub remove_group
{
	my ($self, $group_id) = @_;
	$self->_check_connection || return undef;

	if (defined $group_id && exists $self->msn->{_group}->{$group_id})
	{
		$self->msn->send_packet('remove_group', $group_id, "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->remove_user ( PASSPORT, CONTACT_LIST [, GROUP_ID ] )

Remove an user from a contact list. The parameters are the passport, contact list and
optionally the group identification number (for forward list).

=cut

# Net::MsnMessenger::Event->remove_user
sub remove_user
{
	my ($self, $passport, $c_list, $group_id) = @_;
	$self->_check_connection || return undef;

	if (defined $passport)
	{
		$c_list = 'forward_list' if !defined $c_list || !exists $Contact_List{$c_list};

		my $r_user = $self->get_user($passport);
		return undef if !defined $r_user || !$r_user->is_user_in_list($c_list);

		(defined $group_id && defined $self->get_group($group_id))
		    ? $self->msn->send_packet('remove_user', $Contact_List{$c_list}, $passport, $group_id, "\r\n")
		    : $self->msn->send_packet('remove_user', $Contact_List{$c_list}, $passport, "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->rename_group ( GROUP_ID, NEW_NAME )

Rename a group. The parameters are the group identification number and the new name
for the group (should be encoded in UTF-8).

=cut

# Net::MsnMessenger::Event->rename_group
sub rename_group
{
	my ($self, $group_id, $new_name) = @_;
	$self->_check_connection || return undef;

	if (defined $group_id && defined $new_name)
	{
		$self->msn->send_packet('rename_group', $group_id, $self->msn->url_encode($new_name), 0, "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->send_clientcaps ( SWB_SESSION, %ATTRIBUTES )

Send the text/x-clientcaps message to a switchboard session. This message should be
only send at a start of a switchboard session. The parameters are the switchboard session
number and message attributes. The attributes can be:

 * client_name   => The name and version of your client
 * chat_logging  => Y - chat logging enabled, N - disabled, S - secure
 * buddy_icons   => 1 - buddy icons available

=cut

# Net::MsnMessenger::Event->send_clientcaps
sub send_clientcaps
{
	my ($self, $swb_session, %attributes) = @_;

	if (defined $swb_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_swb = $self->msn->{_swb}->[$swb_session];
		my $r_message = Net::MsnMessenger::Message->new;

		$this_swb->send_packet('message', $r_message->create_clientcaps(%attributes));
		return 1;
	}
	undef;
}

=head2 $msn->send_email_invitation ( EMAIL )

Send an e-mail inviting a person to install MSN Messenger and add you to his/her contact list.
The parameters is the e-mail address of the person. The EMAIL_INVITE_SUCCESS callback confirms
the success.

=cut

# Net::MsnMessenger::Event->send_email_invitation
sub send_email_invitation
{
	my ($self, $email) = @_;
	$self->_check_connection || return undef;

	if (defined $email)
	{
		my $LCID = (defined $self->msn->LCID)? sprintf("0x%x", $self->msn->LCID) : '0x409';

		$self->msn->send_packet('send_invitation', $email, $LCID, 'MSMSGS', 'MSMSGS', "\r\n");
		return 1;
	}
	undef;
}

=head2 $msn->send_message ( SWB_SESSION, MESSAGE, %ATTRIBUTES )

Send a classic message to a switchboard session. The paramters are the switchboard
session number, the message text (should be UTF-8 encoded) and the attributes:

  * EF =>  The font effect (B - bold, I - italic, U - underline)
  * FN =>  The URL encoded font name
  * CO =>  The rgb value of the font color (RRRGGGBBB)
  * CS =>  The number of the charset
  * PF =>  Pitch & Family
  * RL =>  Right-to-Left typing (1 if enabled)

=cut

# Net::MsnMessenger::Event->send_message
sub send_message
{
	my ($self, $swb_session, $message, %attributes) = @_;

	if (defined $swb_session && defined $message)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_swb = $self->msn->{_swb}->[$swb_session];
		my $r_message = Net::MsnMessenger::Message->new(message => $message);

		if (defined $attributes{FN})
		{
			$attributes{FN} = $self->msn->url_encode($attributes{FN});
		}

		while (defined $r_message->message)
		{
			$this_swb->send_packet('message', $r_message->create_switchboard(%attributes));

			$this_swb->{outgoing}->{$this_swb->trans_id - 1}->{message} = $r_message->just_sent;
			$this_swb->{outgoing}->{$this_swb->trans_id - 1}->{delivered} = 0;

			$self->_callback('SEND_MESSAGE_SENT', $swb_session, $r_message->just_sent);
		}
		return 1;
	}
	undef;
}

=head2 $msn->send_pager ( PASSPORT, MESSAGE )

Send a pager (cell phone) message to a user. The user must be in your forward list and
be signed in to MSN Mobile (see Net::MsnMessenger::Contact manual). The parameters are the
switchboard session number and the message text (should be UTF-8 encoded).

=cut

# Net::MsnMessenger::Event->send_pager
sub send_pager
{
	my ($self, $passport, $number_type, $message) = @_;
	$self->_check_connection || return undef;

	my $r_message = Net::MsnMessenger::Message->new(message => $self->msn->xml_encode($message));
	my %attributes;

	if (defined $passport && defined $message)
	{
		my $real_user = $self->get_user($passport);

		if (!defined $real_user || $real_user->mobile_device ne 'Y')
		{
			$self->msn->error("User is not on your contact list or not using MSN Mobile");
			return undef;
		}
		$attributes{passport} = $real_user->passport;

		if (defined $number_type)
		{
			my $number = $self->get_phone_own(lc $number_type);
			if (defined $number)
			{
				$attributes{number_type} = lc $number_type;
				$attributes{number} = $number;
			}
		}

		while (defined $r_message->message)
		{
			$self->msn->send_packet('pager', $real_user->passport, $r_message->create_pager(%attributes));
		}
		return 1;
	}
	undef;
}

=head2 $msn->send_typing_user ( SWB_SESSION )

Tell the users in the switchboard session that you are typing a message. This message should be
sent every 5 seconds while you are typing. The parameter is the switchboard session number.

=cut

# Net::MsnMessenger::Event->send_typing_user
sub send_typing_user
{
	my ($self, $swb_session) = @_;

	if (defined $swb_session)
	{
		$self->_check_connection($swb_session) || return undef;
		my $this_swb = $self->msn->{_swb}->[$swb_session];
		my %attributes;

		$attributes{typinguser} = $self->msn->passport;

		my $r_message = Net::MsnMessenger::Message->new;
		$this_swb->send_packet('message', $r_message->create_control(%attributes));
		return 1;
	}
	undef;
}

=head2 $msn->unblock_user ( PASSPORT )

Unblock a user - remove from the allow list (if there) and add to the block list. The parameter
is the switchboard session number.

=cut

# Net::MsnMessenger::Event->unblock_user
sub unblock_user
{
	my ($self, $passport) = @_;
	$self->_check_connection || return undef;

	if (defined $passport && exists $self->msn->{_contact}->{$passport})
	{
		return undef if !$self->msn->{_contact}->{$passport}->is_user_in_list('block_list') ||
		    $self->msn->{_contact}->{$passport}->is_user_in_list('allow_list');

		$self->remove_user($passport, 'block_list');
		$self->add_user($passport, 'allow_list');
		return 1;
	}
	undef;
}

# Net::MsnMessenger::Event->voice_accept
sub voice_accept
{
	my ($self, $swb_session, $voice_session) = @_;

	if (defined $swb_session && defined $voice_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_voice = $self->msn->{_swb}->[$swb_session]->{_voice}->[$voice_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_voice}->[$voice_session];

		if (defined $this_voice)
		{
			$this_voice->invite_accept;
			return 1;
		}
	}
	undef;
}

# Net::MsnMessenger::Event->voice_cancel
sub voice_cancel
{
	my ($self, $swb_session, $voice_session) = @_;

	if (defined $swb_session && defined $voice_session)
	{
		$self->_check_connection($swb_session) || return undef;
		my $this_voice = $self->msn->{_swb}->[$swb_session]->{_voice}->[$voice_session] if
		    defined $self->{_swb}->[$swb_session] &&
		    defined $self->{_swb}->[$swb_session]->{_voice}->[$voice_session];

		if (defined $this_voice)
		{
			if ($this_voice->connected)
			{
				# TODO: Finish
			}
			else { $this_voice->invite_cancel }
			return 1;
		}
	}
	undef;
}

# Net::MsnMessenger::Event->voice_invite
sub voice_invite
{
	my ($self, $swb_session) = @_;

	if (defined $swb_session)
	{
		$self->_check_connection($swb_session) || return undef;
		my $this_swb = $self->msn->{_swb}->[$swb_session];

		$this_swb->{_voice}->[$this_swb->{_voice_sessions}] = Net::MsnMessenger::Voice->new(
			swb_session   => $swb_session,
			voice_session => $this_swb->{_voice_sessions},
			outgoing      => 1,
			msn           => $self->msn,
		);

		$this_swb->{_voice}->[$this_swb->{_voice_sessions}++]->invite;  # Send the initial invitation
		return $this_swb->{_voice_sessions}++;
	}
	undef;
}

# Net::MsnMessenger::Event->voice_reject
sub voice_reject
{
	my ($self, $swb_session, $voice_session) = @_;

	if (defined $swb_session && defined $voice_session)
	{
		$self->_check_connection($swb_session) || return undef;

		my $this_voice = $self->msn->{_swb}->[$swb_session]->{_voice}->[$voice_session] if
		    defined $self->msn->{_swb}->[$swb_session] &&
		    defined $self->msn->{_swb}->[$swb_session]->{_voice}-[$voice_session];

		if (defined $this_voice)
		{
			$this_voice->invite_reject;
			return 1;
		}
	}
	undef;
}

# ---------- Methods to return parts of the contact lists ---------- #

=head1 METHODS THAT RETURN PARTS OF THE CONTACT LISTS

=head2 $msn->get_all_groups ()

Return an array with all the groups (Net::MsnMessenger::Group objects).

=cut

# Net::MsnMessenger::Event->get_all_groups
sub get_all_groups
{
	my $self = shift;
	my @groups = ();

	if ($self->msn->{_group})
	{
		push @groups, $self->msn->{_group}->{$_} for sort keys %{$self->msn->{_group}};
	}
	return @groups;
}

=head2 $msn->get_all_users ()

Return an array with all the users in all the contact lists (Net::MsnMessenger::Contact
objects).

=cut

# Net::MsnMessenger::Event->get_all_users
sub get_all_users
{
	my $self = shift;
	my @users = ();

	if ($self->msn->{_contact})
	{
		push @users, $self->msn->{_contact}->{$_} for sort keys %{$self->msn->{_contact}};
	}
	return @users;
}

=head2 $msn->get_group ( GROUP_ID )

Return a Net::MsnMessenger::Group object for the given group identification number.

=cut

# Net::MsnMessenger::Event->get_group
sub get_group
{
	my ($self, $group_id) = @_;
	my $r_group;

	if (defined $group_id)
	{
		for (sort keys %{$self->msn->{_group}})
		{
			$r_group = $self->msn->{_group}->{$_} if $self->msn->{_group}->{$_}->ID eq $group_id;
		}
	}
	return $r_group;
}

=head2 $msn->get_group_id_by_name ( GROUP_NAME )

Return the group identification number for the name of the group. The name should be URL and
UTF-8 encoded.

=cut

# Net::MsnMessenger::Event->get_group_id_by_name
sub get_group_id_by_name
{
	my ($self, $g_name) = @_;
	my $g_id;

	if (defined $g_name)
	{
		for (sort keys %{$self->msn->{_group}})
		{
			$g_id = $self->msn->{_group}->{$_}->ID if $self->msn->{_group}->{$_}->name eq $g_name;
		}
	}
	return $g_id;
}

=head2 $msn->get_phone ( PASSPORT [, NUMBER_TYPE ] )

Return a hash reference with phone numbers. The first parameter is the passport of a user,
and the second (optional one) is the type of the number (returns *only* the number).
All the numbers are URL and UTF-8 encoded.
Number types are (home, mobile, work).

=cut

# Net::MsnMessenger::Event->get_phone
sub get_phone
{
	my ($self, $passport, $number_type) = @_;
	$number_type = lc($number_type) if defined $number_type;

	if (defined $passport && defined $self->msn->{_contact} && defined $self->msn->{_contact}->{phone})
	{
		if (defined $number_type)
		{
			return undef if $number_type !~ /^(home|mobile|work)$/;
			return $self->{_contact}->{$passport}->{phone}->{$number_type} || undef;
		}
		return $self->{_contact}->{$passport}->{phone};
	}
	undef;
}

=head2 $msn->get_phone_own ( [ NUMBER_TYPE ] )

Same like above, but work with your own phone numbers.

=cut

# Net::MsnMessenger::Event->get_phone_own
sub get_phone_own
{
	my ($self, $number_type) = @_;
	$number_type = lc($number_type) if defined $number_type;

	if (defined $number_type)
	{
		return undef if lc($number_type) !~ /^(home|mobile|work)$/;
		return $self->msn->phone->{$number_type} || undef;
	}
	return $self->msn->phone;
}

=head2 $msn->get_user ( PASSPORT )

Return a Net::MsnMessenger::Contact object for the given passport of a contact.

=cut

# Net::MsnMessenger::Event->get_user
sub get_user
{
	my ($self, $passport) = @_;

	if (defined $passport)
	{
		return $self->msn->{_contact}->{$passport} if defined $self->msn->{_contact}->{$passport};
	}
	return undef;
}

=head2 $msn->get_users_group ( GROUP_ID )

Return an array with all the users in the given group (Net::MsnMessenger::Contact
objects).

=cut

# Net::MsnMessenger::Event->get_users_group
sub get_users_group
{
	my ($self, $group_id) = @_;
	my @users;

	if (defined $group_id && defined $self->msn->{_group}->{$group_id})
	{
		@users = map {$self->get_user($_)} @{$self->msn->{_group}->{$group_id}->{_users_list}};
	}
	return (@users) ? @users : ();
}

=head2 $msn->get_users_list ( CONTACT_LIST )

Return an array with all the users in the given contact list (Net::MsnMessenger::Contact
objects).

=cut

# Net::MsnMessenger::Event->get_users_list
sub get_users_list
{
	my ($self, $c_list) = @_;
	my @users;

	if (defined $c_list)
	{
		for (sort keys %{$self->msn->{_contact}})
		{
			push @users, $self->msn->{_contact}->{$_} if
			    $self->msn->{_contact}->{$_}->is_user_in_list($c_list);
		}
	}
	return (@users) ? @users : ();
}

# -------------------- Private Methods -------------------- #

# Net::MsnMessenger::Event->_check_connection
sub _check_connection
{
	my ($self, $swb_session) = @_;
	return undef if defined $swb_session && $swb_session !~ /^\d+$/;

	my $c = (defined $swb_session) ? (defined $self->msn->{_swb}->[$swb_session])
	    ? $self->msn->{_swb}->[$swb_session]->connected : 0 : $self->msn->connected;

	if (!$c)
	{
		(defined $swb_session)
		    ? $self->msn->error("Not connected to the switchboard server")
		    : $self->msn->error("Not connected to the server");
		return undef;
	}
	1;
}

# Net::MsnMessenger::Event->_get_hotmail_page
sub _get_hotmail_page
{
	my ($self, $param1, $param2, $param3) = @_;
	my $temp_file;

	require File::Spec;
	require FileHandle;

	while (!defined $temp_file || -f $temp_file)
	{
		$temp_file = File::Spec->catfile(File::Spec->tmpdir(), 'm_hotmail_' . int(rand 9999) . '.html');
	}
	my $temp_file_fh = new FileHandle "> $temp_file";

	if (!defined $temp_file_fh)
	{
		$self->msn->error("Couldn't write to $temp_file: $!");
		return undef;
	}

	# Delay calculating the sl and creds fields as it's (probably) required that they are up-to-date
	my $sl = int (time - $self->msn->{hotmail}->{logintime});
	my $creds = Digest::MD5::md5_hex($self->msn->{hotmail}->{mspauth} . $sl . $self->msn->password);

	print $temp_file_fh <<EOF;

<html>
 <head>
  <noscript>
   <meta http-equiv=Refresh content=\"0; url=http://www.hotmail.com\">
  </noscript>
 </head>

 <body onload=\"document.pform.submit(); \">
  <form name=\"pform\" action=\"$param2\" method=\"POST\">
   <input type=\"hidden\" name=\"mode\" value=\"ttl\">
   <input type=\"hidden\" name=\"login\" value=\"$self->{msn}->{hotmail}->{login}\">
   <input type=\"hidden\" name=\"username\" value=\"$self->{msn}->{hotmail}->{username}\">
   <input type=\"hidden\" name=\"sid\" value=\"$self->{msn}->{hotmail}->{sid}\">

   <input type=\"hidden\" name=\"kv\" value=\"$self->{msn}->{hotmail}->{kv}\">
   <input type=\"hidden\" name=\"id\" value=\"$param3\">
   <input type=\"hidden\" name=\"sl\" value=\"$sl\">
   <input type=\"hidden\" name=\"rru\" value=\"$param1\">
   <input type=\"hidden\" name=\"auth\" value=\"$self->{msn}->{hotmail}->{mspauth}\">
   <input type=\"hidden\" name=\"creds\" value=\"$creds\">

   <input type=\"hidden\" name=\"svc\" value=\"mail\">
   <input type=\"hidden\" name=\"js\" value=\"yes\">
  </form>
 </body>
</html>

EOF

        $temp_file_fh->close;
	return $temp_file;
}

# Net::MsnMessenger::Event->_reject_ni_cookie
sub _reject_ni_cookie
{
	my ($self, $swb_session, $cookie, $session_id) = @_;

	if (defined $swb_session && defined $cookie)
	{
		my $message = Net::MsnMessenger::Message->new;
		my %attributes;

		$attributes{invitation_command} = 'CANCEL';
		$attributes{invitation_cookie} = $cookie;
		$attributes{cancel_code} = 'REJECT_NOT_INSTALLED';
		$attributes{session_id} = $session_id if defined $session_id;

		$self->msn->{_swb}->[$swb_session]->send_packet('message', $message->create_invite(%attributes));
		return 1;
	}
	undef;
}

# Net::MsnMessenger::Event->_send_client_version
sub _send_client_version
{
	my ($self, $os_type, $os_version, $os_arch, $client_name, $client_version, $lcid) = @_;
	$self->_check_connection || return undef;

	return undef if !defined $os_type || !defined $os_version || !defined $client_name ||
	    !defined $client_version;

	$lcid = '0x409' if !defined $lcid;

	$self->msn->send_packet('client_version', $lcid, $os_type, $os_version, $os_arch, $client_name,
				$client_version, 'MSMSGS', "\r\n");
	1;
}


"Net::MsnMessenger::Event";
__END__

=head1 SEE ALSO

Net::MsnMessenger(3), Net::MsnMessenger::Contact(3), Net::MsnMessenger::Group(3)

=head1 AUTHOR

E-mail: B<incoming@tiscali.cz>

=head1 COPYRIGHT

This module is free software; you can redistribute it and/or modify it under
the same terms as Perl itself.

=cut

