# -*- Mode: perl; perl-indent-level: 8; coding: utf-8 -*-
#
# Net::MsnMessenger::Contact
#
# Copyright (C) 2003 <incoming@tiscali.cz>  All rights reserved.
# This module is free software; You can redistribute and/or modify it under
# the same terms as Perl itself.
#
# $Id: Contact.pm,v 1.11 2003/07/15 21:34:05 incoming Exp $

package Net::MsnMessenger::Contact;

=head1 NAME

Net::MsnMessenger::Contact - Object representing a user in a contact list

=head1 SYNOPSIS

  my @users = $msn->get_all_users;

  do_something($users[0]->passport);
  do_something($users[0]->fname);
  do_something($users[0]->ip_address);

=head1 DESCRIPTION

Every contact in Net::MsnMessenger is represented by an object. This manual page describes
what can you do with an object representing a contact. You may also want to read the
Net::MsnMessenger::Event manual page for more information.

=cut

use Net::MsnMessenger::Data;
use strict qw(subs vars);
use vars   qw($AUTOLOAD);

sub AUTOLOAD
{
	my $self = shift;
	my $name = $AUTOLOAD;
	$name =~ s/.*:://;
	return if $name =~ /DESTROY$/;

	if (exists $self->{$name})
	{
		$self->{$name} = shift if @_;
		return $self->{$name};
	}
	if (exists $self->{phone}->{$name})
	{
		$self->{phone}->{$name} = shift if @_;
		return $self->{phone}->{$name};
	}
	Carp::croak("AUTOLOAD: $name is not a valid method\n");

	$self->{$name} = shift if @_;
	$self->{$name};
}

=head1 METHODS

In this section are documented all the public methods of the Net::MsnMessenger::Contact class.

=head2 Net::MsnMessenger::Contact->new

Create a new Net::MsnMessenger::Contact object. It can be called with the following arguments:

 passport  => user's passport
 fname     => user's URL and UTF-8 encoded friendly name
 status    => user's current status
 connected => 1 if the user is online
 group     => a group the user is in
 c_list    => a contact list the user is in

=cut

# Net::MsnMessenger::Contact->new
sub new
{
	my ($this, %args) = @_;
	my $self = {};

	$self->{passport} = $args{passport} || undef;
	$self->{fname} = $args{fname} || undef;
	$self->{group} = ();
	$self->{c_list} = ();

	$self->{phone} = {
		home   => undef,
		mobile => undef,
		work   => undef,
	};

	$self->{mobile_device}    = undef;
	$self->{mobile_device_my} = undef;
	$self->{ip_address}       = undef;
	$self->{user_agent}       = undef;
	$self->{chat_logging}     = undef;
	$self->{buddy_icons}      = undef;

	$self->{age}        = undef;          # Own profile information
	$self->{birthday}   = undef;
	$self->{country}    = undef;
	$self->{gender}     = undef;
	$self->{kid}        = undef;
	$self->{postalcode} = undef;
	$self->{wallet}     = undef;
	$self->{LCID}       = undef;

	$self->{connected} = $args{connected} || 0;
	$self->{status} = $args{status} || 'offline';

	bless $self, $this;

	$self->add_to_list($args{c_list}) if defined $args{c_list};
	$self->add_to_group($args{group}) if defined $args{group};

	return $self;
}

=head2 $contact->add_to_group ( GROUP_ID )

Add the contact to a group. The parameter is the identification number of the group.

=cut

# Net::MsnMessenger::Contact->add_to_group
sub add_to_group
{
	my $self = shift;
	my $group_id = shift;

	if (defined $group_id)
	{
		if ($self->{group} && grep {$group_id eq $_} @{$self->{group}})
		{
			return undef;
		}
		push @{$self->{group}}, $group_id;
		return 1;
	}
	undef;
}

=head2 $contact->add_to_list ( C_LIST )

Add the contact to a contact list. It can be one of the followings:

 * forward_list
 * allow_list
 * block_list
 * reverse_list

=cut

# Net::MsnMessenger::Contact->add_to_list
sub add_to_list
{
	my $self = shift;
	my $c_list = lc(shift) if defined $_[0];

	if (defined $c_list && exists $Contact_List{$c_list})
	{
		if ($self->{c_list} && grep {$c_list eq $_} @{$self->{c_list}})
		{
			return undef;
		}
		push @{$self->{c_list}}, $c_list;
		return 1;
	}
	undef;
}

=head2 $contact->is_user_in_group ( GROUP_ID )

Return 1 if the contact is in the specified group. The parameter is the identification
number of the group.

=cut

# Net::MsnMessenger::Conect->is_user_in_group
sub is_user_in_group
{
	my $self = shift;
	my $group_id = shift;

	if (defined $group_id && $self->{group})
	{
		return 1 if grep {$group_id eq $_} @{$self->{group}};
	}
	undef;
}

=head2 $contact->is_user_in_list ( C_LIST )

Return 1 if the contact is in the specified contact list.

=cut

# Net::MsnMessenger::Contact->is_user_in_list
sub is_user_in_list
{
	my $self = shift;
	my $c_list = lc(shift) if defined $_[0];

	if ($self->{c_list} && defined $c_list && exists $Contact_List{$c_list})
	{
		return 1 if grep {$c_list eq $_} @{$self->{c_list}};
	}
	undef;
}

=head2 $contact->remove_from_group ( GROUP_ID )

Remove the contact from the specified group.

=cut

# Net::MsnMessenger::Contact->remove_from_group
sub remove_from_group
{
	my $self = shift;
	my $group_id = shift;

	if (defined $group_id && $self->{group})
	{
		for (my $i = 0; $i < @{$self->{group}}; $i++)
		{
			next if $self->{group}->[$i] ne $group_id;

			splice @{$self->{group}}, $i, 1;
			return 1;
		}
	}
	undef;
}

=head2 $contact->remove_from_list ( C_LIST )

Remove the contact from the specified contact list.

=cut

# Net::MsnMessenger::Contact->remove_from_list
sub remove_from_list
{
	my $self = shift;
	my $c_list = lc(shift) if defined $_[0];
	return undef if !$self->{c_list} || !defined $c_list || !exists $Contact_List{$c_list};

	for (my $i = 0; $i < @{$self->{c_list}}; $i++)
	{
		next if $self->{c_list}->[$i] ne $c_list;

		splice @{$self->{c_list}}, $i, 1;
		return 1;
	}
	undef;
}

"Net::MsnMessenger::Contact";
__END__

=head1 AUTOLOADED METHODS

An object attributes that are accessible by methods.

=head2 $contact->passport

The contact's passport (e-mail address).

=head2 $contact->fname

The contact's friendly name (URL and UTF-8 encoded).

=head2 $contact->connected

It is set to 1 if the user is connected (doesn't have the offline status set).

=head2 $contact->status

The user's current status. Available states are:

 * online
 * offline
 * away
 * idle
 * busy
 * be_right_back
 * on_the_phone
 * out_to_lunch
 * appear_offline  (only for your own passport)

=head2 $contact->group

A list (array) of the groups (group IDs) the user is in.

=head2 $contact->c_list

A list (array) of the contact lists the user is in.
Available contact lists:

 * allow_list
 * block_list
 * forward_list
 * reverse_list

=head2 $contact->mobile_device

Set to Y if the user allows other to send him/her pager messages on the mobile device.

=head2 $contact->ip_address

IP Address of the contact, only if known (otherwise it returns undef).

=head2 $contact->user_agent

If you receive a message from this person and the message contains the User-Agent MIME
field, it is saved. Otherwise the method returns undef.

=head2 $contact->chat_logging

 Y - chat is logged
 N - chat is not logged
 S - secure logging (log is encrypted)

=head2 $contact->buddy_icons

 1 - Buddy icons are supported
 0 - Buddy icons are not supported

=head2 Own account only

Use these methods only for your own account (use the Net::MsnMessenger object). Some of
them are optional and are not always set.

=head2 $msn->age

Your age.

=head2 $msn->birthday

Your numerical birthday.

=head2 $msn->country

Your country (two-letter country code).

=head2 $msn->gender

Your geneder (M or F).

=head2 $msn->kid

Set to 1 if you are under-age (use kids passport).

=head2 $msn->LCID

Locale (numeric) used by the MSN. It can be converted to a string and back using
$msn->lcid2string and $msn->string2lcid.

=head2 $msn->mobile_device_my

Set to Y if you are have a mobile device available and other users can send you
messages on it.

=head2 $msn->postalcode

Your postal code.

=head2 $msn->wallet

Set to 1 if you use MS Wallet.

=head1 SEE ALSO

Net::MsnMessenger(3), Net::MsnMessenger::Event(3), Net::MsnMessenger::Group(3)

=head1 AUTHOR

E-mail: B<incoming@tiscali.cz>

=head1 COPYRIGHT

This module is free software; you can redistribute it and/or modify it under
the same terms as Perl itself.

=cut

