# -*- Mode: perl; perl-indent-level: 8; coding: utf-8 -*-
#
# Net::MsnMessenger::Connection
#
# Copyright (C) 2003 <incoming@tiscali.cz>  All rights reserved.
# This module is free software; You can redistribute and/or modify it under
# the same terms as Perl itself.
#
# $Id: Connection.pm,v 1.16 2003/07/17 08:12:48 incoming Exp $

package Net::MsnMessenger::Connection;

use Net::MsnMessenger::Data;
use strict qw(subs vars);
use vars   qw($AUTOLOAD);

sub AUTOLOAD
{
	my $self = shift;
	my $name = $AUTOLOAD;
	$name =~ s/.*:://;
	return if $name =~ /DESTROY$/;

	if (!exists $self->{$name})
	{
		Carp::confess("AUTOLOAD: $name is not a valid method\r\n");
	}
	$self->{$name} = shift if @_;
	$self->{$name};
}

# Net::MsnMessenger::Connection->new
sub new
{
	my ($this, %args) = @_;
	my $self = {};

	$self->{_connhash} = {
		address    => $args{address},
		port       => $args{port},
		socket     => undef,
		srv_socket => undef,
		con_type   => $args{connection_type},          # Connection type - client/server
		srv_type   => $args{server_type},              # Server type DS/NS/SS/FTP/Voice
		srv_ip     => $args{srv_ip},
		proto      => lc($args{protocol}),             # Protocol (TCP,UDP)
		socks      => undef,                           # SOCKS connection
	};

	$self->{_event} = {                                # Logged incoming/outgoing events
		incoming => {},
		outgoing => {},
	};

	$self->{_queue_send} = ();
	$self->{_sock_info}  = {};

	$self->{trans_id}         = 0;        # Transaction ID
	$self->{connected}        = 0;
	$self->{select}           = undef;    # IO::Select
	$self->{file_transfering} = 0;
	$self->{last_packet}      = 0;        # UNIX time of the last packet
	$self->{conn_since}       = undef;    # UNIX time when the connection was established

	$self->{_swb_session}   = $args{swb_session} if defined $args{swb_session};
	$self->{_file_session}  = $args{file_session} if defined $args{file_session};
	$self->{_voice_session} = $args{voice_session} if defined $args{voice_session};

	$self->{msn} = $args{msn};

	bless $self, $this;

	$self->{swb} = $self->msn->{_swb}->[$self->{_swb_session}] if defined $self->{_swb_session};
	$self->{file} = $self->{swb}->{_file}->[$self->{_file_session}] if defined $self->{_file_session};
	$self->{voice} = $self->{swb}->{_voice}->[$self->{_voice_session}] if defined $self->{_voice_session};

	return $self;
}

# Net::MsnMessenger::Connection->create_connection
sub create_connection
{
	my $self = shift;

	if (!defined $self->{_connhash}->{con_type})
	{
		$self->msn->error("Connection type not defined. Couldn't create the connection");
		return undef;
	}

	if ($self->{_connhash}->{con_type} ne 'client' && $self->{_connhash}->{con_type} ne 'server')
	{
		$self->msn->error("Invalid connection type. Couldn't create the connection");
		return undef;
	}

	if (!defined $self->{_connhash}->{address} || !defined $self->{_connhash}->{port})
	{
		$self->msn->error("Address or port not specified. Couldn't create the connection");
		return undef;
	}

	return $self->create_connection_client if $self->{_connhash}->{con_type} eq 'client';
	return $self->create_connection_server if $self->{_connhash}->{con_type} eq 'server';

	undef;
}

# Net::MsnMessenger::Connection->create_connection_client
sub create_connection_client
{
	my $self = shift;
	my $error;

	$self->_debug_inf("Connecting to $self->{_connhash}->{address}:$self->{_connhash}->{port}");

	if ($self->msn->{_socks_use})   # Use the SOCKS server
	{
		# TODO: Someone who understands it better should test/fix it

		require Net::SOCKS;
		my $non_anon = (defined $self->msn->socks_user) ? 1 : 0;

		$self->{_connhash}->{socks} = Net::SOCKS->new(
			socks_addr         => $self->msn->socks_host,
			socks_port         => $self->msn->socks_port,
			user_id            => $self->msn->socks_user,
			user_password      => $self->msn->socks_password,
			force_nonanonymous => $non_anon,
			protocol_version   => $self->msn->socks_version,
		);
		$self->{_connhash}->{socket} = $self->{_connhash}->{socks}->connect(
			peer_addr => $self->{_connhash}->{address},
			peer_port => $self->{_connhash}->{port},
		);
	}

	else    # Direct connection
	{
		$self->{_connhash}->{socket} = IO::Socket::INET->new(
			PeerAddr => $self->{_connhash}->{address},
			PeerPort => $self->{_connhash}->{port},
			Proto    => $self->{_connhash}->{proto},
			Timeout  => 10,
		);
	}

	if (!defined $self->{_connhash}->{socket})
	{
		$self->msn->error("Couldn't connect to $self->{_connhash}->{address}: $!");
		$error++;
	}

	elsif (!$self->{_connhash}->{socket}->blocking(0))
	{
		$self->msn->error("Couldn't set the socket to non-blocking: $!");
		$error++;
	}

	if ($error)
	{
		if (defined $self->{_file_session})      # Couldn't establish a file session
		{
			$self->msn->_callback('FILE_RECEIVE_CANCEL', $self->{_swb_session}, $self->{_file_session},
					      "Couldn't connect: $!");
		}
		elsif (defined $self->{_voice_session})  # Couldn't establish a voice session
		{
			$self->msn->_callback('VOICE_CANCEL', $self->{_swb_session}, $self->{_voice_session},
					      "Couldn't connect: $!");
		}
		return undef;
	}

	# All the sockets on Msn need to be readable/writable so set in at once
	$self->msn->_add_fh($self);
	$self->connected(1);
	$self->conn_since(time);
	$self->select(IO::Select->new($self->{_connhash}->{socket}));

	$self->_debug_inf("Connected to $self->{_connhash}->{address}:$self->{_connhash}->{port}");
	1;
}

# Net::MsnMessenger::Connection->create_connection_server
sub create_connection_server
{
	my $self = shift;
	my $error;

	# TODO: Add socks support

	$self->{_connhash}->{socket} = IO::Socket::INET->new(
		LocalAddr => $self->{_connhash}->{address},
		LocalPort => $self->{_connhash}->{port},
		Proto     => $self->{_connhash}->{proto},
		Listen    => 1,
		Reuse     => 1,
	);

	if (!defined $self->{_connhash}->{socket})
	{
		eval { require Errno; };
		if (!$@ && $! == Errno::EADDRINUSE())
		{
			# The local port is already in use, keep increasing the port number until
			# the server is successfully created

			$self->{_connhash}->{port}++;
			$self->_debug_inf("Listening on a new port: $self->{_connhash}->{port}");

			if ($self->{_connhash}->{srv_type} eq 'FTP')   # File server
			{
				$self->file->port($self->{_connhash}->{port});
			}

			return $self->create_connection_server;
		}

		$self->msn->error("Couldn't create the server: $!");
		$error++;
	}

	elsif (!$self->{_connhash}->{socket}->blocking(0))
	{
		$self->msn->error("Couldn't set the socket to non-blocking: $!");
		$error++;
	}

	if ($error)
	{
		if (defined $self->{_file_session})       # Couldn't create a file server
		{
			$self->msn->_callback('FILE_SEND_CANCEL', $self->{_swb_session}, $self->{_file_session},
					      "Couldn't create the server: $!");
		}
		elsif (defined $self->{_voice_session})   # Couldn't create a voice server
		{
			$self->msn->_callback('VOICE_CANCEL', $self->{_swb_session}, $self->{_voice_session},
					      "Couldn't create the server: $!");
		}
		return undef;
	}

	$self->msn->_add_fh($self);
	$self->conn_since(time);
	$self->select(IO::Select->new($self->{_connhash}->{socket}));

	$self->_debug_inf("Server socket created: $self->{_connhash}->{address}:$self->{_connhash}->{port}");
	1;
}

# Net::MsnMessenger::Connection->disconnect
sub disconnect
{
	my $self = shift;
	return undef if !$self->connected;

	# See if it's still connected
	if ($self->{_connhash}->{srv_type} ne 'FTP' && defined fileno($self->{_connhash}->{socket}))
	{
		$self->send_packet('quit');
	}

	$self->disconnect_real;
	1;
}

# Net::MsnMessenger::Connection->disconnect_real
sub disconnect_real
{
	my $self = shift;
	return undef if !$self->connected;

	if ($self->{_connhash}->{con_type} eq 'server')
	{
		close $self->{_connhash}->{srv_socket};
	}

	close $self->{_connhash}->{socket};
	$self->msn->_rem_fh($self);
	$self->connected(0);
	$self->trans_id(0);
	$self->last_packet(0);
	$self->conn_since(0);
	$self->file_transfering(0);

	$self->_debug_inf("Disconnected from $self->{_connhash}->{address}");
	1;
}

# Net::MsnMessenger::Connection->new_client
sub new_client
{
	my $self = shift;

	# New client connected to our server, create a new socket
	$self->{_connhash}->{srv_socket} = $self->{_connhash}->{socket}->accept;

	if (!$self->{_connhash}->{srv_socket}->blocking(0))
	{
		$self->msn->error("Couldn't set the socket to non-blocking: $!");
		return undef;
	}

	$self->select(IO::Select->new($self->{_connhash}->{srv_socket}));
	$self->connected(1);
	$self->_get_conn_ip();   # Get the client's ip address

	if (defined $self->{_connhash}->{srv_ip} && defined $self->file->sending_to)   # Save the IP
	{
		if (exists $self->msn->{_contact}->{$self->file->sending_to})
		{
			$self->msn->{_contact}->{$self->file->sending_to}->ip_address($self->{_connhash}->{srv_ip});
		}
		else
		{
			my $contact_ref;
			for ($self->swb->get_users)
			{
				if ($_->passport eq $self->file->sending_to)
				{
					$contact_ref = $_;  last;
				}
			}
			$contact_ref->ip_address($self->{_connhash}->{srv_ip}) if defined $contact_ref;
		}
	}
	1;
}

# Net::MsnMessenger::Connection->send_packet
sub send_packet
{
	my ($self, $command) = (shift, shift);
	my $message = join ' ', grep {defined} @_ if @_;

	return undef if !$self->connected || !defined $command;

	my $r_command = $self->_get_server_command_by_command($command);
	my $packet = (defined $r_command) ? $r_command : $command;

	# For the following the transaction IDs are ignored
	if ($command ne 'quit' && $command ne 'ping' && $self->{_connhash}->{srv_type} ne 'FTP')
	{
		$packet .= ' ' . $self->trans_id;
	}

	if (defined $message)   # Add the message
	{
		$packet .= ' ' if $message ne "\r\n";   # Skip the extra space if it's only the newline
		$packet .= $message;
	}

	$self->_send($packet);
	$self->{_event}->{outgoing}->{$self->{trans_id}++} = $packet;
	1;
}

# Net::MsnMessenger::Connection->send_packet_file
sub send_packet_file
{
	my ($self, $file_part, $part_len) = @_;

	return undef if !$self->connected;

	# Send the packet header with the part of the file
	return $self->_send(chr(0) . chr($part_len & 0xFF) . chr($part_len >> 8) . $file_part);
}

# Net::MsnMessenger::Connection->send_packet_file_cancel
sub send_packet_file_cancel
{
	my $self = shift;
	return undef if !$self->connected;

	$self->file_transfering(0);
	$self->_send(chr(1) . chr(0) . chr(0));
	$self->msn->disconnect_file($self->{_swb_session}, $self->{_file_session});
	1;
}

# Net::MsnMessenger::Connection->send_receive
sub send_receive
{
	my $self = shift;
	return if !$self->connected;

	if (IO::Select->select($self->select, undef, undef, .00001))
	{
		(defined $self->{file} && $self->file_transfering && $self->{_connhash}->{con_type} eq 'client')
		    ? $self->_receive_file    # Receiving a file
		    : $self->_receive;        # Receiving from a server
	}

	if ($self->file_transfering && $self->{_connhash}->{srv_type} eq 'FTP' &&
	    $self->{_connhash}->{con_type} eq 'server')
	{
		# Sending a file
		$self->_send_file if IO::Select->select(undef, $self->select, undef, .00001);
	}
	1;
}

# -------------------- Private methods -------------------- #

# Net::MsnMessenger::Connection->_get_conn_ip
sub _get_conn_ip
{
	my $self = shift;

	# Get an IP of the client connected to our server
	$self->{_connhash}->{srv_ip} = Socket::inet_ntoa(
		(Socket::sockaddr_in($self->{_connhash}->{srv_socket}->sockname()))[1]);
	1;
}

# Net::MsnMessenger::Connection->_get_category_by_command
sub _get_category_by_command
{
	my ($self, $command) = @_;

	for (keys %{$Command})
	{
		for my $c(keys %{$Command->{$_}})
		{
			return $_ if $Command->{$_}->{$c} eq $command || $c eq $command;
		}
	}
	undef;
}

# Net::MsnMessenger::Connection->_get_server_command_by_command
sub _get_server_command_by_command
{
	my ($self, $command) = @_;

	for (keys %{$Command})
	{
		for my $c(keys %{$Command->{$_}})
		{
			# Find and use real server command for the given name
			return $Command->{$_}->{$c} if $c eq $command;
		}
	}
	undef;
}

# Net::MsnMessenger::Connection->_got_disconnected
sub _got_disconnected
{
	my $self = shift;

	if (defined $self->{_file_session})
	{
		$self->msn->disconnect_file($self->{_swb_session}, $self->{_file_session}, 1);

		($self->{_connhash}->{con_type} eq 'server')
		    ? $self->msn->_callback('FILE_SEND_CANCEL', $self->{_swb_session}, $self->{_file_session},
					    "The client disconnected for an unknown reason")

		    : $self->msn->_callback('FILE_RECEIVE_CANCEL', $self->{_swb_session}, $self->{_file_session},
					    "The server disconnected for an unknown reason");
	}

	elsif (defined $self->{_voice_session})
	{
		$self->msn->disconnect_voice($self->{_swb_session}, $self->{_voice_session}, 1);
	}

	elsif (defined $self->{_swb_session})
	{
		$self->msn->disconnect_swb($self->{_swb_session}, 1);
		$self->msn->_callback('CLOSE_CHAT', $self->{_swb_session});
	}

	else
	{
		$self->msn->disconnect(1);
		$self->msn->_callback('DISCONNECT_FORCED', "The server disconnected for an unknown reason");
	}
	1;
}

# Net::MsnMessenger::Connection->_parse_buffer
sub _parse_buffer
{
	my ($self, $buffer) = @_;
	my @to_parse;

	return if !defined $buffer;

	$self->{_buffer} .= $buffer if defined $self->{_buffer};
	$self->{_buffer}  = $buffer if !defined $self->{_buffer};

	# Message
	if ($self->{_buffer} =~ /$Command->{Event}->{message}.[^\r\n]*\s+\d+/)
	{
		while ($self->{_buffer} =~ /$Command->{Event}->{message}.[^\r\n]*\s+(\d+)\r\n/)
		{
			$self->{_buffer} =~ s/($Command->{Event}->{message}.[^\r\n]*\r\n.{$1})//s || return undef;
			push @to_parse, $1;
		}
		return undef if !@to_parse;
	}

	# Notification (Alert)
	if ($self->{_buffer} =~ /$Command->{Event}->{notification}\s+\d+/)
	{
		while ($self->{_buffer} =~ /$Command->{Event}->{notification}\s+(\d+)\r\n/)
		{
			$self->{_buffer} =~
			    s/($Command->{Event}->{notification}.+?<NOTIFICATION.+?<\/NOTIFICATION>\r\n)//s ||
			    return undef;
			
			push @to_parse, $1;
		}
		return undef if !@to_parse;
	}
	push @to_parse, $1 while $self->{_buffer} =~ s/^(.+\r\n)//;

	if (@to_parse)
	{
		$self->_receive_find_handler($_) for grep {length $_} @to_parse;
	}
	1;
}

# Net::MsnMessenger::Connection->_receive
sub _receive
{
	my $self = shift;
	my ($buffer, $rv);

	if ($self->{_connhash}->{con_type} eq 'server')   # Receiving from a client connected to our server
	{
		$rv = $self->{_connhash}->{srv_socket}->recv($buffer, 1000000);
	}

	else    # Receiving from a server
	{
		$rv = $self->{_connhash}->{socket}->recv($buffer, 1000000);
	}

	if (!defined $rv)
	{
		eval { require Errno; };
		if ($@ || ($! != Errno::EAGAIN()))
		{
			$self->_debug_inf("Receive error. Disconnecting");

			$self->{_buffer} = undef;
			$self->_got_disconnected;
			return undef;
		}
	}

	return undef if !length($buffer);

	$self->last_packet(time);
	$self->_debug_pkt_inc($buffer);
	$self->_parse_buffer($buffer);
	1;
}

# Net::MsnMessenger::Connection->_receive_file
sub _receive_file
{
	my $self = shift;
	my $buffer;

	if (!$self->{_connhash}->{socket}->recv($buffer, 1000000))
	{
		$self->_debug_inf("Receive error. Disconnecting");
		$self->_got_disconnected;
		return undef;
	}

	return if !$buffer;

	$self->_debug_inf("Incoming file transfer: received " . length($buffer) . " from the socket");

	$self->last_packet(time);
	$self->file->_handle_file($buffer);   # Process the received file packet
	1;
}

# Net::MsnMessenger::Connection->_receive_find_handler
sub _receive_find_handler
{
	my ($self, $buffer) = @_;
	my $old_buffer = $buffer;
	my $to_handle = {};
	my $category = undef;

	return if !defined $buffer || $buffer !~ /\S/;

	if (defined $self->{_swb_session})
	{
		$category = 'Switchboard';
		$to_handle->{swb_session} = $self->{_swb_session};
	}
	if (defined $self->{_file_session} && $self->{_connhash}->{srv_type} eq 'FTP')
	{
		$category = ($self->{_connhash}->{con_type} eq 'server') ? 'File_Send' : 'File';
		$to_handle->{file_session} = $self->{_file_session};
	}

	# ---------- Create the info structure ---------- #

	if ($buffer =~ s/^(\S{3})\s*//)
	{
		# Command
		$to_handle->{command} = $1;

		$category = 'Message' if $to_handle->{command} eq $Command->{Event}->{message};
		$category = 'Notification' if $to_handle->{command} eq $Command->{Event}->{notification};

		$category ||= $self->_get_category_by_command($to_handle->{command});
	}

	# Transaction ID
	if ($buffer =~ /^(\d+)\s*/ && exists $self->{_event}->{outgoing}->{$1})
	{
		$to_handle->{trans_id} = $1;
		$buffer =~ s/^\d+\s+//;
	}

	# The rest of the buffer should be the server message
	$to_handle->{message} = $buffer if defined $buffer;

	if (exists $to_handle->{trans_id})
	{
		$self->{_event}->{incoming}->{$to_handle->{trans_id}} = $buffer;
	}

	return $self->msn->_handle_Error($to_handle) if defined $to_handle->{command} &&
	    $to_handle->{command} =~ /^\d+$/;

	if (defined $category)
	{
		my $h_sub = "_handle_" . $category;

		if ($self->msn->can($h_sub))
		{
			return $self->msn->$h_sub($to_handle);
		}
	}

	# Unimplemented event - pass the untouched buffer
	return $self->msn->_callback('UNIMPLEMENTED', $old_buffer);
}

# Net::MsnMessenger::Connection->_send_file
sub _send_file
{
	my $self = shift;
	my $file_part;

	if (!$self->file->transfered || $self->file->transfered < $self->file->file_size)
	{
		my $real_len = read $self->{file}->{file_handle}, $file_part, 2045;
		$self->send_packet_file($file_part, $real_len);

		$self->file->{transfered} += $real_len;
		$self->msn->_callback('FILE_SEND_PROGRESS', $self->{_swb_session}, $self->{_file_session},
				      $self->file->transfered, $self->file->file_size);
	}

	elsif ($self->file->transfered >= $self->file->file_size)
	{
		$self->file_transfering(0);
	}
	1;
}

# Net::MsnMessenger::Connection->_send
sub _send
{
	my ($self, $packet) = @_;
	return if !$self->connected;

	if ($self->{_connhash}->{con_type} eq 'server')
	{
		if ($self->{_connhash}->{srv_socket}->send($packet))
		{
			$self->_debug_pkt_out_srv($packet);
		}
		else
		{
			eval { require Errno; };
			if (!$@ && $! == Errno::EAGAIN())
			{
				$self->_send($packet);
			}
			else
			{
				$self->_debug_inf("Couldn't send. The server is disconnected.");
				$self->_got_disconnected;
				return undef;
			}
		}
	}

	else
	{
		if ($self->{_connhash}->{socket}->send($packet))
		{
			$self->_debug_pkt_out($packet);
		}
		else
		{
			$self->_debug_inf("Couldn't send. Disconnected.");
			$self->_got_disconnected;
			return undef;
		}
	}

	$self->last_packet(time);
	1;
}

# Net::MsnMessenger::Connection->_debug
sub _debug
{
	my ($self, $message) = @_;

	if ($self->msn->debug_connection)
	{
		my $debug = "\r\n";

		$debug .= "Connection Type: $self->{_connhash}->{con_type}\r\n";
		$debug .= "Server Type    : $self->{_connhash}->{srv_type}\r\n";
		$debug .= "Client IP      : $self->{_connhash}->{srv_ip}\r\n" if $self->{_sock_info}->{ip_addr};
		$debug .= "SWB Session    : $self->{_swb_session}\r\n"        if defined $self->{_swb_session};
		$debug .= "File Session   : $self->{_file_session}\r\n"       if defined $self->{_file_session};
		$debug .= "Voice Session  : $self->{_voice_session}\r\n"      if defined $self->{_voice_session};

		$debug .= "\r\n---- END HEADER ----\r\n\r\n";
		$debug .= "Debug message:\r\n$message\r\n";
		$debug .= "\r\n---- END MESSAGE ----\r\n";

		$self->msn->_callback('DEBUG_CONNECTION', $debug);
		return 1;
	}
	undef;
}

# Net::MsnMessenger::Connection->_debug_inf
sub _debug_inf
{
	my $self = shift;
	return $self->_debug("INFO: " . shift);
}

# Net::MsnMessenger::Connection->_debug_pkt_inc
sub _debug_pkt_inc
{
	my ($self, $packet) = @_;
	return $self->_debug("Incoming Packet: $packet" . "!END!\r\n");
}

# Net::MsnMessenger::Connection->_debug_pkt_out
sub _debug_pkt_out
{
	my ($self, $packet) = @_;
	return $self->_debug("Outgoing Packet: $packet" . "!END!\r\n");
}

# Net::MsnMessenger::Connection->_debug_pkt_out_srv
sub _debug_pkt_out_srv
{
	my ($self, $packet) = @_;
	return $self->_debug("Outgoing Server Packet: $packet" . "!END!\r\n");
}


"Net::MsnMessenger::Connection";
__END__

