# -*- Mode: perl; perl-indent-level: 8; coding: utf-8 -*-
#
# Net::MsnMessenger
#
# Copyright (C) 2003 <incoming@tiscali.cz>  All rights reserved.
# This module is free software; You can redistribute and/or modify it under
# the same terms as Perl itself.
#
# $Id: MsnMessenger.pm,v 1.18 2003/07/17 08:12:48 incoming Exp $

package Net::MsnMessenger;

=head1 NAME

Net::MsnMessenger - MSN Instant Messenger protocol implementation

=head1 SYNOPSIS

  use Net::MsnMessenger;

  my $msn = Net::MsnMessenger->new;
  $msn->passport('me@hotmail.com');
  $msn->password('my_secret_password');

  $msn->connect or die $msn->get_error;
  $msn->add_callback('RECEIVE_MESSAGE', \&receive_msg);

  $msn->start;

=head1 DESCRIPTION

Net::MsnMessenger is a perl implementation of MSN Instant Messenger protocol. It
works with version 7 of the protocol and supports many features including complete
contact/groups management, instant messaging and file transfer.

=cut

use Carp;
use Digest::MD5;
use IO::Select;
use IO::Socket;

use Net::MsnMessenger::Connection;
use Net::MsnMessenger::Contact;
use Net::MsnMessenger::Data;
use Net::MsnMessenger::Event;
use Net::MsnMessenger::File;
use Net::MsnMessenger::Group;
use Net::MsnMessenger::Message;
use Net::MsnMessenger::NetMeeting;
use Net::MsnMessenger::RemoteAssistance;
use Net::MsnMessenger::Switchboard;
use Net::MsnMessenger::Voice;

use strict qw(subs vars);
use vars   qw($AUTOLOAD $VERSION);

BEGIN
{
	$VERSION = '0.91';
}

sub AUTOLOAD
{
	my $self = shift;
	my $name = $AUTOLOAD;
	$name =~ s/.*:://;
	return if $name =~ /DESTROY$/;

	if (exists $self->{$name})
	{
		$self->{$name} = shift if @_;
		return $self->{$name};
	}

	for (grep {defined $self->{$_}} '_client', '_account', '_event')
	{
		return $self->{$_}->$name(@_) if $self->{$_}->can($name);

		if (exists $self->{$_}->{$name})
		{
			$self->{$_}->{$name} = shift if @_;
			return $self->{$_}->{$name};
		}
	}
	Carp::confess("AUTOLOAD: $name is not a valid method\n");
}

=head1 METHODS

In this section are documented all the public methods of the Net::MsnMessenger class.

=head2 Net::MsnMessenger->new ( ARGUMENTS )

Create a new Net::MsnMessenger object. The object is created for one single user to be able
to connect to the MSN server.

It can be called with the following arguments:

 passport => User login passport (E-mail address)
 password => User login password

 Optional:
 initial_status => Status to set after login (case insensitive)
 auto_allow     => 1 - automatically add new contacts to the list
 block_privacy  => 1 - receive messages only from allowed users

 debug            => Debug the protocol
 debug_connection => Debug the connection

 send_client_version => 1 - send client info after logged in

 For the client version:
 os_type        => Type (name) of your operating system
 os_version     => Version of your operating system
 os_arch        => Architecture (ie. i386)
 client_name    => Name of your client
 client_version => Version of your client

 For the socks server usage:
 socks_host     => Host (IP address) of the socks server
 socks_port     => Port of the socks server
 socks_user     => User to authenticate on the socks server (if needed)
 socks_password => Password for the authentication (if needed)
 socks_version  => Version of the socks server (4 or 5)

=cut

# Net::MsnMessenger->new
sub new
{
	my $this = shift;
	my %args = @_;
	my $self = {};

	$self->{server}   = $args{server} || 'messenger.hotmail.com';
	$self->{port}     = $args{port} || 1863;
	$self->{password} = $args{password} || undef;

	$self->{initial_status}   = $args{initial_status} || 'online';
	$self->{auto_allow}       = $args{auto_allow} || 0;
	$self->{block_privacy}    = $args{block_privacy} || 0;
	$self->{debug}            = $args{debug} || 0;
	$self->{debug_connection} = $args{debug_connection} || 0;
	$self->{error}            = undef;
	$self->{fname}            = undef;
	$self->{signed_in}        = 0;
	$self->{session_id}       = undef;
	$self->{testing}          = $args{testing} || 0;

	# Client version stuff
	$self->{send_client_version} = $args{send_client_version} || 0;

	$self->{os_type}        = $args{os_type} || undef;
	$self->{os_version}     = $args{os_version} || undef;
	$self->{os_arch}        = $args{os_arch} || undef;
	$self->{client_name}    = $args{client_name} || undef;
	$self->{client_version} = $args{client_version} || undef;

	# Socks stuff
	$self->{socks_host}     = $args{socks_host} || undef;
	$self->{socks_port}     = $args{socks_port} || undef;
	$self->{socks_user}     = $args{socks_user} || undef;
	$self->{socks_password} = $args{socks_password} || undef;
	$self->{socks_version}  = $args{socks_version} || undef;

	# Private
	$self->{_handle}          = ();
	$self->{_contact}         = {};            # Contact List
	$self->{_group}           = {};            # Groups List
	$self->{_callback}        = {};            # Callback functions
	$self->{_swb}             = ();            # Switchboard sessions
	$self->{_swb_sessions}    = 0;             # Number of swb sessions
	$self->{_f_results}       = {};
	$self->{_socks_use}       = 0;
	$self->{_err209}          = {};
	$self->{_inviting}        = ();            # Users that are being invited to a swb session

	bless $self, $this;
	$self->{_account} = Net::MsnMessenger::Contact->new();          # Own account
	$self->{_event} = Net::MsnMessenger::Event->new(msn => $self);

	if (defined $self->{socks_server} && defined $self->{socks_port})
	{
		eval { require Net::SOCKS };
		if ($@)
		{
			$self->error("You need to have Net::SOCKS module. SOCKS5 server will _not_ be used");
		}

		else { $self->{_socks_use}++ }

		$self->{socks_password} = undef if !defined $self->{socks_user};

		if (!defined $self->{socks_version} ||
		    ($self->{socks_version} ne 4 && $self->{socks_version} ne 5))
		{
			$self->{socks_version} = 5;
		}
	}

	$self->passport($args{passport}) if defined $args{passport};
	return $self;
}

=head2 $msn->connect ()

Connect to the dispatch server, send the initial command and continue until logged in. While
logging in 'LOGIN' callbacks are sent to indicate the process. If the connection to the dispatch
server could not be established the method returns undef and sets the error (see get_error).

=cut

# Net::MsnMessenger->connect
sub connect
{
	my $self = shift;

	if (!defined $self->passport)
	{
		$self->error("Couldn't connect: The passport is not set");
		return undef;
	}
	if (!defined $self->password)
	{
		$self->error("Couldn't connect: The password is not set");
		return undef;
	}

	if (defined $self->{_client} && $self->connected)  # Already connected/connecting
	{
		if ($self->signed_in)
		{
			$self->error("Couldn't connect: Already logged in");
			return undef;
		}
		else
		{
			$self->error("Couldn't connect: Login in progress");
			return undef;
		}
	}

	$self->{_client} = Net::MsnMessenger::Connection->new(
		address         => $self->server,
		port            => $self->port,
		msn             => $self,
		connection_type => 'client',
		server_type     => 'DS',
		protocol        => 'tcp',
	);

	$self->_callback('LOGIN_PROGRESS', "Connecting to server $self->{server} port $self->{port}");
	$self->create_connection or return undef;

	# Get own IP Address
	if (!defined $self->ip_address)
	{
		$self->ip_address(Socket::inet_ntoa(
		(Socket::sockaddr_in($self->{_client}->{_connhash}->{socket}->sockname()))[1])) ||
		$self->error("Couldn't read own IP Address");
	}

	# Connected - send the initial message (protocol version)
	$self->send_packet('version', "$Msn_version\r\n");
	1;
}

=head2 $msn->disconnect ()

Log out from the main server and disconnect all the existing (established) connections.

=cut

# Net::MsnMessenger->disconnect
sub disconnect
{
	my $self = shift;
	my $disconnected = shift;

	if ($self->{_handle})
	{
		while (my $conn = shift @{$self->{_handle}})
		{
			# Disconnect every connection, all sockets are read/write
			($disconnected)
			    ? $conn->disconnect_real : $conn->disconnect;
		}
	}

	# Clean the values (neccessary for later reconnect)
	$self->{_contact} = {};
	$self->{_group} = {};
	$self->{_swb} = ();

	$self->signed_in(0);
	$self->connected(0);

	$self->{_swb_sessions} = 0;
	$self->{_inviting} = ();

	$self->status('offline');
	1;
}

=head2 $msn->disconnect_file ( SWB_SESSION, FILE_SESSION )

Disconnect an established file transfer session. The method takes switchboard session and file
transfer session as the parameters.

=cut

# Net::MsnMessenger->disconnect_file
sub disconnect_file
{
	my ($self, $swb_session, $file_session, $_disconnected) = @_;

	return undef if !defined $swb_session || !defined $self->{_swb}->[$swb_session] ||
	    !defined $file_session || !defined $self->{_swb}->[$swb_session]->{_file}->[$file_session];

	eval
	{
		if ($self->{_swb}->[$swb_session]->{_file}->[$file_session]->connected)
		{
			($_disconnected)
			    ? $self->{_swb}->[$swb_session]->{_file}->[$file_session]->disconnect_real
			    : $self->{_swb}->[$swb_session]->{_file}->[$file_session]->disconnect;
		}
	};

	# See if we need to close the file handle
	my $this_file = $self->{_swb}->[$swb_session]->{_file}->[$file_session];

	if (defined $this_file && defined $this_file->file_handle && defined fileno($this_file->file_handle))
	{
		$this_file->file_handle->close;
	}

	$self->{_swb}->[$swb_session]->{_file}->[$file_session] = undef;
	1;
}

=head2 $msn->disconnect_swb ( SWB_SESSION )

Disconnect an established switchboard session. The method takes switchboard session as the
parameter.

=cut

# Net::MsnMessenger->disconnect_swb
sub disconnect_swb
{
	my ($self, $session, $_disconnected) = @_;

	return undef if !defined $session || $session !~ /^\d+$/ || !$self->{_swb_sessions} ||
	    !exists $self->{_swb}->[$session];

	eval
	{
		if ($self->{_swb}->[$session]->connected)
		{
			($_disconnected)
			    ? $self->{_swb}->[$session]->disconnect_real : $self->{_swb}->[$session]->disconnect;
		}
	};

	$self->{_swb}->[$session] = undef;
	1;
}

=head2 $msn->disconnect_swb_pending ( PASSPORT )

When you invite someone to a switchboard session and you want to cancel it before it gets established,
use this method. The parameter is the passport of the user you invited.

=cut

# Net::MsnMessenger->disconnect_swb_pending
sub disconnect_swb_pending
{
	my ($self, $passport) = @_;

	if ($self->{_swb_sessions})   # Already connected
	{
		for (my $i = 0; $i < @{$self->{_swb}}; $i++)
		{
			$self->disconnect_swb($i)
			    if defined $self->{_swb}->[$i] && defined $self->{_swb}->[$i]->inviting &&
			    $self->{_swb}->[$i]->inviting eq $passport;
		}
	}

	if ($self->{_inviting})       # Waiting for a connection
	{
		for (my $i = 0; $i < @{$self->{_inviting}}; $i++)
		{
			splice @{$self->{_inviting}}, $i--, 1 if $self->{_inviting}->[$i] eq $passport;
		}
	}
	1;
}

# Net::MsnMessenger->disconnect_voice
sub disconnect_voice
{
	my ($self, $swb_session, $voice_session, $_disconnected) = @_;

	return undef if !defined $swb_session || !defined $self->{_swb}->[$swb_session] ||
	    !defined $voice_session || !defined $self->{_swb}->[$swb_session]->{_voice}->[$voice_session];

	eval
	{
		if ($self->{_swb}->[$swb_session]->{_voice}->[$voice_session]->connected)
		{
			($_disconnected)
			    ? $self->{_swb}->[$swb_session]->{_voice}->[$voice_session]->disconnect_real
			    : $self->{_swb}->[$swb_session]->{_voice}->[$voice_session]->disconnect;
		}
	};
	$self->{_swb}->[$swb_session]->{_voice}->[$voice_session] = undef;
	1;
}

=head2 $msn->start ()

If you are writing a fairly simple application you can use this method to start communicating with
the server.
Note that this method does not return until you are disconnected from the server by yourself
(using the disconnect() method), or by the server.

=cut

# Net::MsnMessenger->start
sub start
{
	my $self = shift;
	$self->do_one_loop while $self->connected;
}

=head2 $msn->do_one_loop ()

If you do not run the start() method you have to continuously call this one. This method does one
processing loop watching the connected sockets, waiting for incoming data and processing them. It
is important to call this method fast enough to not cause any unwanted delays.

=cut

# Net::MsnMessenger->do_one_loop
sub do_one_loop
{
	my $self = shift;
	return undef if !$self->connected;

	for my $conn(@{$self->{_handle}})
	{
		if ($conn->{_connhash}->{con_type} eq 'server' && !$conn->connected)
		{
			$conn->new_client if IO::Select->select($conn->select, undef, undef, .0001);
		}

		$conn->send_receive if $conn->connected;
	}
	1;
}

=head2 $msn->get_error ()

If a protocol or server error occurs, use this method to read the error message.

 Example:

 $msn->do_one_loop;
 my $error = $msn->get_error;
 print $error if defined $error;

=cut

# Net::MsnMessenger->get_error
sub get_error
{
	my $self = shift;
	my $error = $self->error;
	$self->{error} = undef;
	$error;
}

=head2 $msn->add_callback ( CALLBACK_NAME, CODE_REF )

Associate a code reference to a callback. Every time an event occurs, Net::MsnMessenger calls
the function(s) associated for the callback name with the appropriate parameters. For a full
list of the callbacks, see the CALLBACKS section below.

 Example:

 $msn->add_callback('RECEIVE_MESSAGE', \&callback_receive);

 sub receive_message {
     my ($session, $sender, $message, %attributes) = @_;
     # Do something with the message
 }

=cut

# Net::MsnMessenger->add_callback
sub add_callback
{
	my ($self, $name, $func) = @_;

	if (!defined $name || !defined $func)
	{
		Carp::croak("Usage: \$msn->add_callback('EVENT_NAME', &FUNCTION)\n");
		return undef;
	}

	push @{$self->{_callback}->{$name}}, $func;
	1;
}

=head2 $msn->url_decode ( TEXT_STRING )

Decode an URL encoded text string and return it.

=cut

# Net::MsnMessenger->url_decode
sub url_decode
{
	my ($self, $str) = @_;

	if (!defined $str)
	{
		Carp::croak("Usage: \$msn->url_decode(string_to_decode)\r\n");
		return undef;
	}

	$str =~ s/\%([A-Fa-f\d]{2})/chr hex $1/eg;
	$str;
}

=head2 $msn->url_encode ( TEXT_STRING )

URL encode a text string and return it.

=cut

# Net::MsnMessenger->url_encode
sub url_encode
{
	my ($self, $str) = @_;

	if (!defined $str)
	{
		Carp::croak("Usage: \$msn->url_encode(string_to_encode)\r\n");
		return undef;
	}

	$str =~ s/([^\w()\'*~!.,-])/uc sprintf '%%%2x', ord $1/eg;
	$str;
}

=head2 $msn->xml_decode ( TEXT_STRING )

XML encode a text string and return it.

=cut

# Net::MsnMessenger->xml_decode
sub xml_decode
{
	my ($self, $str) = @_;

	if (!defined $str)
	{
		Carp::croak("Usage: \$msn->xml_decode(string_to_decode)\r\n");
		return undef;
	}

	$str =~ s/&lt;/</g;
	$str =~ s/&gt;/>/g;
	$str =~ s/&quot;/\"/g;
	$str =~ s/&apos;/\'/g;

	$str =~ s/&\#x(\d+);/chr(hex($1))/eg;
	$str =~ s/&\#(\d+);/chr($1)/eg;

	$str =~ s/&amp;/&/g;
	$str;
}

=head2 $msn->xml_encode ( TEXT_STRING )

XML decode a text string and return it.

=cut

# Net::MsnMessenger->xml_encode
sub xml_encode
{
	my ($self, $str) = @_;

	if (!defined $str)
	{
		Carp::croak("Usage: \$msn->xml_encode(string_to_encode)\r\n");
		return undef;
	}

	$str =~ s/&/&amp;/g;
	$str =~ s/</&lt;/g;
	$str =~ s/>/&gt;/g;
	$str =~ s/\"/&quot;/g;
	$str =~ s/\'/&apos;/g;

	$str =~ s/([\x00-\x19])/sprintf "&#x%x;", ord $1/eg;
	$str =~ s/([\x80-\xff])/sprintf "&#x%x;", ord $1/eg;
	$str;
}

=head2 $msn->lcid2string ( LCID_NUMBER )

Convert a language identification number to a name of the language and return it.

=cut

# Net::MsnMessenger->lcid2string
sub lcid2string
{
	my ($self, $lcid) = @_;

	if (!defined $lcid)
	{
		Carp::croak("Usage: \$msn->lcid2string(lcid_number)\r\n");
		return undef;
	}

	$lcid = hex($1) if $lcid =~ /^0x(\d+)$/i;
	return (exists $R_LCID{$lcid}) ? $R_LCID{$lcid} : undef;
}

=head2 $msn->string2lcid ( TEXT_STRING )

Convert a text representation of a language identification number to the number and return it.

=cut

# Net::MsnMessenger->string2lcid
sub string2lcid
{
	my ($self, $string) = @_;

	if (!defined $string)
	{
		Carp::croak("Usage: \$msn->string2lcid(text_string)\r\n");
		return undef;
	}

	return (exists $LCID{$string}) ? $LCID{$string} : undef;
}

# -------------------- Private methods -------------------- #

# Net::MsnMessenger->_add_fh
sub _add_fh
{
	my $self = shift;
	my $connection = shift;

	push @{$self->{_handle}}, $connection;
	1;
}

# Net::MsnMessenger->_rem_fh
sub _rem_fh
{
	my $self = shift;
	my $connection = shift;

	for (my $i = 0; $i < @{$self->{_handle}}; $i++)
	{
		return splice @{$self->{_handle}}, $i, 1 if $self->{_handle}->[$i] eq $connection;
	}
	undef;
}

# Net::MsnMessenger->_callback
sub _callback
{
	my ($self, $name) = (shift, shift);

	return undef if !defined $name;

	if ($name ne 'DEBUG' && $name ne 'DEBUG_CONNECTION')
	{
		my $debug_msg = "Calling $name callback";
		if (@_)
		{
			$debug_msg .= "\r\n";
			$debug_msg .= "Parameter: $_\r\n" for @_;
		}
		$self->_callback('DEBUG', $debug_msg);
	}

	local $^W;

	for my $cb(@{$self->{_callback}->{$name}})
	{
		eval { &{$cb}(@_) };
	}
	1;
}

# Net::MsnMessenger->_handle_Contact_List
sub _handle_Contact_List
{
	my ($self, $to_handle) = @_;
	my @m = split /\s+/, $to_handle->{message} if defined $to_handle->{message};

	# ---------- Sync response ---------- #

	if ($to_handle->{command} eq $Command->{Contact_List}->{sync})
	{
		return 1;
	}

	# ---------- Privacy ---------- #

	if ($to_handle->{command} eq $Command->{Contact_List}->{block_privacy})
	{
		return 1 if $m[1] eq 'AL' && !$self->block_privacy;
		return 1 if $m[1] eq 'BL' && $self->block_privacy;

		# AL - allow all the users to send message except those in block list
		# BL - allow only users in the allow list

		my $cmd = ($self->block_privacy) ? 'BL' : 'AL';
		$self->send_packet('block_privacy', "$cmd\r\n");
		return 1;
	}

	# ---------- Own phone numbers ---------- #

	if ($to_handle->{command} eq $Command->{Contact_List}->{phone_list_own})
	{
		return 1 if !defined $m[2];

		# 0 - contact list version
		# 1 - phone number type
		# 2 - phone number

		$self->phone->{home} = $m[2] if $m[1] eq $Command->{Contact_List}->{phone_home};
		$self->phone->{mobile} = $m[2] if $m[1] eq $Command->{Contact_List}->{phone_mobile};
		$self->phone->{work} = $m[2] if $m[1] eq $Command->{Contact_List}->{phone_work};
		$self->{mobile_device} = $m[2] if $m[1] eq $Command->{Contact_List}->{mobile_device};
		$self->{mobile_device_my} = $m[2] if $m[1] eq $Command->{Contact_List}->{mobile_device_my};
		return 1;
	}

	# ---------- User's phone numbers ---------- #

	if ($to_handle->{command} eq $Command->{Contact_List}->{phone_list_user})
	{
		return 1 if !defined $m[3];

		# 0 - contact list version
		# 1 - user's passport
		# 2 - phone number type
		# 3 - phone number

		my $c = $self->{_contact}->{$m[1]};

		$c->{phone}->{home} = $m[3] if $m[2] eq $Command->{Contact_List}->{phone_home};
		$c->{phone}->{mobile} = $m[3] if $m[2] eq $Command->{Contact_List}->{phone_mobile};
		$c->{phone}->{work} = $m[3] if $m[2] eq $Command->{Contact_List}->{phone_work};
		$c->{mobile_device} = $m[3] if $m[2] eq $Command->{Contact_List}->{mobile_device};
		return 1;
	}

	# ---------- Group info ---------- #

	if ($to_handle->{command} eq $Command->{Contact_List}->{list_group})
	{
		# 0 - contact list version
		# 1 - number of the group
		# 2 - total groups
		# 3 - group ID
		# 4 - group name

		$self->{_group}->{$m[3]} = Net::MsnMessenger::Group->new(ID => $m[3], name => $m[4]);
		return 1;
	}

	# ---------- Users list ---------- #

	if ($to_handle->{command} eq $Command->{Contact_List}->{list})
	{
		if (!$m[3] && $R_Contact_List{$m[0]} eq 'reverse_list')
		{
			$self->_callback('SYNC_DONE');
			return 1;
		}
		return 1 if !$m[3];

		# 0 - contact list
		# 1 - contact list version
		# 2 - number of the contact list member
		# 3 - total contact list members
		# 4 - user's passport
		# 5 - user's friendly name
		# 6 - group ID the users is in (or groups separated by a space)

		# The contact could be already created for the initial status
		$self->{_contact}->{$m[4]} ||= Net::MsnMessenger::Contact->new(passport => $m[4]);

		my $c = $self->{_contact}->{$m[4]};
		$c->add_to_list($R_Contact_List{$m[0]});

		# The friendly name could be already changed (ILN)
		# If it is force the server to change it to the newest one
		if (defined $c->fname && $c->fname ne $m[5])
		{
			$self->change_friendly_name($c->passport, $c->fname, 1);
		}
		else { $c->fname($m[5]) }

		# Add the user to a group if the group exists
		if (defined $m[6])
		{
			for (split /,/, $m[6])
			{
				$c->add_to_group($_);
				$self->{_group}->{$_}->add_user($c->passport);
			}
		}

		$c->status('offline') if !$c->status;
		$c->connected(1)      if $c->status ne 'offline';

		# See if there are users that added this passport to their contact lists

		if ($m[2] == $m[3] && $R_Contact_List{$m[0]} eq 'reverse_list')
		{
			$self->_callback('SYNC_DONE');
			for (sort keys %{$self->{_contact}})
			{
				my $cc = $self->{_contact}->{$_};

				if (!$cc->is_user_in_list('allow_list') && !$cc->is_user_in_list('block_list'))
				{
					$self->_callback('ADD_BY_USER', $cc->passport, $cc->fname) if
					    $cc->is_user_in_list('reverse_list');

					$self->add_user($cc->passport, 'allow_list') if
					    $cc->is_user_in_list('forward_list');
				}
			}
		}
		return 1;
	}

	if ($to_handle->{command} eq $Command->{Contact_List}->{auto_allow})
	{
		# 0 - contact list version
		# 1 - A | N

		# A - auto_allow == 0   asks in the client (default)
		# N - auto_allow == 1   client should add the user to the allow list

		my $aa = (defined $self->auto_allow && $self->auto_allow) ? 'N' : 'A';

		if (defined $self->auto_allow)
		{
			$self->send_packet('auto_allow', $aa, "\r\n") if $m[1] ne $aa;
		}
		return 1;
	}
	$self->_callback('DEBUG', "BUG: _handle_Contact_List; command: $to_handle->{command}") if
	    $self->debug;

	undef;
	# _handle_Contact_List
}

# Net::MsnMessenger->_handle_Error
sub _handle_Error
{
	my ($self, $to_handle) = @_;
	my $error = (exists $Error{$to_handle->{command}})
	    ? $Error{$to_handle->{command}} : "Unknown error ($to_handle->{command})";

	if ($to_handle->{command} eq 208 || $to_handle->{command} eq 216 || $to_handle->{command} eq 217 &&
	    !$self->{_swb}->[$to_handle->{swb_session}]->get_users)
	{
		$self->_callback('CLOSE_CHAT', $to_handle->{swb_session},
				 $self->{_swb}->[$to_handle->{swb_session}]->inviting);
		$self->disconnect_swb($to_handle->{swb_session});
	}

	elsif ($to_handle->{command} eq 209 && defined $to_handle->{trans_id} && !defined $to_handle->{swb_session} &&
	       !defined $to_handle->{file_session})
	{
		# Try to get around the restricted words, if it fails the seconds time, give up
		my $old_pkt = $self->{_client}->{_event}->{outgoing}->{$to_handle->{trans_id}};

		if (defined $old_pkt)
		{
			my ($passport, $fname) = $old_pkt =~ /(\S+)\s+(\S+)\r\n$/;

			if (defined $self->{_err209}->{$passport} && $self->{_err209}->{$passport} eq $fname)
			{
				delete $self->{_err209}->{$passport};
			}
			else
			{
				$fname =~ s/(^|\s)(\S)/$1 . sprintf "%%%2x", ord($2)/eg;
				$self->send_packet('friendly_name', $passport, $fname, "\r\n");

				$self->{_err209}->{$passport} = $fname;
				return 1;
			}
		}
	}

	if ($to_handle->{command} eq 911)
	{
		$self->disconnect;
		$self->_callback('DISCONNECT', "Authentication failed. Probably a misspelled password.");
		return 1;
	}

	# If the server is too busy it usualy sends the error as a reply to USR, try to catch these and
	# disconnect.

	if (defined $to_handle->{trans_id} && !defined $to_handle->{swb_session} &&
	    !defined $to_handle->{file_session})
	{
		my $old_pkt = $self->{_client}->{_event}->{outgoing}->{$to_handle->{trans_id}};
		my ($old_cmd) = $old_pkt =~ /^(\S+)/;

		if (defined $old_cmd && $old_cmd eq $Command->{Login}->{user})
		{
			$self->disconnect;
			$self->_callback('DISCONNECT_FORCED', $error);
			return 1;
		}
	}

	$self->_callback('SERVER_ERROR', $error);
	1;
}

# Net::MsnMessenger->_handle_Event
sub _handle_Event
{
	my ($self, $to_handle) = @_;
	my @m = split /\s+/, $to_handle->{message} if defined $to_handle->{message};

	# ---------- Disconnected from the server ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{quit})
	{
		# Already disconnected - correct the flags
		$self->disconnect;

		if (defined $m[0] && $m[0] eq 'OTH')
		{
			$self->_callback('DISCONNECT', "Logged in from another location");
			return 1;
		}
		if (defined $m[0] && $m[0] eq 'SSD')
		{
			$self->_callback('DISCONNECT', "Server is going down for maintenance");
			return 1;
		}

		# Unknown reason
		$self->_callback('DISCONNECT', "The server disconnected for unknown reason");
		return 1;
	}

	# ---------- Own status was changed ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{change_my_status})
	{
		if ($R_Status{$m[0]} eq $self->initial_status && $self->status eq 'offline')
		{
			$self->status($self->initial_status);
			$self->_callback('SIGNED_IN');

			# Initial status (still login response). Continue synchronizing the lists
			$self->send_packet('sync', 0, "\r\n");
			return 1;
		}

		if ($R_Status{$m[0]} ne $self->status)
		{
			$self->status($R_Status{$m[0]});
			$self->_callback('CHANGE_MY_STATUS', $self->status);
		}
		return 1;
	}

	# ---------- Initial status of users in contact list ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{initial_status})
	{
		# 0 - user's status
		# 1 - passport
		# 2 - friendly name

		# The user probably doesn't yet exist in the contact list
		$self->{_contact}->{$m[1]} ||= Net::MsnMessenger::Contact->new(passport => $m[1]);

		my $c = $self->{_contact}->{$m[1]};
		$c->connected(1);
		$c->status($R_Status{$m[0]});
		$c->fname($m[2]);

		$self->_callback('INITIAL_STATUS', $c->passport, $c->fname, $c->status);
		return 1;
	}

	# ---------- User changed status ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{login})
	{
		# 0 - user's status
		# 1 - passport
		# 2 - friendly name

		$self->{_contact}->{$m[1]} ||= Net::MsnMessenger::Contact->new(passport => $m[1]);

		my $c = $self->{_contact}->{$m[1]};
		my $old_status = $c->status if defined $c->status;

		$c->status($R_Status{$m[0]});
		$c->connected(1);

		# If the current friendly name is not the one we have cached it was changed
		# by the user since last time. If this happens use the newer friendly name and
		# force the server to change it physically
		if (defined $c->fname && $c->fname ne $m[2])
		{
			$self->change_friendly_name($c->passport, $m[2], 1);
		}

		$c->fname($m[2]);
		$self->_callback('CHANGE_STATUS', $c->passport, $c->fname, $c->status, $old_status);
		return 1;
	}

	# ---------- User changed status to offline (logged off) ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{logout})
	{
		# 0 - user's passport

		$self->{_contact}->{$m[0]} ||= Net::MsnMessenger::Contact->new(passport => $m[0]);

		$self->{_contact}->{$m[0]}->status('offline');
		$self->{_contact}->{$m[0]}->connected(0);

		$self->{_contact}->{$m[0]}->{$_} = undef for 'ip_address', 'user_agent';

		$self->_callback('CHANGE_STATUS_OFFLINE', $m[0]);
		return 1;
	}

	# ---------- Answer the challenge ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{challenge})
	{
		# 0 - challenge magic

		my $checksum = Digest::MD5::md5_hex("$m[0]$Challenge");
		$self->send_packet('query', 'msmsgs@msnmsgr.com', "32\r\n$checksum");
		return 1;
	}

	# ---------- Changed friendly name ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{friendly_name})
	{
		# 0 - new contact list version
		# 1 - user's passport
		# 2 - friendly name

		# Own friendly name
		if ($m[1] eq $self->passport)
		{
			$self->fname($m[2]);
			$self->_callback('CHANGE_MY_FRIENDLY_NAME', $self->fname);
			return 1;
		}

		# User's friendly name
		$self->{_contact}->{$m[1]}->fname($m[2]);
		return 1;
	}

	# ---------- Added (by) a user ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{add_user})
	{
		# 0 - contact list
		# 1 - new contact list version
		# 2 - passport
		# 3 - friendly name
		# 4 - group

		if ($to_handle->{trans_id} eq 0 && $m[0] eq $Contact_List{reverse_list})
		{
			$self->{_contact}->{$m[2]} ||= Net::MsnMessenger::Contact->new(passport => $m[2]);

			my $c = $self->{_contact}->{$m[2]};
			if (!defined $c->fname || $c->fname ne $m[3])
			{
				$self->change_friendly_name($c->passport, $m[3], 1);
				$c->fname($m[3]);
			}
			$c->add_to_list('reverse_list');
			$self->_callback('ADD_BY_USER', $c->passport, $c->fname);
		}
		else
		{
			$self->{_contact}->{$m[2]} ||= Net::MsnMessenger::Contact->new(passport => $m[2]);

			my $c = $self->{_contact}->{$m[2]};
			$c->add_to_list($R_Contact_List{$m[0]});

			$c->fname($m[3]) if $m[2] ne $m[3];
			$c->status('offline') if !$c->status;

			# Add the user to group for forward list
			if (defined $m[4])
			{
				$self->{_group}->{$m[4]}->add_user($c->passport);
				$c->add_to_group($m[4]);
			}

			# User was added to forward list
			if ($R_Contact_List{$m[0]} eq 'forward_list')
			{
				(defined $m[4])
				    ? $self->_callback('ADD_USER', $c->passport, $self->{_group}->{$m[4]}->name)
				    : $self->_callback('ADD_USER', $c->passport);
			}

			# User was blocked
			if ($R_Contact_List{$m[0]} eq 'block_list')
			{
				$self->_callback('BLOCK_USER', $c->passport);
			}
			if ($R_Contact_List{$m[0]} eq 'allow_list')
			{
				$self->_callback('ALLOW_USER', $c->passport);
			}
		}
		return 1;
	}

	# ---------- Added a group ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{add_group})
	{
		# 0 - contact list version
		# 1 - new group's name
		# 2 - new group's ID
		# 3 - some string (what the hell is that???)

		$self->{_group}->{$m[2]} = Net::MsnMessenger::Group->new(ID => $m[2], name => $m[1]);
		$self->_callback('ADD_GROUP', $self->{_group}->{$m[2]}->name);
		return 1;
	}

	# ---------- Removed (by) a user ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{remove_user})
	{
		# 0 - contact list
		# 1 - contact list version
		# 2 - passport
		# 3 - group ID

		if ($to_handle->{trans_id} eq 0 && $m[0] eq $Contact_List{reverse_list})
		{
			$self->{_contact}->{$m[2]}->remove_from_list('reverse_list');
			$self->_callback('REMOVE_BY_USER', $m[2]);
		}
		else
		{
			if ($R_Contact_List{$m[0]} eq 'forward_list')
			{
				if (defined $m[3])
				{
					$self->{_group}->{$m[3]}->remove_user($m[2]);
					$self->{_contact}->{$m[2]}->remove_from_group($m[3]);

					if (!$self->{_contact}->{$m[2]}->group)
					{
						$self->{_contact}->{$m[2]}->remove_from_list('forward_list');
					}
					$self->_callback('REMOVE_USER', $m[2], $self->{_group}->{$m[3]}->name);
				}
				else
				{
					$self->{_contact}->{$m[2]}->remove_from_list('forward_list');
					$self->_callback('REMOVE_USER', $m[2]);
				}
			}
			else
			{
				$self->{_contact}->{$m[2]}->remove_from_list($R_Contact_List{$m[0]});
			}

			if ($R_Contact_List{$m[0]} eq 'allow_list')
			{
				$self->_callback('UNALLOW_USER', $m[2]);
			}
			if ($R_Contact_List{$m[0]} eq 'block_list')
			{
				$self->_callback('UNBLOCK_USER', $m[2]);
			}
		}

		if (!$self->{_contact}->{$m[2]}->c_list)
		{
			# The user is in no contact lists anymore, delete the contact
			delete $self->{_contact}->{$m[2]};
		}

		return 1;
	}

	# ---------- Removed a group ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{remove_group})
	{
		# 0 - contact list version
		# 1 - group ID

		my $name = $self->{_group}->{$m[1]}->name;
		delete $self->{_group}->{$m[1]};

		$self->_callback('REMOVE_GROUP', $name);
		return 1;
	}

	# ---------- Renamed a group ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{rename_group})
	{
		# 0 - contact list version
		# 1 - group ID
		# 2 - new group's name
		# 3 - 0

		my $old_name = $self->{_group}->{$m[1]}->name;
		$self->{_group}->{$m[1]}->name($m[2]);

		$self->_callback('RENAME_GROUP', $m[1], $old_name, $m[2]);
		return 1;
	}

	# ---------- Member search results ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{find})
	{
		# 0 - number of the result
		# 1 - total number of results
		# 2-6 - first name, last name, city, state, country (some may not be filled)

		my $r_hash;

		for (2 .. @m-1)
		{
			if (defined $m[$_] && $m[$_] =~ /^(\S+?)=(\S+)$/)
			{
				$r_hash->{lc $1} = $2;
			}
		}

		push @{$self->{_f_results}->{$to_handle->{trans_id}}}, $r_hash;
		$self->_callback('FIND_RESULTS', @{$self->{_f_results}->{$to_handle->{trans_id}}}) if $m[0] == $m[1];
		return 1;
	}

	# ---------- Ping response ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{pong})
	{
		$self->_callback('PONG');
		return 1;
	}

	# ---------- Response on a successful challenge ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{query})
	{
		$self->_callback('DEBUG', "Challenge successful") if $self->debug;
		return 1;
	}

	# ---------- Hotmail or Profile URLs ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{url})
	{
		# 0 - rru
		# 1 - action
		# 2 - id (both for the hotmail page)

		my $new_page = $self->_get_hotmail_page($m[0], $m[1], $m[2]);
		$self->_callback('URL', $new_page);
		return 1;
	}

	# ---------- E-mail invitation confirmed ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{send_invitation})
	{
		# 0 - OK

		if (defined $m[0] && $m[0] eq 'OK')
		{
			my $passport = $1 if
			    $self->{_client}->{_event}->{outgoing}->{$to_handle->{trans_id}} =~ /^\S+\s+\d+\s+(\S+)/;

			$self->_callback('EMAIL_INVITE_SUCCESS', $passport) if defined $passport;
		}
		return 1;
	}

	# ---------- E-mail invitation confirmed (member search) ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{find_invite})
	{
		# 0 - OK

		if (defined $m[0] && $m[0] eq 'OK')
		{
			my $find_nr = $1 if
			    $self->{_client}->{_event}->{outgoing}->{$to_handle->{trans_id}} =~ /^\S+\s+\d+\s+(\d+)/;

			$self->_callback('FIND_INVITE_SUCCESS', $find_nr) if defined $find_nr;
		}
		return 1;
	}

	# ---------- Version response ---------- #

	if ($to_handle->{command} eq $Command->{Event}->{client_version})
	{
		# 0 - recommended version of the client
		# 1 - the same as 0
		# 2 - 1.0.0000
		# 3 - URL to download the recommended version from
		# 4 - URL to find more information about the client

		return 1;
	}

	$self->_callback('DEBUG', "BUG: _handle_Event; command: $to_handle->{command}") if
	    $self->debug;

	undef;
	# _handle_Event
}

# Net::MsnMessenger->_handle_File
sub _handle_File
{
	my ($self, $to_handle) = @_;
	my @m = split /\s+/, $to_handle->{message} if defined $to_handle->{message};

	my $this_swb = $self->{_swb}->[$to_handle->{swb_session}];
	my $this_file = $this_swb->{_file}->[$to_handle->{file_session}];

	if ($to_handle->{command} eq $Command->{Login}->{version})
	{
		# 0 - MSNFTP

		if (@m && grep {$_ eq 'MSNFTP'} @m)
		{
			$this_file->send_packet('user', $self->passport, $this_file->authcookie, "\r\n");
			return 1;
		}

		$self->disconnect_file($to_handle->{swb_session}, $to_handle->{file_session});

		$self->_callback('FILE_RECEIVE_CANCEL', $to_handle->{swb_session}, $to_handle->{file_session},
				 "Invalid FTP version response");
		return undef;
	}

	if ($to_handle->{command} eq $Command->{File}->{file})
	{
		# 0 - number of bytes in the file

		if ($m[0] ne $this_file->file_size)
		{
			$self->disconnect_file($to_handle->{swb_session}, $to_handle->{file_session});
			$self->_callback('FILE_RECEIVE_CANCEL', $to_handle->{swb_session}, $to_handle->{file_session},
					 "Invalid file length: $m[0]");
			return undef;
		}

		$this_file->send_packet('file_transfer', "\r\n");
		$this_file->file_transfering(1);

		$self->_callback('FILE_RECEIVE_START', $to_handle->{swb_session}, $to_handle->{file_session});
		return 1;
	}

	$self->_callback('DEBUG', "BUG: _handle_File; command: $to_handle->{command}") if
	    $self->debug;

	undef;
	# _handle_File
}

# Net::MsnMessenger->_handle_File_Send
sub _handle_File_Send
{
	my ($self, $to_handle) = @_;
	my @m = split /\s+/, $to_handle->{message} if defined $to_handle->{message};

	my $this_swb = $self->{_swb}->[$to_handle->{swb_session}];
	my $this_file = $this_swb->{_file}->[$to_handle->{file_session}];

	if ($to_handle->{command} eq $Command->{Login}->{version})
	{
		# 0 - MSNFTP

		if (@m && grep {$_ eq 'MSNFTP'} @m)
		{
			$this_file->send_packet('version', "MSNFTP\r\n");
			return 1;
		}

		$self->disconnect_file($to_handle->{swb_session}, $to_handle->{file_session});
		$self->_callback('FILE_SEND_CANCEL', $to_handle->{swb_session}, $to_handle->{file_session},
				 "Invalid FTP version");
		return undef;
	}

	if ($to_handle->{command} eq $Command->{Login}->{user})
	{
		# 0 - user's passport
		# 1 - auth cookie

		if ($m[1] ne $this_file->authcookie)
		{
			$self->disconnect_file($to_handle->{swb_session}, $to_handle->{file_session});
			$self->_callback('FILE_SEND_CANCEL', $to_handle->{swb_session}, $to_handle->{file_session},
					 "Invalid AuthCookie received: $m[1]");
			return undef;
		}

		$this_file->sending_to($m[0]);
		$this_file->send_packet('file', $this_file->file_size, "\r\n");
		return 1;
	}

	if ($to_handle->{command} eq $Command->{File}->{file_transfer})
	{
		$self->_callback('FILE_SEND_START', $to_handle->{swb_session}, $to_handle->{file_session},
				 $this_file->sending_to);
		$this_file->file_transfering(1);
		return 1;
	}

	if ($to_handle->{command} eq $Command->{File}->{file_cancel})
	{
		$self->disconnect_file($to_handle->{swb_session}, $to_handle->{file_session});
		$self->_callback('FILE_SEND_CANCEL', $to_handle->{swb_session}, $to_handle->{file_session},
				 "The recipient cancelled the file transfer");
		return 1;
	}

	if ($to_handle->{command} eq $Command->{Switchboard}->{leave})
	{
		$self->disconnect_file($to_handle->{swb_session}, $to_handle->{file_session});

		if ($m[0] eq '16777987' || $m[0] eq '16777989')
		{
			# File successfully transfered
			$self->_callback('FILE_SEND_SUCCESS', $to_handle->{swb_session}, $to_handle->{file_session});
		}
		elsif ($m[0] eq '2147942405')
		{
			$self->_callback('FILE_SEND_CANCEL', $to_handle->{swb_session}, $to_handle->{file_session},
					 "The recipient is out of disk space");
		}
		elsif ($m[0] eq '2164261682')
		{
			$self->_callback('FILE_SEND_CANCEL', $to_handle->{swb_session}, $to_handle->{file_session},
					 "The recipient cancelled the file transfer");
		}
		elsif ($m[0] eq '2164261683')
		{
			$self->_callback('FILE_SEND_CANCEL', $to_handle->{swb_session}, $to_handle->{file_session},
					 "The sender cancelled the file transfer");
		}
		elsif ($m[0] eq '2164261694')
		{
			$self->_callback('FILE_SEND_CANCEL', $to_handle->{swb_session}, $to_handle->{file_session},
					 "The connection is blocked");
		}
		else
		{
			if (!defined $this_file->transfered || $this_file->transfered < $this_file->file_size)
			{
				$self->_callback('FILE_SEND_CANCEL', $to_handle->{swb_session},
						 $to_handle->{file_session});
			}
			else
			{
				$self->_callback('DEBUG', "Unkown BYE response (probably SUCCESS)") if $self->debug;
				$self->_callback('FILE_SEND_SUCCESS', $to_handle->{swb_session},
						 $to_handle->{file_session});
			}
		}
		return 1;
	}

	$self->_callback('DEBUG', "BUG: _handle_File_Send; command: $to_handle->{command}") if
	    $self->debug;

	undef;
	# _handle_File_Send
}

# Net::MsnMessenger->_handle_Login
sub _handle_Login
{
	my ($self, $to_handle) = @_;
	my @m = split /\s+/, $to_handle->{message} if defined $to_handle->{message};

	# ---------- Protocol Version ---------- #

	if ($to_handle->{command} eq $Command->{Login}->{version})
	{
		# For success:
		# S: VER TransactionID version

		# For failure:
		# S: VER TransactionID 0 version

		# Set an error if the response doesn't include our version or if it
		# begins with a value of 0

		if ($to_handle->{message} !~ $Msn_version || $m[0] eq 0)
		{
			$self->_callback('SERVER_ERROR', "Invalid version response");
			return undef;
		}
		$self->send_packet('info', "\r\n");
		return 1;
	}

	# ---------- Server policy Info ---------- #

	if ($to_handle->{command} eq $Command->{Login}->{info})
	{
		# Possible responses:
		# 0 - MD5 - Notification server (main server login)
		# 0 - CKI - Switchboard server (switchboard server login)

		# Reply with user name to the info response
		if ($m[0] eq 'MD5')
		{
			$self->send_packet('user', "MD5 I", $self->passport, "\r\n");
		}
		return 1;
	}

	# ---------- Redirect to notification or switchboard server ---------- #

	if ($to_handle->{command} eq $Command->{Login}->{redirect})
	{
		if ($m[0] ne 'NS' && $m[0] ne 'SB')
		{
			$self->_callback('SERVER_ERROR', "Invalid redirect response");
			return undef;
		}
		my ($server, $port) = split /:/, $m[1];

		# Follow the redirect to the received server address

		# 0 - server type
		# 1 - server ip adress and port

		# ---------- Notification server ---------- #

		# 2 - 0 (???)
		# 3 - another ip address (no clue what is it for)

		if ($m[0] eq 'NS')
		{
			$self->_callback('LOGIN_PROGRESS', "Following redirect to server $server port $port");
			
			# Disconnect from the dispatch server
			$self->disconnect;

			# Connect to the notification server
			$self->{_client} = Net::MsnMessenger::Connection->new(
				address         => $server,
				port            => $port,
				msn             => $self,
				connection_type => 'client',
				server_type     => 'NS',
				protocol        => 'tcp',
			);
			$self->create_connection or return undef;
			$self->connected(1);
			$self->send_packet('version', "$Msn_version\r\n");
			return 1;
		}

		# ---------- Switchboard server ---------- #

		# 2 - CKI
		# 3 - hash to use when logging in to the server

		$self->{_swb}->[$self->{_swb_sessions}] = Net::MsnMessenger::Switchboard->new(
			server    => $server,
			port      => $port,
			conn_hash => $m[3],
			swb_id    => $self->{_swb_sessions},
			msn       => $self,
		);
		$self->{_swb}->[$self->{_swb_sessions}++]->invite_login;
		return 1;
	}

	# ---------- User authentication ---------- #

	if ($to_handle->{command} eq $Command->{Login}->{user})
	{
		# 0 - MD5
		# 1 - I (Initiate message) - only to send
		# 1 - S (Subsequent message)

		if ($m[0] ne 'MD5' && $m[0] ne 'OK')
		{
			$self->_callback('SERVER_ERROR', "Invalid user response");
			return undef;
		}

		# ----- MD5 encrypted password request

		if ($m[0] eq 'MD5')
		{
			# 0 - MD5
			# 1 - S (Subsequent message)
			# 2 - magic to add to the password

			$self->_callback('LOGIN_PROGRESS', "Sending the password for " . $self->passport);

			my $checksum = Digest::MD5::md5_hex("$m[2]$self->{password}");
			$self->send_packet('user', "MD5 S $checksum\r\n");
			return 1;
		}

		# ----- Done authorizing on the Notification server

		# 0 - OK
		# 1 - own passport
		# 2 - own friendly name
		# 3 - 1 (???)

		$self->signed_in(1);

		# If the current friendly name got from server is not the same as $self->fname
		# force the server to change it
		if (defined $self->fname && $self->fname ne $m[2])
		{
			$self->_callback('DEBUG', "Fname defined as ".$self->fname."; changing on the server");
			$self->change_friendly_name($self->passport, $self->fname);
		}
		$self->fname($m[2]);

		# >> Otherwise wait for the server reply
		$self->{initial_status} = 'online' if !defined $self->{initial_status};
		if (!exists $Status{lc($self->{initial_status})})
		{
			$self->{initial_status} = lc(join '_', (split /\s+/, $self->{initial_status}));
			$self->{initial_status} = 'online' if !exists $Status{$self->{initial_status}};
		}
		my $p_status = ucfirst $self->initial_status;  $p_status =~ s/_/ /g;

		$self->_callback('LOGIN_PROGRESS', "Setting the initial status to $p_status");
		$self->change_status(lc $self->initial_status);

		$self->send_client_version(0) if $self->send_client_version &&
		    (!defined $self->os_type || !defined $self->os_version || !defined $self->os_arch ||
		     !defined $self->client_name || !defined $self->client_version);
										
		$self->_send_client_version($self->os_type, $self->os_version, $self->os_arch,
					    $self->client_name, $self->client_version);
		return 1;
	}

	$self->_callback('DEBUG', "BUG: _handle_Login; command: $to_handle->{command}") if
	    $self->debug;

	undef;
	# _handle_Login
}

# Net::MsnMessenger->_handle_Message
sub _handle_Message
{
	my ($self, $to_handle) = @_;
	my $this_swb = $self->{_swb}->[$to_handle->{swb_session}] if defined $to_handle->{swb_session};
	my ($cookie_obj, %temp_m);

	($temp_m{mime_version}) = $to_handle->{message} =~ /^MIME-Version:\s*(\S+)/m;
	($temp_m{content}) = $to_handle->{message} =~ /^Content-Type:\s*(\S+?)(?:;|\s|\r|\n)/m;
	($temp_m{charset}) = $to_handle->{message} =~ /charset=(.+);?\r\n/;

	if ($to_handle->{message} =~ /^(\S+)\s+(\S+).[^\r\n]*\r\n/)  # Message sender
	{
		$temp_m{sender} = $1;
		$temp_m{sender_name} = (defined $2 && $1 ne $2) ? $2 : $1;
	}

	if ($temp_m{content} eq 'text/plain' && defined $to_handle->{swb_session})
	{
		# Classic switchboard message
		my %temp_m_format;

		my ($t_format) = $1 if $to_handle->{message} =~ /^X-MMS-IM-Format:\s*(.+?)(?<!\;)\r\n/ms;
		my ($t_ua) = $1 if $to_handle->{message} =~ /^user-?agent:\s*(\S+?)\r\n/im;
		my ($t_message) = $to_handle->{message} =~ /^.*?\r\n\r\n(.*)$/s;

		if (defined $t_format)
		{
			$t_format =~ s/\r\n//g;
			$temp_m_format{$1} = $2 while $t_format =~ s/(\S+?)=\s*(\S*?);?(\s|$)//;
		}

		if (defined $t_ua)
		{
			if (exists $self->{_contact}->{$temp_m{sender}})
			{
				$self->{_contact}->{$temp_m{sender}}->user_agent($t_ua);
			}
			else  # Try to find the user in the current switchboard session
			{
				my $contact_ref;
				for ($this_swb->get_users)
				{
					if ($_->passport eq $temp_m{sender})
					{
						$contact_ref = $_;  last;
					}
				}
				$contact_ref->user_agent($t_ua) if defined $contact_ref;
			}
		}
		$this_swb->{incoming}->[scalar @{$this_swb->{incoming}}]->{message} = $t_message;
		$self->_callback('RECEIVE_MESSAGE', $to_handle->{swb_session}, $temp_m{sender},
				 $t_message, %temp_m_format);
		return 1;
	}

	# ---------- Hotmail messages ---------- #

	if ($temp_m{sender} eq 'Hotmail')
	{
		# ----- Initial e-mail notification

		if ($temp_m{content} =~ /(application|text)\/x-msmsgsinitialemailnotification/)
		{
			$self->_callback('DEBUG', "Parsing x-msmsgsinitialemailnotification") if $self->debug;

			$to_handle->{message} =~ /^Inbox-Unread:\s*(.+?)\r\n/m;
			$self->_callback('EMAIL_UNREAD', $1);
		}

		# ----- An e-mail received

		if ($temp_m{content} =~ /(application|text)\/x-msmsgsemailnotification/)
		{
			$self->_callback('DEBUG', "Parsing x-msmsgsemailnotification") if $self->debug;

			my ($from_addr) = $to_handle->{message} =~ /^From-Addr:\s*(\S+?)\r\n/m;
			my ($from)      = $to_handle->{message} =~ /^From:\s*(\S.*?)\r\n/m;
			my ($subject)   = $to_handle->{message} =~ /^Subject:\s*(\S.*?)\r\n/m;

			$self->_callback('EMAIL_NEW', $from_addr, $from, $subject);
		}

		# ----- E-mail activity

		if ($temp_m{content} =~ /(application|text)\/x-msmsgsactivemailnotification/)
		{
			$self->_callback('DEBUG', "Parsing x-msmsgsactiveemailnotification") if $self->debug;

			$to_handle->{message} =~
			    /^Src-Folder:\s*(.*?)\r\n.*^Dest-Folder:\s*(.*?)\r\n.*^Message-Delta:\s*(.*?)\r\n/sm;

			$self->_callback('EMAIL_ACTIVE', $1, $2, $3);
		}

		# ----- Profile message

		if ($temp_m{content} eq 'text/x-msmsgsprofile')
		{
			$self->_callback('DEBUG', "Parsing x-msmsgsprofile") if $self->debug;

			for ('LoginTime', 'sid', 'kv', 'MSPAuth')
			{
				$self->{hotmail}->{lc $_} = $1 if $to_handle->{message} =~ /^$_:\s*(.+?)\r\n/m;
			}
			$self->{hotmail}->{username} = $self->passport;
			($self->{hotmail}->{login}) = $self->{hotmail}->{username} =~ /^(\S+?)\@/;
			($self->{hotmail}->{enabled}) = $to_handle->{message} =~ /^EmailEnabled:\s*(.+?)\r\n/m;

			if ($to_handle->{message} =~ /^lang_preference:\s*(.+?)\r\n/m && length($1))
			{
				$self->{_account}->{LCID} = $1;
				$self->_callback('DEBUG', "Language: " . $self->lcid2string($1));
			}

			for ('age', 'birthday', 'country', 'gender', 'kid', 'postalcode', 'wallet')
			{
				if ($to_handle->{message} =~ /^$_:\s*(\S+?)\r\n/mi)
				{
					$self->{_account}->{$_} = $1;
					$self->_callback('DEBUG', "$_ defined to $1");
				}
			}
		}
		return 1;
	}

	# ---------- Control messages ---------- #

	if ($temp_m{content} eq 'text/x-msmsgscontrol')
	{
		$self->_callback('DEBUG', "Parsing x-msmsgscontrol") if $self->debug;
		return $self->_callback('TYPING_USER', $to_handle->{swb_session}, $temp_m{sender});
	}

	# ---------- System messages ---------- #

	elsif ($temp_m{content} =~ /(application|text)\/x-msmsgssystemmessage/ && $temp_m{sender} eq 'Hotmail')
	{
		$self->_callback('DEBUG', "Parsing x-msmsgssystemmessage") if $self->debug;

		($temp_m{type}) = $to_handle->{message} =~ /^Type:\s*(.+?)\r\n/m;
		($temp_m{arg1}) = $to_handle->{message} =~ /^Arg1:\s*(.+?)\r\n/m;

		if ($temp_m{type} eq '1' && defined $temp_m{arg1})
		{
			$self->_callback('SERVER_MESSAGE',
					 "The server will shut down for maintenance in $temp_m{arg1} ".
					 "minutes(s). At the time you will be automatically signed out");
		}
		else
		{
			$self->_callback('UNIMPLEMENTED', "Unknown system message:\r\n$to_handle->{message}");
		}
		return 1;
	}

	# ---------- Invitation messages ---------- #

	elsif ($temp_m{content} eq 'text/x-msmsgsinvite')
	{
		$self->_callback('DEBUG', "Parsing x-msmsgsinvite") if $self->debug;

		($temp_m{application_name})   = $to_handle->{message} =~ /^Application-Name:\s*(.+?)\r\n/m;
		($temp_m{application_guid})   = $to_handle->{message} =~ /^Application-GUID:\s*(.+?)\r\n/m;
		($temp_m{invitation_command}) = $to_handle->{message} =~ /^Invitation-Command:\s*(.+?)\r\n/m;
		($temp_m{invitation_cookie})  = $to_handle->{message} =~ /^Invitation-Cookie:\s*(.+?)\r\n/m;

		my ($inv_type, $cookie_obj);

		if (uc($temp_m{invitation_command}) eq 'INVITE' && defined $temp_m{application_guid})
		{
			for my $type(keys %Net::MsnMessenger::Data::Guid)
			{
				$inv_type = $type if
				    $temp_m{application_guid} eq $Net::MsnMessenger::Data::Guid{$type};
			}
		}

		# ---------- Invited to file transfer

		if (defined $inv_type && $inv_type eq 'File Transfer')
		{
			($temp_m{application_file}) = $to_handle->{message} =~/^Application-File:\s*(.+?)\r\n/m;
			($temp_m{application_filesize}) = $to_handle->{message} =~
			    /^Application-FileSize:\s*(.+?)\r\n/m;

			$this_swb->{_file}->[$this_swb->{_file_sessions}] = Net::MsnMessenger::File->new(
				file              => $temp_m{application_file},
				file_size         => $temp_m{application_filesize},
				invitation_cookie => $temp_m{invitation_cookie},
				swb_session       => $to_handle->{swb_session},
				file_session      => $this_swb->{_file_sessions},
				incoming          => 1,
				msn               => $self,
			);

			$self->_callback('FILE_INVITED', $to_handle->{swb_session}, $this_swb->{_file_sessions}++,
					 $temp_m{sender}, $temp_m{sender_name}, $temp_m{application_file},
					 $temp_m{application_filesize});
			return 1;
		}

		# ---------- NetMeeting

		elsif (defined $inv_type && $inv_type eq 'NetMeeting')
		{
			($temp_m{session_id}) = $to_handle->{message} =~ /^Session-ID:\s*(.+?)\r\n/m;

			$this_swb->{_netmeeting}->[$this_swb->{_netmeeting_sessions}] =
			    Net::MsnMessenger::NetMeeting->new(
				invitation_cookie  => $temp_m{invitation_cookie},
				session_id         => $temp_m{session_id},
				swb_session        => $to_handle->{swb_session},
				netmeeting_session => $this_swb->{_netmeeting_sessions},
				incoming           => 1,
				msn                => $self,
			);
			$self->_callback('NETMEETING_INVITED', $to_handle->{swb_session},
					 $this_swb->{_netmeeting_sessions}++, $temp_m{sender}, $temp_m{sender_name});
		}

		# ---------- Voice Conversation

		elsif (     $self->testing &&         defined $inv_type && $inv_type eq 'Voice Conversation')
		{
			($temp_m{session_id}) = $to_handle->{message} =~ /^Session-ID:\s*(.+?)\r\n/m;
			($temp_m{session_protocol}) = $to_handle->{message} =~ /^Session-Protocol:\s*(.+?)\r\n/m;

			$this_swb->{_voice}->[$this_swb->{_voice_sessions}] = Net::MsnMessenger::Voice->new(
				invitation_cookie => $temp_m{invitation_cookie},
				session_id        => $temp_m{session_id},
				session_protocol  => $temp_m{session_protocol},
				swb_session       => $to_handle->{swb_session},
				voice_session     => $this_swb->{_voice_sessions},
				incoming          => 1,
				msn               => $self,
			);

			$self->_callback('VOICE_INVITED', $to_handle->{swb_session}, $this_swb->{_voice_sessions}++,
					 $temp_m{sender}, $temp_m{sender_name});
			return 1;
		}


		# ---------- Remote Assistance

		elsif (     $self->testing &&         defined $inv_type && $inv_type eq 'Remote Assistance')
		{
			($temp_m{session_id}) = $to_handle->{message} =~ /^Session-ID:\s*(.+?)\r\n/m;

			$this_swb->{_rassistance}->[$this_swb->{_rassistance_sessions}] =
			    Net::MsnMessenger::RemoteAssistance->new(
				invitation_cookie   => $temp_m{invitation_cookie},
				session_id          => $temp_m{session_id},
				swb_session         => $to_handle->{swb_session},
				rassistance_session => $this_swb->{_rassistance_sessions},
				incoming            => 1,
				msn                 => $self,
			);

			$self->_callback('RA_INVITED', $to_handle->{swb_session}, $this_swb->{_rassistance_sessions}++,
					 $temp_m{sender}, $temp_m{sender_name});
		}

		# ---------- Unknown (reject automatically as not installed)

		elsif (uc($temp_m{invitation_command}) eq 'INVITE')
		{
			$self->_reject_ni_cookie($temp_m{invitation_cookie});
			$self->_callback('UNKNOWN_REJECT', $to_handle->{swb_session}, $temp_m{sender},
					 $temp_m{sender_name}, $temp_m{application_name});
			return 1;
		}

		# ---------- Accept/Cancel/Context messages

		$cookie_obj = $this_swb->find_cookie($temp_m{invitation_cookie}) if
		    $temp_m{invitation_command} =~ /(accept|cancel|context)/i;

		if (uc($temp_m{invitation_command}) eq 'ACCEPT')
		{
			if (!defined $cookie_obj)
			{
				$self->error("Couldn't find matching invitation for: $temp_m{invitation_cookie}");
				return undef;
			}
			($temp_m{session_protocol}) = $to_handle->{message} =~ /^Session-Protocol:\s*(.+?)\r\n/m;
			($temp_m{conn_type}) = $to_handle->{message} =~ /^Conn-Type:\s*(.+?)\r\n/m;

			if (defined $temp_m{session_protocol} && $temp_m{session_protocol} eq 'SM1')
			{
				# The SM1 protocol uses xxx.xxx.xxx.xxx:xxxx ip/port format
				if ($to_handle->{message} =~ /IP-Address:\s*(.+?)\r\n/)
				{
					($temp_m{ip_address}, $temp_m{port}) = split /:/, $1;
				}
			}
			else
			{
				($temp_m{ip_address}) = $to_handle->{message} =~ /^IP-Address:\s*(.+?)\r\n/m;
				($temp_m{lc $_}) = $to_handle->{message} =~ /^$_:\s*(.+?)\r\n/m
				    for 'Port', 'AuthCookie';
			}
		}

		if (uc($temp_m{invitation_command} eq 'ACCEPT') && $cookie_obj->type eq 'File Transfer')
		{
			if ($cookie_obj->incoming)
			{
				if (!defined $temp_m{ip_address} || !defined $temp_m{port})
				{
					$self->error("IP or Port not defined for: $temp_m{invitation_cookie}");
					return undef;
				}

				# Save the IP Address
				if (exists $self->{_contact}->{$temp_m{sender}})
				{
					$self->{_contact}->{$temp_m{sender}}->ip_address($temp_m{ip_address});
				}
				else
				{
					my $contact_ref;
					for ($this_swb->get_users)
					{
						if ($_->passport eq $temp_m{sender})
						{
							$contact_ref = $_;  last;
						}
					}
					$contact_ref->ip_address($temp_m{ip_address}) if defined $contact_ref;
				}

				$cookie_obj->ip_address($temp_m{ip_address});
				$cookie_obj->authcookie($temp_m{authcookie});
				$cookie_obj->port($temp_m{port});

				$cookie_obj->create_client;
			}
			else
			{
				$self->_callback('FILE_ACCEPT', $cookie_obj->swb_session, $cookie_obj->file_session,
						 $temp_m{sender});
				$cookie_obj->sending_to($temp_m{sender});
				$cookie_obj->invite_accept_confirm;
			}
		}

		elsif (uc($temp_m{invitation_command}) eq 'ACCEPT' && $cookie_obj->type eq 'NetMeeting')
		{
			if ($cookie_obj->outgoing)
			{
				$self->_callback('NETMEETING_ACCEPT', $cookie_obj->swb_session,
						 $cookie_obj->netmeeting_session, $temp_m{sender});
			}
			else
			{
				$cookie_obj->invite_accept_confirm;
			}
			$cookie_obj->ip_address($temp_m{ip_address});

			$self->_callback('NETMEETING_LAUNCH', $cookie_obj->swb_session,
					 $cookie_obj->netmeeting_session, $cookie_obj->ip_address);
		}

		elsif ($self->testing &&
		       uc($temp_m{invitation_command}) eq 'ACCEPT' && $cookie_obj->type eq 'Voice Conversation')
		{
			# TODO: Finish

			$cookie_obj->ip_address($temp_m{ip_address});
			$cookie_obj->port($temp_m{port});
			$cookie_obj->conn_type($temp_m{conn_type}) if defined $temp_m{conn_type};

			if ($cookie_obj->incoming)
			{
				$cookie_obj->create_client;
			}
			else
			{
				$self->_callback('VOICE_ACCEPT', $cookie_obj->swb_session,
						 $cookie_obj->voice_session, $temp_m{sender});

				$cookie_obj->invite_accept_confirm;
			}
		}
		elsif ($self->testing &&
		       uc($temp_m{invitation_command}) eq 'ACCEPT' && $cookie_obj->type eq 'Remote Assistance')
		{
			# TODO: Finish

			if ($cookie_obj->incoming)
			{
				$cookie_obj->ip_address($temp_m{ip_address});
				$cookie_obj->port($temp_m{port});

				$cookie_obj->invite_context;
			}
			else
			{
				$self->_callback('RA_ACCEPT', $cookie_obj->swb_session,
						 $cookie_obj->rassistance_session, $temp_m{sender});

				$cookie_obj->ip_address($temp_m{ip_address});
				$cookie_obj->port($temp_m{port});

				$cookie_obj->invite_accept_confirm;
			}
		}

		if (uc($temp_m{invitation_command}) eq 'CANCEL')
		{
			($temp_m{cancel_code}) = $to_handle->{message} =~ /^Cancel-Code:\s*(.+?)\r\n/m;
			return undef if !defined $temp_m{cancel_code};

			if (!defined $cookie_obj)
			{
				$this_swb->remove_invite($temp_m{invitation_cookie});
				return 1;
			}

			my $c_reason;
			if (uc($temp_m{cancel_code}) eq 'REJECT')
			{
				$self->_callback('FILE_REJECT', $cookie_obj->swb_session, $cookie_obj->file_session,
				    $temp_m{sender}) if
					$cookie_obj->type eq 'File Transfer';

				$self->_callback('VOICE_REJECT', $cookie_obj->swb_session, $cookie_obj->voice_session,
				    $temp_m{sender}) if
					$cookie_obj->type eq 'Voice Conversation';

				$self->_callback('NETMEETING_REJECT', $cookie_obj->swb_session,
				    $cookie_obj->netmeeting_session, $temp_m{sender}) if
					$cookie_obj->type eq 'NetMeeting';

				$self->_callback('RA_REJECT', $cookie_obj->swb_session,
				    $cookie_obj->rassistance_session, $temp_m{sender}) if
					$cookie_obj->type eq 'Remote Assistance';
			}

			# The messages for the error codes here are mostly guesses or I didn't want to bother
			# with it. If someone knows better it will be appreciated.

			elsif (uc($temp_m{cancel_code}) eq 'CANCEL' || uc($temp_m{cancel_code}) eq 'TIMEOUT')
			{
				$c_reason = "The Invitation was cancelled by the user";
			}
			elsif (uc($temp_m{cancel_code}) eq 'OUTBANDCANCEL')
			{
				$c_reason = "The File Transfer was cancel by the user";
			}
			elsif (uc($temp_m{cancel_code}) eq 'FTTIMEOUT')
			{
				$c_reason = "The File Transfer timed out";
			}
			elsif (uc($temp_m{cancel_code}) eq 'FAIL')
			{
				$c_reason = "Failed for unknown reason";
			}
			elsif (uc($temp_m{cancel_code}) eq 'REJECT_NOT_INSTALLED')
			{
				$c_reason = "The user doesn't have the application installed";
			}
			elsif (uc($temp_m{cancel_code}) eq 'NETWORK_CONFIG_FAILURE')
			{
				$c_reason = "Couldn't invite the user due to network problems";
			}
			else
			{
				$c_reason = "Unknown reason: $temp_m{cancel_code}";
			}

			if ($cookie_obj->type eq 'File Transfer')
			{
				$self->disconnect_file($cookie_obj->swb_session, $cookie_obj->file_session);

				my $cb_name = ($cookie_obj->incoming) ? 'FILE_RECEIVE_CANCEL' : 'FILE_SEND_CANCEL';
				$self->_callback($cb_name, $cookie_obj->swb_session, $cookie_obj->file_session,
						 $c_reason);
			}
			elsif ($cookie_obj->type eq 'NetMeeting')
			{
				$self->_callback('NETMEETING_CANCEL', $cookie_obj->swb_session,
						 $cookie_obj->netmeeting_session, $c_reason);
			}

			elsif ($self->testing &&      $cookie_obj->type eq 'Voice Conversation')
			{
				$self->disconnect_voice($cookie_obj->swb_session, $cookie_obj->voice_session);
				$self->_callback('VOICE_CANCEL', $cookie_obj->swb_session, $cookie_obj->voice_session,
						 $c_reason);
			}
			elsif ($self->testing &&      $cookie_obj->type eq 'Remote Assistance')
			{
				$self->_callback('RA_CANCEL', $cookie_obj->swb_session,
						 $cookie_obj->rassistance_session, $c_reason);
			}
			$this_swb->remove_invite($temp_m{invitation_cookie});
			return 1;
		}

		if (uc($temp_m{invitation_command}) eq 'CONTEXT')
		{
			if ($cookie_obj->type eq 'Remote Assistance')
			{
				# TODO: Finish
			}
			return 1;
		}

		return 1;
	}

	# ---------- Client capabilities ---------- #

	elsif ($temp_m{content} eq 'text/x-clientcaps')
	{
		$self->_callback('DEBUG', "Parsing x-clientcaps") if $self->debug;
		my $contact_ref;

		if (defined $self->{_contact}->{$temp_m{sender}})
		{
			$contact_ref = $self->{_contact}->{$temp_m{sender}};
		}
		else
		{
			for ($this_swb->get_users)
			{
				if ($_->passport eq $temp_m{sender})
				{
					$contact_ref = $_;  last;
				}
			}
		}

		if (defined $contact_ref)
		{
			($contact_ref->{user_agent})   = $to_handle->{message} =~ /^Client-Name:\s*(.+?)\r\n/m;
			($contact_ref->{chat_logging}) = $to_handle->{message} =~ /^Chat-Logging:\s*(.+?)\r\n/m;
			($contact_ref->{buddy_icons})  = $to_handle->{message} =~ /^Buddy-Icons:\s*(.+?)\r\n/m;
		}
	}

	else
	{
		$self->_callback('UNIMPLEMENTED', "Unsupported message type: $temp_m{content}:\r\n".
				 $to_handle->{message});
	}
	1;
}

# Net::MsnMessenger->_handle_Notification
sub _handle_Notification
{
	my ($self, $to_handle) = @_;

	my ($site_url)   = $to_handle->{message} =~ m!  siteurl=\"(\S+?)\"         !x;
	my ($action_url) = $to_handle->{message} =~ m!  <ACTION\s+url=\"(\S+)\"/>  !x;
	my ($text)       = $to_handle->{message} =~ m!  <TEXT>(.+)</TEXT>          !xs;

	$text =~ s/\r\n//g;

	$self->_callback('ALERT', $site_url, $action_url, $text);
	1;
}

# Net::MsnMessenger->_handle_Switchboard
sub _handle_Switchboard
{
	my ($self, $to_handle) = @_;
	my @m = split /\s+/, $to_handle->{message} if defined $to_handle->{message};

	my $this_swb = $self->{_swb}->[$to_handle->{swb_session}] if defined $to_handle->{swb_session};

	# ---------- Switchboard authentication (inviting a user) ---------- #

	if ($to_handle->{command} eq $Command->{Login}->{user})
	{
		# 0 - OK
		# 1 - own passport
		# 2 - own friendly name

		if (!$self->{_inviting})
		{
			$self->_callback('DEBUG', "BUG: _handle_Switchboard - no users to invite");
			$this_swb->disconnect;
			return undef;
		}
		if ($m[0] eq 'OK' && $m[1] eq $self->passport && $m[2] eq $self->fname)
		{
			my $u_inviting = shift @{$self->{_inviting}};
			$this_swb->inviting($u_inviting);
			$this_swb->invite_to_session($u_inviting);
		}
		else
		{
			$self->_callback('SERVER_ERROR', "Failed to login to the switchboard server");
			$this_swb->disconnect;
			return undef;
		}
		return 1;
	}

	# ---------- Invited to the session ---------- #

	if ($to_handle->{command} eq $Command->{Switchboard}->{invited})
	{
		# 0 - server session ID
		# 1 - switchboard server ip and port
		# 2 - CKI
		# 3 - hash to use to login to the server
		# 4 - user's passport
		# 5 - user's friendly name

		my ($server, $port) = split /:/, $m[1];

		$self->{_swb}->[$self->{_swb_sessions}] = Net::MsnMessenger::Switchboard->new(
				server     => $server,
				port       => $port,
				swb_id     => $self->{_swb_sessions},
				swb_id_srv => $m[0],
				conn_hash  => $m[3],
				msn        => $self,
		);

		$self->{_swb}->[$self->{_swb_sessions}++]->join_session();
		return 1;
	}

	# ---------- Session established ---------- #

	if ($to_handle->{command} eq $Command->{Switchboard}->{answer})
	{
		# 0 - OK

		if ($m[0] ne 'OK')
		{
			$self->_callback('SERVER_ERROR', "Invalid ANSWER response: $m[0]");
			$this_swb->disconnect;
			return undef;
		}
		$self->_callback('OPEN_CHAT', $to_handle->{swb_session});
		return 1;
	}

	# ---------- Message delivered ---------- #

	if ($to_handle->{command} eq $Command->{Switchboard}->{acknowledge})
	{
		my $out = $this_swb->{outgoing}->{$to_handle->{trans_id}};

		$self->_callback('SEND_MESSAGE_RECEIVED', $to_handle->{swb_session}, $out->{message});
		$out->{delivered} = 1;
		return 1;
	}

	# ---------- Message not delievered ---------- #

	if ($to_handle->{command} eq $Command->{Switchboard}->{not_acknowledge})
	{
		my $out = $this_swb->{outgoing}->{$to_handle->{trans_id}};

		$self->_callback('SEND_MESSAGE_NOT_RECEIVED', $to_handle->{swb_session}, $out->{message});
		$out->{delivered} = 0;
		return 1;
	}

	# ---------- User joined a session ---------- #

	if ($to_handle->{command} eq $Command->{Switchboard}->{join})
	{
		# 0 - user's passport
		# 1 - user's friendly name

		# Do not add own passport
		if ($m[0] ne $self->passport)
		{
			my $fname = (defined $self->{_contact}->{$m[0]} && defined $self->{_contact}->{$m[0]}->fname)
			    ? $self->{_contact}->{$m[0]}->fname : $m[1];

			$this_swb->add_user($m[0], $fname);
			$this_swb->{inviting} = undef;

			$self->_callback('JOIN_SESSION_USER', $to_handle->{swb_session}, $m[0], $fname);
			$self->_callback('OPEN_CHAT', $to_handle->{swb_session}) if $this_swb->get_users == 1;
		}
		return 1;
	}

	# ---------- List of Switchboard users ---------- #

	if ($to_handle->{command} eq $Command->{Switchboard}->{list_swb})
	{
		# 0 - number of user
		# 1 - total number of users in the session
		# 2 - user's passport
		# 3 - user's friendly name

		# Do not add own passport
		if ($m[2] ne $self->passport)
		{
			my $fname = (defined $self->{_contact}->{$m[2]} && defined $self->{_contact}->{$m[2]}->fname)
			    ? $self->{_contact}->{$m[2]}->fname : $m[3];

			$this_swb->add_user($m[2], $m[3]);
			$self->_callback('JOIN_SESSION_USER', $to_handle->{swb_session}, $m[2], $fname);
		}
		return 1;
	}

	# ---------- User left a session ---------- #

	if ($to_handle->{command} eq $Command->{Switchboard}->{leave})
	{
		# 0 - user's passport
		# 1 - 1 (automatically closed - timeout)

		if (defined $m[1] && $m[1] eq 1)
		{
			$self->_callback('SWB_TIMEOUT', $to_handle->{swb_session});
			$self->disconnect_swb($to_handle->{swb_session});  # Already disconnected
		}
		else
		{
			$self->_callback('LEAVE_SESSION_USER', $to_handle->{swb_session}, $m[0]);
		}

		# Delete the user from switchboard users list
		$this_swb->remove_user($m[0]);

		# Close the session if there are no user left
		if (!$this_swb->get_users)
		{
			$self->_callback('CLOSE_CHAT', $to_handle->{swb_session});
			$self->disconnect_swb($to_handle->{swb_session});
		}
		return 1;
	}

	if ($to_handle->{command} eq $Command->{Switchboard}->{invite})
	{
		return 1;
	}

	$self->_callback('DEBUG', "BUG: _handle_Switchboard; command: $to_handle->{command}") if
	    $self->debug;

	undef;
	# _handle_Switchboard
}

"Net::MsnMessenger";
__END__

=head1 CALLBACKS

Callbacks are called every time an event occurs. They can be associated to your own functions
using the add_callback() method. In this section are documented all the callbacks used by
Net::MsnMessenger including parameters the functions are called with.

=over 3

=item B<ADD_BY_USER> ( PASSPORT, FRIENDLY_NAME )

An user added you to his/her contact list. The parameters are the user's passport and the friendly
name (URL and UTF-8 encoded).

=item B<ADD_GROUP> ( GROUP_NAME )

A group was added to your contact list. The parameter is the name of the group (URL and UTF-8
encoded).

=item B<ADD_USER> ( PASSPORT [, GROUP_NAME ] )

An user was added to your contact list. The parameters are his passport and optionally the group
name (URL and UTF-8 encoded).

=item B<ALERT> ( SOURCE_URL, ACTION_URL, ALERT_TEXT )



=item B<ALLOW_USER> ( PASSPORT )

An user was allowed (added to the allow list). The parameter is the user's passport.

=item B<BLOCK_USER> ( PASSPORT )

An user was blocked (added to the block list). The parameter is the user's passport.

=item B<CHANGE_MY_FRIENDLY_NAME> ( NEW_FRIENDLY_NAME )

Your friendly name was successfully changed. The parameter is your new (URL and UTF-8 encoded)
friendly name.

=item B<CHANGE_MY_STATUS> ( NEW_STATUS )

Your status was successfully changed. The parameter is your new status.
Possible states are:

  * online
  * away
  * idle
  * busy
  * be_right_back
  * on_the_phone
  * out_to_lunch
  * appear_offline

=item B<CHANGE_STATUS> ( PASSSPORT, FRIENDLY_NAME, NEW_STATUS, OLD_STATUS )

An user changed the status (possiblly logged in). The parameters are the user's passport, the
user's (URL and UTF-8 encoded) friendly name, the new status and the old status. The states
are the same like above (except the appear_offline status).

=item B<CHANGE_STATUS_OFFLINE> ( PASSPORT )

An user logged off (changed the status to offline) or changed the status to appear_offline.
The parameter is the user's passport.

=item B<CLOSE_CHAT> ( SWB_SESSION )

A chat session was closed (disconnected). The parameter is the switchboard session number.

=item B<DEBUG> ( MESSAGE )

Debug message of the protocol (only sent if 'debug' attribute is set to true).

=item B<DEBUG_CONNECTION> ( MESSAGE )

Debug message of the connection (only sent if 'debug_connection' attribute is set to true).

=item B<DISCONNECT> ( REASON )

Disconnected by the server. The parameter is the reason for disconnecting.

=item B<DISCONNECT_FORCED> ( REASON )

Disconnected by the server usualy because of some client-server failure or possibly a bug
in the protocol. This is called if the server disconnects without notifying about it.

=item B<EMAIL_ACTIVE> ( SRC_FOLDER, DEST_FOLDER, MESSAGE_DELTA )

E-mail (hotmail) activity. It notifies you when an unread mail was read or deleted.
SRC_FOLDER is the folder the message was originally in. DEST_FOLDER is the name of the folder
the message was moved to. MESSAGE_DELTA is the number of moved messages.

=item B<EMAIL_INVITE_SUCCESS> ( PASSPORT )

E-mail invitation was successfully sent. The parameter is the passport of the user you
sent the invitation to.

=item B<EMAIL_NEW> ( FROM_ADDRESS, FROM_NAME [, SUBJECT ] )

You just received a new e-mail (only for hotmail). The parameters are the sender's e-mail
address, the sender's name in the e-mail (if any) and subject (if any).

=item B<EMAIL_UNREAD> ( UNREAD_MESSAGES )

A number of unread e-mails in your hotmail inbox. This is sent only if there are 1 or more
unread e-mails in your inbox. The parameter is the number of unread e-mails.

=item B<FILE_ACCEPT> ( SWB_SESSION, FILE_SESSION, USER )

A file transfer was accepted by the recipient. After this callback the file transfer is
automatically started. The parameters are the switchboard, file session numbers and
the user (passport) who accepted it.

=item B<FILE_INVITED> ( SWB_SESSION, FILE_SESSION, SENDER, SENDER_NAME, F_NAME, F_SIZE )

You were invited to a file transfer. The parameters are the switchboard session number, the
file session number, the sender's passport, the sender's friendly name (URL and UTF-8 encoded),
file name and file size. After this you should accept or reject the file transfer.

=item B<FILE_RECEIVE_CANCEL> ( SWB_SESSION, FILE_SESSION, CANCEL_REASON )

The file transfer was cancelled (mostly by the sender). The parameters are the switchboard
session, the file session and the reason for canceling the file transfer.

=item B<FILE_RECEIVE_PROGRESS> ( SWB_SESSION, FILE_SESSION, TRANSFERED, TOTAL )

This callback indicates the progress of file receiving. It is sent after every received
packet. The parameters are the switchboard session, the file session, number of bytes
currently transfered and total file size.

=item B<FILE_RECEIVE_START> ( SWB_SESSION, FILE_SESSION )

The file receiving was just started (can be useful for timing). The parameters are the
switchboard session and the file session numbers.

=item B<FILE_RECEIVE_SUCCESS> ( SWB_SESSION, FILE_SESSION )

The file was successfully received. The parameters are the switchboard session and the
file session numbers.

=item B<FILE_REJECT> ( SWB_SESSION, FILE_SESSION )

A file transfer was rejected by the recipient and it cannot proceed. The parameters are
the switchboard session, file session numbers and the user (passport) who rejected it.

=item B<FILE_SEND_CANCEL> ( SWB_SESSION, FILE_SESSION, CANCEL_REASON )

The file sending was cancelled. The parameters are the switchboard and file session numbers and
the reason for canceling the file transfer.

=item B<FILE_SEND_PROGRESS> ( SWB_SESSION, FILE_SESSION, TRANSFERED, TOTAL )

File sending progress (same as for receiving). The parameters are the switchboard and file
sessions, number of bytes currently transfered and total file size.

=item B<FILE_SEND_START> ( SWB_SESSION, FILE_SESSION, USER )

The file sending was just started. The parameters are the switchboard and the file session
numbers.

=item B<FILE_SEND_SUCCESS> ( SWB_SESSION, FILE_SESSION )

The file was successully sent. The parameters are the switchboard session and the file
session numbers.

=item B<FIND_INVITE_SUCCESS> ( FIND_RESULT_NUMBER )

The invitation to the person found in member search was successfully sent. The parameter is
the number of the search result.

=item B<FIND_RESULTS> ( @RESULTS )

Results from the find_member functions. Very often there are too many users matching and
the server returns are error (SERVER_ERROR callback). If there are no matches it returns
an empty array. The results are in an array of hash references with the following format:

  * $results[$number]->{fname}    - First name
  * $results[$number]->{lname}    - Last name
  * $results[$number]->{city}     - City (optional)
  * $results[$number]->{state}    - State (optional)
  * $results[$number]->{country}  - Country

=item B<INITIAL_STATUS> ( PASSPORT, FRIENDLY_NAME, STATUS )

Initial status of an user. This is sent either after logged in to the server and before
synchronizing the contact lists (the users logged in before you went online) or after
you add a new user and the user is online. The parameters are the user's passport, the
friendly name (URL and UTF-8 encoded) and the user's current status.

=item B<JOIN_SESSION_USER> ( SESSION, PASSPORT, FRIENDLY_NAME )

An user joined a switchboard (chat) session. The parameters are the switchboard session
number, user's passport and friendly name (URL and UTF-8 encoded).

=item B<LEAVE_SESSION_USER> ( SESSION, PASSPORT )

An user left a switchboard session. The parameters are the switchboard session and the
user's passport.

=item B<LOGIN_PROGRESS> ( LOGIN_MESSAGE )

Login message. This messages 'very simply' indicates the login process. The parameter is
the current progress message.

=item B<NETMEETING_ACCEPT> ( SWB_SESSION, NETMEETING_SESSION, USER )

A NetMeeting invitation was accepted by an user. The parameters are the switchboard,
netmeeting session numbers and the user (passport) who accepted it.

=item B<NETMEETING_CANCEL> ( SWB_SESSION, NETMEETING_SESSION, CANCEL_REASON )

A NetMeeting session was canceled. The parameters are the switchboard, netmeeting session
numbers and reason for canceling it.

=item B<NETMEETING_INVITED> ( SWB_SESSION, NETMEETING_SESSION, USER, FNAME )

You were invited to use NetMeeting. The parameters are the switchboard and netmeeting
session numbers, the user's passport and the user's friendly name (URL and UTF-8 encoded).
After this you should accept or reject the invitation.

=item B<NETMEETING_LAUNCH> ( SWB_SESSION, NETMEETING_SESSION, IP_ADDRESS )

You are now asked to launch your netmeeting application. The parameters are the switchboard
and netmeeting session numbers and the user's IP Address.

=item B<NETMEETING_REJECT> ( SWB_SESSION, NETMEETING_SESSION, USER )

A NetMeeting invitation was rejected by an user. The parameters are the switchboard,
netmeeting session numbers and the user (passport) who rejected it.

=item B<OPEN_CHAT> ( SESSION )

A switchboard session was successfully established (opened). The parameter is the
switchboard session number.

=item B<PONG> ()

A pong (reply to ping) was just received from the server.

=item B<RECEIVE_MESSAGE> ( SESSION, SENDER, MESSAGE, %ATTRIBUTES )

A chat message was just received. The parameters are the switchboard session, the
sender's passport, the message, and attributes. The attributes are in a simple hash and
are interpreted as follows:

  * $attributes{EF} - The font effect (B - bold, I - italic, U - underline)
  * $attributes{FN} - The URL encoded font name
  * $attributes{CO} - The rgb value of the font color (RRRGGGBBB)
  * $attributes{CS} - The number of the charset
  * $attributes{PF} - Pitch & Family
  * $attributes{RL} - Right-to-Left typing (1 if enabled)

=item B<REMOVE_BY_USER> ( PASSPORT )

An user removed you from his/her contact list (removed from the reverse list). The parameter
is the user's passport.

=item B<REMOVE_GROUP> ( GROUP_NAME )

A group was successfully removed. The parameter is the URL and UTF-8 encoded name of the group.

=item B<REMOVE_USER> ( PASSPORT [, GROUP_NAME ] )

An user was successfully removed from your contact list (forward list). The parameters are
the user's passport and optionally URL and UTF-8 encoded name of the group the user was removed
from.

=item B<RENAME_GROUP> ( GROUP_ID, OLD_NAME, NEW_NAME )

A group was successfully renamed. The parameters are the group ID, the old name of the group
and the new name (both URL and UTF-8 encoded).

=item B<SEND_MESSAGE_NOT_RECEIVED> ( SESSION, MESSAGE )

A message was not delivered to one or more recipients. The parameters are the switchboard
session number and the message (UTF-8 encoded).

=item B<SEND_MESSAGE_RECEIVED> ( SESSION, MESSAGE )

A message was successfully delived to all the recipients. The parameters are the switchboard
session number and the message (UTF-8 encoded).

=item B<SEND_MESSAGE_SENT> ( SESSION, MESSAGE )

A message was sent to the server (but wasn't yet acknowledged). The parameters are the switchboard
session and the message (UTF-8 encoded).

=item B<SERVER_ERROR> ( ERROR_MESSAGE )

Server error that happens mostly because of invalid parameters or busy servers. Sometimes the
server disconnects after this.

=item B<SERVER_MESSAGE> ( MESSAGE )

Server's system message. Currently there is only one type supported and it notifies you that
you will be signed off because of server's maintenance.

=item B<SIGNED_IN> ()

You were successfully signed in to the server. After this Net::MsnMessenger starts synchronizing
the contact lists.

=item B<SWB_TIMEOUT> ( SWB_SESSION )

A switchboard session timed out and was disconnected. This has the same effect as CLOSE_CHAT but
the session wasn't disconnected because the user closed it but because of server timeout. The
timeout is 5 minutes (no sent/received data) for two users and 15 minutes for more users.

=item B<SYNC_DONE> ()

The contact list synchronization was finished.

=item B<TYPING_USER> ( SESSION, PASSPORT )

An user is typing a message. This message is usualy sent once every 10 seconds if the user keeps
typing. The parameters are the switchboard session number and user's passport.

=item B<UNBLOCK_USER> ( PASSPORT )

An user was successfully unblocked. The parameter is the user's passport.

=item B<UNIMPLEMENTED> ( MESSAGE )

This callback is used when some event occurs that is unknown to Net::MsnMessenger. The parameter
is the server message.

=item B<UNKNOWN_REJECT> ( SWB_SESSION, SENDER, SENDER_NAME, APP_NAME )

An unknown invitatation was received and automatically rejected as not installed. The parameters
are the switchboard session number, sender's passport, sender' friendly name (URL and UTF-8
encoded) and the name of the application.

=item B<URL> ( PAGE_PATH, CMD_PARAMETERS )

This callback is a response for get_hotmail(). The first parameter is a path to (local) generated
html page. The page is generated automatically when the response arrives. The rest of the parameters
are parameters that you passed to get_hotmail (to know identify the page).

=back

=head1 SEE ALSO

Net::MsnMessenger::Event(3), Net::MsnMessenger::Contact(3), Net::MsnMessenger::Group(3)

=head1 AUTHOR

E-mail: B<incoming@tiscali.cz>

=head1 COPYRIGHT

This module is free software; you can redistribute it and/or modify it under
the same terms as Perl itself.

=cut

