/*
 * MGTERM -- Terminal Emulator for MobileGear -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include "mgl2.h"

char *picts[6] ={
"\
#MGR000200160016
++++++++++++++++
+++++++@@+++++++
+++++@@BB@@@++++
+++@@BBBBBBB@+++
++@~@@B@B@B@@@++
+@~~~~@~@~@~~~@+
++@~+~~~+~~+~~@+
++@~~~~~~~~~~@++
++@~~+~~+~~+~@++
+++@~~~~~~~~~@++
+++@~~+~~~~~@+++
++++@~~~~+~@++++
+++++@~~+~~@++++
++++++@~~~@+++++
+++++++@@@++++++
++++++++++++++++
","\
#MGR000200160016
++++++++++++++++
++++++++++++@@++
++++++++++++@@++
++@@@@++++++@@++
+@@::@@+++@@+@++
+@::::@@@@@++@++
+@::::@++++++@++
+@@:::@+++++@@++
++@:::@++++@@+++
+++@@@@@@@@@++++
++++++@:::@@++++
+++++@@::::@++++
+++++@@::::@++++
++++++@:::@@++++
+++++++@@@@+++++
++++++++++++++++
","\
#MGR000200160016
++++++++++++++++
++++++++++++++++
++++@@@@@@@@++++
+++@CCCBCBCB@+++
++@CCCBBCBCBB@++
+@BCCBBCCBBCBB@+
+@BCCBBCCBBCBB@+
+@BCCBBCCBBCBB@+
+@BCCBBCCBBCCB@+
+@BBCBBCCBBCCB@+
+@BBCBBCCBBCBB@+
++@BCBBCCCBCB@++
+++@BCBCCBBB@+++
++++@@@@@@@@++++
++++++++++++++++
++++++++++++++++
","\
#MGR000200160016
++++++++++++++++
++++++++++++++++
++++++++++++@@++
+++++@@@@+++@@++
++++@YYY@+++@@++
++++@YYY@++@@@++
++++@@@Y@@@@@+++
+++@YYY@YYY@++++
+++@YYY@YYY@++++
+++@YYY@YY@@@+++
++@@@Y@@@@YYY@++
+@YYY@YYY@YYY@++
+@YYY@YYY@YYY@++
+@YYY@YYY@@@@+++
++@@@+@@@+++++++
++++++++++++++++
","\
#MGR000200160016
++++++++++++++++
+++++++@@@++++++
+++++@@BBB@+++++
++++@l@BBB@@++++
+++@lll@@@ll@+++
++@ljlljlllll@++
+@llllllllljll@+
+@lljlljlllllj@+
+@jlllllllllll@+
+@llljllllljll@+
+@ljllllljllll@+
++@lljllllljl@++
+++@llljllll@+++
++++@@@@@@@@++++
++++++++++++++++
++++++++++++++++
","\
#MGR000200160016
++++++++++++++++
++++++++@+++++++
+++++@@@@@@@++++
+++@@ee@eeee@+++
++@eeee@eeeee@++
++@ee@eeee@eee@+
+@eeee@@@@eeee@+
+@eeeeeeeeeeee@+
+@eeeeeeeeeeee@+
+@eeeeeeeeeeee@+
+@eeeeeeeeeeee@+
++@eeeeeeeeee@++
++@eeeeeeeeee@++
+++@eeeeeeee@+++
++++@@@@@@@@++++
++++++++++++++++
"};
static int vk_attached=0;
static struct virtual_key *vk_canvas;
static struct screen *ps[6];
static int knum=4;

static void ps_init() {
    int i;
    if (!ps[0]) {
	for (i=0; i<6; i++) {
	    /* ps[i] = conv_screen_from_mgr(picts[i],STK_GENERIC_192COLOR); */
	    ps[i] = conv_screen_from_mgr(picts[i], STK_NATIVE);
	}
    }
}

#define MAXHEIGHT	12
#define WIDTH	10
#define HEIGHT	10

static int rest;
static char map[WIDTH][HEIGHT];
static char mark[WIDTH][HEIGHT];
static int markcnt=0;
static int curx,cury;
static int point=0;
static int auto_mode=0;

#define XOFF	480
#define YOFF	8
#define XSIZE	160
static int ysize = 224;
static int from_main = 0;
#define YSIZE	(ysize)
static void help(void);
static void auto_play(void);
static void flush_undo(int e);

#ifdef MINIAPLI
#undef XOFF
#undef YOFF
#define XOFF	0
#define YOFF	0

void msame();

main(int argc,char *argv[]) {
	int c;
	int mini=1;
	int k;
	
	while ((c=getopt(argc,argv,"mk:")) != EOF) {
	    switch(c) {
	    case 'm':
		mini = !mini;
		break;
	    case 'k':
		if ((sscanf(optarg,"%d",&k) == 1)) {
		    if ((3 <= k) && (k <= 6)) {
			knum = k;
		    }
		}
	    }
	}
	if (mini) {
	    mgl_apli_type = AT_MINIAPLI;
	}
	open_graph();
	if (!mgl_client) {
		printf("server not running \n");
		exit(2);
	}
	set_icon(picts[0],"msame");

	from_main = 1;
	msame(-2);
	for (;;) {
		c = get_key(10);
		msame(c);
	}
}
#endif


static void help() {
	static char *help_msg = "\
᤬ɤǤ\n\
\n\
륭\n\
     \n\
      h j k l\n\
\n\
u     ɥ\n\
SPACE ä\n\
R     ȥ饤\n\
N     ƥ\n\
a     auto ⡼\n\
4,5,6 μ\n\
   (फͭ)";

	struct textscreen *ts;
	ts = create_textscreen(0,XOFF,YOFF,XSIZE,YSIZE,TS_BORDER|TS_BLINE|TS_SAVE);
	ts_clear(ts);
	set_color(COLOR_BLACK);
	set_font(12,0);
	ts_put_string(ts,help_msg,0);
	refresh();
	get_key(50);
	free_textscreen(ts);
	refresh();
}

static void show_point() {
	char buf[100];
	int xpos,ypos;

	xpos = XOFF + 40;
	ypos = YOFF + 12;
	set_color(COLOR_LIGHTGRAY);
	fill_rect(xpos-2,ypos-2,100,20);
	set_color(COLOR_BLACK);
	set_font(16,FA_BOLD);

	sprintf(buf,"%12d",point);
	draw_string(xpos,ypos,buf,DIR_NORTH);
}

static void show_cursor() {
	int xpos,ypos;

	if (mark[curx][cury]) return;
	xpos = XOFF + 10+ curx*14;
	ypos = YOFF + (YSIZE-24) - (cury*14);
	set_color(COLOR_REVERSE);
	draw_rect(xpos,ypos,14-1,14-1);
	draw_rect(xpos+1,ypos+1,14-3,14-3);
}

static void hide_cursor() {
	int xpos,ypos;

	if (mark[curx][cury]) return;
	xpos = XOFF + 10+ curx*14;
	ypos = YOFF + (YSIZE-24) - (cury*14);
	set_color(COLOR_REVERSE);
	draw_rect(xpos,ypos,14-1,14-1);
	draw_rect(xpos+1,ypos+1,14-3,14-3);
}

static void draw_icon(x,y,sel) {
	int xpos,ypos;
	int c,code;

	ps_init();
	xpos = XOFF + 10+ x*14;
	ypos = YOFF + (YSIZE-24) - (y*14);
	set_color(sel?COLOR_WHITE:COLOR_LIGHTGRAY);
	fill_rect(xpos,ypos,14,14);
	c = map[x][y];
	set_color(COLOR_BLACK);
	if (c > 0) {
	    bitblt(NULL,xpos,ypos,ps[c-1],1,1,14,14,BLT_MASKING|COLOR_LIGHTGRAY);
	}
}

static void mark_same(x,y,lvl,dir) {
	int i,j;
	char c = map[x][y];
	if (lvl == 0) {
		for (i=0; i< WIDTH; i++) for (j=0; j<HEIGHT; j++) {
			mark[i][j] = 0;
		}
		markcnt = 0;
		if (c == 0) return;
		dir = 0;
	}
	if (x > 0 && !mark[x-1][y] && map[x-1][y] == c) {
		mark[x-1][y] = 1;
		markcnt++;
		if (!(dir & 1))
		mark_same(x-1,y,lvl+1, 4);
	}
	if (y > 0 && !mark[x][y-1] && map[x][y-1] == c) {
		mark[x][y-1] = 1;
		markcnt++;
		if (!(dir & 2))
		mark_same(x,y-1,lvl+1, 8);
	}
	if (x < WIDTH-1 && !mark[x+1][y] && map[x+1][y] == c) {
		mark[x+1][y] = 1;
		markcnt++;
		if (!(dir & 4))
		mark_same(x+1,y,lvl+1, 1);
	}
	if (y < HEIGHT-1 && !mark[x][y+1] && map[x][y+1] == c) {
		mark[x][y+1] = 1;
		markcnt++;
		if (!(dir & 8))
		mark_same(x,y+1,lvl+1, 2);
	}
}

static void show_all(force) {
	int i,j;

	if (!vk_canvas) {
		vk_canvas = create_virtual_key3(XOFF+10,YOFF+(YSIZE-24)-14*9,140,140
				,MK_V1,MK_V2,MK_V3);
		vk_attach(NULL,vk_canvas);
		vk_attached = 1;
	}
	if (!force && rest == 0) {
		long t;
		time(&t);
		srandom(t);
		for (i=0; i<WIDTH; i++) {
			for (j=0; j<HEIGHT; j++) {
				map[i][j] = (random()&0x7fffffff)%knum + 1;
			}
		}
		rest = WIDTH*HEIGHT;
		point = 0;
	}
	set_color(COLOR_LIGHTGRAY);
	fill_rect(XOFF+1,YOFF+1,160-2,YSIZE-2);
	set_color(COLOR_DARKGRAY);
	draw_rect(XOFF+4,YOFF+4,160-8,YSIZE-8);
	mark_same(curx,cury,0,0);
	for (i=0; i<WIDTH; i++) {
		for (j=0; j<HEIGHT; j++) {
			draw_icon(i,j,0);
		}
	}
	show_point();
}

static void show_mark() {
	int i,j;

	mark_same(curx,cury,0,0);
	for (i=0; i<WIDTH; i++) {
		for (j=0; j<HEIGHT; j++) {
			if (mark[i][j]) {
				draw_icon(i,j,mark[i][j]);
			}
		}
	}
}

static void hide_mark() {
	int i,j;

	for (i=0; i<WIDTH; i++) {
		for (j=0; j<HEIGHT; j++) {
			if (mark[i][j]) {
				draw_icon(i,j,!mark[i][j]);
			}
		}
	}
}

struct undo_buf {
	struct undo_buf *next;
	int rest;
	int point;
	int curx,cury;
	unsigned int map[WIDTH];
} *undo_top;

static void push_undo() {
	extern char *malloc();
	struct undo_buf *buf;
	int i,j;
	unsigned int x;

	buf = (struct undo_buf *)malloc(sizeof(*buf));
	for (i=0; i<WIDTH; i++) {
		x = 0;
		for (j=0; j<HEIGHT;j++) {
			x *= 6;
			x += map[i][j];
		}
		buf->map[i] = x;
	}
	buf->rest = rest;
	buf->point = point;
	buf->curx = curx;
	buf->cury = cury;
	buf->next = undo_top;
	undo_top = buf;
}

static int pop_undo(e) {
	struct undo_buf *buf;
	int i,j;
	unsigned int x;

	buf = undo_top;
	if (!buf) return -1;
	undo_top = buf->next;

	if (e) {
		for (i=0; i<WIDTH; i++) {
			x = buf->map[i];
			for (j=HEIGHT-1; j>=0 ;j--) {
				map[i][j] = x % 6;
				x /= 6;
			}
		}
		rest = buf->rest;
		point = buf->point;
		curx = buf->curx;
		cury = buf->cury;
	}
	free(buf);
	return 0;
}

static void flush_undo(e) {
	while (pop_undo(e) == 0)
		;
}


static void delete_mark() {
	int last;
	int i,j,k,m,n;
	int ii;

	push_undo();
	for (last=0; last<WIDTH; last++) {
		if (!map[last][0]) break;
	}
	for (i=0; i<last; i++) {
		k = 0;
		n = 0;
		for (j=0; j<HEIGHT; j++) {
		    if (mark[i][j] == 0) {
			if (map[i][j]) {
				n++;
			}
			map[i][k] = map[i][j];
			draw_icon(i,k,0);
			k++;
		    }
		}
		if (n == 0) {
			for (ii=i; ii<last-1; ii++) {
				for (j=0; j<HEIGHT; j++) {
					map[ii][j] = map[ii+1][j];
					mark[ii][j] = mark[ii+1][j];
					draw_icon(ii,j,mark[ii][j]);
				}
			}
			for (j=0; j<HEIGHT; j++) {
				map[last-1][j] = 0;
				mark[last-1][j] = 0;
				draw_icon(last-1,j,0);
			}
			i--;
			last--;
		} else {
			for (; k < HEIGHT; k++) {
				map[i][k] = 0;
				draw_icon(i,k,0);
			}
		}
	}
	rest -= markcnt;
	point += (markcnt -1)*(markcnt -1);
	markcnt = 0;
}


static void auto_play() {
	int x,y;
	int retry_cnt = 0;
	x = y = -1;

	for (;;) {
retry:
		retry_cnt++;
		if (retry_cnt > 100) break;
		if (markcnt) hide_mark();
		cury++;
		if (cury > HEIGHT) {
			cury=0;
			curx++;
			if (curx > WIDTH) {
				pop_undo(1);
				show_all(1);
				x = curx;
				y = cury;
				goto retry;
			}
		}
		show_mark();
refresh();
		if (markcnt) {
			if (x >= 0 && y >= 0 && mark[x][y]) goto retry;
			delete_mark();
			curx = cury = 0;
			show_mark();
refresh();
		} else {
			goto retry;
		}
#if 0
			rest = 0;
			point = 0;
			flush_undo(0);
			show_all(0);
			show_mark();
refresh();
#endif
		break;
	}
}

void msame(c) int c; {
	int tx,ty;
	switch (c) {
	case '4':
	case '5':
	case '6':
		knum = c - '0';
		break;
	case MK_V3:	/* up */
		hide_cursor();
		if (markcnt) hide_mark();
		curx = vk_x/14;
		cury = (139 - vk_y)/14;
//printf("up   to %2d %2d\n",curx,cury);
		if (markcnt) {
			delete_mark();
			show_mark();
			show_cursor();
			show_point();
		} else {
			show_cursor();
		}
		break;
	case MK_V2:	/* move */
		tx = vk_x/14;
		ty = (139  - vk_y)/14;
		if ((curx == tx) && (cury == ty)) break;
		hide_cursor();
		if (markcnt) hide_mark();
		curx = tx;
		cury = ty;
//printf("move to %2d %2d\n",curx,cury);
		show_mark();
		show_cursor();
		break;
	case MK_V1:	/* down */
		hide_cursor();
		if (markcnt) hide_mark();
		curx = vk_x/14;
		cury = (139 - vk_y)/14;
//printf("down to %2d %2d\n",curx,cury);
		show_mark();
		show_cursor();
		break;
	case MK_RIGHT:
	case 'l':
		if (curx < WIDTH-1) {
			hide_cursor();
			if (markcnt) hide_mark();
			curx++;
			show_mark();
			show_cursor();
		}
		break;
	case MK_LEFT:
	case 'h':
		if (curx > 0) {
			hide_cursor();
			if (markcnt) hide_mark();
			curx--;
			show_mark();
			show_cursor();
		}
		break;
	case MK_UP:
	case 'k':
		if (cury < HEIGHT-1) {
			hide_cursor();
			if (markcnt) hide_mark();
			cury++;
			show_mark();
			show_cursor();
		}
		break;
	case MK_DOWN:
	case 'j':
		if (cury >0) {
			hide_cursor();
			if (markcnt) hide_mark();
			cury--;
			show_mark();
			show_cursor();
		}
		break;
		
	case ' ':
		if (markcnt) {
			hide_cursor();
			delete_mark();
			show_mark();
			show_cursor();
			show_point();
		}
		break;
	case 'u':
		pop_undo(1);
		show_all(1);
		show_mark();
		show_cursor();
		break;
	case 'a':
		auto_mode = !auto_mode;
		break;
	case '?':
		help();
		break;
	case 'R':
		rest = 0;
		point = 0;
		flush_undo(1);
		show_all(1);
		show_mark();
		show_cursor();
		break;
	case 'N':
		rest = 0;
		point = 0;
		flush_undo(0);
		show_all(0);
		show_mark();
		show_cursor();
		break;
	case (-2):
		if (SCREEN_HEIGHT < 224) {
			YSIZE = SCREEN_HEIGHT;
			if (!from_main) YSIZE -= 16;
		}
		show_all(0);
		show_mark();
		show_cursor();
		break;
	case (-3):
		if (!vk_attached) {
			vk_attach(NULL,vk_canvas);
			vk_attached = 1;
		}
		set_color(COLOR_BLACK);
		draw_rect(XOFF,YOFF,160-1,YSIZE-1);
		break;
	case (-4):
		if (vk_attached) {
			vk_detach(vk_canvas,0);
			vk_attached = 0;
		}
		set_color(COLOR_LIGHTGRAY);
		draw_rect(XOFF,YOFF,160-1,YSIZE-1);
		break;
	case (-1):
		if (auto_mode) {
			hide_cursor();
			auto_play();
			show_cursor();
		}
		break;
	}
}
