/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
/* 2 plane 4color (for vga) */

#define MGL_PREFIX
#include "config.h"
#include "mgl2.h"
#include "draw_engine.h"
#include <stdlib.h>

#ifdef USE_LOCAL_MEMSET
#define memset	mgl_memset
#endif
#ifdef USE_LOCAL_MEMMOVE
#define memmove	mgl_memmove
#endif

#define lp_color	(s->_pen_color.local_color)
#define lp_col0		(*(int *)(&(s->_pen_color.opt)))
#define lp_col1		(*(int *)(&(s->_pen_color.link)))
#define lp_dodither	(s->_pen_color.flags)

#ifndef NULL
#define NULL ((void *)0)
#endif

static struct draw_engine depl2;

#define PN	2		/* plane number */

/*
  0 %       20%       40%       60%    80%     100%
  .....    #....    #..#.    #.##.    ####.    #####
  .....    ..#..    ..#.#    #.#.#    #.###    #####

*/
static const int dither_map[2][5][16] = {
{  {0,1,1,1,1   , 1,2,2,2,2,  2,3,3,3,3,  3},
   {0,0,0,0,1   , 1,1,1,1,2,  2,2,2,2,3,  3},
   {0,0,0,1,1   , 1,1,1,2,2,  2,2,2,3,3,  3},
   {0,0,1,1,1   , 1,1,2,2,2,  2,2,3,3,3,  3},
   {0,0,0,0,0   , 1,1,1,1,1,  2,2,2,2,2,  3},
}, {
   {0,0,0,1,1   , 1,1,1,2,2,  2,2,2,3,3,  3},
   {0,0,0,0,0   , 1,1,1,1,1,  2,2,2,2,2,  3},
   {0,1,1,1,1   , 1,2,2,2,2,  2,3,3,3,3,  3},
   {0,0,0,0,1   , 1,1,1,1,2,  2,2,2,2,3,  3},
   {0,0,1,1,1   , 1,1,2,2,2,  2,2,3,3,3,  3},
}
};

static struct screen *depl2_create_memscreen(int xs,int ys,char *bitmap,int op) {
	struct screen *ret;
	int wb;
	int type = STK_GENERIC_4COLOR | ST_ALLOCED;

	wb = (xs+7)/8;
	//wb = (wb + 7) & ~7;	/* align to int */
	ret = de_create_memscreen(xs,ys,bitmap,op);
	if (!ret) return NULL;
	if (!bitmap) {
		bitmap = (char *)malloc(ys * wb * PN);
        	if(!bitmap){
			perror("malloc");
			return (struct screen *)NULL;
        	}
		type |= ST_ALLOCED_BITMAP;
		memset(bitmap,0,ys * wb * PN);
	}

	ret->type |= type;
	ret->de = &depl2;
	ret->wbytes = wb;
	ret->bitmap = bitmap;
	ret->plbytes = wb * ys;
	return ret;
}

static struct screen *depl2_create_subscreen(struct screen *org, int x, int y,
                                int xs, int ys,int opt) {
	struct screen *ret;

	ret = de_create_subscreen(org,x,y,xs,ys,opt);
	if (!ret) return NULL;
	if (opt & CSS_AS_MEMSCREEN) {
		ret->type &= ~ST_SUBSCREEN;
		ret->bitmap = org->bitmap + (ret->off_y * ret->wbytes)
			+ (ret->off_x /8);
		ret->off_y = 0;
		ret->off_x = 0;
	}
	return ret;
}

static void depl2_free_screen(struct screen *s) {
	de_free_screen(s);
}


static int depl2_get_pixel(struct screen *s,int x, int y,int op) {
	unsigned char *p;
	int col,ret,d;
	int plbytes;

	if (x < 0 || x >= s->width
	   || y < 0 || y >= s->height) return 0;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
		s->plbytes = 
			((struct screen *)(s->bitmap))->plbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + (x>>3);
	d = x & 7;
	plbytes = s->plbytes;

	col = (((*p >> d) & 1) << 1)
		 | ((*(p+plbytes) >> d) & 1);
	ret = CONV_FROM_COL4(col);
	if ((op & BLT_MASKING) && (ret == (op & BLT_MASKCOL))) {
		ret = COLOR_TRANSPARENT;
	}
	return ret;
}

static void depl2_put_pixel(struct screen *s,int x, int y, int col) {
	char *p;
	int mask,d;
	int plbytes;

	if (col & COLOR_TRANSPARENT) return;
	if (col < 0) return;

	CLIP_POINT(s,x,y);

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
		plbytes = s->plbytes = 
			((struct screen *)(s->bitmap))->plbytes;
	} else {
		p = (char *)s->bitmap;
		plbytes = s->plbytes;
	}
	p += y * s->wbytes + (x>>3);
	d = x & 7;
	mask = 1 << d;

	if (col & COLOR_DITHER) {
		int dx = (x + (y>>2) + (y>>4))%5;
		int dy = y % 2;
		col = dither_map[dy][dx][col & 0xf];
		*p = (*p & ~mask)| ((col>>1)<<d);
		*(p+plbytes) = (*(p+plbytes) & ~mask)| ((col & 1)<<d);
	} else {
	    col >>= 2;
	    *p = (*p & ~mask)| (((col >>1)& 1)<<d);
	    *(p+plbytes) = (*(p+plbytes) & ~mask)| ((col & 1)<<d);
	}
	return;
}

static void depl2_set_color(struct screen *s,int col) {
	lp_color = CONV_TO_COL4(col);
	lp_col0 = (lp_color & 2)?0xffffffff:0;
	lp_col1 = (lp_color & 1)?0xffffffff:0;
	lp_dodither = 0;

	if (col & COLOR_DITHER) lp_dodither = 1;
	if ((CONV_FROM_COL4(lp_color) | COLOR_DITHER) == (col & ~0xff0))
		lp_dodither = 0;
	return;
}

static void depl2_draw_pixel(struct screen *s,int x, int y) {
	char *p;
	int mask,col,d;
	int plbytes;

//printf("depl2_draw_pixel (%d,%d) %d\n",x,y,lp_color);
	CLIP_POINT(s,x,y);

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
		plbytes = s->plbytes = 
			((struct screen *)(s->bitmap))->plbytes;
	} else {
		p = (char *)s->bitmap;
		plbytes = s->plbytes;
	}
	p += y * s->wbytes + (x>>3);
	d = x & 7;
	mask = 1 << d;
	if (pen_color == COLOR_REVERSE) {
		*p = *p ^ mask;
		*(p+plbytes) = *(p+plbytes) ^ mask;
	} else if (lp_dodither) {
		int dx = (x + (y>>2) + (y>>4))%5;
		int dy = y % 2;
		col = dither_map[dy][dx][pen_color & 0xf];
		*p = (*p & ~mask)| ((col>>1)<<d);
		*(p+plbytes) = (*(p+plbytes) & ~mask)|((col&1)<<d);
	} else {
		*p = (*p & ~mask)| (lp_col0 & mask);
		*(p+plbytes) = (*(p+plbytes) & ~mask)| (lp_col1 & mask);
	}
      return;
}

static void depl2_get_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir,int op
		,struct draw_engine *self) {
	unsigned char *p;
	int i;
	int d;
	int col,r;
	int plbytes;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
		plbytes = s->plbytes = 
			((struct screen *)(s->bitmap))->plbytes;
	} else {
		p = (char *)s->bitmap;
		plbytes = s->plbytes;
	}
	p += y * s->wbytes + (x>>3);

	switch(dir) {
	case DIR_NORTH:
		for (i=0; i<length; i++,x++) {
			d = (x & 7);
			col = (((*p >> d ) & 1)<<1) | ((*(p+plbytes) >> d) & 1);
			r = CONV_FROM_COL4(col);
			if ((op & BLT_MASKING) && (r == (op & BLT_MASKCOL))) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			if (d == 7) p++;
		}
		break;
	case DIR_WEST:
		d = (x & 7);
		for (i=0; i<length; i++) {
			col = (((*p >> d ) & 1)<<1)|((*(p+plbytes) >> d) & 1);
			r = CONV_FROM_COL4(col);
			if ((op & BLT_MASKING) && (r == (op & BLT_MASKCOL))) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p -= s->wbytes;
		}
		break;
	case DIR_SOUTH:
		for (i=0; i<length; i++,x--) {
			d = (x & 7);
			col = (((*p >> d ) & 1)<<1) | ((*(p+plbytes) >> d) &1);
			r = CONV_FROM_COL4(col);
			if ((op & BLT_MASKING) && (r == (op & BLT_MASKCOL))) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			if (d == 0) p--;
		}
		break;
	case DIR_EAST:
		d = (x & 7);
		for (i=0; i<length; i++) {
			col = (((*p >> d ) & 1)<<1) |((*(p+plbytes) >> d) & 1);
			r = CONV_FROM_COL4(col);
			if ((op & BLT_MASKING) && (r == (op & BLT_MASKCOL))) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p += s->wbytes;
		}
		break;
	}
}

static void depl2_put_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir
		,struct draw_engine *self) {
	char *p;
	int col,mask;
	int d,i;
	int dx,dy;
	int dodither = 0;
	int plbytes;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
		plbytes = s->plbytes = 
			((struct screen *)(s->bitmap))->plbytes;
	} else {
		p = (char *)s->bitmap;
		plbytes = s->plbytes;
	}
	p += y * s->wbytes + (x>>3);

	if (*buf & COLOR_DITHER) {
	    	dy = y % 2;
	    	dx = (x + (y>>2) + (y>>4))%5;
		dodither = 1;
	}
	switch(dir) {
	case DIR_NORTH:
	    for (i=0; i<length;i++,x++) {
		col = *buf++;
		d = (x & 7);
		if (!(col & COLOR_TRANSPARENT)) {
		    mask = 1 << d;
		    if (dodither) {
			col = dither_map[dy][dx][col & 0xf];
	        	*p = (*p & ~mask)|((col>>1)<<d);
	        	*(p+plbytes) = (*(p+plbytes) & ~mask)|((col&1)<<d);
		    } else {
			col >>= 2;
		        *p = (*p & ~mask) | (((col>>1)&1)<<d);
		        *(p+plbytes) = (*(p+plbytes) & ~mask) | ((col&1)<<d);
		    }
		}
		if (d == 7) p++;
		if (dodither) if (++dx >= 5) dx = 0;
	    }
	    break;
	case DIR_WEST:
	    d = (x & 7);
	    mask = 1 << d;
	    for (i=0; i<length;i++) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		    if (col & COLOR_DITHER) {
			dx = (x + (y>>2) + (y>>4))%5;
			dy = y % 2;
			col = dither_map[dy][dx][col & 0xf];
	        	*p = (*p & ~mask)|((col>>1)<<d);
	        	*(p+plbytes) = (*(p+plbytes) & ~mask)|((col&1)<<d);
		    } else {
			col >>= 2;
	        	*p = (*p & ~mask)|(((col>>1)&1)<<d);
	        	*(p+plbytes) = (*(p+plbytes) & ~mask)|((col&1)<<d);
		    }
		}
		p -= s->wbytes;
	    }
	    break;
	case DIR_SOUTH:
	    for (i=0; i<length;i++,x--) {
		col = *buf++;
		d = (x & 7);
		if (!(col & COLOR_TRANSPARENT)) {
		    mask = 1 << d;
		    if (dodither) {
			col = dither_map[dy][dx][col & 0xf];
	        	*p = (*p & ~mask)|((col>>1)<<d);
	        	*(p+plbytes) = (*(p+plbytes) & ~mask)|((col&1)<<d);
		    } else {
			col >>= 2;
	        	*p = (*p & ~mask)|(((col>>1)&1)<<d);
	        	*(p+plbytes) = (*(p+plbytes) & ~mask)|((col&1)<<d);
		    }
		}
	        if (d == 0) p--;
	        if (dodither) if (--dx < 0) dx = 4;
	    }
	    break;
	case DIR_EAST:
	    d = (x & 7);
	    mask = 1 << d;
	    for (i=0; i<length;i++) {
		col = *buf++;
		if (!(col & COLOR_TRANSPARENT)) {
		    if (col & COLOR_DITHER) {
			dx = (x + (y>>2) + (y>>4))%5;
			dy = y % 2;
			col = dither_map[dy][dx][col & 0xf];
	        	*p = (*p & ~mask)|(((col>>1)&1)<<d);
	        	*(p+plbytes) = (*(p+plbytes) & ~mask)|((col&1)<<d);
		    } else {
			col >>= 2;
	        	*p = (*p & ~mask)|(((col>>1)&1)<<d);
	        	*(p+plbytes) = (*(p+plbytes) & ~mask)|((col&1)<<d);
		    }
		}
		p += s->wbytes;
	    }
	    break;
	}
	return;
}


#define ABS(a) (((a)<0) ? -(a) : (a))

static void depl2_draw_line_vertical(struct screen *s,int x1, int y1, int x2, int y2,struct draw_engine *self) {
	char *p;
	int mask,col;
	int i;
	int n = y2 - y1 +1;
	int plbytes;

	CLIP_VLINE(s,x1,y1,x2,y2);

	x1 += s->off_x;
	y1 += s->off_y;
	x2 += s->off_x;
	y2 += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
		plbytes = s->plbytes = 
			((struct screen *)(s->bitmap))->plbytes;
	} else {
		p = (char *)s->bitmap;
		plbytes = s->plbytes;
	}
	p += y1 * s->wbytes + (x1>>3);

	mask = 1 << (x1 & 7);
	if (pen_color == COLOR_REVERSE) {
		for (i=0; i<n; i++,p+=s->wbytes) {
			*p = (*p ^ mask);
			*(p+plbytes) = (*(p+plbytes) ^ mask);
		}
	} else if (lp_dodither) {
		for (i=0; i<n; i++,p+=s->wbytes,y1++) {
		    int dx = (x1 + (y1>>2) + (y1>>4))%5;
		    int dy = y1 % 2;
		    col = dither_map[dy][dx][pen_color & 0xf];
		    *p = (*p & ~mask) | (((col>>1)&1)<<(x1 & 7));
		    *(p+plbytes) = (*(p+plbytes) & ~mask)|((col&1)<<(x1 & 7));
		}
	} else {
		for (i=0; i<n; i++,p+=s->wbytes) {
		    *p = (*p & ~mask) | (lp_col0 & mask);
		    *(p+plbytes) = (*(p+plbytes) & ~mask) | (lp_col1 & mask);
		}
	}
}

static void depl2_draw_line_horizontal(struct screen *s,int x1, int y1, int x2, int y2,struct draw_engine *self) {
	char *p;
	int wb;
	int i,j;
	int off_x,off_y;
	int plbytes;

//printf("depl2_draw_line_horizontal \n",x1,y1,x2,y2);
	CLIP_HLINE(s,x1,y1,x2,y2);
//printf("clip pass \n");
	s->need_clipping--;

	off_x = s->off_x;
	off_y = s->off_y;

	x1 += off_x;
	x2 += off_x;
	y1 += off_y;
	y2 += off_y;

	while ((x1 % 8) != 0) {
	    draw_pixel(s,x1-off_x,y1-off_y);
	    x1++;
	}
	wb = (x2 + 1 - x1)/8;
	if (wb) {
	     if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
		plbytes = s->plbytes = 
			((struct screen *)(s->bitmap))->plbytes;
	     } else {
		p = (char *)s->bitmap;
	        plbytes = s->plbytes;
	    }
	    p += y1 * s->wbytes + (x1>>3);

	    if (pen_color == COLOR_REVERSE) {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		for (i=0; i< wb; i++,p++) {
		    *p ^= 0xff;
		    *(p+plbytes) ^= 0xff;
			
		}
#else
		mgl_memxor(p, 0xff, wb);
		mgl_memxor(p+plbytes, 0xff, wb);
#endif
	    } else if (lp_dodither) {
		int dx = (x1 + (y1>>2) + (y1>>4))%5;
		int dy = y1 % 2;
		int m0,m1;
		for (i=0; i< wb; i++) {
		    m0 = m1 = 0;
		    for (j=0; j<8; j++) {
			int c;
			m0 <<= 1;
			m1 <<= 1;
			c = dither_map[dy][dx][pen_color & 0xf];
			m0 |= (c >> 1);
			m1 |= (c &  1);
			dx = (dx + 1)%5;
		    }
		    *p = m0;
		    *(p+plbytes) = m1;
		    p++;
		}
	    } else {
		memset(p, lp_col0 & 0xff, wb);
		memset(p+plbytes, lp_col1 & 0xff, wb);
	    }
	}
	x1 += wb * 8;
	while (x1 <= x2) {
	    draw_pixel(s,x1-off_x, y1-off_y);
	    x1++;
	}
	s->need_clipping++;
    return;
}

#if 0
static void depl2_clear_screen(struct screen *s,struct draw_engine *self) {
	int x1,y1,x2,y2;
	int off_x,off_y;
	char *p,*pp;
	int wb;
	int i;
	int y;

	x1 = 0;
	x2 = s->width-1;
	y1 = 0;
	y2 = s->height-1;
	s->need_clipping--;

	off_x = s->off_x;
	off_y = s->off_y;
	x1 += off_x;
	x2 += off_x;
	y1 += off_y;
	y2 += off_y;

	while (x1 % 4 != 0) {
	    draw_line_vertical(x1-off_x,y1-off_y,x1-off_x,y2-off_y,self);
	    x1++;
	}
	wb = (x2 + 1 - x1)/4;
	if (wb) {
	    if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
		s->plbytes = 
			((struct screen *)(s->bitmap))->plbytes;
	    } else {
		p = (char *)s->bitmap;
	    }
	    p += y1 * s->wbytes + (x1>>3);
	    for (y=y1; y<=y2; y++) {
		if (pen_color == COLOR_REVERSE) {
#if (OPTIMIZE_FOR == OPTTYPE_SLOW)
		    pp = p;
		    for (i=0; i< wb; i++) {
			*pp++ ^= 0xff;
		    }
#else
		    mgl_memxor(p, 0xff, wb);
#endif
		} else if (lp_dodither) {
		    int dx = (x1 + (y>>2) + (y>>4))%5;
		    int dy = y % 2;
		    int m;
		    pp = p;
		    for (i=0; i< wb; i++) {
			m = dither_mask_map[dy][dx][pen_color & 0xf] & 0x3;
			dx = (dx + 1)%5;
			m |= dither_mask_map[dy][dx][pen_color & 0xf] & 0xc;
			dx = (dx + 1)%5;
			m |= dither_mask_map[dy][dx][pen_color & 0xf] & 0x30;
			dx = (dx + 1)%5;
			m |= dither_mask_map[dy][dx][pen_color & 0xf] & 0xc0;
			dx = (dx + 1)%5;
		        *pp++ = m;
		    }
		} else {
		    memset(p, lp_mask & 0xff, wb);
	        }
		p += s->wbytes;
	    }
	}
	x1 += wb * 4;
	while (x1 <= x2) {
	    draw_line_vertical(x1-off_x,y1-off_y,x1-off_x,y2-off_y,self);
	    x1++;
	}
	s->need_clipping++;
}
#endif

static void depl2_bitblt_copy(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *dp,*sp;
	int i,len,d,e;
	int mask_d,mask_e;
	int dplbytes,splbytes;
	if ((dx - sx) & 0x7) goto gen;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		dp = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
		dplbytes = dst->plbytes = 
			((struct screen *)(dst->bitmap))->plbytes;
	} else {
		dp = (char *)dst->bitmap;
		dplbytes = dst->plbytes;
	}
	if (src->type & ST_SUBSCREEN) {
		sp = ((struct screen *)(src->bitmap))->bitmap;
		src->wbytes = 
			((struct screen *)(src->bitmap))->wbytes;
		splbytes = src->plbytes = 
			((struct screen *)(src->bitmap))->plbytes;
	} else {
		sp = (char *)src->bitmap;
		splbytes = src->plbytes;
	}
	d = 0;
	if (dx & 7) {
		d = 8 - (dx & 7);
	}
	dx += d;
	sx += d;
	xsize -= d;
	e = (xsize & 7);
	xsize -= e;
	dp += dy * dst->wbytes + (dx>>3);
	sp += sy * src->wbytes + (sx>>3);
	len = xsize>>3;
	mask_d = mask_e = 0;
	for (i=0; i<d; i++) {
		mask_d |= (1<<7)>>i;
	}
	for (i=0; i<e; i++) {
		mask_e |= 1<<i;
	}
	for (i=0; i< ysize; i++) {
		if (d) {
		    *(dp-1) = (*(dp-1) & ~mask_d) | (*(sp-1) & mask_d);
		    *(dp+dplbytes-1) = (*(dp+dplbytes-1) & ~mask_d)
					 | (*(sp+splbytes-1) & mask_d);
		}
		memmove(dp,sp,len);
		memmove(dp+dplbytes,sp+splbytes,len);
		if (e) {
		    *(dp+len) = (*(dp+len) & ~mask_e) | (*(sp+len) & mask_e);
		    *(dp+dplbytes+len) = (*(dp+dplbytes+len) & ~mask_e)
					 | (*(sp+splbytes+len) & mask_e);
		}
		dp += dst->wbytes;
		sp += src->wbytes;
	}
	return;
gen:
	bitblt_generic(dst,dx,dy,src,sx,sy,xsize,ysize,op);
}

static void depl2_bitblt_scroll_forward(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *p,*dp,*sp;
	int i,len,d,e;
	int mask_d,mask_e;
	int plbytes;

	if ((dx - sx) & 0x7) goto gen;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		p = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
		plbytes = dst->plbytes = 
			((struct screen *)(dst->bitmap))->plbytes;
	} else {
		p = (char *)dst->bitmap;
		plbytes = dst->plbytes;
	}
	d = 0;
	if (dx & 7) {
		d = 8 - (dx & 7);
	}
	dx += d;
	sx += d;
	xsize -= d;
	e = (xsize & 7);
	xsize -= e;

	dp = p + dy * dst->wbytes + (dx>>3);
	sp = p + sy * dst->wbytes + (sx>>3);
	len = xsize>>3;
	mask_d = mask_e = 0;
	for (i=0; i<d; i++) {
		mask_d |= (1<<7)>>i;
	}
	for (i=0; i<e; i++) {
		mask_e |= 1<<i;
	}
	for (i=0; i< ysize; i++) {
		if (d) {
		    *(dp-1) = (*(dp-1) & ~mask_d) | (*(sp-1) & mask_d);
		    *(dp+plbytes-1) = (*(dp+plbytes-1) & ~mask_d) 
					| (*(sp+plbytes-1) & mask_d);
		}
		memmove(dp,sp,len);
		memmove(dp+plbytes,sp+plbytes,len);
		if (e) {
		    *(dp+len) = (*(dp+len) & ~mask_e) | (*(sp+len) & mask_e);
		    *(dp+plbytes+len) = (*(dp+plbytes+len) & ~mask_e)
				 | (*(sp+plbytes+len) & mask_e);
		}
		dp += dst->wbytes;
		sp += dst->wbytes;
	}
	return;
gen:
	bitblt_generic(dst,dx,dy,src,sx,sy,xsize,ysize,op);
}

static void depl2_bitblt_scroll_backward(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *p,*dp,*sp;
	int i,len,d,e;
	int mask_d,mask_e;
	int plbytes;

	if ((dx - sx) & 0x7) goto gen;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		p = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
		plbytes = dst->plbytes = 
			((struct screen *)(dst->bitmap))->plbytes;
	} else {
		p = (char *)dst->bitmap;
		plbytes = dst->plbytes;
	}
	d = 0;
	if (dx & 7) {
		d = 8 - (dx & 7);
	}
	dx += d;
	sx += d;
	xsize -= d;
	e = (xsize & 7);
	xsize -= e;
	dy += ysize - 1;
	sy += ysize - 1;
	dp = p + dy * dst->wbytes + (dx>>3);
	sp = p + sy * dst->wbytes + (sx>>3);
	len = xsize>>3;
	mask_d = mask_e = 0;
	for (i=0; i<d; i++) {
		mask_d |= (1<<7)>>i;
	}
	for (i=0; i<e; i++) {
		mask_e |= 1<<i;
	}
	for (i=0; i< ysize; i++) {
		if (e) {
		    *(dp+len) = (*(dp+len) & ~mask_e) | (*(sp+len) & mask_e);
		    *(dp+plbytes+len) = (*(dp+plbytes+len) & ~mask_e) 
					| (*(sp+plbytes+len) & mask_e);
		}
		memmove(dp,sp,len);
		memmove(dp+plbytes,sp+plbytes,len);
		if (d) {
		    *(dp-1) = (*(dp-1) & ~mask_d) | (*(sp-1) & mask_d);
		    *(dp+plbytes-1) = (*(dp+plbytes-1) & ~mask_d) 
					| (*(sp+plbytes-1) & mask_d);
		}
		dp -= dst->wbytes;
		sp -= dst->wbytes;
	}
	return;
gen:
	bitblt_generic(dst,dx,dy,src,sx,sy,xsize,ysize,op);
}

void depl2_init(int type) {
	depl2._create_subscreen = depl2_create_subscreen;
	depl2._free_screen = depl2_free_screen;
	depl2._put_pixel = depl2_put_pixel;
	depl2._get_pixel = depl2_get_pixel;

	depl2._set_color = depl2_set_color;
	depl2._draw_pixel = depl2_draw_pixel;

	//depl2._draw_line = depl2_draw_line;

	depl2._draw_line_vertical = depl2_draw_line_vertical;
	depl2._draw_line_horizontal = depl2_draw_line_horizontal;

	//depl2._clear_screen = depl2_clear_screen;

	depl2._get_pixstream = depl2_get_pixstream;
	depl2._put_pixstream = depl2_put_pixstream;
	//depl2._put_pixstream_rect = depl2_put_pixstream_rect;

	//depl2._bitblt = depl2_bitblt;
	depl2._bitblt_scroll_forward = depl2_bitblt_scroll_forward;
	depl2._bitblt_scroll_backward = depl2_bitblt_scroll_backward;
	depl2._bitblt_copy = depl2_bitblt_copy;
	//depl2._bitblt_reserved_mask = depl2_bitblt_reserved_mask;
	//depl2._bitblt_reserved_masktile = depl2_bitblt_reserver_masktile;
	//depl2._bitblt_gen = depl2_bitblt_gen;
	setup_draw_engine(&depl2,0);
	_create_memscreen[type] = depl2_create_memscreen;
}
