/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <stdlib.h>
#include "mgl2.h"
#include "mglcol.h"

#define COLOR_BG	packMC(MCH_ORANGE,5,15)
#define COLOR_LINE	packMC(MCH_ORANGE,5,5)
/*
ɸ७Υ

13 x 18 = 234

        +---+
        | 1 | 20
        +---+
         20

+-----------------------------------------------+
|Esc| 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 0 | BS|
+---+---+---+---+---+---+---+---+---+---+---+---+
|Tab| q | w | e | r | t | y | u | i | o | p |Ent|
+---+---+---+---+---+---+---+---+---+---+---+-  |
|Caps| a | s | d | f | g | h | j | k | l | ; |er|
+----+---+---+---+---+---+---+---+---+---+---+--|
|Shift| z | x | c | v | b | n | m | , | . ||@ |
+-----+---+---+---+---+---+---+---+---+---+--+--+
|Ctrl |Alt  |Hide |             |Kanji ||||
+-----+-----+-----+-------------+-----------+---+
*/


static char *kb_layout1="\
-----------------------------------------------+
                                               |
-----------------------------------------------+
Esc| 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 | 0 | BS|
---+---+---+---+---+---+---+---+---+---+---+---+
Tab| q | w | e | r | t | y | u | i | o | p |Ent|
---+---+---+---+---+---+---+---+---+---+---+-  |
Caps| a | s | d | f | g | h | j | k | l | ; |er|
----+---+---+---+---+---+---+---+---+---+---+--|
Shift| z | x | c | v | b | n | m | , | . ||@ |
-----+---+---+---+---+---+---+---+---+---+--+--+
Ctrl |Alt  |Hide |             |Kanji ||||
-----+-----+-----+-------------+-----------+---+
";

static char *kb_layout2="\
-----------------------------------------------+
                                               |
-----------------------------------------------+
Esc| ! | \" | # | $ | % | & | ' | ( | ) | ~ | BS|
---+---+---+---+---+---+---+---+---+---+---+---+
Tab| Q | W | E | R | T | Y | U | I | O | P |Ent|
---+---+---+---+---+---+---+---+---+---+---+-  |
Caps| A | S | D | F | G | H | J | K | L | + |er|
----+---+---+---+---+---+---+---+---+---+---+--|
Shift| Z | X | C | V | B | N | M | < | > ||` |
-----+---+---+---+---+---+---+---+---+---+--+--+
Ctrl |Alt  |Hide |             |Kanji ||||
-----+-----+-----+-------------+-----------+---+
";

static char *kb_layout3="\
-----------------------------------------------+
                                               |
-----------------------------------------------+
Esc|                                       | BS|
---+---+---+---+---+---+---+---+---+---+---+---+
Han| q | w | e | r | t | y | u | i | o | p |Ent|
---+---+---+---+---+---+---+---+---+---+---+-  |
Zen | a | s | d | f | g | h | j | k | l | ; |er|
----+---+---+---+---+---+---+---+---+---+---+--|
Kata | z | x | c | v | b | n | m | , | . ||/ |
-----+---+---+---+---+---+---+---+---+---+--+--+
Dic | \\ | @ | - | = |          |Alpha ||||
-----+-----+-----+-------------+-----------+---+
";

#define MK_CAPS		MK_V1
#define MK_SHIFT	MK_V2
#define MK_CTRL		MK_V3
#define MK_ALT		MK_V4
#define MK_KANJI	MK_V6

struct {
	char *str;
	char *alt;
	int size;
	int attr;
	int val;
} conv_table[] = {
{"Esc"	,NULL	,12,0		,033},
{"BS"	,NULL	,12,0		,'\b'},
{"Tab"	,NULL	,12,0		,'\t'},
{"Caps"	,NULL	,12,0		,MK_CAPS},
{"Shift",NULL	,12,0		,MK_SHIFT},
{"Ctrl"	,NULL	,12,0		,MK_CTRL},
{"Alt"	,NULL	,12,0		,MK_ALT},
{"Hide"	,""	,12,0		,MK_V5},
{"Kanji",""	,12,0		,MK_V6},
{"Alpha","ѿ"	,12,0		,MK_V6},
{"Kata"	,""	,12,0		,MK_V7},
{"Han","Ⱦ"	,12,0		,MK_V9},
{"Zen",""	,12,0		,MK_V10},
{"Dic","Ͽ"	,12,0		,MK_V11},
{""	,NULL	,12,FA_BOLD	,MK_UP},
{""	,NULL	,12,FA_BOLD	,MK_LEFT},
{""	,NULL	,12,FA_BOLD	,MK_RIGHT},
{""	,NULL	,12,FA_BOLD	,MK_DOWN},
{"Ent"	,NULL 	,12,0		,'\n'},
{"er"	,NULL 	,12,0		,'\n'},
};

#define XSIZE	240
#define YSIZE	(6*16+1)

static struct virtual_key *vk_alt,*vk_ctrl;

static int conv_keyname(s) char *s; {
	int i;
	for (i=0; i<sizeof(conv_table)/sizeof(conv_table[0]); i++) {
		if (!strcmp(conv_table[i].str,s)) {
			if (conv_table[i].alt) {
				strcpy(s,conv_table[i].alt);
			}
			set_font(conv_table[i].size,conv_table[i].attr);
			return conv_table[i].val;
		}
	}
	set_font(12,0);
	return s[0];
}

static struct screen *create_keymap(layout,v) char *layout; struct virtual_key *v; {
	static int cell_w = 5;
	static int ch = 16;
	char buf[1024];
	char *p,*top,*line_buf[10];
	int line_size[10];
	char key_name[20];
	int line = 0;
	int ln=0,ls=0;
	int x,xx,kx,w=0;
	int i,j,jmax,code;
	struct virtual_key *vk;
	struct screen *s;

	s = create_memscreen(XSIZE,YSIZE,NULL,STK_NATIVE,0);
	if (!s) return NULL;
	push_screen(s);
	set_color(COLOR_BG);
	clear_screen();
	set_color(COLOR_BLACK);
//printf("create_keymap\n");
	strcpy(buf,layout);
	p = buf;
	top = buf;
	ls = 0;
	while (*p) {
		if ((*p == '-') || (*p == '+')) {
			ln++;
		}
		if (!ls && (*p == ' ')) ls = p - top;
		if (*p == '\n') {
			*p = 0;
			if (ln < 10) {
				line_buf[line] = top;
				line++;
			} else {
				line_size[line] = ls;
			}
			top = p+1;
			ln = ls = 0;
		}
		p++;
	}
	set_font(12,0);
	set_color(COLOR_LINE);
	draw_line(0,0,current_screen->width-1,0);
	draw_line(0,0,0,current_screen->height-1);
	for (i=0; i<line; i++) {
		jmax = strlen(line_buf[i]);
		xx = 0;
		p = line_buf[i];
		kx = 0;
		for (x=0; *p; p++,x++) {
			if (*p == '|' && p[1] != '|') {
				if (!kx) {
					key_name[kx++] = ' ';
				}
				key_name[kx] = 0;
				code = conv_keyname(key_name);
				w = ((x - xx)*cell_w) - (strlen(key_name)*6);
				w /= 2;
				vk = create_virtual_key(3+xx*cell_w,i*ch
						,(x-xx)*cell_w,ch,code);
				vk_attach(v,vk);
				if (code == MK_CTRL) {
					vk_ctrl = vk;
				}
				if (code == MK_ALT) {
					vk_alt = vk;
				}
				set_color(COLOR_BLACK);
				draw_string(3+xx*cell_w+w,i*ch+2,key_name,0);
				set_color(COLOR_LINE);
				draw_line(3+x*cell_w,i*ch,3+x*cell_w,i*ch+ch);
				xx = x;
				kx = 0;
			}
			else if (*p == '|' && p[1] == '|') {
				key_name[kx++] = *p;
				p++;
			} else if (*p != ' ') {
				key_name[kx++] = *p;
			}
		}
		set_color(COLOR_LINE);
		if (line_size[i+1])
			draw_line(0,i*ch+ch,3+(line_size[i+1]-1)*cell_w,i*ch+ch);
		else
			draw_line(0,i*ch+ch,3+(x-1)*cell_w,i*ch+ch);
	}
	pop_screen();
	return s;
}

static struct screen *keymap_s[3];
static struct virtual_key *keymap_v[3];
static int cur_layout = 0;
static struct screen *scr_sav;

static int c_line[48];
static int c_pos;
static int c_mode;

static int put_line(int ch) {
	if (ch == '\b') {
	    if (c_pos == 0) {
		c_line[c_pos++] = ch;
		return 1;
	    }
	    c_pos--;
	    if (c_line[c_pos] & 0x80) {
		c_pos--;
		set_color(COLOR_BG);
		fill_rect(c_pos * 6 + 2, 2, 12, 12);
		return 0;
	    } else {
		set_color(COLOR_BG);
		fill_rect(c_pos * 6 + 2, 2, 6, 12);
		return 0;
	    }
	} else if (ch < 0x20) {
		c_line[c_pos++] = ch;
		return 1;
	} else if (ch & 0x80) {
	    if (c_mode) {
		set_color(COLOR_BLACK);
		draw_font(c_pos * 6 + 2, 2, (c_mode << 8) | ch , DIR_NORTH);
		c_line[c_pos++] = c_mode;
		c_line[c_pos++] = ch;
		c_mode = 0;
	    } else {
		c_mode = ch;
		return 0;
	    }
	} else {
		set_color(COLOR_BLACK);
		draw_font(c_pos * 6 + 2, 2, ch , DIR_NORTH);
		c_line[c_pos++] = ch;
		
	}
	if (c_pos < 39) {
		return 0;
	}
	return 1;
}

static int get_line(char *p,int len) {
	int ret = 0;
	set_color(COLOR_BG);
	fill_rect(2,2,c_pos * 6,12);
	for (ret=0; (ret< c_pos) && (ret < len); ret++) {
		if (c_line[ret] & 0x80) {
			if (ret + 2 >= len) break;
			*p++ = c_line[ret++];
		}
		*p++ = c_line[ret];
	}
	c_pos = 0;
	c_mode = 0;
	return ret;
}

static void redraw_line() {
	int ret = 0;
	set_color(COLOR_BLACK);
	for (ret=0; ret< c_pos ; ret++) {
	    if (c_line[ret] & 0x80) {
		draw_font(ret * 6 + 2, 2, (c_line[ret] << 8) | c_line[ret+1]
			 , DIR_NORTH);
		ret++;
	     } else {
		draw_font(ret * 6 + 2, 2, c_line[ret] , DIR_NORTH);
	     }
	}
	return;
}

static int mode_ctrl = 0;
static int mode_alt = 0;

static int imparted = 0;
static struct virtual_key *imv;
static struct screen *ims;

int im_read(char *ibuf, int ibufsize) {
	int i,c,n,r;
	struct virtual_key *v,*vp;
	struct screen *s;
	static int initialized;
	char *layout;
	int lock = 0;
	int kanji = 0;
	int new_layout = 0;
	int vx,vy;
	int pos = 0;
	int next_unlock = 0;

	set_font(12,0);
	r = im_view_point(XSIZE,YSIZE-1,&vx,&vy);
	if (r != 2) {
		r = im_view_point(SCREEN_WIDTH,YSIZE,&vx,&vy);
	}
	if (r == 2) {
	    imparted = 1;
	    if (ims == NULL) {
		ims = create_subscreen(NULL,vx,vy,XSIZE,YSIZE-1);
		imv = create_virtual_key(vx,vy,XSIZE,YSIZE-1,MK_V0);
	        vk_attach(NULL,imv);
	    }
	}
	if (!initialized) {
	    for (i=0; i<3; i++) {
		v = create_virtual_key(0,0,XSIZE,YSIZE,MK_V0);
		s = create_keymap((i==0)?kb_layout1:
				  (i==1)?kb_layout2:kb_layout3,v);
		keymap_s[i] = s;
		keymap_v[i] = v;
	    }
	    initialized = 1;
	}

	if (!imparted) {
	    s = create_subscreen(NULL,vx,vy,XSIZE,YSIZE);
	    vp = create_virtual_key(vx,vy,XSIZE,YSIZE,MK_V0);
	    vk_attach(NULL,vp);
	} else {
	    s = ims;
	    vp = imv;
	}
	vk_attach(vp,keymap_v[cur_layout]);

	if (!imparted) {
	    if (!scr_sav) {	
		scr_sav  = create_memscreen(XSIZE,YSIZE,NULL,STK_NATIVE,0);
	    }
	    bitblt(scr_sav,0,0,s,0,0,-1,-1,0);
	}
	bitblt(s,0,0,keymap_s[cur_layout],0,0,-1,-1,0);
	push_screen(s);

	for(;;) {
	    if (next_unlock) {
		lock = !lock;
		next_unlock = 0;
	    }
	    c = get_key(-1);
	    if (c == MK_CAPS) {
		lock = !lock;
	    }
	    else if (c == MK_CTRL) {
		mode_ctrl = !mode_ctrl;
		set_color(COLOR_REVERSE);
		if (vk_ctrl) {
		    fill_rect(vk_ctrl->x,vk_ctrl->y,vk_ctrl->xs,vk_ctrl->ys);
		}
	    }
	    else if (c == MK_ALT) {
		mode_alt = !mode_alt;
		set_color(COLOR_REVERSE);
		if (vk_alt) {
		    fill_rect(vk_alt->x,vk_alt->y,vk_alt->xs,vk_alt->ys);
		}
	    }
	    else if (c == MK_SHIFT) {
		lock = !lock;
		next_unlock = 1;
	    }
	    else if (c == MK_KANJI) {
			kanji = !kanji;
	    } else {
		if (put_line(c) || mode_alt) {
			n = get_line(ibuf,ibufsize);
			pos += n;
			if (mode_ctrl && (ibuf[pos] != 'O'&0x1f))
				 ibuf[pos++] = 'O'&0x1f;
			break;
		}
	    }
	    if (kanji) new_layout = 2;
	    else if (lock) new_layout = 1;
	    else new_layout = 0;
	    if (new_layout != cur_layout) {
		vk_detach(keymap_v[cur_layout],0);
		vk_attach(vp,keymap_v[new_layout]);
		bitblt(s,0,0,keymap_s[new_layout],0,0,-1,-1,0);
		redraw_line();
		cur_layout = new_layout;
		mode_ctrl = 0;
		if ((cur_layout != kanji) && mode_alt && vk_alt) {
		    fill_rect(vk_alt->x,vk_alt->y,vk_alt->xs,vk_alt->ys);
		}
	    }
	}
	vk_detach(keymap_v[cur_layout],0);
	if (!imparted) {
	    bitblt(s,0,0,scr_sav,0,0,-1,-1,0);
	}
	pop_screen();
	if (!imparted) {
	    free_screen(s);
	    vk_detach(vp,1);
	}
	return pos;
}
