/*
 * MGTERM -- Terminal Emulator for MobileGear -
 * Copyright (C) 1998, 1999, 2000
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      SATO Kazumi (sato@netbsd.org)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <time.h>
#include "cal.h"
#include "mgl2.h"
#include "mgterm.h"


static char *month_names[2][12] = {
	{
	"January", "February", "March", "April", "May", "June",
	"July", "August", "September", "October", "November", "December",
	}
	,{
	"", "", "", "", "", "",
	"", "", "", "", "", "",
	}};

static char *weektitle[2] = {"  S  M Tu  W Th  F  S"
		    ,"       "};

#define XOFF	480
#define YOFF	8
#define XSIZE	160
static int ysize = 224;
static int from_main=0;
#define YSIZE	(ysize)
#define IXOFF	(XOFF+4)
#define IYOFF	(YOFF+4)
#define IXSIZE	(XSIZE-8)
#define IYSIZE	(YSIZE-8)

#ifdef MINIAPLI
#undef XOFF
#undef YOFF
#define XOFF	0
#define YOFF	0

void mdate();

static char *icon_mdate="\
#MGR000200160016
++++++++++++++++
+@@@@@@@@@@@@@++
+@...........@++
+@...........@++
+@.@.@.@.@.@.@++
+@.@.@.@.@.@.@++
+@...........@++
+@.@.@.@.@.@.@++
+@...........@++
+@.@.@.@.@.@.@++
+@...........@++
+@.@.@.@.@.@.@++
+@...........@++
+@@@@@@@@@@@@@++
++++++++++++++++
++++++++++++++++
";
main() {
	int c;
	mgl_apli_type = AT_MINIAPLI;
	open_graph();
	if (!mgl_client) {
		printf("server not running \n");
		exit(2);
	}
	set_icon(icon_mdate,"mdate");

	from_main = 1;
	mdate(-2);
	for (;;) {
		c = get_key(10);
		mdate(c);
	}
}
#endif

void mem_draw(void);

static void help() {
	static char *help_msg = "\
פǤ\n\
\n\
  ηΥ\n\
  ηΥ\n\
s     ץ⡼ѹ()\n";

	struct textscreen *ts;
	ts = create_textscreen(0,XOFF,YOFF,XSIZE,YSIZE,TS_BORDER|TS_BLINE|TS_SAVE);
	ts_clear(ts);
	set_color(COLOR_BLACK);
	set_font(12,0);
	ts_put_string(ts,help_msg,0);
	refresh();
	get_key(50);
	free_textscreen(ts);
	refresh();
}
static int mode_sec = 0;
static int mode_mem = 0;
static int mode_rmem = 0;
static struct textscreen *ts;

void mdate(c) {
	time_t t;
	struct tm *tm;
	char buf[512];
	int days[42];
	int i,j,d;
	int ri,rj;
	static int month;
	static int year;
	static int mday;
	static int hour;
	static int min;
	int cc = 0;

	if (c == -3) {
		set_color(COLOR_BLACK);
		draw_rect(XOFF,YOFF,XSIZE-1,YSIZE-1);
		return;
	} else if (c == -4) {
		set_color(ts->bgcolor);
		draw_rect(XOFF,YOFF,XSIZE-1,YSIZE-1);
		return;
	}

	time(&t);
	tm = localtime(&t);
	if (c == -2) {
		if (SCREEN_HEIGHT < 224) {
			YSIZE = SCREEN_HEIGHT;
			if (!from_main) YSIZE -= 16;
		}
		if (ts == NULL) {
			month = tm->tm_mon;
			year = tm->tm_year;
			mday = tm->tm_mday;
			hour = -1;
			min = -1;
			ts = create_textscreen(NULL,XOFF,YOFF,XSIZE,YSIZE,TS_BORDER|TS_BLINE);
			ts_set_bgcolor(ts,COLOR_LIGHTGRAY);
		}
		ts_clear(ts);
	}
	if (mday != tm->tm_mday) {
	    mday = tm->tm_mday;
	    cc = 1;
#ifdef RESET_CALENDAR
	    /* դѤäȤˡɽƤ륫η򺣷ˤ */
	    if (month != tm->tm_mon) { /* Ǥʤɤ¾ɽƤ */
		month = tm->tm_mon;
		if (year != tm->tm_year) {
		    year = tm->tm_year;
		}
	    }
#endif
	}

	switch(c) {
	case MK_PAGE_DOWN:
	case 'f'&0x1f:
		month++;
		if (month > 11) {
			year++;
			month = 0;
		}
		cc = 1;
		break;
	case MK_PAGE_UP:
	case 'b'&0x1f:
		month--;
		if (month < 0) {
			year--;
			month = 11;
		}
		cc = 1;
		break;
	case 's':
		mode_sec = !mode_sec;
		break;
	case 'm':
		mode_mem = !mode_mem;
		break;
	case 'r':
		mode_rmem = !mode_rmem;
		break;
	case '?':
		help();
		break;
	}

	set_font(12,0);
	set_color(COLOR_BLACK);
	set_font(24,FA_BOLD);
	if (mode_sec) {
		sprintf(buf,"%2d:%02d:%02d",tm->tm_hour,tm->tm_min,tm->tm_sec);
	} else {
		sprintf(buf,"  %2d:%02d ",tm->tm_hour,tm->tm_min);
	}
	ts_goto(ts,4+20,4);
	ts_put_string(ts,buf,0);


	if (c == -2 || cc) {
		set_color(COLOR_LIGHTGRAY);
		fill_rect(IXOFF,IYOFF+4+40,IXSIZE,IYSIZE-4-40);
		set_color(COLOR_BLACK);
		set_font(12,0);
		day_array(month+1, year+1900, days);
#if 0
		ts_goto(ts,4+40,4+8*5);
		sprintf(buf,"%s %4d\n",month_names[0][month],year+1900);
#else
		ts_goto(ts,4+40,4+8*5);
		sprintf(buf,"%4d ǯ %s\n",year+1900,month_names[1][month]);
#endif
		ts_put_string(ts,buf,0);
		
		ts_goto(ts,4+6,4+16*4);
		ts_put_string(ts,weektitle[1],0);

		ri = rj = -1;
		for (i=0; i<6; i++) {
			buf[0] = 0;
			for (j=0; j<7; j++) {
				d = days[i*7+j];
				if (d<0) 
					sprintf(buf+strlen(buf),"   ");
				else
					sprintf(buf+strlen(buf)," %2d",d);
				if (d == tm->tm_mday) {
					ri = i;
					rj = j;
				}
			}
			ts_goto(ts,4+6,4+16*(5+i));
			ts_put_string(ts,buf,0);
		}
		if (tm->tm_mon == month && tm->tm_year == year && ri >= 0) {
			set_color(COLOR_REVERSE);
			fill_rect(IXOFF+4+6+rj*18+6-2,IYOFF+4+16*(5+ri)-2,12+4,12+4);
			set_color(COLOR_BLACK);
		}
refresh();
	}
	if (YSIZE >= 200)
		mem_draw();
}


#ifdef __FreeBSD__
#include <sys/types.h>
#include <nlist.h>
#include <fcntl.h>
#include <kvm.h>
#include <sys/vmmeter.h>
#if (__FreeBSD__ <= 3)
#include <sys/rlist.h>
#endif
#include <sys/conf.h>
#include <osreldate.h>

static int init_kvm();
int swapmode();

static struct nlist nlst[] = {
	{ "_cnt" },
/* Swap */
#define VM_SWAPLIST     1
        { "_swaplist" },/* list of free swap areas */
#define VM_SWDEVT       2
        { "_swdevt" },  /* list of swap devices and sizes */
#define VM_NSWAP        3
        { "_nswap" },   /* size of largest swap device */
#define VM_NSWDEV       4
        { "_nswdev" },  /* number of swap devices */
#define VM_DMMAX        5
        { "_dmmax" },   /* maximum size of a swap block */
	{ NULL },
};

static kvm_t *kd;
static int init_fail;

static int init_kvm() {
    int fd;

    fd = open("/dev/kmem",O_RDONLY);
    if (fd < 0) {
	init_fail = 1;
	return -1;
    }
    close(fd);

    if (init_fail || kd) return -1;
    if ((kd = kvm_open(NULL, NULL, NULL, O_RDONLY, "kvm_open")) == NULL) {
	init_fail = 1;
	return -1;
    }

    /* get the list of symbols we want to access in the kernel */
    (void) kvm_nlist(kd, nlst);
    if (nlst[0].n_type == 0)
    {
	kvm_close(kd);
	kd = NULL;
	init_fail = 1;
	return(-1);
    }
    return 0;
}

void mem_draw() {
	int ret;
	int swpavail,swpfree;
	char buf[30];
	struct vmmeter vmm;
	int m;

	if (init_fail) return;
	init_kvm();

	if (kd == NULL) return;
	ret = kvm_read(kd, nlst[0].n_value, (char *)(&vmm), sizeof(vmm));
	if (ret != sizeof(vmm)) return;
	swapmode(&swpavail, &swpfree);

	m = vmm.v_free_count-vmm.v_free_reserved;
	if (m < 0) m = 0;
	m *= 4;
	if (!mode_rmem)
	m +=vmm.v_cache_count;
	set_font(12,FA_BOLD);
	set_color(COLOR_DARKGRAY);
	if (mode_mem) {
		ts_goto(ts,24,YSIZE-40);
		ts_put_string(ts,"                 ",0);
		sprintf(buf,"ĥ%6d KB ",m);
		ts_goto(ts,24,YSIZE-20);
		ts_put_string(ts,buf,0);
	} else {
		sprintf(buf,"ĥ%6d KB ",m);
		ts_goto(ts,24,YSIZE-40);
		ts_put_string(ts,buf,0);
		sprintf(buf,"å%6d KB ", swpfree);
		ts_goto(ts,24,YSIZE-20);
		ts_put_string(ts,buf,0);
	}
/*
δطϼΤ褦ˤʤäƤ롣

		vmm.v_page_count (Ѳǽ)
		=
		vmm.v_active_count
		 + vmm.v_inactive_count
		 + vmm.v_wire_count 
		 + vmm.v_cache_count( ĤǤ Free Ǥڡ)
		 + vmm.v_free_count (Free ڡ)

Ĥϡ
	vmm.v_cache_count(ĤǤFreeǤ)
	 + vmm.v_free_count

ɽ뤬
ºݤϡ vmm.v_free_reserver ʲˤ(ۤȤ)ʤʤ
ΤᤳͤƤʤФʤʤ

ʤsysctl -w vmm.v_free_reserver=n ǤͤѹǤ뤬
إѹȡƥबϥ󥰥åפ롣

*/
}


#if (__FreeBSD__ <= 3)
/*
/* this swapmode is based on FreeBSD packages's top-3.3/machine.c
 * swapmode is based on a program called swapinfo written
 * by Kevin Lahey <kml@rokkaku.atl.ga.us>.
 */

#define	SVAR(var) __STRING(var)	/* to force expansion */
#define	KGET(idx, var)							\
	KGET1(idx, &var, sizeof(var), SVAR(var))
#define	KGET1(idx, p, s, msg)						\
	KGET2(nlst[idx].n_value, p, s, msg)
#define	KGET2(addr, p, s, msg)						\
	if (kvm_read(kd, (u_long)(addr), p, s) != s) {		        \
		warnx("cannot read %s: %s", msg, kvm_geterr(kd));       \
		return (0);                                             \
       }
#define	KGETRET(addr, p, s, msg)					\
	if (kvm_read(kd, (u_long)(addr), p, s) != s) {			\
		warnx("cannot read %s: %s", msg, kvm_geterr(kd));	\
		return (0);						\
	}

int
swapmode(retavail, retfree)
	int *retavail;
	int *retfree;
{
	char *header;
	int hlen, nswap, nswdev, dmmax;
	int i, div, avail, nfree, npfree, used;
	struct swdevt *sw;
	long blocksize, *perdev;
	u_long ptr;
	struct rlist head;
	struct rlisthdr swaplist;
	struct rlist *swapptr;

	/*
	 * Counter for error messages. If we reach the limit,
	 * stop reading information from swap devices and
	 * return zero. This prevent endless 'bad address'
	 * messages.
	 */
	static warning = 10;

	if (warning <= 0) {
	    /* a single warning */
	    if (!warning) {
		warning--;
		fprintf(stderr, 
			"Too much errors, stop reading swap devices ...\n");
		(void)sleep(3);
	    }
	    return(0);
	}
	warning--; /* decrease counter, see end of function */

	KGET(VM_NSWAP, nswap);
	if (!nswap) {
		fprintf(stderr, "No swap space available\n");
		return(0);
	}

	KGET(VM_NSWDEV, nswdev);
	KGET(VM_DMMAX, dmmax);
	KGET1(VM_SWAPLIST, &swaplist, sizeof(swaplist), "swaplist");
	if ((sw = (struct swdevt *)malloc(nswdev * sizeof(*sw))) == NULL ||
	    (perdev = (long *)malloc(nswdev * sizeof(*perdev))) == NULL)
		err(1, "malloc");
	KGET1(VM_SWDEVT, &ptr, sizeof ptr, "swdevt");
	KGET2(ptr, sw, nswdev * sizeof(*sw), "*swdevt");

	/* Count up swap space. */
	nfree = 0;
	memset(perdev, 0, nswdev * sizeof(*perdev));
	swapptr = swaplist.rlh_list;
	while (swapptr) {
		int	top, bottom, next_block;
		KGET2(swapptr, &head, sizeof(struct rlist), "swapptr");

		top = head.rl_end;
		bottom = head.rl_start;

		nfree += top - bottom + 1;

		/*
		 * Swap space is split up among the configured disks.
		 *
		 * For interleaved swap devices, the first dmmax blocks
		 * of swap space some from the first disk, the next dmmax
		 * blocks from the next, and so on up to nswap blocks.
		 *
		 * The list of free space joins adjacent free blocks,
		 * ignoring device boundries.  If we want to keep track
		 * of this information per device, we'll just have to
		 * extract it ourselves.
		 */
		while (top / dmmax != bottom / dmmax) {
			next_block = ((bottom + dmmax) / dmmax);
			perdev[(bottom / dmmax) % nswdev] +=
				next_block * dmmax - bottom;
			bottom = next_block * dmmax;
		}
		perdev[(bottom / dmmax) % nswdev] +=
			top - bottom + 1;

		swapptr = head.rl_next;
	}

	header = getbsize(&hlen, &blocksize);
	div = blocksize / 512;
	avail = npfree = 0;
	for (i = 0; i < nswdev; i++) {
		int xsize, xfree;

		/*
		 * Don't report statistics for partitions which have not
		 * yet been activated via swapon(8).
		 */

		xsize = sw[i].sw_nblks;
		xfree = perdev[i];
		used = xsize - xfree;
		npfree++;
		avail += xsize;
	}

	/* 
	 * If only one partition has been set up via swapon(8), we don't
	 * need to bother with totals.
	 */
	*retavail = avail / 2;
	*retfree = nfree / 2;
	used = avail - nfree;
	free(sw); free(perdev);

	/* increase counter, no errors occurs */
	warning++; 

	return  (int)(((double)used / (double)avail * 100.0) + 0.5);
}
#else
/* this swapmode is based on FreeBSD's top
 *
 * AUTHOR:  Christos Zoulas <christos@ee.cornell.edu>
 *          Steven Wallace  <swallace@freebsd.org>
 *          Wolfram Schneider <wosch@FreeBSD.org>
 *
 * $FreeBSD: src/usr.bin/top/machine.c,v 1.29 1999/11/17 16:31:51 davidn Exp $
 */
int
swapmode(retavail, retfree)
        int *retavail;
        int *retfree;
{
	int n;
	int pagesize = getpagesize();
	struct kvm_swap swapary[1];

	*retavail = 0;
	*retfree = 0;

#define CONVERT(v)      ((quad_t)(v) * pagesize / 1024)

	n = kvm_getswapinfo(kd, swapary, 1, 0);
	if (n < 0 || swapary[0].ksw_total == 0)
		return(0);

	*retavail = CONVERT(swapary[0].ksw_total);
	*retfree = CONVERT(swapary[0].ksw_total - swapary[0].ksw_used);
	n = (int)((double)swapary[0].ksw_used * 100.0 /
		(double)swapary[0].ksw_total);
	return(n);
}
#endif
#elif defined (__linux__)
/*
% cat /proc/meminfo
        total:    used:    free:  shared: buffers:  cached:
Mem:  146210816 34295808 111915008 19546112  3051520 19623936
Swap: 123826176        0 123826176
MemTotal:    142784 kB
MemFree:     109292 kB
MemShared:    19088 kB
Buffers:       2980 kB
Cached:       19164 kB
SwapTotal:   120924 kB
SwapFree:    120924 kB
*/
/*
used free and cached
no reserved
*/
#define MEMINFO "/proc/meminfo"
int
mem_read(m, cache, swapfree)
int *m;
int *cache;
int *swapfree;
{
	FILE *fp;
	char buffer[128];
	char label[64];
	char unit[64];
	int  value;

	/* device open */
	fp = fopen(MEMINFO, "r");
	if(!fp){
		return(0);
	}

	*m = *cache = *swapfree = 0;

	while(fgets(buffer, 127, fp) != NULL){
		if(sscanf(buffer, "%s %d %s\n", label, &value, unit) == 3){
			if(!strcmp(label, "MemFree:")){
				*m = value;
			}else
			if(!strcmp(label, "Cached:")){
				*cache = value;
			}else
			if(!strcmp(label, "SwapFree:")){
				*swapfree = value;
			}
		}
	}
	fclose(fp);
	return(1);
}
void mem_draw() {
	char buf[30];
	int swpfree;
	int cache;
	int m;

	if(mem_read(&m, &cache, &swpfree) == 0){
		return;
	}

	if (!mode_rmem)
	m +=cache;
	set_font(12,FA_BOLD);
	set_color(COLOR_DARKGRAY);
	if (mode_mem) {
		ts_goto(ts,24,184);
		ts_put_string(ts,"                 ",0);
		sprintf(buf,"ĥ%6d KB ",m);
		ts_goto(ts,24,200);
		ts_put_string(ts,buf,0);
	} else {
		sprintf(buf,"ĥ%6d KB ",m);
		ts_goto(ts,24,184);
		ts_put_string(ts,buf,0);
		sprintf(buf,"å%6d KB ", swpfree);
		ts_goto(ts,24,200);
		ts_put_string(ts,buf,0);
	}
}
#elif defined(__NetBSD__)
/*
 * Copyright (C) 2000.
 * 	SATO Kazumi (sato@netbsd.org)
 */
#include <unistd.h>
#include <sys/swap.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <vm/vm_param.h>
#ifdef UVM
#include <uvm/uvm_extern.h>
#endif
#include <errno.h>

#define NSWAPS	10

struct swapent swaps[NSWAPS];

int
memmode()
{
	struct uvmexp uvmexp;
	int mib[2];
	int size = sizeof(uvmexp);
	int pagesize = getpagesize();
	int nfree;

	static int memmode_ok = 1;

	if (!memmode_ok)
		return 0;
	mib[0] = CTL_VM;
	mib[1] = VM_UVMEXP;
	if (sysctl(mib, 2, &uvmexp, &size, NULL, 0) < 0) {
		printf("Can't get uvmexp: %s\n", strerror(errno));
		memset(&uvmexp, 0, sizeof(uvmexp));
		memmode_ok = 0;
	}
#define pgtok(a) ((long)((a)*(pagesize>>10)))
	return pgtok(uvmexp.inactive + uvmexp.free);
}

int
swapmode(retavail, retfree)
	int *retavail;
	int *retfree;
{
	int nswdev;
	int i, nfree;
	static int avail = 0, used = 0;

	static int swapmode_ok = 1;

	if (!swapmode_ok) {
		if (avail != 0 && used != 0)
			return  (int)(((double)used / (double)avail * 100.0) + 0.5);
		else
			return 0;
	}

	*retavail = avail = 0;
	*retfree = used = 0;
	nswdev = swapctl(SWAP_NSWAP, NULL, 0);
	if (nswdev == -1) {
		fprintf(stderr, "no swap devices... \n");
		swapmode_ok = 0;
		return 0;
	}
	if (nswdev > NSWAPS) {
		fprintf(stderr, "too many swap devices... \n");
		nswdev = NSWAPS;
	}
	if (swapctl(SWAP_STATS, swaps, nswdev) == -1) {
		perror("swapctl(SWAP_NSWAP)");
		swapmode_ok = 0;
		return 0;
	}
	for (i = 0; i < nswdev; i++) {
		avail += swaps[i].se_nblks;
		used += swaps[i].se_inuse;
	}

	*retavail = avail/2; /* blocks -> KB */
	nfree = avail - used;
	*retfree = nfree/2;

	return  (int)(((double)used / (double)avail * 100.0) + 0.5);
}

void 
mem_draw()
{
	int ret;
	int swpavail,swpfree;
	int memavail, memfree;
	char buf[80];

	memfree = memmode();
	set_font(12,FA_BOLD);
	set_color(COLOR_DARKGRAY);
	if (mode_mem) {
		ts_goto(ts,24,184);
		ts_put_string(ts,"                 ",0);
		sprintf(buf,"ĥ%6d KB ",memfree);
		ts_goto(ts,24,200);
		ts_put_string(ts,buf,0);
	} else {
		swapmode(&swpavail, &swpfree);
		sprintf(buf,"ĥ%6d KB ",memfree);
		ts_goto(ts,24,184);
		ts_put_string(ts,buf,0);
		sprintf(buf,"å%6d KB ", swpfree);
		ts_goto(ts,24,200);
		ts_put_string(ts,buf,0);
	}
}


#else 
void mem_draw()
{
}
#endif

