/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#include "mgl2.h"
#include <stdlib.h>

#ifndef NULL
#define NULL	(void *)0
#endif

static struct textscreen *t_freelist;

extern struct screen *current_screen;

struct textscreen *create_textscreen(struct screen *s, int x,
                                     int y, int xs, int ys, int attr) {
	struct screen *ss,*sc;
	struct textscreen *t;

	if (s == NULL) s = current_screen;
	ss = create_subscreen(s,x,y,xs,ys);
	if (ss == NULL) return NULL;
	if ((attr & TS_BORDER ) && (attr & TS_BLINE)) {
		sc = create_subscreen(s,x+4,y+4,xs-8,ys-8);
		if (!sc) {
			free_screen(ss);
			return NULL;
		}
	} else {
		sc = ss;
	}
	if (t_freelist) {
		t = t_freelist;
		t_freelist = *(struct textscreen **)t_freelist;
	} else {
		t = (struct textscreen *)malloc(sizeof(*t));
	}
       	if(!t){
		perror("malloc");
		free_screen(ss);
		return (struct textscreen *)NULL;
       	}
	t->attr = attr;
	t->ss = ss;
	t->sc = sc;
	t->x = x;
	t->y = y;
	t->xs = xs;
	t->ys = ys;
	t->xpos = 0;
	t->ypos = 0;
	t->lh = 0;
	t->bgcolor = COLOR_WHITE;
	if (t->attr & TS_SAVE) {
		t->save = create_memscreen(xs,ys,NULL,STK_NATIVE,0);
		bitblt(t->save,0,0,s,x,y,xs,ys,0);
	} else {
		t->save = NULL;
	}
	return t;
}

void free_textscreen(struct textscreen *t) {
	if (t->save) {
		bitblt(t->ss,0,0,t->save,0,0,t->xs,t->ys,0);
		free_screen(t->save);
	}
	if (t->ss != t->sc) free_screen(t->sc);
	free_screen(t->ss);
	*(struct textscreen **)t = t_freelist;
        t_freelist = t;
	return;
}

static void ts_scroll(struct screen *t, int dy) {
	int c;
	bitblt(NULL,0,0,NULL,0,dy,t->width,t->height-dy,0);
	c = pen_color;
	set_color(COLOR_WHITE);
	fill_rect(0,t->height-dy,t->width,dy);
	set_color(c);
	return;
}


void ts_clear(struct textscreen *t) {
	int c;
	struct screen *ss;

	ss = t->ss;
	push_screen(ss);
	set_color(t->bgcolor);
	clear_screen();
	if ((t->attr & TS_BORDER ) && (t->attr & TS_BLINE)) {
		int d=2;
#if 0
		set_color(COLOR_BLACK);
		draw_rect(d,d,t->xs-d*2,t->ys-d*2);
#else
		set_color(COLOR_DARKGRAY);
		draw_rect(d,d,t->xs-d*2+1,t->ys-d*2+1);
		set_color(COLOR_LIGHTGRAY);
		draw_rect(d-1,d-1,t->xs-d*2+1,t->ys-d*2+1);
#endif
	}
	pop_screen();
	return;
}

void ts_put_string(struct textscreen *t, char *str, int op) {
	struct screen *ss;
	int code;
	int w,h;
	int d=0;
	if (t->attr & TS_BORDER) d = 4;
	ss = t->sc;
	push_screen(ss);
	h = pen_font->height;
	while (*str) {
		if (*str & 0x80) {
			if ((unsigned char)*str == 0x8E){
				code = (*str++ << 8) & 0xffff;
				code |= *str++ & 0xff;
				w = pen_font->height/2;
			}else{
				code = (*str++ << 8) & 0xffff;
				code |= *str++ & 0xff;
				w = pen_font->height;
			}
		} else {
			code = *str++;
			w = pen_font->height/2;
		}
		if (t->xpos + w > ss->width || code == '\n') {
			t->xpos = 0;
			if (h > t->lh) t->lh = h;
			t->ypos += t->lh;
			t->lh = 0;
		}
		if (code >= 0x20) {
			if (t->ypos + h > ss->height) {
				ts_scroll(ss,t->ypos + h - ss->height);
				t->ypos = ss->height - h;
			}
			if (op==0) {
				int c = pen_color;
				int hh = h;
				if (t->lh > hh) hh = t->lh;
				set_color(t->bgcolor);
				fill_rect(t->xpos,t->ypos,w,hh);
				set_color(c);
			}
			draw_font(t->xpos,t->ypos,code,DIR_NORTH);
			t->xpos += w;
			if (h > t->lh) t->lh = h;
		}
	}
	pop_screen();
	return;
}

void ts_goto(struct textscreen *t, int x, int y) {
	t->xpos = x;
	t->ypos = y;
	t->lh = 0;
	return;
}

void ts_set_bgcolor(struct textscreen *t, int c) {
	t->bgcolor = c;
	return;
}

void ts_put_image(struct textscreen *t, struct screen *s,
                  int sx, int sy, int w, int h, int op) {
	struct screen *ss;
	int d=0;

	if (t->attr & TS_BORDER) d = 4;
	ss = t->sc;
	push_screen(ss);

	if (w <= 0) w = s->width - sx;
	if (h <= 0) h = s->height - sy;
	if (w > ss->width) w = ss->width;
	if (h > ss->height) h = ss->height;

	if (t->xpos + w > ss->width) {
		t->xpos = 0;
		t->ypos += t->lh;
		t->lh = 0;
	}
	if (h > t->lh) t->lh = h;

	if (t->ypos + h > ss->height) {
		ts_scroll(ss,t->ypos + h - ss->height);
		t->ypos = ss->height - h;
	}
	if ((op==0) && (h < t->lh)) {
		int c = pen_color;
		int hh = h;
		if (t->lh > hh) hh = t->lh;
		set_color(COLOR_WHITE);
		fill_rect(t->xpos,t->ypos,w,hh);
		set_color(c);
	}
	bitblt(NULL,t->xpos,t->ypos,s,sx,sy,w,h,op);
	t->xpos += w;
	pop_screen();
	return;
}

