/*
 * Copyright (c) 2004, 2005 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <strings.h>
#include <time.h>

#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>


#include "config.h"
#include "packets.h"
#include "mdd.h"

int quiet = 0;
int debug = 0;
int verbose = 0;

static char *humanize_offset(off_t);

int main (argc, argv)
	int argc;
	char **argv;
{
	char ch;
	int net_block_size = NET_BLOCK_SIZE;
	int write_block_size = WRITE_BLOCK_SIZE;
	int block_size = 0;
	int do_sender = 0;
	int do_sniffer = 0;
	int do_stat = 0;
	int do_receiver = 0;
	struct timeval timeout = { TIMEOUT_SEC, TIMEOUT_USEC };
	char *file_name = NULL;
	in_addr_t group_addr;
	u_char ttl = 1;

	while((ch = getopt(argc, argv, "qdvr:s:f:b:t:T:w:W:")) != (char)-1) {
		switch (ch) {
		case 'q':
			quiet = 1;
			break;
		case 'd':
			debug = 1;
			verbose = 1;
			break;
		case 'v':
			verbose = 1;
			break;
		case 'r':
			do_receiver = 1;
			if ((inet_pton(AF_INET, optarg, &group_addr)) != 1) {
				printf("Invalid address \"%s\"\n", optarg);
				exit(-1);
			}
			break;
		case 's':
			do_sender = 1;
			if ((inet_pton(AF_INET, optarg, &group_addr)) != 1) {
				printf("Invalid address \"%s\"\n", optarg);
				exit(-1);
			}
			break;
		case 'f':
			file_name = optarg;
			break;
		case 'b':
			block_size = atoi(optarg);
			break;
		case 't':
			timeout.tv_usec = atol(optarg);
			timeout.tv_sec = timeout.tv_usec / 1000000;
			timeout.tv_usec = timeout.tv_usec % 1000000;
			break;
		case 'T':
			ttl = (u_char)atoi(optarg);
			break;
		case 'w':
			if ((inet_pton(AF_INET, optarg, &group_addr)) != 1) {
				printf("Invalid address \"%s\"\n", optarg);
				exit(-1);
			}
			do_sniffer = 1;
			break;
		case 'W':
			if ((inet_pton(AF_INET, optarg, &group_addr)) != 1) {
				printf("Invalid address \"%s\"\n", optarg);
				exit(-1);
			}
			do_stat = 1;
			break;
		default:
			printf("Usage: mdd -w addr [-d]\n");
			printf("       mdd -W addr [-d]\n");
			printf("       mdd -s addr [-d|-v|-q] [-T ttl] "
			       "[-b blocksize] -f file \n");
			printf("       mdd -r addr [-d|-v|-q] [-T ttl] "
			       "[-b blocksize] [-t timeout] -f file \n");
			exit(-1);
			break;
		}
	}

	if (do_sender) {
		if (block_size != 0)
			net_block_size = block_size;

		if (verbose) {
			printf("Running as sender, ");
			printf("file name: \"%s\", ", file_name);
			printf("block size: %d\n", net_block_size);
		}
		sender(file_name, net_block_size, group_addr, ttl);
		printf("\n");
		return 0;
	}

	if (do_sniffer) {
		if (verbose)
			printf("Running as sniffer\n");
		sniffer(group_addr, 0);
		printf("\n");
		return 0;
	}

	if (do_stat) {
		if (verbose)
			printf("Collecting statistics\n");
		sniffer(group_addr, 1);
		printf("\n");
		return 0;
	}

	if (do_receiver) {
		/* Else: receiver */
		if (block_size != 0)
			write_block_size = block_size;

		if (verbose) {
			printf("Running as receiver, ");
			printf("file name: \"%s\", ", file_name);
			printf("timeout: %ld.%06ld sec\n", 
			    timeout.tv_sec, timeout.tv_usec);
			printf("block size: %d\n",  write_block_size);
		}
		receiver(file_name, write_block_size, timeout, group_addr, ttl);
		printf("\n");
		return 0;
	}

	printf("Either -r, -s or -w is required\n");
	return -1;
}

in_addr_t
local_addr(dst_addr)
	in_addr_t dst_addr;
{
	struct sockaddr_in src;
	struct sockaddr_in dst;
	int sock = -1;
	socklen_t socklen;

	bzero(&src, sizeof(src));
	src.sin_len = sizeof(src);
	src.sin_family = AF_INET;

	dst.sin_len = sizeof(dst);
	dst.sin_family = AF_INET;
	dst.sin_port = htons(DATA_SEND_PORT);
	dst.sin_addr.s_addr = dst_addr;

	if ((sock = socket(dst.sin_family, SOCK_DGRAM, 0)) == -1) {
		perror("Cannot open socket");
		goto err;
	}

	if (connect(sock, (struct sockaddr *)&dst, dst.sin_len) != 0) {
		perror("connect failed");
		goto err;
	}

	socklen = src.sin_len;
	if (getsockname(sock, (struct sockaddr *)&src, &socklen) != 0) {
		perror("getsockname failed");
		goto err;
	}

err:
	close(sock);
	return src.sin_addr.s_addr;
}

#define MAXSTRLEN 80
static char *
humanize_offset(offset)
	off_t offset;
{
	static char str[MAXSTRLEN + 1];
	off_t b, kb, mb, gb, tb;

	kb = offset / 1024;
	b = offset % 1024;

	mb = kb / 1024;
	kb = kb % 1024;

	gb = mb / 1024;
	mb = mb % 1024;

	tb = gb / 1024;
	gb = gb % 1024;

	if (tb != 0) {
		snprintf(str, MAXSTRLEN, 
		    "Progress: %lld.%03lld TB    ", tb, gb * 1000 / 1024); 
		return str;
	}

	if (gb != 0) {
		snprintf(str, MAXSTRLEN, 
		    "Progress: %lld.%03lld GB    ", gb, mb * 1000 / 1024); 
		return str;
	}

	if (mb != 0) {
		snprintf(str, MAXSTRLEN, 
		    "Progress: %lld.%03lld MB    ", mb, kb * 1000 / 1024); 
		return str;
	}

	snprintf(str, MAXSTRLEN, 
	    "Progress: %lld.%03lld kB    ", kb, b * 1000 / 1024); 
	return str;
}


void
print_report(offset)
	off_t offset;
{
	struct timeval now;
	static struct timeval last = { 0, 0 };
	struct timeval interval = { REPORT_INTERVAL_SEC, REPORT_INTERVAL_USEC };
	char *report;
	int i;

	gettimeofday(&now, NULL);
	if (timercmp(&now, &last, >)) {
		report = humanize_offset(offset);
		printf("%s", report);

		for (i = 0; i < strlen(report); i++)
			printf("%c", 8);

		fflush(stdout);
		timeradd(&interval, &now, &last);
	}

	return;
}
