/*	$Id: mad123.c,v 1.24 2004/09/22 11:04:34 steve Exp $	*/

/*
 Command Line Front-end for Robert Leslie's `libmad' mp3 decoder.
 Copyright (C) 2001-2003 Steve Woodford.

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/poll.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <mad.h>

#include "audio_dev.h"
#include "mp3_stream.h"

#define MAD123_AUDIO_SLAVE_PRIORITY_INCR	8
#define MAD123_STREAMER_PRIORITY_INCR		4


static unsigned int	mp3buf_len = 2048;
static void		*mp3fp;
static int		mp3ended;
static unsigned char	*mp3buf;
static unsigned int	mp3len;

static void		*audio_ctx;

static int		total_tracks;
static int		playing_track;

static volatile int	player_stop, player_quit, player_controls_error;
static int		player_paused, player_next, player_prev, player_rewind;
static int		player_controls, player_quiet;
static int		read_ahead_ok;

static struct termios	stdin_tio_orig;
static int		parent_pid;

static void		stdin_cleanup(void);
static void		play_event(void *);
static void		play_print_stats(int, int);
static enum mad_flow	play_input(void *, struct mad_stream *);
static enum mad_flow	play_output(void *, struct mad_header const *,
			    struct mad_pcm *);
static void		play_next(int);
static int		process_key_controls(void *);
static char * const *	read_playlist(const char *, int *);

extern char		*optarg;
extern int		optind;


static void
stdin_cleanup(void)
{

	if (player_controls && getpid() == parent_pid)
		tcsetattr(fileno(stdin), TCSANOW, &stdin_tio_orig);
}

static void
play_event(void *arg)
{

	if (player_controls) {
		struct pollfd infd;
		int rv;

		infd.fd = fileno(stdin);
		infd.events = POLLRDNORM;
		infd.revents = 0;

		while ((rv = poll(&infd, 1, 0)) < 0 && errno == EINTR)
			;

		if (rv < 0 ||
		    ((infd.revents & POLLRDNORM) != 0 &&
		     process_key_controls(arg) < 0)) {
			player_controls_error = 1;
		}
	}

	play_print_stats(audio_dev_played_time(arg),
			 audio_dev_buffer_time(arg));

}

static void
play_print_stats(int played, int reserve)
{
	static int last, lastpause;

	if (player_quiet)
		return;

	if (abs(played - last) >= 3 || lastpause != player_paused) {
		printf(
	    " %s: %2d:%02d:%d0  Buffered: %2d seconds  Read-ahead: %2d%% \r",
		    player_paused ? " Paused" : "Playing",
		    played / 600, (played / 10) % 60,
		    played % 10, reserve / 10,
		    mp3_stream_read_ahead_size(mp3fp));
		fflush(stdout);
		last = played;
	}

	if (lastpause != player_paused)
		lastpause = player_paused;
}

static enum mad_flow
play_input(void *arg, struct mad_stream *mstream)
{
	ssize_t rv;

	if (player_stop)
		return (MAD_FLOW_STOP);

	if (mstream->next_frame) {
		if ((&mp3buf[mp3len] - mstream->next_frame) == mp3buf_len) {
			/*
			 * libmad is more than likely skipping some corruption
			 * in the mp3 stream, and needs a larger input buffer
			 * to do so.
			 * Double-check the error just to be sure, before
			 * increasing our input buffer size.
			 */
			unsigned char *tmp;
			if (mstream->error != MAD_ERROR_BUFLEN)
				return (MAD_FLOW_BREAK);
			if (mp3buf_len >= (1*1024*1024)) {
				fprintf(stderr, "mad123: run-away input "
				    "buffer size. Corrupt mp3?\n");
				fflush(stderr);
				return (MAD_FLOW_BREAK);
			}
			tmp = realloc(mp3buf, mp3buf_len * 2);
			if (tmp == NULL)
				return (MAD_FLOW_BREAK);
			mp3buf = tmp;
			mp3buf_len *= 2;
		} else
			memmove(mp3buf, mstream->next_frame,
			    mp3len = &mp3buf[mp3len] - mstream->next_frame);
	} else
	if (mp3ended)
		return (MAD_FLOW_STOP);

	do {
		rv = mp3_stream_read(mp3fp, &mp3buf[mp3len],
		    mp3buf_len - mp3len, read_ahead_ok);
	} while (rv < 0 && errno == EINTR);

	if (rv < 0)
		return (MAD_FLOW_BREAK);
	else
	if (rv == 0 && mp3ended == 0) {
		while (rv < MAD_BUFFER_GUARD)
			mp3buf[mp3len + rv++] = 0;
		mp3ended = 1;
	}

	if (rv >= 0 && mp3ended < 2)
		mad_stream_buffer(mstream, mp3buf, mp3len += rv);
	else
		return (MAD_FLOW_STOP);

	if (rv == 0)
		mp3ended++;

	return (MAD_FLOW_CONTINUE);
}

static enum mad_flow
play_output(void *arg, struct mad_header const *mhead, struct mad_pcm *mpcm)
{
	struct pollfd infd[2];
	int rv, fd, nfds, reserve, played;
	char foo;

	while ((rv = audio_dev_write(arg, mhead, mpcm, &fd)) == AD_WOULD_BLOCK){
		if (player_stop)
			return (MAD_FLOW_STOP);

		infd[0].fd = fd;
		infd[0].events = POLLRDNORM;
		infd[0].revents = 0;
		nfds = 1;

		if (player_controls) {
			infd[1].fd = fileno(stdin);
			infd[1].events = POLLRDNORM;
			infd[1].revents = 0;
			nfds = 2;
		}

		while ((rv = poll(infd, nfds, INFTIM)) < 0 && errno == EINTR)
			;

		if (rv < 0)
			return (MAD_FLOW_STOP);

		if ((infd[0].revents & POLLRDNORM) != 0) {
			do
				rv = read(fd, &foo, sizeof(foo));
			while (rv < 0 && errno == EINTR);

			if (rv <= 0)
				return (MAD_FLOW_BREAK);

		}

		play_event(audio_ctx);

		if (player_stop || player_controls_error)
			return (MAD_FLOW_STOP);
	}

	if (rv != AD_NO_ERROR)
		return (MAD_FLOW_BREAK);

	/*
	 * Dynamically enable and disable read-ahead depending
	 * on how much buffered `reserve' data is currently sitting
	 * in the output ring-buffer.
	 * Anything over 5 seconds worth of audio is enough to
	 * allow full read-ahead. Anything less than 3 seconds and
	 * we really can't afford the extra cycles read-ahead will
	 * take.
	 */
	reserve = audio_dev_buffer_time(arg);
	played = audio_dev_played_time(arg);

	if (reserve >= 50 && read_ahead_ok == 0)
		read_ahead_ok = 1;
	else
	if (reserve < 30 && read_ahead_ok)
		read_ahead_ok = 0;

	play_event(audio_ctx);

	return (mp3ended ? MAD_FLOW_STOP : MAD_FLOW_CONTINUE);
}

/* ARGSUSED */
static void
play_next(int arg)
{
	static struct timeval last_tv;
	struct timeval tv;

	if (gettimeofday(&tv, NULL) < 0) {
		perror("gettimeofday");
		exit(1);
	}

	if ((tv.tv_sec - last_tv.tv_sec) < 2)
		player_quit = 1;

	player_next = 1;
	player_stop = 1;

	last_tv = tv;
}

static int
process_key_controls(void *arg)
{
	char ch;
	int rv;

	while ((rv = read(fileno(stdin), &ch, 1)) < 0 && errno == EINTR)
		;

	if (rv == 0)
		return (-1);
	else
	if (rv < 0) {
		perror("standard input");
		return (-1);
	}

	switch (ch) {
	case 'n':
	case 'N':
		if ((playing_track + 1) < total_tracks) {
			player_next = 1;
			player_stop = 1;
		}
		break;

	case 'b':
	case 'B':
		if (playing_track > 0) {
			player_prev = 1;
			player_stop = 1;
		}
		break;

	case 'q':
	case 'Q':
		player_quit = 1;
		player_stop = 1;
		break;

	case 'r':
	case 'R':
		player_rewind = 1;
		player_stop = 1;
		break;

	case 'p':
	case 'P':
		if (audio_dev_send_command(arg, ADC_PAUSE, NULL, 1) < 0) {
			perror("pause command");
			return (-1);
		}

		player_paused = 1 - player_paused;
		break;
	}

	return (0);
}

static void
usage(void)
{
	fprintf(stderr, "usage: mad123 [-cq] [-b <read-ahead-bytes>] "
	    "[-f <audio-dev>] [-p <playlist-file>] <mp3path> ...\n");

	exit(1);
}

int
main(int argc, char **argv)
{
	struct mad_decoder mdec;
	struct termios tio;
	char *id3, *p;
	const char *adev;
	char * const *playlist = NULL;
	int ch, pri = 255;
	int input_buffer = mp3buf_len;
	int playlist_len = 0;

	/*
	 * If started by root or if installed set-uid root, try to
	 * raise priority.
	 * The audio device slave is set to a higher priority than
	 * the main mp3 streaming loop (parent).
	 */
	if (geteuid() == 0) {
		pri = getpriority(PRIO_PROCESS, 0);
		if (pri == -1 && errno) {
			perror("getpriority");
			fprintf(stderr, "%s: warning: can't raise priority\n",
			    argv[0]);
			fflush(stderr);
			pri = 255;
		} else {
			/*
			 * Raise our priority
			 */
			pri -= MAD123_AUDIO_SLAVE_PRIORITY_INCR;
			if (pri < -20)
				pri = -20;

			if (setpriority(PRIO_PROCESS, 0, pri) < 0) {
				perror("getpriority");
				fprintf(stderr,
				    "%s: warning: can't raise priority\n",
				    argv[0]);
				fflush(stderr);
				pri = 255;
			}
		}

		/*
		 * If we're running at an elevated priviledge,
		 * drop back down to the user's real privilege
		 * level. Bomb out immediately on failure.
		 */
		if (getuid() != geteuid() && setuid(getuid()) < 0) {
			perror("setuid");
			fprintf(stderr, "%s: Can't drop privs!! Aborting!\n",
			    argv[0]);
			exit (1);
		}
	}

	adev = "/dev/audio";

	while ((ch = getopt(argc, argv, "cqb:f:p:")) != -1) {
		switch (ch) {
		case 'c':
			player_controls = 1;
			break;

		case 'q':
			player_quiet = 1;
			break;

		case 'f':
			adev = optarg;
			break;

		case 'b':
			input_buffer = strtoul(optarg, &p, 0);
			if (*p != '\0') {
				fprintf(stderr, "%s: bad buffer size\n",
				    argv[0]);
				exit(1);
			}

			if (input_buffer < mp3buf_len) {
				fprintf(stderr,
				    "%s: using %d bytes for input buffer\n",
				    argv[0], mp3buf_len);
				input_buffer = mp3buf_len;
			}
			break;

		case 'p':
			if (playlist != NULL) {
				fprintf(stderr, "%s: multiple playlists"
				    "are not supported\n", argv[0]);
				exit(1);
			}
			playlist = read_playlist(optarg, &playlist_len);
			break;

		case '?':
		default:
			usage();
		}
	}

	if (playlist == NULL) {
		argc -= optind;
		argv += optind;

		if (argc == 0)
			usage();

		playlist = argv;
		playlist_len = argc;
	}

	/*
	 * Fork off the audio device slave at the (possibly) elevated priority
	 */
	if ((audio_ctx = audio_dev_init(adev)) == NULL) {
		perror("audio_dev_init");
		return (1);
	}

	audio_dev_register_read_callback(audio_ctx, play_event, audio_ctx);

	/*
	 * If we managed to elevate our priority, lower it slightly here
	 * to below that of the audio device slave.
	 */
	if (pri != 255)
		setpriority(PRIO_PROCESS, 0,
		    pri + MAD123_STREAMER_PRIORITY_INCR);

	if (player_controls) {
		parent_pid = getpid();

		if (tcgetattr(fileno(stdin), &tio) < 0) {
			perror("tcgetattr");
			exit(1);
		}

		stdin_tio_orig = tio;

		tio.c_lflag &= ~(ECHO | ICANON);
		tio.c_lflag |= ECHONL;

		if (tcsetattr(fileno(stdin), TCSANOW, &tio) < 0) {
			perror("tcsetattr");
			exit(1);
		}

		if (atexit(stdin_cleanup) < 0) {
			perror("atexit");
			tcsetattr(fileno(stdin), TCSANOW, &tio);
			exit(1);
		}
	}

	signal(SIGHUP, play_next);
	signal(SIGQUIT, play_next);
	signal(SIGINT, play_next);

	if ((mp3buf = malloc(mp3buf_len)) == NULL) {
		perror("malloc");
		audio_dev_close(audio_ctx);
		return (1);
	}

	mad_decoder_init(&mdec, audio_ctx, play_input,
	    NULL, NULL, play_output, NULL, NULL);

	playing_track = -1;
	total_tracks = playlist_len;

	while (player_quit == 0 && ++playing_track < total_tracks) {

		if (player_paused &&
		    audio_dev_send_command(audio_ctx, ADC_PAUSE, NULL, 1) < 0) {
			perror("pause command");
			return (-1);
		}

		player_paused = 0;

		mp3fp = mp3_stream_open(playlist[playing_track], input_buffer,
		    !player_controls);
		if (mp3fp == NULL) {
			perror(playlist[playing_track]);
			continue;
		}

		if (player_quiet == 0) {
			printf("\n\nSource: %s\n", playlist[playing_track]);

			if (mp3_stream_id3(mp3fp, MP3_STREAM_TAG_ARTIST,
			    &id3) == 0 && id3 != NULL)
				printf("Artist: %s\n", id3);

			if (mp3_stream_id3(mp3fp, MP3_STREAM_TAG_TITLE,
			    &id3) == 0 && id3 != NULL)
				printf("Title:  %s\n", id3);

			putchar('\n');
		}

		mp3ended = 0;

		mad_decoder_run(&mdec, MAD_DECODER_MODE_SYNC);
		mad_decoder_finish(&mdec);

		if (player_quit || player_next || player_prev || player_rewind){
			if (audio_dev_purge_wait(audio_ctx) < 0) {
				perror("audio_dev_purge_wait");
				break;
			}

			if (player_prev)
				playing_track -= 2;
			else
			if (player_rewind)
				playing_track -= 1;

			player_next = player_prev = player_rewind = 0;
		} else
		if (audio_dev_flush_wait(audio_ctx) < 0) {
			perror("audio_dev_flush_wait");
			break;
		}

		player_stop = 0;
		mp3len = 0;
		read_ahead_ok = 0;

		mp3_stream_close(mp3fp);
	}

	audio_dev_close(audio_ctx);

	if (player_quiet == 0)
		putchar('\n');

	return (0);
}

static char * const *
read_playlist(const char *fn, int *lenp)
{
	struct stat st;
	char *fbuf, *p;
	char **rv;
	FILE *fp;
	int i, len;

	if (stat(fn, &st) < 0) {
		perror(fn);
		exit(1);
	}

	if ((fp = fopen(fn, "r")) == NULL) {
		perror(fn);
		exit(1);
	}

	if ((fbuf = malloc(st.st_size + 1)) == NULL) {
		perror("malloc");
		exit(1);
	}

	if (fread(fbuf, sizeof(char), st.st_size, fp) != st.st_size) {
		perror("fread playlist");
		exit(1);
	}

	for (p = fbuf, i = 0; (p - fbuf) < st.st_size; p++) {
		if (*p == '\n') {
			i++;
			*p = '\0';
		}
	}

	*p = '\0';

	if ((rv = malloc(i * sizeof(*rv))) == NULL) {
		perror("malloc");
		exit(1);
	}

	for (len = 0, p = fbuf; (p - fbuf) < st.st_size; ) {
		if (*p == '\0') {
			p++;
			continue;
		}

		rv[len++] = p;
		while (*p != '\0' && (p - fbuf) < st.st_size)
			p++;
	}

	*lenp = len;
	return (rv);
}
