/*  Lziprecover - Data recovery tool for lzip files
    Copyright (C) 2009, 2010, 2011, 2012, 2013 Antonio Diaz Diaz.

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _FILE_OFFSET_BITS 64

#include <cerrno>
#include <climits>
#include <cstdio>
#include <cstring>
#include <string>
#include <vector>
#include <stdint.h>
#include <unistd.h>
#include <sys/stat.h>

#include "lzip.h"
#include "file_index.h"


int seek_read( const int fd, uint8_t * const buf, const int size,
               const long long pos )
  {
  if( lseek( fd, pos, SEEK_SET ) == pos )
    return readblock( fd, buf, size );
  return 0;
  }


int seek_write( const int fd, const uint8_t * const buf, const int size,
                const long long pos )
  {
  if( lseek( fd, pos, SEEK_SET ) == pos )
    return writeblock( fd, buf, size );
  return 0;
  }


int repair_file( const std::string & input_filename,
                 const std::string & output_filename, const int verbosity,
                 const bool force )
  {
  struct stat in_stats;
  const int infd = open_instream( input_filename, &in_stats, true, true );
  if( infd < 0 ) return 1;

  const File_index file_index( infd );
  if( file_index.retval() != 0 )
    { show_error( file_index.error().c_str() ); return file_index.retval(); }

  int outfd = -1;
  for( int i = 0; i < file_index.members(); ++i )
    {
    const long long mpos = file_index.mblock( i ).pos();
    const long long msize = file_index.mblock( i ).size();
    if( !safe_seek( infd, mpos ) )
      cleanup_and_fail( output_filename, outfd, 1 );
    long long failure_pos = 0;
    if( try_decompress_member( infd, msize, &failure_pos ) ) continue;
    if( failure_pos >= msize - 8 ) failure_pos = msize - 8 - 1;
    if( failure_pos < File_header::size )
      { show_error( "Can't repair error in input file." );
        cleanup_and_fail( output_filename, outfd, 2 ); }

    if( outfd < 0 )			// first damaged member found
      {
      if( !safe_seek( infd, 0 ) ) return 1;
      outfd = open_outstream_rw( output_filename, force );
      if( outfd < 0 ) { close( infd ); return 1; }
      if( !copy_file( infd, outfd ) )		// copy whole file
        cleanup_and_fail( output_filename, outfd, 1 );
      }

    if( verbosity >= 1 )
      {
      std::printf( "Repairing member %d\n", i + 1 );
      std::fflush( stdout );
      }
    const long long min_pos =
      std::max( (long long)File_header::size, failure_pos - 1000 );
    bool done = false;
    for( long long pos = failure_pos; pos >= min_pos && !done ; --pos )
      {
      if( verbosity >= 1 )
        {
        std::printf( "Trying position %llu \r", mpos + pos );
        std::fflush( stdout );
        }
      uint8_t byte;
      if( seek_read( outfd, &byte, 1, mpos + pos ) != 1 )
        { show_error( "Error reading output file", errno );
          cleanup_and_fail( output_filename, outfd, 1 ); }
      for( int i = 0; i < 256; ++i )
        {
        ++byte;
        if( seek_write( outfd, &byte, 1, mpos + pos ) != 1 ||
            lseek( outfd, mpos, SEEK_SET ) < 0 )
          { show_error( "Error writing output file", errno );
            cleanup_and_fail( output_filename, outfd, 1 ); }
        if( i == 255 ) break;
        if( try_decompress_member( outfd, msize ) )
          { done = true; break; }
        }
      }
    if( verbosity >= 1 ) std::printf( "\n" );
    if( !done )
      {
      show_error( "Error is larger than 1 byte. Can't repair input file." );
      cleanup_and_fail( output_filename, outfd, 2 );
      }
    }

  if( outfd < 0 )
    {
    if( verbosity >= 1 )
      std::printf( "Input file has no errors. Recovery is not needed.\n" );
    return 0;
    }
  if( close( outfd ) != 0 )
    {
    show_error( "Error closing output file", errno );
    cleanup_and_fail( output_filename, -1, 1 );
    }
  if( verbosity >= 1 )
    std::printf( "Copy of input file repaired successfully.\n" );
  return 0;
  }
