/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <gdk/gdkkeysyms.h>
#include "sg.h"
#include "sg_worksheet.h"
#include "sg_worksheet_window.h"
#include "sg_worksheet_menu.h"
#include "sg_worksheet_toolbar.h"
#include "sg_matrix_menu.h"
#include "sg_matrix_toolbar.h"
#include "sg_formula_dialog.h"
#include "pixmaps/sheet_icon2.xpm"

static void     sg_worksheet_window_class_init               (SGworksheetWindowClass *klass);
static void     sg_worksheet_window_init                     (SGworksheetWindow *worksheet);
static void     sg_worksheet_window_init_gui             (SGworksheetWindow *worksheet);

static void 	sg_worksheet_window_realize			(GtkWidget *widget); 
static void 	sg_worksheet_window_map			(GtkWidget *widget); 
static void 	sg_worksheet_window_destroy		(GtkObject *object); 
static gboolean activate_cell				(GtkSheet *sheet, 
							 gint row, gint col, 
							 gpointer data);

static GtkWindowClass *parent_class = NULL;

GtkType
sg_worksheet_window_get_type (void)
{
  static GtkType sg_worksheet_window_type = 0;

  if (!sg_worksheet_window_type)
    {
      GtkTypeInfo sg_worksheet_window_info =
      {
        "SGworksheetWindow",
        sizeof (SGworksheetWindow),
        sizeof (SGworksheetWindowClass),
        (GtkClassInitFunc) sg_worksheet_window_class_init,
        (GtkObjectInitFunc) sg_worksheet_window_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_worksheet_window_type = gtk_type_unique (gtk_window_get_type(), &sg_worksheet_window_info);
    }

  return sg_worksheet_window_type;
}

static void
sg_worksheet_window_class_init (SGworksheetWindowClass *klass)
{
  GtkWidgetClass *widget_class;
  GtkObjectClass *object_class;

  widget_class = (GtkWidgetClass*) klass;
  object_class = (GtkObjectClass*) klass;

  parent_class = (GtkWindowClass *)gtk_type_class (gtk_window_get_type ());

  object_class->destroy = sg_worksheet_window_destroy;

  widget_class->realize = sg_worksheet_window_realize;
  widget_class->map = sg_worksheet_window_map;
}

SGworksheetWindow *
sg_worksheet_window_new(SGworksheet *worksheet)
{
  SGworksheetWindow *window;

  window = SG_WORKSHEET_WINDOW(gtk_widget_new(sg_worksheet_window_get_type(), NULL));

  window->worksheet = worksheet;
  sg_worksheet_window_init_gui(window);
  g_object_set_data(G_OBJECT(worksheet), "window", window);

  return window;
}  

static void
sg_worksheet_window_init(SGworksheetWindow *window)
{
  window->x = window->y = 0;
  window->width = 500;
  window->height = 350;
}

static void
sg_worksheet_window_init_gui(SGworksheetWindow *window)
{
  SGworksheet *worksheet = window->worksheet;
  GtkWidget *vbox;
  GtkWidget *frame;
  GtkWidget *toolbar;
  GtkWidget *hb;
  gchar window_title[500];

  if(GTK_IS_SG_MATRIX(worksheet))
    window->menu = sg_matrix_menu_new(SG_MATRIX(worksheet));
  else
    window->menu = sg_worksheet_menu_new(worksheet);

  g_snprintf(window_title, 500, "SciGraphica: %s", worksheet->name);
  gtk_window_set_title(GTK_WINDOW(window), window_title);
  gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
 
  window->vbox = vbox = gtk_vbox_new(FALSE, 5);

  gtk_container_add(GTK_CONTAINER(window),vbox);

  if(GTK_IS_SG_MATRIX(worksheet))
    toolbar = sg_matrix_toolbar_new(window);
  else
    toolbar = sg_worksheet_toolbar_new(window);
  gtk_object_set_data(GTK_OBJECT(window),"menubar",toolbar);

  gtk_box_pack_start(GTK_BOX(vbox), hb = gtk_handle_box_new(), FALSE, FALSE, 0);
  gtk_container_add(GTK_CONTAINER(hb), toolbar);
  gtk_widget_show_all(hb);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
  window->label = gtk_label_new(" ");
  gtk_misc_set_alignment(GTK_MISC(window->label), 0., .5);
  gtk_container_add(GTK_CONTAINER(frame), window->label);
 
  window->sw = gtk_scrolled_window_new(NULL, NULL);
 
  gtk_container_add(GTK_CONTAINER(window->sw), GTK_WIDGET(worksheet));
  gtk_box_pack_start(GTK_BOX(vbox), window->sw, TRUE, TRUE, 0);
  gtk_box_pack_end(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
  
  gtk_widget_ensure_style(GTK_WIDGET(worksheet));

  gtk_widget_show(window->sw);
  gtk_widget_show(GTK_WIDGET(worksheet));
  gtk_widget_show(window->label);
  gtk_widget_show(frame);
  gtk_widget_show(vbox);

  if(window->x > 0 && window->y > 0){
           gtk_widget_set_uposition(GTK_WIDGET(window), 
                                    window->x, window->y);
           gtk_widget_set_usize(GTK_WIDGET(window),
                                window->width,
                                window->height);
  }else{
           gtk_widget_set_usize(GTK_WIDGET(window),
                                500,
                                350);
  }

  gtk_signal_connect(GTK_OBJECT(worksheet), "activate",
                     GTK_SIGNAL_FUNC(activate_cell), window);
  gtk_signal_connect_object(GTK_OBJECT(worksheet), "button_press_event",
                     GTK_SIGNAL_FUNC(sg_worksheet_menu_show),
                     GTK_OBJECT(window->menu));
}

static gboolean
activate_cell(GtkSheet *sheet, gint row, gint col, gpointer data)
{
  SGworksheetWindow *window = SG_WORKSHEET_WINDOW(data);
  gchar label[500];

  sprintf(label, "  %s:%d", sheet->column[col].name, row);
  gtk_label_set(GTK_LABEL(window->label), label);

  return TRUE;
}

static void
sg_worksheet_window_map(GtkWidget *widget)
{
 SGworksheetWindow *window = SG_WORKSHEET_WINDOW(widget);
 SGworksheet *worksheet = window->worksheet;
 gchar window_title[500];

 GTK_WIDGET_CLASS(parent_class)->map(widget);

 g_snprintf(window_title, 500, "SciGraphica: %s", worksheet->name);
 gtk_window_set_title(GTK_WINDOW(window), window_title);
 gdk_window_set_icon_name(GTK_WIDGET(window)->window, worksheet->name);
}


static void
sg_worksheet_window_realize(GtkWidget *widget)
{
  GdkPixmap *sheet_icon_pixmap;
  GdkBitmap *sheet_icon_mask;
  SGworksheetWindow *window;
  
  window = SG_WORKSHEET_WINDOW(widget);

  GTK_WIDGET_CLASS(parent_class)->realize(widget);

  sheet_icon_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                    gdk_colormap_get_system(),
                                    &sheet_icon_mask, NULL, sheet_icon2_xpm);


  gdk_window_set_icon(GTK_WIDGET(window)->window, NULL,
                      sheet_icon_pixmap, sheet_icon_mask);
}

static void
sg_worksheet_window_destroy(GtkObject *object)
{
  SGworksheetWindow *window = SG_WORKSHEET_WINDOW(object);
  window->menu = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
     (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

