/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <gdk/gdkkeysyms.h>
#include "sg_locale.h"
#include "sg_worksheet.h"
#include "sg_formula_dialog.h"
#include "sg_python.h"
#include "sg_clipboard.h"
#include "sg_python_worksheet.h"
#include "sg_python_expr.h"
#define PY_ARRAY_UNIQUE_SYMBOL PyArrayXXX
#define NO_IMPORT_ARRAY
#include <arrayobject.h>
#define DEFAULT_PRECISION 3
#define DEFAULT_ROWS 20 
#define DEFAULT_COLUMNS 5 

static void     sg_worksheet_class_init               (SGworksheetClass *klass);
static void     sg_worksheet_init                     (SGworksheet *worksheet);
static void	sg_worksheet_destroy			(GtkObject *object); 
static gboolean sg_worksheet_real_cell_update_format	(SGworksheet *sheet, 
                                     			 gint row, gint col);
static gboolean sg_worksheet_update			(SGworksheet *sheet, 
                    					 gint row0, gint rowi,
                    					 gint col0, gint coli);

static gboolean activate_cell				(GtkSheet *sheet, 
							 gint row, gint col,
							 gpointer data);
static gboolean deactivate_cell                         (GtkSheet *sheet,
                                                         gint row, gint col,
                                                         gpointer data);
static void set_cell                         (GtkSheet *sheet,
                                                         gint row, gint col,
                                                         gpointer data);

static gboolean click_on_column				(GtkWidget *widget, 
							 GdkEventButton *event,
							 gpointer data);
static void clear_cell                         		(GtkSheet *sheet,
                                                         gint row, gint col,
                                                         gpointer data);


static GtkSheetClass *parent_class = NULL;
extern gboolean sg_report_python_error;

GtkType
sg_worksheet_get_type (void)
{
  static GtkType sg_worksheet_type = 0;

  if (!sg_worksheet_type)
    {
      GtkTypeInfo sg_worksheet_info =
      {
        "SGworksheet",
        sizeof (SGworksheet),
        sizeof (SGworksheetClass),
        (GtkClassInitFunc) sg_worksheet_class_init,
        (GtkObjectInitFunc) sg_worksheet_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_worksheet_type = gtk_type_unique (gtk_sheet_get_type(), &sg_worksheet_info);
    }

  return sg_worksheet_type;
}

static void
sg_worksheet_class_init (SGworksheetClass *klass)
{
  GtkWidgetClass *widget_class;
  GtkObjectClass *object_class;
  SGworksheetClass *worksheet_class;

  widget_class = (GtkWidgetClass*) klass;
  object_class = (GtkObjectClass*) klass;
  worksheet_class = (SGworksheetClass*) klass;

  parent_class = (GtkSheetClass *)gtk_type_class (gtk_sheet_get_type ());

  object_class->destroy = sg_worksheet_destroy;

  worksheet_class->update_exp = sg_worksheet_update;
  worksheet_class->update_cell_format = sg_worksheet_real_cell_update_format;
}

static void
sg_worksheet_init (SGworksheet *worksheet)
{
  GtkSheet *sheet;
  gint c;
  gint nrows, ncols;

  gtk_sheet_construct(GTK_SHEET(worksheet), DEFAULT_ROWS, DEFAULT_COLUMNS, worksheet->name ? worksheet->name : _("Data") );

  worksheet->cell_save=NULL;
  worksheet->is_frozen = FALSE;
  worksheet->begin = -1;
  worksheet->end = -1;

  nrows = 20;
  ncols = 5;

  sg_worksheet_add_rows(worksheet, nrows - DEFAULT_ROWS);
  sg_worksheet_add_columns(worksheet, ncols - DEFAULT_COLUMNS);

  worksheet->last_column = ncols-1;

  sheet = GTK_SHEET(worksheet);
  gtk_sheet_set_row_titles_width(sheet, 60);
  gtk_sheet_set_justify_entry(sheet, FALSE);
  gtk_sheet_rows_set_resizable(sheet, FALSE);

  worksheet->column = g_new(SGcolumn, ncols);

  for(c = 0; c < ncols; c++){
     gchar label[4];
     gint max = 'Z'-'A'+1;
     gint n;

     n = 0;
     if(c >= max*max){
        label[0] = c/(max*max)+'A'-1;
        label[1] = '\0';
        n = 1;
        c -= (c/(max*max))*max*max;
     }
     if(c >= max){
        label[n] = c/max+'A'-1;
        label[n+1] = '\0';
        n++ ;
        c -= (c/max)*max;
     }
     if(c < max){
        label[n] = c+'A';
        label[n+1] = '\0';
     }

    gtk_sheet_column_button_add_label(sheet, c, label);
    gtk_sheet_set_column_title(sheet, c, label);
    sg_worksheet_column_set_format(worksheet, c, 
                                   SG_TYPE_NUMBER,
                                   SG_FORMAT_DECIMAL,
                                   SG_INTERNAL_DOUBLE,
                                   DEFAULT_PRECISION);
    worksheet->column[c].exp = NULL;
  }

  gtk_signal_connect(GTK_OBJECT(worksheet), "activate",
                     GTK_SIGNAL_FUNC(activate_cell), worksheet);

/*
  gtk_signal_connect(GTK_OBJECT(worksheet), "deactivate",
                     GTK_SIGNAL_FUNC(deactivate_cell), worksheet);
*/

  gtk_signal_connect(GTK_OBJECT(worksheet), "set_cell",
                     GTK_SIGNAL_FUNC(set_cell), worksheet);

  gtk_signal_connect(GTK_OBJECT(worksheet), "clear_cell",
                     GTK_SIGNAL_FUNC(clear_cell), worksheet);

  gtk_signal_connect(GTK_OBJECT(worksheet), "button_press_event",
                     GTK_SIGNAL_FUNC(click_on_column), worksheet);

  worksheet->clipboard = G_OBJECT(sg_clipboard_new());
  g_object_ref(worksheet->clipboard);
}   

SGworksheet *
sg_worksheet_new(const gchar *name, gint nrows, gint ncols)
{
  SGworksheet *worksheet;
  gint old_rows, old_cols;

  worksheet = SG_WORKSHEET(gtk_widget_new(sg_worksheet_get_type(), NULL));
  old_rows = gtk_sheet_get_rows_count(GTK_SHEET(worksheet));
  old_cols = gtk_sheet_get_columns_count(GTK_SHEET(worksheet));
  sg_worksheet_add_rows(worksheet, nrows-old_rows);
  sg_worksheet_add_columns(worksheet, ncols-old_cols);

  if(name){
    sg_worksheet_rename(worksheet, name);
    gtk_sheet_set_title(GTK_SHEET(worksheet), name);
  }

  return worksheet;
}  

gint
sg_worksheet_get_column(SGworksheet *worksheet, const gchar *col_name)
{
  GtkSheet *sheet;
  gint i;

  sheet = GTK_SHEET(worksheet);

  for(i = 0; i <= sheet->maxcol; i++)
    if(strcmp(sheet->column[i].name, col_name) == 0) return i;

  return -1;
}

void
sg_worksheet_column_set_format(SGworksheet *worksheet,
                               gint column, 
                               SGcolumntype type,
                               SGcolumnformat format,
                               SGcolumninternal internal,
                               gint precision)
{
  GtkSheet *sheet;
  gint row;

  sheet = GTK_SHEET(worksheet);

  worksheet->column[column].type = type; 
  worksheet->column[column].format = format; 
  worksheet->column[column].internal = internal; 
  worksheet->column[column].precision = precision;

  switch(worksheet->column[column].type){
    case SG_TYPE_NUMBER:
       gtk_sheet_column_set_justification(sheet, column, GTK_JUSTIFY_RIGHT);
       break;
    case SG_TYPE_DATE:
    case SG_TYPE_TIME:
       gtk_sheet_column_set_justification(sheet, column, GTK_JUSTIFY_CENTER);
       break;
    case SG_TYPE_TEXT:
    default:
       gtk_sheet_column_set_justification(sheet, column, GTK_JUSTIFY_LEFT);
  }

  gtk_sheet_freeze(sheet);
  for(row = 0; row <= sheet->maxallocrow; row++){
    sg_worksheet_cell_update_format(worksheet, row, column);
  }

  gtk_sheet_thaw(sheet);
}

void 
sg_worksheet_hidden_cell_clear(SGworksheet *sheet, gint row, gint col)
{
  SGhiddencell *hidden;
  hidden = (SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(sheet), row, col);
  if(hidden)
  { 
    if (hidden->formula){
      g_free(hidden->formula);
      hidden->formula = NULL;
    }
    switch(hidden->type){
      case SG_TYPE_DATE:
      case SG_TYPE_TIME:
      case SG_TYPE_TEXT:
        if (hidden->value.val_char){
          g_free(hidden->value.val_char);
          hidden->value.val_char = NULL;
        }
        break;
      default: 
        break;
    }
    g_free(hidden);
    gtk_sheet_link_cell(GTK_SHEET(sheet), row, col, NULL);
  }
}

void 
sg_worksheet_cell_clear(SGworksheet *sheet, gint row, gint col)
{ 
  sg_worksheet_hidden_cell_clear(sheet, row, col);
  gtk_sheet_cell_clear(GTK_SHEET(sheet), row, col);
}

static gboolean
click_on_column(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
  SGworksheet *worksheet;
  GtkSheet *sheet;
  GdkModifierType mods;
  gint x, y;
  gint r, c;

  worksheet = SG_WORKSHEET(data);
  sheet = GTK_SHEET(worksheet);

  if(event->window != sheet->column_title_window) return FALSE;
  gdk_window_get_pointer(widget->window, &x, &y, &mods);
  if(!(mods & GDK_BUTTON1_MASK)) return FALSE;
  if(event->type != GDK_2BUTTON_PRESS) return FALSE;

  sheet->flags = 0;
  gdk_pointer_ungrab(event->time);
  gtk_grab_remove(widget);

  gtk_sheet_get_pixel_info(sheet, x, y, &r, &c);

  if(c >= 0)
     sg_formula_dialog(worksheet, c);

  return FALSE;
}

static void 
clear_cell(GtkSheet *sheet, gint row, gint col, gpointer data)
{
  SGhiddencell *hidden;

  hidden = (SGhiddencell *)gtk_sheet_get_link(sheet, row, col);
  if(hidden)
    { 
      if (hidden->formula){
        g_free(hidden->formula);
        hidden->formula = NULL;
      }
      switch(hidden->type){
        case SG_TYPE_DATE:
        case SG_TYPE_TIME:
        case SG_TYPE_TEXT:
          if (hidden->value.val_char){
            g_free(hidden->value.val_char);
            hidden->value.val_char = NULL;
          }
          break;
        default: break;
      }
      g_free(hidden);
      gtk_sheet_link_cell(sheet, row, col, NULL);
    }
}

static gboolean
activate_cell(GtkSheet *sheet, gint row, gint col, gpointer data)
{
  SGworksheet *worksheet;
  GtkEntry *entry;
  gchar *text=NULL,fpnum[40];
  SGhiddencell *link = NULL;
  PyObject *floato;

  entry = GTK_ENTRY(gtk_sheet_get_entry(sheet));
  worksheet = (SGworksheet *)data;

  if(worksheet->cell_save){
    g_free(worksheet->cell_save);
  }
  worksheet->cell_save = g_strdup(gtk_entry_get_text(entry));
  if(worksheet->column[col].type == SG_TYPE_TEXT) return TRUE;

  link=(SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(worksheet),row,col);

  fpnum[0]='\0';

  if (link)
   {  
      if (link->formula){
        text = g_strdup(link->formula);
      }
      else if(link->type == SG_TYPE_NUMBER){
        if(link->internal == SG_INTERNAL_INTEGER){
          switch(link->format){
            case SG_FORMAT_DECIMAL:
              g_snprintf(fpnum,40,"%ld",(glong)link->value.val_long);
              text=g_strdup(fpnum);
              break;
            case SG_FORMAT_SCIENTIFIC:
              floato=PyObject_Repr(PyFloat_FromDouble((double)link->value.val_long));
              text=g_strdup(PyString_AsString(floato));
              break;
          }
        } else {
          switch(link->format){
            case SG_FORMAT_DECIMAL:
              floato=PyObject_Repr(PyFloat_FromDouble(link->value.val_double));
              text=g_strdup(PyString_AsString(floato));
              break;
            case SG_FORMAT_SCIENTIFIC:
              g_snprintf(fpnum,40,"%2.20e",link->value.val_double);
              text=g_strdup(fpnum);
              break;
          }
        }
      } else {
          text=g_strdup(link->value.val_char);
      }
      if(!worksheet->is_frozen){
        gtk_sheet_set_cell_text(GTK_SHEET(worksheet),row,col,text);
/*
        gtk_item_entry_set_text(GTK_ITEM_ENTRY(entry),text,GTK_JUSTIFY_LEFT);
*/
      }
   }
  else
   { 
      text = gtk_sheet_cell_get_text(GTK_SHEET(worksheet),row,col);
      if(text && strlen(text) > 0) text = g_strdup(text);
      if(!worksheet->is_frozen){
        if(text && strlen(text) > 0) 
          gtk_item_entry_set_text(GTK_ITEM_ENTRY(entry),text,GTK_JUSTIFY_LEFT);
        else
          gtk_item_entry_set_text(GTK_ITEM_ENTRY(entry),"",GTK_JUSTIFY_LEFT);
      }
   }

  if(text && strlen(text)){
/*
    gtk_entry_select_region(entry, 0, strlen(text));
    gtk_entry_set_position(entry, 0);
*/
    g_free(text);
  }

  return TRUE;
}

static gboolean
deactivate_cell(GtkSheet *sheet, gint row, gint col, gpointer data)
{
  SGworksheet *worksheet;
  gchar *text;

  worksheet = SG_WORKSHEET(sheet);
  text = gtk_sheet_cell_get_text(sheet, row, col);

  if(text && strlen(text) > 0){
    gchar *aux_text = g_strdup(text);
    sg_worksheet_cell_set(worksheet, row, col, aux_text, TRUE, TRUE);
    g_free(aux_text);
  }

  return TRUE;
}


static void 
set_cell(GtkSheet *sheet, gint row, gint col, gpointer data)
{
  SGworksheet *worksheet;
  gchar *text;

  worksheet = SG_WORKSHEET(sheet);
  text = gtk_sheet_cell_get_text(sheet, row, col);

  if(text && strlen(text) > 0)
    sg_worksheet_cell_set(worksheet, row, col, text, TRUE, TRUE);
}

void 
sg_worksheet_cell_set_text(SGworksheet *worksheet, gint row, gint col, const gchar *text)
{ 
  gint arow,acol;

  gtk_sheet_set_cell_text(GTK_SHEET(worksheet), row, col, text);

  gtk_sheet_get_active_cell(GTK_SHEET(worksheet), &arow, &acol);

  if (arow==row && acol==col)
   {  
      gtk_entry_set_text(GTK_ENTRY(GTK_SHEET(worksheet)->sheet_entry),text);
   }

}

void
sg_worksheet_cell_set(SGworksheet *worksheet, gint row, gint col,
                      const gchar *text, gboolean formula, gboolean eval)
{
  gchar *save;
  SGhiddencell *link = NULL;
  GtkSheet *sheet;
  gboolean myformula=formula, myeval=eval;

  sheet = GTK_SHEET(worksheet);
  link = (SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(worksheet), row, col);

  if (text && strlen(text))
  { 
     if (!link) {
            myformula=formula;
            myeval=eval;
            link = (SGhiddencell *)g_new0(SGhiddencell, 1);
            link->formula = NULL;
            link->updated = FALSE;
            link->format =  worksheet->column[col].format;
            link->type =  worksheet->column[col].type;
            link->internal =  worksheet->column[col].internal;
            link->precision = worksheet->column[col].precision;
            gtk_sheet_link_cell(GTK_SHEET(worksheet), row, col, link);
     }

     save = g_strdup(text);
     if (!myformula)
     {  
        sg_worksheet_cell_set_text(worksheet, row, col, save);
     }
     else
     {  
       link->updated=FALSE;
       switch(link->type){
	case SG_TYPE_DATE:
	case SG_TYPE_TIME:
	case SG_TYPE_TEXT:
	  if (link->value.val_char){
  	    g_free(link->value.val_char);
	    link->value.val_char = NULL;
	  }
          myeval = FALSE;
	  break;
        default:
	  break;
       }
       if (myeval){
            worksheet->is_frozen = TRUE;  
            if (python_sheet(worksheet, row, col, save, GTK_ORIENTATION_VERTICAL)){
                link->updated = TRUE;
            }
            worksheet->is_frozen = FALSE;
       }
       else /* currently used for dates, text, and time */
       {          
            sg_worksheet_cell_set_text(worksheet, row, col, save?save:text);
       }

       if (link->formula)
          g_free(link->formula);
       link->formula = save;
     }
   }
  else /* We were passed an empty cell */
   { 
     if (link){
       sg_worksheet_cell_clear(worksheet, row, col);
     }
     else
     {  
        if (worksheet->cell_save && strlen(worksheet->cell_save)){
          sg_worksheet_cell_set_text(worksheet,
                                row, col, g_strdup(worksheet->cell_save));
        }else{
          gtk_sheet_cell_clear(GTK_SHEET(worksheet), row, col);
        }
     }
   }


  if(worksheet->cell_save) g_free(worksheet->cell_save);
  worksheet->cell_save = NULL;
}

gboolean
sg_worksheet_cell_update_format(SGworksheet *worksheet, 
                                gint row, gint col)
{
  return SG_WORKSHEET_CLASS(GTK_OBJECT_GET_CLASS(GTK_OBJECT(worksheet)))->update_cell_format(worksheet, row, col);
}

static gboolean
sg_worksheet_real_cell_update_format(SGworksheet *worksheet, 
                                     gint row, gint col)
{
  GtkSheet *sheet;
  gchar *string = NULL, fpnum[40], pspec[20];
  SGhiddencell *hidden;
  guint type, format, internal, precision;

  sheet = GTK_SHEET(worksheet);
  hidden = (SGhiddencell *)gtk_sheet_get_link(sheet, row, col);
  if (!hidden) return TRUE;
  fpnum[0]='\0';
  pspec[0]='\0';

  type = worksheet->column[col].type;
  format = worksheet->column[col].format;
  internal = worksheet->column[col].internal;
  precision = worksheet->column[col].precision;

/* Convert to new format */
  if(type == SG_TYPE_NUMBER && hidden->type == SG_TYPE_NUMBER) {
    if(hidden->internal != internal){ 
      if(internal != SG_INTERNAL_INTEGER){
        glong v = hidden->value.val_long;
        hidden->value.val_double = v;
      } else {
        gdouble v = hidden->value.val_double;
        hidden->value.val_long = v;
      }     
    }
  } else if(type == SG_TYPE_NUMBER && hidden->type != SG_TYPE_NUMBER) {
    gchar *s = hidden->value.val_char;
    hidden->value.val_char = NULL;
    if(internal == SG_INTERNAL_INTEGER)
        hidden->value.val_long = atoi(s);
    else 
        hidden->value.val_double = atof(s);
    g_free(s);
  } else if(type != SG_TYPE_NUMBER && hidden->type == SG_TYPE_NUMBER) {
    if(hidden->formula)
      hidden->value.val_char = g_strdup(hidden->formula);
    else
      hidden->value.val_char = g_strdup(gtk_sheet_cell_get_text(sheet,row,col));
  }

/* Set string */

  hidden->type = type;
  hidden->format = format;
  hidden->internal = internal;
  hidden->precision = precision;

  if(hidden->type == SG_TYPE_NUMBER){
    if(hidden->internal == SG_INTERNAL_INTEGER){
      switch(hidden->format){
        case SG_FORMAT_DECIMAL:
          g_snprintf(pspec,20,"%%ld");
          g_snprintf(fpnum,40,pspec,hidden->value.val_long);
          string=fpnum;
          break;
        case SG_FORMAT_SCIENTIFIC:
          g_snprintf(pspec,20,"%%1.%de",hidden->precision);
          g_snprintf(fpnum,40,pspec,(double)hidden->value.val_long);
          string=fpnum;
          break;
      }
    } else {
      switch(hidden->format){
        case SG_FORMAT_DECIMAL:
          g_snprintf(pspec,20,"%%1.%df",hidden->precision);
          g_snprintf(fpnum,40,pspec,hidden->value.val_double);
          string=fpnum;
          break;
        case SG_FORMAT_SCIENTIFIC:
          g_snprintf(pspec,20,"%%1.%de",hidden->precision);
          g_snprintf(fpnum,40,pspec,hidden->value.val_double);
          string=fpnum;
          break;
      }
    }
  } else {
      string=hidden->value.val_char;
  }
  sg_worksheet_cell_set_text(worksheet,row, col, string);

  
  return TRUE;
}


const gchar *
sg_worksheet_cell_get_text(SGworksheet *worksheet, gint row, gint col)
{ 
  gint arow,acol;
  gchar *text = NULL;

  gtk_sheet_get_active_cell(GTK_SHEET(worksheet), &arow, &acol);
  
  if (GTK_SHEET(worksheet)->state == GTK_STATE_NORMAL && 
      arow==row && acol==col && worksheet->cell_save){
   text=worksheet->cell_save;
  }else{
   text = gtk_sheet_cell_get_text(GTK_SHEET(worksheet), row, col);
  }
  return text;
}

const gchar *
sg_worksheet_cell_get_formula(SGworksheet *worksheet, gint row, gint col)
{
  gchar *text = NULL;
  SGhiddencell *link;

  link = (SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(worksheet), row, col);

  if(link) text = link->formula;

  return text;
}

gdouble
sg_worksheet_cell_get_double(SGworksheet *worksheet, gint row, gint col, gboolean *error)
{
  SGhiddencell *link;

  *error = FALSE;

  if(!worksheet) {
      *error=TRUE;
      return 0.0;
  }

  link = (SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(worksheet), row, col);
                  
  if(!link) {
      *error=TRUE;
      return 0.0;
  }
  switch(link->type){
      case SG_TYPE_NUMBER:
      switch (link->internal){
        case SG_INTERNAL_INTEGER: 
          return (gdouble)link->value.val_long;
        case SG_INTERNAL_DOUBLE:  
          return (gdouble)link->value.val_double;
        default: 
          return 0.0;
      }
      case SG_TYPE_TEXT:        
      case SG_TYPE_TIME:        
      case SG_TYPE_DATE:        
      default:
          return 0.0;
  }
}

gint
sg_worksheet_cell_get_int(SGworksheet *worksheet, gint row, gint col, gboolean *error)
{
  SGhiddencell *link;

  *error = FALSE;

  if(!worksheet) {
      *error=TRUE;
      return 0;
  }


  link = (SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(worksheet), row, col);
  if(!link) {
      *error=TRUE;
      return 0;
  }

  switch(link->type){
      case SG_TYPE_NUMBER:
      switch (link->internal){
          case SG_INTERNAL_INTEGER: return (gint)link->value.val_long;
          case SG_INTERNAL_DOUBLE:  return (gint)link->value.val_double;
          default: return 0;
      }
      case SG_TYPE_TEXT:        
      case SG_TYPE_TIME:        
      case SG_TYPE_DATE:        
      default:
          return 0;
  }
}

void
sg_worksheet_set_column_data(SGworksheet *worksheet, gint col, GtkPlotArray *array)
{
  gint i;
  gint size;
  gint nrows;

  size = gtk_plot_array_get_size(array);
  nrows = gtk_sheet_get_rows_count(GTK_SHEET(worksheet));

  gtk_sheet_freeze(GTK_SHEET(worksheet));

  if(size > nrows){
    gtk_sheet_add_row(GTK_SHEET(worksheet), size - nrows); 
  } 

  if(array->type == GTK_TYPE_STRING)
    sg_worksheet_column_set_format(worksheet, col, SG_TYPE_TEXT, 0, 0, 0);

  for(i = 0; i < size; i++){
    if(array->type == GTK_TYPE_STRING && array->data.data_string){
      if(array->data.data_string[i])
        sg_worksheet_cell_set_text(worksheet, i, col, array->data.data_string[i]);
    }
    if(array->type == GTK_TYPE_DOUBLE && array->data.data_double){
      gchar text[200];
      g_snprintf(text, 200, "%g", array->data.data_double[i]);
      sg_worksheet_cell_set(worksheet, i, col, text, TRUE, TRUE);
    }
  }

  gtk_sheet_thaw(GTK_SHEET(worksheet));
}


GtkPlotArray *
sg_worksheet_get_column_data(SGworksheet *worksheet, gint col, GtkType type)
{
  GtkSheet *sheet = GTK_SHEET(worksheet);
  GtkPlotArray *array = NULL;
  gint begin, end;
  gint n = 0, i = 0;

  begin = MAX(0,worksheet->begin);
  end = MAX(sheet->maxrow, worksheet->end);

  if(type == GTK_TYPE_DOUBLE){
    gdouble *data = g_new0(gdouble,end-begin+1);
    for(i = begin; i <= end; i++){
      gdouble val;
      gboolean error;
      val = sg_worksheet_cell_get_double(worksheet, i, col, &error);
      if(!error){
        data[n] = val;
        n++;
      } else {
        break;
      }
    }
    if(n > 0) {
      array = GTK_PLOT_ARRAY(gtk_plot_array_new(sheet->column[col].name,(gpointer)data,n,type,TRUE));
    } else {
      g_free(data);
    }
  }
  if(type == GTK_TYPE_STRING){
    gchar **data = g_new0(gchar *,end-begin+1);
    for(i = begin; i <= end; i++){
      const gchar *val;
      val = sg_worksheet_cell_get_text(worksheet, i, col);
      if(val){
        data[n] = g_strdup(val);
        n++;
      } else {
        break;
      }
    }
    if(n > 0){
      array = GTK_PLOT_ARRAY(gtk_plot_array_new("column",(gpointer)data,n,type,TRUE));
    } else {
      g_free(data);
    }
  }
  return array;
}

gint
sg_worksheet_rename(SGworksheet *worksheet, const gchar *name)
{
  if(worksheet->name){
     g_free(worksheet->name);
     worksheet->name = NULL;
  }

  worksheet->name = g_strdup(name);
  gtk_sheet_set_title(GTK_SHEET(worksheet), name);

  return TRUE;
}

void
sg_worksheet_set_begin(SGworksheet *worksheet, gint row)
{
  GtkSheet *sheet;
  GdkColor bg;
  GtkSheetRange range;
  gchar label[100];

  sheet = GTK_SHEET(worksheet);

  if(row > worksheet->end &&
     worksheet->end != -1) return;

  gtk_sheet_freeze(sheet);

  if(worksheet->begin != -1){
    range.col0 = 0;
    range.coli = sheet->maxcol;
    range.row0 = worksheet->begin;
    range.rowi = worksheet->begin;
    gtk_sheet_row_button_add_label(sheet, worksheet->begin, NULL);
    gtk_sheet_range_set_background(sheet, &range, &sheet->bg_color);
  }

  if(row > -1){
    gdk_color_parse("light blue", &bg);
    gdk_color_alloc(gdk_colormap_get_system(), &bg);
    worksheet->begin = row;
    range.col0 = 0;
    range.coli = sheet->maxcol;
    range.row0 = row;
    range.rowi = row;
  
    sprintf(label, _("(Begin)") );

    gtk_sheet_row_button_add_label(sheet, row, label);
    gtk_sheet_range_set_background(sheet, &range, &bg);
  }
  gtk_sheet_thaw(sheet);
}

void
sg_worksheet_set_end(SGworksheet *worksheet, gint row)
{
  GtkSheet *sheet;
  GdkColor bg;
  GtkSheetRange range;
  gchar label[100];


  sheet = GTK_SHEET(worksheet);

  if(row != -1 && row < worksheet->begin) return;

  gtk_sheet_freeze(sheet);

  if(worksheet->end != -1){
    range.col0 = 0;
    range.coli = sheet->maxcol;
    range.row0 = worksheet->end;
    range.rowi = worksheet->end;
    gtk_sheet_row_button_add_label(sheet, worksheet->end, NULL);
    gtk_sheet_range_set_background(sheet, &range, &sheet->bg_color);
  }

  if(row > -1){
    gdk_color_parse("light blue", &bg);
    gdk_color_alloc(gdk_colormap_get_system(), &bg);
    worksheet->end = row;
    range.col0 = 0;
    range.coli = sheet->maxcol;
    range.row0 = row;
    range.rowi = row;
  
    sprintf(label, _("(End)") );

    gtk_sheet_row_button_add_label(sheet, row, label);
    gtk_sheet_range_set_background(sheet, &range, &bg);
  }

  gtk_sheet_thaw(sheet);
}


void
sg_worksheet_reset(SGworksheet *worksheet)
{
  sg_worksheet_set_begin(worksheet, -1);
  sg_worksheet_set_end(worksheet, -1);
}

void
sg_worksheet_destroy(GtkObject *object)
{
  SGworksheet *worksheet;

  worksheet = SG_WORKSHEET(object);

  if(worksheet->name) g_free(worksheet->name);
  worksheet->name = NULL;
  if(worksheet->column) g_free(worksheet->column);
  worksheet->column = NULL;

  if(worksheet->clipboard) g_object_unref(worksheet->clipboard);
  worksheet->clipboard = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

void
sg_worksheet_add_rows(SGworksheet *worksheet, gint nrows)
{
  GtkSheet *sheet;

  sheet = GTK_SHEET(worksheet);
  if(nrows == 0) return;

  if(nrows < 0){
    sg_worksheet_delete_rows(worksheet, sheet->maxrow+nrows+1, -nrows);
    return;
  } 

  gtk_sheet_freeze(sheet);

  gtk_sheet_add_row(sheet, nrows);

  sg_worksheet_set_begin(worksheet, worksheet->begin);
  sg_worksheet_set_end(worksheet, worksheet->end);

  gtk_sheet_thaw(sheet);
}

void
sg_worksheet_add_columns(SGworksheet *worksheet, gint ncols)
{
  GtkSheet *sheet;
  gint i, c, n = 0, max;
  gchar label[4];

  sheet = GTK_SHEET(worksheet);
  if(ncols == 0) return;

  if(ncols < 0){
    sg_worksheet_delete_columns(worksheet, sheet->maxcol+ncols+1, -ncols);
    return;
  } 

  gtk_sheet_freeze(sheet);

  max = 'Z'-'A'+1;

  gtk_sheet_add_column(sheet, ncols);

  worksheet->column = g_renew(SGcolumn, worksheet->column, (sheet->maxcol + 1));

  for(i = sheet->maxcol - ncols + 1; i <= sheet->maxcol ; i++){
      worksheet->last_column++;
      c = worksheet->last_column;
      n = 0;
      if(c >= max*max){
         label[0] = c/(max*max)+'A'-1;
         label[1] = '\0';
         n = 1;
         c -= (c/(max*max))*max*max;
      }
      if(c >= max){
         label[n] = c/max+'A'-1;
         label[n+1] = '\0';
         n++ ;
         c -= (c/max)*max;
      }
      if(c < max){
         label[n] = c+'A';
         label[n+1] = '\0';
      }
      gtk_sheet_column_button_add_label(sheet, i, label);
      gtk_sheet_set_column_title(sheet, i, label);
      gtk_sheet_column_set_justification(sheet, i, GTK_JUSTIFY_RIGHT);
      sg_worksheet_column_set_format(worksheet, i, 
                                     SG_TYPE_NUMBER,
                                     SG_FORMAT_DECIMAL,
                                     SG_INTERNAL_DOUBLE,
                                     DEFAULT_PRECISION);
      worksheet->column[i].exp = NULL;
  }

  sg_worksheet_set_begin(worksheet, worksheet->begin);
  sg_worksheet_set_end(worksheet, worksheet->end);

  gtk_sheet_thaw(sheet);

}

void
sg_worksheet_set_column_name(SGworksheet *worksheet, gint col, const gchar *name)
{
  GtkSheet *sheet;

  sheet = GTK_SHEET(worksheet);

  gtk_sheet_column_button_add_label(sheet, col, name);
  gtk_sheet_set_column_title(sheet, col, name);
}


void
sg_worksheet_insert_rows(SGworksheet *worksheet, 
                         gint row,
                         gint nrows)
{
  GtkSheet *sheet;

  sheet = GTK_SHEET(worksheet);

  gtk_sheet_insert_rows(sheet, row, nrows);

  if(worksheet->end >= row)
      sg_worksheet_set_end(worksheet, worksheet->end + nrows);

  if(worksheet->begin >= row)
      sg_worksheet_set_begin(worksheet, worksheet->begin + nrows);

}

void
sg_worksheet_insert_columns(SGworksheet *worksheet, 
                            gint col,
                            gint ncols)
{
  GtkSheet *sheet;
  gint i, c, n, max;
  gchar label[4];
  SGcolumn auxcol;

  sheet = GTK_SHEET(worksheet);

  gtk_sheet_freeze(sheet);

  gtk_sheet_insert_columns(sheet, col, ncols);

  worksheet->column = g_renew(SGcolumn, worksheet->column, (sheet->maxcol + 1));

  for(i=sheet->maxcol; i>=col+ncols; i--){
    auxcol = worksheet->column[i];
    worksheet->column[i]=worksheet->column[i-ncols];
    worksheet->column[i-ncols]=auxcol;
  }

  max = 'Z'-'A'+1;
  for(i = col; i < col+ncols; i++){
      worksheet->last_column++;
      c = worksheet->last_column;

      n = 0;
      if(c >= max*max){
         label[0] = c/(max*max)+'A'-1;
         label[1] = '\0';
         n = 1;
         c -= (c/(max*max))*max*max;
      }
      if(c >= max){
         label[n] = c/max+'A'-1;
         label[n+1] = '\0';
         n++ ;
         c -= (c/max)*max;
      }
      if(c < max){
         label[n] = c+'A';
         label[n+1] = '\0';
      }

      gtk_sheet_column_button_add_label(sheet, i, label);
      gtk_sheet_set_column_title(sheet, i, label);
      gtk_sheet_column_set_justification(sheet, i, GTK_JUSTIFY_RIGHT);
      sg_worksheet_column_set_format(worksheet, i, 
                                     SG_TYPE_NUMBER,
                                     SG_FORMAT_DECIMAL,
                                     SG_INTERNAL_DOUBLE,
                                     DEFAULT_PRECISION);
      worksheet->column[i].exp = NULL;
    }

  gtk_sheet_thaw(sheet);

}

void
sg_worksheet_delete_rows(SGworksheet *worksheet, gint row, gint nrows)
{
  GtkSheet *sheet;

  sheet = GTK_SHEET(worksheet);

  gtk_sheet_delete_rows(sheet, row,
                        nrows);

  if(worksheet->begin >= row &&
     worksheet->begin <= row+nrows)
          sg_worksheet_set_begin(worksheet, -1);

  if(worksheet->begin > row+nrows)
          sg_worksheet_set_begin(worksheet,
                                 worksheet->begin - nrows);

  if(worksheet->end >= row &&
     worksheet->end <= row+nrows)
          sg_worksheet_set_end(worksheet, -1);

  if(worksheet->end > row+nrows)
          sg_worksheet_set_end(worksheet,
                               worksheet->end - nrows);
}

void
sg_worksheet_delete_columns(SGworksheet *worksheet, gint col, gint ncols)
{
  GtkSheet *sheet;
  gint i;

  sheet = GTK_SHEET(worksheet);

  gtk_sheet_delete_columns(sheet, 
                           col,
                           ncols);


  for(i = col; i <= sheet->maxcol-ncols; i++){
    sg_worksheet_column_set_exp(worksheet, i, NULL);
    worksheet->column[i]=worksheet->column[i+ncols];
  }

}

void
sg_worksheet_column_set_exp(SGworksheet *worksheet, gint col, const gchar *exp)
{ 
  gchar *new_exp = NULL;
  
  if (exp && strlen(exp)>0)
     new_exp=g_strdup(exp);

  if(GTK_SHEET(worksheet)->maxcol < col){
    g_warning("SGworksheet: col > maxcol");
    return;
  }

  if(worksheet->column[col].exp)
     g_free(worksheet->column[col].exp);

  worksheet->column[col].exp = NULL;

  if(exp && strlen(exp) > 0)
     worksheet->column[col].exp = new_exp;;
}

void
sg_worksheet_update_column_exp (SGworksheet *worksheet, 
                                const gchar *exp, gint column,
                                gint from, gint to)
{ 
  PyObject *object;
  gint i;
  gchar *aux = g_strdup(exp);

  object = (PyObject *)python_eval_expr(aux);
  if (object)
  {
      if (PyArray_Check(object))
       { gtk_sheet_freeze(GTK_SHEET(worksheet));
         python_array(worksheet,0,column,(PyArrayObject *)object,
                      GTK_ORIENTATION_VERTICAL,TRUE);
         gtk_sheet_thaw(GTK_SHEET(worksheet));
       }
      else if (PySequence_Check(object))
       { gtk_sheet_freeze(GTK_SHEET(worksheet));
         python_sequence(worksheet,0,column,object,
                         GTK_ORIENTATION_VERTICAL,TRUE,FALSE);
         gtk_sheet_thaw(GTK_SHEET(worksheet));
       }
      else if (object!=Py_None)
       { gtk_sheet_freeze(GTK_SHEET(worksheet));
         for (i=from;i<=to;i++) /* inclusive */
             python_singleton(worksheet,i,column,object,TRUE,TRUE);
         gtk_sheet_thaw(GTK_SHEET(worksheet));
       }
      else
       { gtk_sheet_freeze(GTK_SHEET(worksheet));
         for (i=from;i<=to;i++) /* inclusive */
          sg_worksheet_cell_set(worksheet, i, column, exp, TRUE, TRUE);
         gtk_sheet_thaw(GTK_SHEET(worksheet));
       }
       Py_XDECREF(object);
   }

   g_free(aux);
}


/* Mark all formulas in a range as not updated */
gint 
sg_worksheet_unupdate_exp_range(SGworksheet *worksheet,
                                gint row0, gint rowi,
                                gint col0, gint coli)
{ 
  gint col = 0,row = 0;
  const gchar *exp;
  SGhiddencell *link = NULL;


  link = (SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(worksheet), row, col);

  for(row = row0; row <= rowi; row++)
      for (col = col0; col <= coli; col++)
      {   
          exp = sg_worksheet_cell_get_formula(worksheet, row, col);
          if (!exp) continue;
          link = (SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(worksheet), row, col);
          if (link) link->updated = FALSE;
      }
  return TRUE;
}

/* Mark all formulas in cells as not updated */
gint 
sg_worksheet_unupdate_exp_all(SGworksheet *worksheet)
{
  return sg_worksheet_unupdate_exp_range(worksheet, 
           0, GTK_SHEET(worksheet)->maxallocrow,
           0, GTK_SHEET(worksheet)->maxalloccol);
}                                   


gboolean 
sg_worksheet_update_column(SGworksheet *worksheet, gint col)
{ 
  gtk_sheet_freeze(GTK_SHEET(worksheet));
  if (worksheet->column[col].exp)
  {  sg_worksheet_update_column_exp(worksheet,worksheet->column[col].exp,col,0,
        gtk_sheet_get_rows_count(GTK_SHEET(worksheet)) - 1);
  }
  gtk_sheet_thaw(GTK_SHEET(worksheet));  
  return TRUE;
}

gboolean
sg_worksheet_update_exp_range(SGworksheet *worksheet, 
                              gint row0, gint rowi,
                              gint col0, gint coli)
{
  return SG_WORKSHEET_CLASS(GTK_OBJECT_GET_CLASS(GTK_OBJECT(worksheet)))->update_exp(worksheet,
			    row0, rowi, col0, coli);
}

static gboolean 
sg_worksheet_update(SGworksheet *worksheet, 
                    gint row0, gint rowi,
                    gint col0, gint coli)
{ 
  gint col, row, uneval1 = -1, uneval2 = 0;
  gchar *exp;
  SGhiddencell *link = NULL;
  gint srow, scol, num_selected = 0;
  GtkSheet *sheet;

  sheet=GTK_SHEET(worksheet);
  gtk_sheet_freeze(sheet);

  gtk_sheet_get_active_cell(GTK_SHEET(worksheet),&srow,&scol);  
  num_selected=((sheet->range.rowi-sheet->range.row0)+1)*
  ((sheet->range.coli-GTK_SHEET(worksheet)->range.col0)+1);

  sg_report_python_error=FALSE;
  while (uneval1!=uneval2)
  {   /* Report problems from the second pass */
      if (uneval1>=0) sg_report_python_error=TRUE;
      uneval1=uneval2;
      for (col=col0;col<=coli;col++)
      { sg_worksheet_update_column(worksheet,col);
          for(row=row0,uneval2=0;row<=rowi;row++)
          {   exp=g_strdup(sg_worksheet_cell_get_formula(worksheet, row, col));
              link = (SGhiddencell *)gtk_sheet_get_link(sheet, row, col);
              if (!link || !exp)
                continue;
              sg_worksheet_cell_set(worksheet, row, col, exp, TRUE, TRUE);
              if (row==srow && col==scol && num_selected==1){
                 gtk_entry_set_text(GTK_ENTRY(sheet->sheet_entry), exp);
              g_free(exp);
              }
              if (!link->updated) uneval2++;
          }
      }
      if (uneval2==0) break;
  }

  gtk_sheet_thaw(GTK_SHEET(worksheet));
  return uneval2;
}

gboolean 
sg_worksheet_update_exp_all(SGworksheet *worksheet)
{ 
  return sg_worksheet_update_exp_range(worksheet, 
           0, gtk_sheet_get_rows_count(GTK_SHEET(worksheet))-1,
           0, gtk_sheet_get_columns_count(GTK_SHEET(worksheet))-1);
}

void
sg_worksheet_copy (SGworksheet* worksheet, gboolean clear)
{
  sg_clipboard_copy(SG_CLIPBOARD(worksheet->clipboard), worksheet, clear);
}

void
sg_worksheet_paste (SGworksheet* worksheet)
{
  sg_clipboard_paste(SG_CLIPBOARD(worksheet->clipboard), worksheet);
}
