/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <stdlib.h>
#include "sg_toggle_combos.h"
#include "sg.h"
#include "pixmaps/fit_page.xpm"
#include "pixmaps/symbol_none.xpm" 
#include "pixmaps/symbol_square.xpm" 
#include "pixmaps/symbol_circle.xpm" 
#include "pixmaps/symbol_uptriangle.xpm" 
#include "pixmaps/symbol_dntriangle.xpm" 
#include "pixmaps/symbol_rtriangle.xpm" 
#include "pixmaps/symbol_ltriangle.xpm" 
#include "pixmaps/symbol_diamond.xpm" 
#include "pixmaps/symbol_plus.xpm" 
#include "pixmaps/symbol_cross.xpm" 
#include "pixmaps/symbol_star.xpm" 
#include "pixmaps/symbol_dot.xpm" 
#include "pixmaps/symbol_impulse.xpm"

#define DATA_ROWS 3
#define DATA_COLS 4
#define SYMBOL_ROWS 4
#define SYMBOL_COLS 4

static gchar **symbol_pixmaps[SYMBOL_ROWS][SYMBOL_COLS];

static void sg_toggle_data_real_init		(GtkToggleCombo *combo, 
						 gint row, gint col,
                                                 gpointer data);
static void sg_toggle_symbol_real_init		(GtkToggleCombo *combo, 
						 gint row, gint col,
                                                 gpointer data);

GtkWidget *
sg_toggle_data_new(gint plot_type)
{
  GtkWidget *combo;
  GdkColormap *colormap;
  gint i, j, n;
  gint nrows, ncols;
  GList *list;
  GtkTable *table;

  colormap = gdk_colormap_get_system();

  n = 0;
  list = sg_plugins();
  while(list){
    if(GTK_IS_SG_PLUGIN_STYLE(list->data)){
      SGpluginStyle *c = SG_PLUGIN_STYLE(list->data);

      if(plot_type == c->layer) n++;
    }
    list = list->next;
  }

  if(n >= 5 && n/5*5 == n) { nrows = n / 5; ncols = 5; }
  else if(n >= 4 && n/4*4 == n) { nrows = n / 4; ncols = 4; }
  else if(n >= 3 && n/3*3 == n) { nrows = n / 3; ncols = 3; }
  else if(n >= 5 && n-n/5*5 == 4) { nrows = n / 5; ncols = 5; }
  else if(n >= 4 && n-n/4*4 == 3) { nrows = n / 4; ncols = 4; }
  else if(n >= 3 && n-n/3*3 == 2) { nrows = n / 3; ncols = 3; }
  else { nrows = MIN(n / 2, 1); ncols = 2; }

  combo = gtk_toggle_combo_new(nrows, ncols);
  table = GTK_TABLE(GTK_TOGGLE_COMBO(combo)->table);

  i = j = 0;
  list = sg_plugins();
  while(list){
    if(GTK_IS_SG_PLUGIN_STYLE(list->data)){
      SGpluginStyle *c = SG_PLUGIN_STYLE(list->data);

      if(plot_type == c->layer){
        GdkPixmap *pixmap = c->pixmap->pixmap;
        GtkWidget *bpixmap;
      
        gtk_container_add(GTK_CONTAINER(GTK_TOGGLE_COMBO(combo)->button[i][j]),
                          bpixmap = GTK_WIDGET(gtk_pixmap_new(pixmap,NULL)));
        g_object_set_data(G_OBJECT(GTK_TOGGLE_COMBO(combo)->button[i][j]),"constructor",c);
        gtk_widget_show(bpixmap);
        j++;
        if(j == ncols) { i++; j = 0; }
      }
    }
    list = list->next;
  }

  gtk_signal_connect(GTK_OBJECT(combo), "changed", 
                     GTK_SIGNAL_FUNC(sg_toggle_data_real_init), NULL);

  sg_toggle_data_real_init(GTK_TOGGLE_COMBO(combo), 0, 0, NULL); 
  return combo;
}

void
sg_toggle_data_init(GtkWidget *widget, gint selection)
{
  GtkToggleCombo *combo;
  gint row, col;

  combo = GTK_TOGGLE_COMBO(widget);

  row = selection / combo->ncols;
  col = selection - row * combo->ncols;

  gtk_toggle_combo_select(combo, row, col);
}

gint
sg_toggle_data_get_selection(GtkWidget *widget)
{
  GtkToggleCombo *combo;

  combo = GTK_TOGGLE_COMBO(widget);
  return (combo->row * combo->ncols + combo->column);
}

static void
sg_toggle_data_real_init(GtkToggleCombo *combo, gint row, gint col, gpointer data)
{
  GtkWidget *pixmap;
  GdkPixmap *tool;
  GdkColormap *colormap;

  colormap = gdk_colormap_get_system();

  if(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child)
    gtk_container_remove(GTK_CONTAINER(GTK_COMBO_BUTTON(combo)->button),
                         GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child);
  
  if(!GTK_BIN(combo->button[row][col])->child) return;
;  
  tool = GTK_PIXMAP(GTK_BIN(combo->button[row][col])->child)->pixmap;  
  pixmap = gtk_pixmap_new(tool, NULL);

  gtk_container_add(GTK_CONTAINER(GTK_COMBO_BUTTON(combo)->button),
                    pixmap);
  gtk_widget_show(pixmap);
}

GtkWidget *
sg_toggle_symbol_new()
{
  const gchar *symbol_text[SYMBOL_ROWS][SYMBOL_COLS] = 
                   {{ _("None"), _("Squares"), _("Circles"), _("Up triangles")},
                    { _("Down Triangles"), _("Right Triangles"), _("Left Triangles"), _("Diamonds")},
                    { _("Plus"), _("Crosses"), _("Stars"), _("Dots")},
                    { _("Impulses"), "", "", ""}, };

  GtkWidget *combo;
  GtkWidget *pixmap;
  GdkPixmap *tool;
  GdkBitmap *mask;
  GdkColormap *colormap;
  GtkTooltips *tooltips;
  gint i, j;

  symbol_pixmaps[0][0] = symbol_none_xpm; 
  symbol_pixmaps[0][1] = symbol_square_xpm; 
  symbol_pixmaps[0][2] = symbol_circle_xpm; 
  symbol_pixmaps[0][3] = symbol_uptriangle_xpm; 
  symbol_pixmaps[1][0] = symbol_dntriangle_xpm; 
  symbol_pixmaps[1][1] = symbol_rtriangle_xpm; 
  symbol_pixmaps[1][2] = symbol_ltriangle_xpm; 
  symbol_pixmaps[1][3] = symbol_diamond_xpm; 
  symbol_pixmaps[2][0] = symbol_plus_xpm; 
  symbol_pixmaps[2][1] = symbol_cross_xpm; 
  symbol_pixmaps[2][2] = symbol_star_xpm; 
  symbol_pixmaps[2][3] = symbol_dot_xpm; 
  symbol_pixmaps[3][0] = symbol_impulse_xpm; 
  symbol_pixmaps[3][1] = NULL; 
  symbol_pixmaps[3][2] = NULL; 
  symbol_pixmaps[3][3] = NULL; 

  colormap = gdk_colormap_get_system();

  combo = gtk_toggle_combo_new(SYMBOL_ROWS, SYMBOL_COLS);

  for(i = 0; i < SYMBOL_ROWS; i++)
    for(j = 0; j < SYMBOL_COLS; j++){

      if(symbol_pixmaps[i][j]){
        tool=gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                                   symbol_pixmaps[i][j]);
        pixmap = gtk_pixmap_new(tool, mask);

        gtk_container_add(GTK_CONTAINER(GTK_TOGGLE_COMBO(combo)->button[i][j]),
                    pixmap);
        gtk_widget_show(pixmap);
        gdk_pixmap_unref(tool);
        gdk_bitmap_unref(mask);

        tooltips = gtk_tooltips_new();
        gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips),
                             GTK_TOGGLE_COMBO(combo)->button[i][j],
                             symbol_text[i][j],
                             symbol_text[i][j]);
        gtk_tooltips_enable(GTK_TOOLTIPS(tooltips));
        gtk_tooltips_set_delay(GTK_TOOLTIPS(tooltips), 0);
      }
    }

  gtk_signal_connect(GTK_OBJECT(combo), "changed", 
                     GTK_SIGNAL_FUNC(sg_toggle_symbol_real_init), symbol_pixmaps);

  sg_toggle_symbol_real_init(GTK_TOGGLE_COMBO(combo), 0, 0, symbol_pixmaps); 
  return combo;
}

void
sg_toggle_symbol_init(GtkWidget *widget, gint selection)
{
  GtkToggleCombo *combo;
  gint row, col;

  combo = GTK_TOGGLE_COMBO(widget);

  row = selection / combo->ncols;
  col = selection - row * combo->ncols;

  gtk_toggle_combo_select(combo, row, col);
}

gint
sg_toggle_symbol_get_selection(GtkWidget *widget)
{
  GtkToggleCombo *combo;

  combo = GTK_TOGGLE_COMBO(widget);
  return (combo->row * combo->ncols + combo->column);
}

static void
sg_toggle_symbol_real_init(GtkToggleCombo *combo, gint row, gint col, gpointer data)
{
  GtkWidget *pixmap;
  GdkPixmap *tool;
  GdkBitmap *mask;
  GdkColormap *colormap;

  colormap = gdk_colormap_get_system();


  if(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child)
    gtk_container_remove(GTK_CONTAINER(GTK_COMBO_BUTTON(combo)->button),
                         GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child);
     
  tool=gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                             symbol_pixmaps[row][col]);
  pixmap = gtk_pixmap_new(tool, mask);

  gtk_container_add(GTK_CONTAINER(GTK_COMBO_BUTTON(combo)->button),
                    pixmap);
  gtk_widget_show(pixmap);
  gdk_pixmap_unref(tool);
}


