/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>
#include <stdio.h>
#include <string.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <gdk/gdkkeysyms.h>
#include "sg_plot_window.h"
#include "sg_plot_menu.h"
#include "sg_ellipse_dialog.h"
#include "sg_rectangle_dialog.h"
#include "sg_text_dialog.h"
#include "sg_line_dialog.h"
#include "sg_misc_dialogs.h"
#include "sg_dataset_dialog.h"
#include "sg_layer.h"
#include "sg_layer_control.h"
#include "sg_locale.h"
#include "sg_toolbox.h"
#include "sg_plot_tools.h"
#include "gtkplotart.h"
#include "pixmaps/plot_icon2.xpm"


#define EVENT_METHOD(i, x) GTK_WIDGET_CLASS(GTK_OBJECT(i)->klass)->x

static const gdouble unit_pt[]={ 1.0, 2.83, 28.35, 72.0, (72.0/SG_DPI)};


static void 	sg_plot_window_class_init 		(SGplotWindowClass *klass);
static void 	sg_plot_window_init			(SGplotWindow *plot);
static void 	sg_plot_window_init_gui			(SGplotWindow *plot);
static void	sg_plot_window_destroy			(GtkObject *object); 
static void 	sg_plot_window_realize			(GtkWidget *widget);
static void 	sg_plot_window_map			(GtkWidget *widget);
static void 	remove_layer				(SGplot *plot, 
							 SGlayer *layer, 
							 gpointer data);
static void 	add_layer				(SGplot *plot, 
							 SGlayer *layer, 
							 gpointer data);
static void 	activate_layer				(SGplot *plot, 
							 SGlayer *layer, 
							 gpointer data);
static void     button_toggled                          (GtkWidget *widget,
                                                         gpointer data);
static gint     button_clicked                          (GtkWidget *widget,
                                                         GdkEventButton *event,
                                                         gpointer data);
static void update_ruler_expose_x			(GtkWidget *scroll, 
							 gpointer data);
static void update_ruler_expose_y			(GtkWidget *scroll, 
							 gpointer data);
static void sg_plot_window_motion			(GtkWidget *widget, 
							 GdkEventMotion *event,
							 gpointer data);
static void activate_dataset				(GtkWidget *widget, 
							 gpointer data);
static void init_datasets_combo				(SGlayer *layer);
static void layer_add_data				(SGlayer *layer, 
							 GtkPlotData *dataset,
							 gpointer data);
static void layer_remove_data				(SGlayer *layer, 
							 GtkPlotData *dataset,
							 gpointer data);
static void layer_activate_data				(SGlayer *layer, 
							 GtkPlotData *dataset,
							 gpointer data);
static void sg_plot_motion				(GtkWidget *widget, 
							 GdkEventMotion *event, 
							 gpointer data);
static void     toolbox_select                          (GtkButton *button,
                                                         gpointer data);
static gint     tool_select_region                      (GtkPlotCanvas *canvas,
                                                         gdouble xmin,
                                                         gdouble xmax,
                                                         gdouble ymin,
                                                         gdouble ymax,
                                                         gpointer data);
static void     init_tool                               (SGplot *plot,
							 SGtoolbox *toolbox,
							 gboolean reset);


static GtkWindowClass *parent_class = NULL;

GtkType
sg_plot_window_get_type (void)
{
  static GtkType sg_plot_window_type = 0;

  if (!sg_plot_window_type)
    {
      GtkTypeInfo sg_plot_window_info =
      {
        "SGplotWindow",
        sizeof (SGplotWindow),
        sizeof (SGplotWindowClass),
        (GtkClassInitFunc) sg_plot_window_class_init,
        (GtkObjectInitFunc) sg_plot_window_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_plot_window_type = gtk_type_unique (gtk_window_get_type(), &sg_plot_window_info);
    }

  return sg_plot_window_type;
}

static void
sg_plot_window_class_init (SGplotWindowClass *klass)
{
  GtkWidgetClass *widget_class;
  GtkObjectClass *object_class;

  widget_class = (GtkWidgetClass*) klass;
  object_class = (GtkObjectClass*) klass;

  parent_class = (GtkWindowClass *)gtk_type_class (gtk_window_get_type ());

  widget_class->realize = sg_plot_window_realize;
  widget_class->map = sg_plot_window_map;
  object_class->destroy = sg_plot_window_destroy;
}

static void
sg_plot_window_init (SGplotWindow *plot)
{
   plot->layer_buttons = NULL;

   plot->hruler = NULL;
   plot->vruler = NULL;

   plot->show_toolbar = TRUE;
   plot->show_menubar = TRUE;

   plot->plot = NULL;

   plot->sw = NULL;
   plot->toolbox = NULL;

   plot->x = 20;
   plot->y = 20;
   plot->width = 600;
   plot->height = 600;

}

void 
sg_plot_window_set_menubar_visible(SGplotWindow *plot, gboolean visible)
{ 
  GtkWidget *widget;

  plot->show_menubar = visible;
  widget = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(plot),"menubar"));
  if (plot->show_menubar)
    gtk_widget_show(widget);
  else
    gtk_widget_hide(widget);

}

void 
sg_plot_window_set_toolbar_visible(SGplotWindow *plot, gboolean visible)
{ 
  plot->show_toolbar = visible;
  if(!plot->toolbox) return;
  if(visible)
    gtk_widget_show(plot->toolbox);
  else
    gtk_widget_hide(plot->toolbox);
}

static void
add_dataset(GtkWidget *button, SGplotWindow *window)
{
  SGplot *plot = window->plot;
  SGlayer *layer;
  SGapplication *app = NULL;
  SGdataset *dataset;
  gpointer data;

  layer = plot->active_layer;
  if(!layer) return;

  data = g_object_get_data(G_OBJECT(plot), "application");
  if(data) app = SG_APPLICATION(data);
  dataset = sg_dataset_dialog(app, layer->plugin);
  if(dataset) {
    GtkPlotData *plot_data;
    sg_list_add(app->datasets, G_OBJECT(dataset), dataset->name);
    plot_data = sg_layer_add_dataset_autosymbol(layer, dataset);
    sg_dataset_refresh(dataset);
  }
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_widget_queue_draw(GTK_WIDGET(plot));
}

static void
sg_plot_window_init_gui(SGplotWindow *plot)
{
 GtkWidget *table, *hbox, *hbox2, *vbox, *top_hruler, *left_vruler;
 GtkWidget *frame,*sbar_vert, *sbar_horiz;
 GtkStyle *style;
 GtkWidget *toolbar;
 GtkWidget *hb;
 GtkWidget *add_button;

 plot->menu = sg_plot_menu_new(plot->plot);

 vbox = gtk_vbox_new(FALSE, 0);
 plot->canvas_box = gtk_vbox_new(TRUE, 0);
 gtk_box_set_spacing(GTK_BOX(vbox), 5);
 hbox = gtk_hbox_new(FALSE, 0);
 hbox2 = gtk_hbox_new(FALSE, 0);
 table = gtk_table_new(4, 3, FALSE);

 gtk_container_add(GTK_CONTAINER(plot), vbox);

 toolbar = sg_plot_toolbar_new(plot);
 gtk_object_set_data(GTK_OBJECT(plot),"menubar",toolbar);
 gtk_box_pack_start(GTK_BOX(vbox), hb = gtk_handle_box_new(), FALSE, FALSE, 0);
 gtk_container_add(GTK_CONTAINER(hb), toolbar);
 gtk_widget_show_all(hb);
 gtk_box_pack_start(GTK_BOX(vbox), hb = gtk_handle_box_new(), FALSE, FALSE, 0);
 gtk_container_set_border_width(GTK_CONTAINER(hb), 0);

 plot->layer_toolbar = gtk_toolbar_new();
 gtk_container_add(GTK_CONTAINER(hb), plot->layer_toolbar);
 gtk_toolbar_append_widget   (GTK_TOOLBAR(plot->layer_toolbar),
                              gtk_label_new(_("Layers: ")), NULL, NULL);
 gtk_toolbar_append_widget   (GTK_TOOLBAR(plot->layer_toolbar),
                              (plot->layer_bbox  = gtk_hbox_new(FALSE,0)),
                              NULL, NULL);
 gtk_toolbar_append_space   (GTK_TOOLBAR(plot->layer_toolbar));
 gtk_toolbar_append_widget   (GTK_TOOLBAR(plot->layer_toolbar),
                              gtk_label_new(_("Active Dataset: ")), NULL, NULL);
 gtk_toolbar_append_widget   (GTK_TOOLBAR(plot->layer_toolbar),
                              (plot->datasets_combo = gtk_combo_new()),
                              NULL, NULL);
 gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(plot->datasets_combo)->entry), FALSE);
 gtk_signal_connect(GTK_OBJECT(GTK_COMBO(plot->datasets_combo)->entry),
                    "changed",
                    (GtkSignalFunc) activate_dataset, plot);

 add_button = gtk_button_new_with_label( _("Add dataset"));
 gtk_toolbar_append_widget   (GTK_TOOLBAR(plot->layer_toolbar),
                              add_button,
                              NULL, NULL);
 gtk_signal_connect(GTK_OBJECT(add_button),
                    "clicked",
                    (GtkSignalFunc) add_dataset, plot);

 gtk_widget_show_all(hb);

 gtk_window_set_policy(GTK_WINDOW(plot), TRUE, TRUE, FALSE);

 gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
 gtk_widget_show (hbox);
 plot->sw=gtk_viewport_new(NULL,NULL);
 sbar_vert=gtk_vscrollbar_new(gtk_viewport_get_vadjustment (GTK_VIEWPORT(plot->sw)));
 sbar_horiz=gtk_hscrollbar_new(gtk_viewport_get_hadjustment (GTK_VIEWPORT(plot->sw)));
 gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(sbar_vert), 3, 4, 1, 2,
                  (GtkAttachOptions)0,
                  (GtkAttachOptions)(GTK_FILL|GTK_SHRINK|GTK_EXPAND),0,0);
 gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(sbar_horiz), 2, 3, 2, 3,
                  (GtkAttachOptions)(GTK_FILL|GTK_SHRINK|GTK_EXPAND),
                  (GtkAttachOptions)0,0,0);

 plot->toolbox = sg_toolbox_new();
 
 gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(plot->toolbox), FALSE, FALSE, 0);
 
 gtk_widget_show_all(plot->toolbox);

 if (!plot->show_toolbar)
    gtk_widget_hide(plot->toolbox);

 gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 0);

 gtk_container_add(GTK_CONTAINER(plot->sw),hbox2);
 gtk_box_pack_start(GTK_BOX(hbox2), GTK_WIDGET(plot->canvas_box), TRUE, FALSE, 0);
 gtk_box_pack_start(GTK_BOX(plot->canvas_box), GTK_WIDGET(plot->plot), TRUE, FALSE, 0);
 gtk_widget_show(GTK_WIDGET(plot->plot));
 gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(plot->sw), 2, 3, 1, 2,
                  (GtkAttachOptions)(GTK_FILL|GTK_SHRINK|GTK_EXPAND),
                  (GtkAttachOptions)(GTK_FILL|GTK_SHRINK|GTK_EXPAND), 0, 0);

/*-------------------------------------------------------------*/
 gtk_signal_connect(GTK_OBJECT(gtk_viewport_get_hadjustment (GTK_VIEWPORT(plot->sw))),
                    "changed",
                    (GtkSignalFunc)update_ruler_expose_x,(gpointer)plot);
 gtk_signal_connect(GTK_OBJECT(gtk_viewport_get_hadjustment (GTK_VIEWPORT(plot->sw))),
                    "value_changed",
                    (GtkSignalFunc)update_ruler_expose_x,(gpointer)plot);
 gtk_signal_connect(GTK_OBJECT(gtk_viewport_get_vadjustment (GTK_VIEWPORT(plot->sw))),
                    "changed",
                    (GtkSignalFunc)update_ruler_expose_y,(gpointer)plot);
 gtk_signal_connect(GTK_OBJECT(gtk_viewport_get_vadjustment (GTK_VIEWPORT(plot->sw))),
                    "value_changed",
                    (GtkSignalFunc)update_ruler_expose_y,(gpointer)plot);
/*---------------------------------------------------------------*/
 plot->hruler = top_hruler = gtk_hruler_new();
 gtk_ruler_set_range(GTK_RULER(top_hruler), 
                     0,
                     SG_PLOT(plot->plot)->page_width, 
                     0, 
                     SG_PLOT(plot->plot)->page_width);

 gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(top_hruler), 2,3,0,1,
                  (GtkAttachOptions)(GTK_SHRINK|GTK_FILL|GTK_EXPAND),
		  (GtkAttachOptions)0,0,0);
 gtk_widget_show(top_hruler);


 plot->vruler = left_vruler = gtk_vruler_new();
 gtk_ruler_set_range(GTK_RULER(left_vruler), 
                     0,
                     SG_PLOT(plot->plot)->page_height, 
                     0, 
                     SG_PLOT(plot->plot)->page_height);


 gtk_table_attach(GTK_TABLE(table), GTK_WIDGET(left_vruler),1,2,1,2,
                  (GtkAttachOptions)0,
		  (GtkAttachOptions)(GTK_SHRINK|GTK_FILL|GTK_EXPAND),0,0);
 gtk_widget_show(left_vruler);
 
 style=gtk_style_copy(GTK_WIDGET(plot)->style);

 if (style){
  gtk_widget_set_style(top_hruler,style);
  gtk_widget_set_style(left_vruler,style);
 }

 gtk_signal_connect(GTK_OBJECT(plot->hruler),
                    "map",
                    (GtkSignalFunc)update_ruler_expose_x,(gpointer)plot);
 gtk_signal_connect(GTK_OBJECT(plot->vruler),
                    "map",
                    (GtkSignalFunc)update_ruler_expose_y,(gpointer)plot);

 gtk_widget_show (left_vruler);

 gtk_signal_connect(GTK_OBJECT(plot->plot),"motion_notify_event",
                    GTK_SIGNAL_FUNC(sg_plot_window_motion), plot);
/*-------------------------------------------------------------*/

 frame = gtk_frame_new(NULL);
 gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
 gtk_box_pack_end(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
 gtk_widget_show(frame);

 plot->label = gtk_label_new("");
 gtk_misc_set_alignment(GTK_MISC(plot->label), 0., .5);
 gtk_container_add(GTK_CONTAINER(frame), plot->label);
 gtk_widget_show(plot->label);

/*-------------------------------------------------------------*/
 gtk_plot_canvas_set_size(GTK_PLOT_CANVAS(plot->plot),
                          GTK_PLOT_CANVAS(plot->plot)->width,
                          GTK_PLOT_CANVAS(plot->plot)->height);

 gtk_widget_ensure_style(GTK_WIDGET(plot->plot));
 gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->plot));
 gtk_widget_queue_draw(GTK_WIDGET(plot->plot));

 
 gtk_widget_show_all(table);
 gtk_widget_show(vbox);

 gtk_signal_connect(GTK_OBJECT(plot->plot), "changed", 
		    GTK_SIGNAL_FUNC(update_ruler_expose_x),
		    plot);
 gtk_signal_connect(GTK_OBJECT(plot->plot), "changed", 
		    GTK_SIGNAL_FUNC(update_ruler_expose_y),
		    plot);
 gtk_signal_connect(GTK_OBJECT(plot->plot), "add_layer", 
		    GTK_SIGNAL_FUNC(add_layer), plot);
 gtk_signal_connect(GTK_OBJECT(plot->plot), "remove_layer", 
		    GTK_SIGNAL_FUNC(remove_layer), plot);
 gtk_signal_connect(GTK_OBJECT(plot->plot), "activate_layer", 
		    GTK_SIGNAL_FUNC(activate_layer), plot);
 gtk_signal_connect(GTK_OBJECT(plot->plot),"motion_notify_event",
                    GTK_SIGNAL_FUNC(sg_plot_motion), plot);
 gtk_signal_connect_object(GTK_OBJECT(plot->plot), "button_press_event",
                    GTK_SIGNAL_FUNC(sg_plot_menu_show),
                    GTK_OBJECT(plot->menu));


}

static void
sg_plot_motion(GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
  SGplot *plot;
  SGplotWindow *window;
  GtkPlotCanvas *canvas;

  plot = SG_PLOT(widget);
  window = SG_PLOT_WINDOW(data);
  canvas = GTK_PLOT_CANVAS(widget);

  if(!plot->layers || !plot->active_layer) return;

  if(!GTK_IS_PLOT3D(GTK_PLOT_CANVAS_PLOT(plot->active_layer)->plot) && plot->tool == SG_TOOL_ARROW){
    gint x, y;
    gdouble px, py;
    gchar text[200];

    gtk_widget_get_pointer(widget, &x, &y);
    gtk_plot_get_point(GTK_PLOT(GTK_PLOT_CANVAS_PLOT(plot->active_layer)->plot),
                       x, y, &px, &py);
    if(GTK_IS_PLOT_POLAR(GTK_PLOT_CANVAS_PLOT(plot->active_layer)->plot))
      g_snprintf(text, 200, "  (R,Angle) = (%f,%f)", px, py);
    else
      g_snprintf(text, 200, "  (X,Y) = (%f,%f)", px, py);
    gtk_label_set(GTK_LABEL(window->label), text);
  }

  gtk_widget_event(window->hruler, (GdkEvent *)event);
  gtk_widget_event(window->vruler, (GdkEvent *)event);
}

static void
sg_plot_window_realize(GtkWidget *widget)
{
 SGplotWindow *plot;
 GdkPixmap *plot_icon_pixmap;
 GdkBitmap *plot_icon_mask;

 plot = SG_PLOT_WINDOW(widget);

 if(plot->width > 0 && plot->height > 0){
          gtk_widget_set_uposition(GTK_WIDGET(plot),
                                   plot->x, plot->y);
          gtk_widget_set_usize(GTK_WIDGET(plot),
                               plot->width,
                               plot->height);
 }else{
          gtk_widget_set_usize(GTK_WIDGET(plot),
                               600,
                               600);
 }

 GTK_WIDGET_CLASS(parent_class)->realize(widget);

 plot_icon_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                                gdk_colormap_get_system(),
                                                &plot_icon_mask, NULL, plot_icon2_xpm);

 gdk_window_set_icon(GTK_WIDGET(plot)->window, NULL, plot_icon_pixmap, plot_icon_mask);

 sg_plot_toolbox_init(SG_TOOLBOX(plot->toolbox), plot->plot); /* this is added by Bryan Cole, taken from sg_plot_window_map() */
}

static void
sg_plot_window_map(GtkWidget *widget)
{
  SGplotWindow *plot;
  gchar window_title[500];
  
  plot = SG_PLOT_WINDOW(widget);
 
  GTK_WIDGET_CLASS(parent_class)->map(widget);
  sg_plot_window_set_menubar_visible(plot, plot->show_menubar);
 
  g_snprintf(window_title, 500, "SciGraphica: %s", SG_PLOT(plot->plot)->name);
  gtk_window_set_title(GTK_WINDOW(plot), window_title);

  gdk_window_set_icon_name(GTK_WIDGET(plot)->window,
                           SG_PLOT(plot->plot)->name);

   
  sg_plot_toolbox_set_tool(SG_TOOLBOX(plot->toolbox), plot->plot);
}
 
SGplotWindow *
sg_plot_window_new(SGplot *plot)
{
   SGplotWindow *window;
   GList *list;

   window = SG_PLOT_WINDOW(gtk_widget_new(sg_plot_window_get_type(), NULL));

   window->plot = plot;
   sg_plot_window_init_gui(window);

   list = plot->layers;
   while(list){
     SGlayer *layer = SG_LAYER(list->data);
     add_layer(plot, layer, window);
     list = list->next;
   }

   g_object_set_data(G_OBJECT(plot), "window", window);
   if(plot->active_layer){
     init_datasets_combo(plot->active_layer);
   } else if(plot->layers){
     init_datasets_combo(SG_LAYER(plot->layers->data));
   }
   if(plot->active_layer) activate_layer(plot, plot->active_layer, NULL);
   return window;
}   

static void
button_set_label(GtkWidget *button, gint num)
{
  gchar title[10];

  sprintf(title,"%i\n",num);

  gtk_misc_set_alignment(GTK_MISC(GTK_BIN(button)->child), .5, .0);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(button)->child), title);
}

static void
add_layer(SGplot *plot, SGlayer*layer, gpointer data)
{
   SGplotWindow *plot_window = SG_PLOT_WINDOW(data);
   GtkWidget *button;
   GtkRequisition req;
   gint size;

   button = gtk_toggle_button_new_with_label(" ");
   gtk_widget_size_request(button, &req);
   size = MAX(req.width,req.height);
   gtk_widget_set_usize(button, size, size);
   g_object_set_data(G_OBJECT(layer), "tbutton", button);
   button_set_label(button, plot->nlayers);

   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
   gtk_box_pack_start(GTK_BOX(plot_window->layer_bbox),
                      button, FALSE, FALSE, 0);
   gtk_widget_show(button);

   gtk_signal_connect(GTK_OBJECT(button), "toggled",
                      GTK_SIGNAL_FUNC(button_toggled), layer);
   gtk_signal_connect(GTK_OBJECT(button), "button_press_event",
                      GTK_SIGNAL_FUNC(button_clicked), layer);
   gtk_signal_connect(GTK_OBJECT(layer), "add_dataset",
                      GTK_SIGNAL_FUNC(layer_add_data), layer);
   gtk_signal_connect(GTK_OBJECT(layer), "remove_dataset",
                      GTK_SIGNAL_FUNC(layer_remove_data), NULL);
   gtk_signal_connect(GTK_OBJECT(layer), "activate_dataset",
                      GTK_SIGNAL_FUNC(layer_activate_data), plot_window);

}

static void
remove_layer(SGplot *plot, SGlayer* layer, gpointer data)
{
  SGplotWindow *plot_window = NULL;
  gpointer button;
  GList *layers = NULL;
  gint nlayers = 0;

  if(!G_IS_OBJECT(data) || !G_IS_OBJECT(layer)) return;

  plot_window = SG_PLOT_WINDOW(data);
  button = g_object_get_data(G_OBJECT(layer), "tbutton");

  if(!plot_window->layer_bbox || !GTK_IS_CONTAINER(plot_window->layer_bbox)) return;

  if(button && GTK_IS_WIDGET(button))
    gtk_container_remove(GTK_CONTAINER(plot_window->layer_bbox), GTK_WIDGET(button)); 

  layers = plot->layers;
  while(layers) {
     GtkWidget *button;
     SGlayer *l = SG_LAYER(layers->data);

     if(l != layer) {
       button = g_object_get_data(G_OBJECT(l), "tbutton");
       if(button && GTK_IS_WIDGET(button))
         button_set_label(GTK_WIDGET(button), nlayers + 1);

       nlayers++;
     }
     layers = layers->next;
  }

  gtk_signal_disconnect_by_func(GTK_OBJECT(layer), 
                      GTK_SIGNAL_FUNC(layer_add_data), layer);
  gtk_signal_disconnect_by_func(GTK_OBJECT(layer), 
                      GTK_SIGNAL_FUNC(layer_remove_data), NULL);
  gtk_signal_disconnect_by_func(GTK_OBJECT(layer), 
                      GTK_SIGNAL_FUNC(layer_activate_data), plot_window);
}

static void
activate_layer(SGplot *plot, SGlayer*layer, gpointer data)
{
  GtkWidget *button = NULL;
  gpointer odata = NULL;
  GList *list;
  SGplotWindow *plot_window = SG_PLOT_WINDOW(g_object_get_data(G_OBJECT(plot), "window"));
  GtkWidget *toolbox;

  if(!layer) return;
  init_datasets_combo(layer);

  list = plot->layers;
  while(list){
    odata = g_object_get_data(G_OBJECT(list->data), "tbutton");
    if(odata && GTK_IS_WIDGET(odata)){
      button = GTK_WIDGET(odata);
      gtk_signal_handler_block_by_func(GTK_OBJECT(button), GTK_SIGNAL_FUNC(button_toggled), list->data);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);
      gtk_signal_handler_unblock_by_func(GTK_OBJECT(button), GTK_SIGNAL_FUNC(button_toggled), list->data);
    }
    list = list->next;
  }

  odata = g_object_get_data(G_OBJECT(layer), "tbutton");
  if(odata && GTK_IS_WIDGET(odata)){
    button = GTK_WIDGET(odata);
    gtk_signal_handler_block_by_func(GTK_OBJECT(button), GTK_SIGNAL_FUNC(button_toggled), layer);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(button), GTK_SIGNAL_FUNC(button_toggled), layer);
  }

  toolbox =  sg_plugin_layer_toolbox(layer->plugin, GTK_OBJECT(layer));
  sg_toolbox_add_layer_toolbox(SG_TOOLBOX(plot_window->toolbox), toolbox);
}

static void
button_toggled(GtkWidget *widget, gpointer data)
{
  SGlayer *layer;

  layer = SG_LAYER(data);
  sg_plot_set_active_layer(SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent), layer);
}

static gint
button_clicked(GtkWidget* widget, GdkEventButton *event, gpointer data)
{
  SGlayer *layer;
  SGplot *plot;
  GList *llink;
  GdkModifierType mods;
  gchar path[1000];
  GtkObject *dialog;
  gpointer odata;

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);
  if(!(mods & GDK_BUTTON1_MASK)) return FALSE;
  if(event->type != GDK_2BUTTON_PRESS) return FALSE;

  layer = SG_LAYER(data);
  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent);
  llink = g_list_find(plot->layers, layer);
  sprintf(path, "%s:%d:dataset", plot->name,
  g_list_position(plot->layers, llink)+1);

  gtk_grab_remove(widget);
  sg_plot_layer_control(plot, path);

  return FALSE;
}

static void
activate_dataset(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  GList *list;
  SGplotWindow *plot_window = SG_PLOT_WINDOW(data);
  SGplot *plot = plot_window->plot;
  SGlayer *layer = plot->active_layer;
  GtkCombo *combo = GTK_COMBO(plot_window->datasets_combo);
  GtkPlotData *child_data;

  child = GTK_WIDGET(GTK_LIST(combo->list)->selection->data);
  child_data = GTK_PLOT_DATA(g_object_get_data(G_OBJECT(child), "data"));

  list = GTK_PLOT_CANVAS_PLOT(layer)->plot->data_sets;
  while(list){
    if(GTK_PLOT_DATA(list->data) == child_data){
      sg_layer_set_active_dataset(layer, child_data);
      break;
    }
    list = list->next;
  }
}

static void
layer_add_data(SGlayer *layer, GtkPlotData *dataset, gpointer data)
{
  init_datasets_combo(layer);
}

static void
layer_remove_data(SGlayer *layer, GtkPlotData *dataset, gpointer data)
{
  init_datasets_combo(layer);
}

static void
layer_activate_data(SGlayer *layer, GtkPlotData *dataset, gpointer data)
{
  gint n = 0;
  GList *list = GTK_PLOT_CANVAS_PLOT(layer)->plot->data_sets;
  SGplotWindow *plot = SG_PLOT_WINDOW(data);

  while(list){
    GtkPlotData *dataset;
 
    dataset = GTK_PLOT_DATA(list->data);
 
    if(dataset == GTK_PLOT_CANVAS_PLOT(layer)->data) break; 
    n++;
    list = list->next;
  }

  if(GTK_PLOT_CANVAS_PLOT(layer)->plot->data_sets)
    gtk_list_select_item(GTK_LIST(GTK_COMBO(plot->datasets_combo)->list), n);
}

static void
init_datasets_combo(SGlayer *layer)
{
  GList *list;
  gint n = 0;
  gint active_data = 0;
  GtkWidget *item = NULL;
  SGplot *plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent);
  SGplotWindow *plot_window = SG_PLOT_WINDOW(g_object_get_data(G_OBJECT(plot), "window"));
  GtkCombo *combo = GTK_COMBO(plot_window->datasets_combo);

  if(!layer) return;
  gtk_signal_disconnect_by_func(GTK_OBJECT(combo->entry),
                                GTK_SIGNAL_FUNC(activate_dataset), plot_window);

  list = GTK_LIST(GTK_COMBO(combo)->list)->children; 
  while(list){
    gtk_container_remove(GTK_CONTAINER(combo->list), 
                         GTK_WIDGET(list->data)); 
    list = GTK_LIST(combo->list)->children; 
  }

  gtk_entry_set_text(GTK_ENTRY(combo->entry), "");

  list = GTK_PLOT_CANVAS_PLOT(layer)->plot->data_sets;
  while(list){
    GtkPlotData *dataset = GTK_PLOT_DATA(list->data);
    GList *aux = GTK_PLOT_CANVAS_PLOT(layer)->plot->data_sets;
    gint n = 1;
 
    while(aux){
      GtkPlotData *aux_data = GTK_PLOT_DATA(aux->data);
      if(aux_data != dataset && dataset->name && aux_data->name && strcmp(dataset->name, aux_data->name) == 0) n++;
      if(aux_data == dataset) break;
      aux = aux->next;
    }
    if(n > 1){
      gchar name[200];
      g_snprintf(name, 200, "%s (%d)", dataset->name, n);
      item = gtk_list_item_new_with_label(name);
    } else {
      item = gtk_list_item_new_with_label(dataset->name);
    }

    if(dataset == GTK_PLOT_CANVAS_PLOT(layer)->data) active_data = n;
    g_object_set_data(G_OBJECT(item), "data", dataset);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(combo->list), item);
    list = list->next;
    n++;
  }

  gtk_list_select_item(GTK_LIST(combo->list), active_data);

  gtk_signal_connect(GTK_OBJECT(combo->entry),
                     "changed",
                     (GtkSignalFunc) activate_dataset, plot_window);
}

void
sg_plot_window_rescale(SGplotWindow *plot, gdouble scale)
{
  if(scale < .15) return;

  SG_PLOT(plot->plot)->scale = scale;
  gtk_plot_canvas_set_magnification(GTK_PLOT_CANVAS(plot->plot),
                                    SG_PLOT(plot->plot)->scale);

  if(plot->hruler) update_ruler_expose_x(NULL, plot);
  if(plot->vruler) update_ruler_expose_y(NULL, plot);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot->plot));
  gtk_widget_queue_draw(GTK_WIDGET(plot->plot));
}

void
sg_plot_window_update_rulers(SGplotWindow *plot)
{
  if(plot->hruler) update_ruler_expose_x(NULL, plot);
  if(plot->vruler) update_ruler_expose_y(NULL, plot);
}

void
sg_plot_window_fit_page(SGplotWindow *plot)
{
  gdouble width, height;
  gdouble w_width, w_height;
  gdouble scalex,scaley;

  w_width = (gdouble)plot->sw->allocation.width;
  w_height = (gdouble)plot->sw->allocation.height;
  width = (gdouble)GTK_PLOT_CANVAS(plot->plot)->pixmap_width;
  height = (gdouble)GTK_PLOT_CANVAS(plot->plot)->pixmap_height;

  scalex = w_width/width;
  scaley = w_height/height;

  if(scalex<scaley)   
    sg_plot_window_rescale(plot, scalex*SG_PLOT(plot->plot)->scale);
  else   
    sg_plot_window_rescale(plot, scaley*SG_PLOT(plot->plot)->scale);

}

void
sg_plot_window_fit_page_h(SGplotWindow *plot)
{
  gdouble width;
  gdouble w_width;
  gdouble scalex;

  w_width=(gdouble)plot->sw->allocation.width;
  width = (gdouble)GTK_PLOT_CANVAS(plot->plot)->pixmap_width;

  scalex=w_width/width;

  sg_plot_window_rescale(plot, scalex*SG_PLOT(plot->plot)->scale);
}

void
sg_plot_window_fit_page_v(SGplotWindow *plot)
{
  gdouble height;
  gdouble w_height;
  gdouble scaley;

  w_height=(gdouble)plot->sw->allocation.height;
  height = (gdouble)GTK_PLOT_CANVAS(plot->plot)->pixmap_height;

  scaley=w_height/height;

  sg_plot_window_rescale(plot, scaley*SG_PLOT(plot->plot)->scale);
}

static void
sg_plot_window_destroy(GtkObject *object)
{
  SGplotWindow *plot_window = SG_PLOT_WINDOW(object);

  if(plot_window->plot)
    gtk_signal_disconnect_by_func(GTK_OBJECT(plot_window->plot), 
  	  	  		  GTK_SIGNAL_FUNC(activate_layer), object);
  plot_window->plot = NULL;

  plot_window->menu = NULL;
  plot_window->layer_bbox = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

/* Events */

static void
update_ruler_expose_x(GtkWidget *scroll, gpointer data)
{
    GtkAdjustment *adj;
    SGplotWindow *plot;
    gdouble start, end, fac, p_size;
    gint s_width;
    GtkAllocation allocation;

    if (!data) return;

    plot=SG_PLOT_WINDOW(data);

    allocation = GTK_WIDGET(plot->plot)->allocation;

    /* width of vscrollbar*/
    s_width=plot->sw->allocation.width;
          
    p_size = GTK_PLOT_CANVAS(plot->plot)->width * GTK_PLOT_CANVAS(plot->plot)->magnification;

    adj=gtk_viewport_get_hadjustment (GTK_VIEWPORT(plot->sw));

    /* Conversion factor from pixels to real physical unit */
    fac=SG_PLOT(plot->plot)->page_width/unit_pt[plot->plot->page_units];
    if (p_size<s_width || adj->upper == 0)
    {  
        start=-((s_width-p_size)/2)/p_size*fac;
        end=((s_width-p_size)/2+p_size)/p_size*fac;
        allocation.x = (gint16)( (s_width - p_size)/2 );
        gtk_widget_size_allocate(plot->canvas_box, &allocation);
    }
    else
    { 
        start=adj->value/adj->upper*fac;
        end=(adj->value+adj->page_size)/adj->upper*fac;
        allocation.x = 0;
        gtk_widget_size_allocate(plot->canvas_box, &allocation);
    }
    gtk_ruler_set_range(GTK_RULER(plot->hruler), start,
                        end, start, end);

    fac = (gdouble)GTK_PLOT_CANVAS(plot->plot)->pixmap_width / (gdouble)SG_PLOT(plot->plot)->page_width * unit_pt[plot->plot->page_units]; 
    switch(plot->plot->page_units){
      case SG_UNIT_PT:
      case SG_UNIT_PIXEL:
        GTK_PLOT_CANVAS(plot->plot)->grid_step = 20. * fac; 
        break;
      case SG_UNIT_MM:
        GTK_PLOT_CANVAS(plot->plot)->grid_step = 10. * fac; 
        break;
      case SG_UNIT_CM:
        GTK_PLOT_CANVAS(plot->plot)->grid_step = 1. * fac; 
        break;
      case SG_UNIT_IN:
        GTK_PLOT_CANVAS(plot->plot)->grid_step = .2 * fac; 
        break;
     }
}

static void
update_ruler_expose_y(GtkWidget *scroll, gpointer data)
{
    GtkAdjustment *adj;
    SGplotWindow *plot;
    gdouble start, end, fac, p_size;
    gint s_height;
    GtkAllocation allocation;

    if (!data) return;

    plot=SG_PLOT_WINDOW(data);

    allocation = GTK_WIDGET(plot->plot)->allocation;

    /* height of hscrollbar*/
    s_height=plot->sw->allocation.height;

    p_size = GTK_PLOT_CANVAS(plot->plot)->height * GTK_PLOT_CANVAS(plot->plot)->magnification;

    adj=gtk_viewport_get_vadjustment (GTK_VIEWPORT(plot->sw));

    /* Conversion factor from pixels to real physical unit */
    fac=SG_PLOT(plot->plot)->page_height/unit_pt[plot->plot->page_units];

    if (p_size<s_height || adj->upper == 0)
    {  
        start=-((s_height-p_size)/2)/p_size*fac;
        end=((s_height-p_size)/2+p_size)/p_size*fac;
        allocation.y = (gint16)( (s_height - p_size)/2 );
        gtk_widget_size_allocate(plot->canvas_box, &allocation);
    }
    else
    { 
        start=adj->value/adj->upper*fac;
        end=(adj->value+adj->page_size)/adj->upper*fac;
        allocation.y = 0;
        gtk_widget_size_allocate(plot->canvas_box, &allocation);
    }
    gtk_ruler_set_range(GTK_RULER(plot->vruler), start,
                        end, start, end);
}

static void
sg_plot_window_motion(GtkWidget *widget, GdkEventMotion *event, gpointer data)
{
  SGplotWindow *plot;
  GtkPlotCanvas *canvas;

  plot = SG_PLOT_WINDOW(data);
  canvas = GTK_PLOT_CANVAS(widget);

  if(!SG_PLOT(canvas)->layers || !SG_PLOT(canvas)->active_layer) return;

  if(!GTK_IS_PLOT3D(GTK_PLOT_CANVAS_PLOT(SG_PLOT(canvas)->active_layer)->plot) && SG_PLOT(canvas)->tool == SG_TOOL_ARROW){
    gint x, y;
    gdouble px, py;
    gchar text[200];

    gtk_widget_get_pointer(widget, &x, &y);
    gtk_plot_get_point(GTK_PLOT_CANVAS_PLOT(SG_PLOT(canvas)->active_layer)->plot, x, y, &px, &py);
    if(GTK_IS_PLOT_POLAR(GTK_PLOT_CANVAS_PLOT(SG_PLOT(canvas)->active_layer)->plot))
      g_snprintf(text, 200, "  (R,Angle) = (%f,%f)", px, py);
    else
      g_snprintf(text, 200, "  (X,Y) = (%f,%f)", px, py);
    gtk_label_set(GTK_LABEL(plot->label), text);
  }

  gtk_widget_event(plot->hruler, (GdkEvent *)event);
  gtk_widget_event(plot->vruler, (GdkEvent *)event);
}


static gint
tool_select_region(GtkPlotCanvas *canvas, 
                   gdouble x1, gdouble y1,
                   gdouble x2, gdouble y2,
                   gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  GtkPlotCanvasChild *child = NULL;
  gdouble xmin, ymin, xmax, ymax;
  gint px1, px2, py1, py2;

  xmin = MIN(x1, x2);
  ymin = MIN(y1, y2);
  xmax = MAX(x1, x2);
  ymax = MAX(y1, y2);

  plot = SG_PLOT(canvas);

  real_plot = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(plot->active_layer)->plot);
  gtk_plot_canvas_unselect(canvas);

  switch(plot->tool){
    case SG_TOOL_ZOOM:
      if(x1 == x2 || y1 == y2){
         gtk_widget_queue_draw(GTK_WIDGET(canvas));
         return FALSE;
      }
      gtk_plot_canvas_get_pixel(canvas, xmin, ymin, &px1, &py1);
      gtk_plot_canvas_get_pixel(canvas, xmax, ymax, &px2, &py2);
      gtk_plot_get_point(real_plot, px1, py1, &xmin, &ymax);
      gtk_plot_get_point(real_plot, px2, py2, &xmax, &ymin);
      /* this is a crude work-arround.
      It would be better to modify gtk_plot_get_point() to check for log-scales*/
      if (real_plot->left->ticks.scale==GTK_PLOT_SCALE_LOG10) {
      ymax = real_plot->ymin * pow(10,
             ((ymax - real_plot->ymin) / (real_plot->ymax - real_plot->ymin))
	     * log10((real_plot->ymax) / (real_plot->ymin)) );
      ymin = real_plot->ymin * pow(10,
             ((ymin - real_plot->ymin) / (real_plot->ymax - real_plot->ymin))
	     * log10((real_plot->ymax) / (real_plot->ymin)) );
      }
      else {
	while(real_plot->left->ticks.step > ymax - ymin){
		gtk_plot_set_major_ticks(real_plot, GTK_PLOT_AXIS_Y,
					 real_plot->left->ticks.step / 2.);
	}
      }

      if (real_plot->bottom->ticks.scale==GTK_PLOT_SCALE_LOG10) {
      xmax = real_plot->xmin * pow(10,
             ((xmax - real_plot->xmin) / (real_plot->xmax - real_plot->xmin))
	     * log10((real_plot->xmax) / (real_plot->xmin)) );

      xmin = real_plot->xmin * pow(10,
             ((xmin - real_plot->xmin) / (real_plot->xmax - real_plot->xmin))
	     * log10((real_plot->xmax) / (real_plot->xmin)) );
      }
      else {
	while(real_plot->bottom->ticks.step > xmax - xmin){
		gtk_plot_set_major_ticks(real_plot, GTK_PLOT_AXIS_X,
					 real_plot->bottom->ticks.step / 2.);
	}
      }
      gtk_plot_set_range(real_plot, xmin, xmax, ymin, ymax);
      gtk_plot_canvas_paint(canvas);
      gtk_widget_queue_draw(GTK_WIDGET(canvas));
      return FALSE;
    case SG_TOOL_RECTANGLE:
      if(x1 == x2 || y1 == y2){
         gtk_widget_queue_draw(GTK_WIDGET(canvas));
         return FALSE;
      }
      child = gtk_plot_canvas_rectangle_new(GTK_PLOT_LINE_SOLID, 0, 
                                            NULL, NULL, (GtkPlotBorderStyle)1, FALSE);
      gtk_plot_canvas_put_child(canvas, child, x1, y1, x2, y2);
      gtk_plot_canvas_paint(canvas);
      gtk_widget_queue_draw(GTK_WIDGET(canvas));
      return TRUE;
    case SG_TOOL_ELLIPSE:
      if(x1 == x2 || y1 == y2){
         gtk_widget_queue_draw(GTK_WIDGET(canvas));
         return FALSE;
      }
      child = gtk_plot_canvas_ellipse_new(GTK_PLOT_LINE_SOLID, 0, 
                                          NULL, NULL, FALSE);
      gtk_plot_canvas_put_child(canvas, child, x1, y1, x2, y2);
      gtk_plot_canvas_paint(canvas);
      gtk_widget_queue_draw(GTK_WIDGET(canvas));
      return FALSE;
    case SG_TOOL_LINE_ARROW:
      child = gtk_plot_canvas_line_new(GTK_PLOT_LINE_SOLID, 0, NULL,
                                       GTK_PLOT_ARROW_END);
      gtk_plot_canvas_put_child(canvas, child, x1, y1, x2, y2);
      gtk_plot_canvas_paint(canvas);
      gtk_widget_queue_draw(GTK_WIDGET(canvas));
      return FALSE;
    case SG_TOOL_LINE:
      child = gtk_plot_canvas_line_new(GTK_PLOT_LINE_SOLID, 0, NULL,
                                       GTK_PLOT_ARROW_NONE);
      gtk_plot_canvas_put_child(canvas, child, x1, y1, x2, y2);
      gtk_plot_canvas_paint(canvas);
      gtk_widget_queue_draw(GTK_WIDGET(canvas));
      return FALSE;
    default:
      break;
  }

  return FALSE;
}


static void
toolbox_select(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  SGtoolbox *toolbox;

  toolbox = SG_TOOLBOX(data);
  plot = SG_PLOT(g_object_get_data(G_OBJECT(toolbox), "plot"));

  if(!plot->active_layer) return;

  real_plot = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(plot->active_layer)->plot);

  GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot),
                              GTK_PLOT_CANVAS_DND_FLAGS);
  GTK_PLOT_CANVAS_PLOT(plot->active_layer)->flags = 0;

  if(button == GTK_BUTTON(toolbox->button[0][0])){
        plot->tool = SG_TOOL_ARROW;
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot),
                                  GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot),
                                    GTK_PLOT_CANVAS_CAN_SELECT); 
  }

  if(button == GTK_BUTTON(toolbox->button[0][1])){
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot),
                                    GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot),
                                  GTK_PLOT_CANVAS_CAN_SELECT);
        plot->tool = SG_TOOL_ZOOM;
  }

  if(button == GTK_BUTTON(toolbox->button[1][0])){
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot),
                                  GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot),
                                    GTK_PLOT_CANVAS_CAN_SELECT); 
        GTK_PLOT_CANVAS_PLOT(plot->active_layer)->flags = 
                             GTK_PLOT_CANVAS_CAN_SELECT;
        plot->tool = SG_TOOL_POINTER;
  }

  if(button == GTK_BUTTON(toolbox->button[1][1])){
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot),
                                  GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot),
                                    GTK_PLOT_CANVAS_CAN_SELECT); 
        GTK_PLOT_CANVAS_PLOT(plot->active_layer)->flags = 
                             GTK_PLOT_CANVAS_CAN_SELECT;

        plot->tool = SG_TOOL_MARKERS;
  }

  if(button == GTK_BUTTON(toolbox->button[2][0])){
        plot->tool = SG_TOOL_TEXT;
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot),
				    GTK_PLOT_CANVAS_CAN_SELECT |
                                    GTK_PLOT_CANVAS_DND_FLAGS);
  }

/*
  if(button == GTK_BUTTON(toolbox->button[2][0])){
        plot->tool = SG_TOOL_LINE;
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot),
                                    GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot),
                                  GTK_PLOT_CANVAS_CAN_SELECT);
  }
*/
  if(button == GTK_BUTTON(toolbox->button[2][1])){
        plot->tool = SG_TOOL_LINE_ARROW;
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot),
                                    GTK_PLOT_CANVAS_CAN_SELECT_ITEM|
                                    GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot),
                                  GTK_PLOT_CANVAS_CAN_SELECT);
  }
  if(button == GTK_BUTTON(toolbox->button[3][0])){
        plot->tool = SG_TOOL_RECTANGLE;
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot),
                                    GTK_PLOT_CANVAS_CAN_SELECT_ITEM|
                                    GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot),
                                  GTK_PLOT_CANVAS_CAN_SELECT);
  }
  if(button == GTK_BUTTON(toolbox->button[3][1])){
        plot->tool = SG_TOOL_ELLIPSE;
        GTK_PLOT_CANVAS_UNSET_FLAGS(GTK_PLOT_CANVAS(plot),
                                    GTK_PLOT_CANVAS_CAN_SELECT_ITEM|
                                    GTK_PLOT_CANVAS_DND_FLAGS);
        GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot),
                                  GTK_PLOT_CANVAS_CAN_SELECT);
  }
}


void
sg_plot_toolbox_init(SGtoolbox *toolbox, SGplot *plot)
{
  gint i, j;

  g_object_set_data(G_OBJECT(toolbox), "plot", plot);

  for(i = 0; i < 2; i++){
    for(j = 0; j < 4; j++){
       gtk_signal_connect(GTK_OBJECT(toolbox->button[j][i]),
                          "toggled",
                          GTK_SIGNAL_FUNC(toolbox_select), toolbox);
    }
  }
 
  gtk_signal_connect(GTK_OBJECT(plot), "select_region",
                     GTK_SIGNAL_FUNC(tool_select_region), plot);

/* 
  init_tool(plot, toolbox, FALSE);
*/
}

void
sg_plot_toolbox_set_tool(SGtoolbox *toolbox, SGplot *plot)
{
  init_tool(plot, toolbox, TRUE); 
}

static void
init_tool(SGplot *plot, SGtoolbox *toolbox, gboolean reset)
{
  gint row = 0, col = 0;
  GtkWidget *button;
  gboolean init = FALSE;

  if(plot->tool == SG_TOOL_NONE){
    plot->tool = SG_TOOL_ARROW;
    init = TRUE;
  }

  switch(plot->tool){
    case SG_TOOL_ARROW:
      row = 0;
      col = 0;
      break;
    case SG_TOOL_ZOOM:
      row = 0;
      col = 1;
      break;
    case SG_TOOL_POINTER:
      row = 1;
      col = 0;
      break;
    case SG_TOOL_MARKERS:
      row = 1;
      col = 1;
      break;
    case SG_TOOL_TEXT:
      row = 2;
      col = 0;
      break;
    case SG_TOOL_LINE:
/*
      row = 2;
      col = 0;
*/
      break;
    case SG_TOOL_LINE_ARROW:
      row = 2;
      col = 1;
      break;
    case SG_TOOL_RECTANGLE:
      row = 3;
      col = 0;
      break;
    case SG_TOOL_ELLIPSE:
      row = 3;
      col = 1;
      break;
    default:
      break;
  }

  button = toolbox->button[row][col];
  GTK_BUTTON(button)->button_down=TRUE;
  GTK_TOGGLE_BUTTON(button)->active=TRUE;
  gtk_widget_set_state(button, GTK_STATE_ACTIVE);
  if(init) gtk_signal_emit_by_name(GTK_OBJECT(button), "toggled", NULL);
}


