/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_list.h"

static void sg_list_class_init (SGlistClass *klass);
static void sg_list_init 	(SGlist *dlist);
static void sg_list_finalize 	(GObject *object);

static GObjectClass *parent_class = NULL;

GType
sg_list_get_type (void)
{
  static GType sg_list_type = 0;
                                                                                
  if (!sg_list_type)
    {
      static const GTypeInfo sg_list_info =
      {
        sizeof (SGlistClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) sg_list_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (SGlist),
        0,              /* n_preallocs */
        (GInstanceInitFunc) sg_list_init,
        NULL,
      };
                                                                                
      sg_list_type = g_type_register_static (G_TYPE_OBJECT, "SGlist",
                                               &sg_list_info, 0);
    }
                                                                                
  return sg_list_type;
}

static void
sg_list_class_init (SGlistClass *klass)
{
  GObjectClass *object_class;
  SGlistClass *dlist_class;

  parent_class = g_type_class_peek_parent (klass);

  object_class = (GObjectClass *) klass;
  dlist_class = (SGlistClass *) klass;

  object_class->finalize = sg_list_finalize;
}


static void
sg_list_init (SGlist *dlist)
{
  dlist->list = NULL;
  dlist->n = 0;
}

SGlist*
sg_list_new (void)
{
  GObject *object;

  object = g_object_new (sg_list_get_type (), NULL);

  return (SG_LIST(object));
}

static void
sg_list_finalize(GObject *object)
{
  SGlist *dlist = SG_LIST(object);

  sg_list_clear(dlist);
}

static GList *
find_object_by_name(SGlist *dlist, const gchar *name)
{
  GList *list;

  if(!dlist->list) return NULL;

  list = dlist->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    if(child->name && strcmp(child->name, name) == 0) return list;
    list = list->next;
  }
  return NULL;
}

static GList *
find_object(SGlist *dlist, GObject *object)
{
  GList *list;

  if(!dlist->list) return NULL;

  list = dlist->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    if(child->object == object) return list;
    list = list->next;
  }
  return NULL;
}

SGlistChild *
sg_list_find(SGlist *dlist, GObject *object)
{
  GList *list;
  list = find_object(dlist, object);
  if(list) {
    SGlistChild *child = (SGlistChild *)list->data;
    return child;
  }
  return NULL;
}

SGlistChild *
sg_list_add(SGlist *dlist, GObject *object, const gchar *name)
{
  GList *list = NULL;
  SGlistChild *child = NULL;

  list = find_object(dlist, object);
  if(list){
	  /* FIXME: emit signal */
    return NULL;
  }

  child = g_new0(SGlistChild, 1);
  if(name) child->name = g_strdup(name);
  child->object = object;

  dlist->list = g_list_append(dlist->list, child);
  if(GTK_IS_WIDGET(object))
    gtk_widget_ref(GTK_WIDGET(object)); 
  else if(G_IS_OBJECT(object))
    g_object_ref(G_OBJECT(object)); 
  dlist->n++;
  child->id = dlist->n;
  return child;
}

gboolean
sg_list_remove(SGlist *dlist, GObject *object)
{
  GList *list;
  list = find_object(dlist, object);
  if(list){
    SGlistChild *child = (SGlistChild *)list->data;
    if(child->name) g_free(child->name);
    g_free(child);
    if(GTK_IS_WIDGET(object))
      gtk_widget_unref(GTK_WIDGET(object));
    else if(G_IS_OBJECT(object))
      g_object_unref(G_OBJECT(object));
    dlist->list = g_list_remove_link(dlist->list, list);
    g_list_free_1(list); 
    return TRUE;
  }
  return FALSE;
}

gboolean
sg_list_set_name(SGlist *dlist, GObject *object, const gchar *name)
{
  GList *list;

  if(!name) return FALSE;

  list = find_object(dlist, object);
  if(list){
    SGlistChild *child = (SGlistChild *)list->data;
    if(child->name) g_free(child->name);
    child->name = g_strdup(name);
    return TRUE;
  }
  return FALSE;
}


GObject *
sg_list_get(SGlist *dlist, const gchar *name)
{
  GList *list = NULL;;
  list = find_object_by_name(dlist, name);
  if(list) {
    SGlistChild *child = (SGlistChild *)list->data;
    return G_OBJECT(child->object);
  }
  return NULL;
}

GObject *
sg_list_get_by_id(SGlist *dlist, gint id)
{
  GList *list;

  if(!dlist->list) return NULL;

  list = dlist->list;
  while(list){
    if(list->data){
      SGlistChild *child = (SGlistChild *)list->data;
      if(child->id == id) return G_OBJECT(child->object);
    }
    list = list->next;
  }
  return NULL;
}

void
sg_list_clear(SGlist *dlist)
{
  GList *list;

  list = dlist->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    if(child->name) g_free(child->name);

    if(child->object){
      if(GTK_IS_WIDGET(child->object)) 
        gtk_widget_unref(GTK_WIDGET(child->object));
      else if(G_IS_OBJECT(child->object)) 
        g_object_unref(G_OBJECT(child->object));
    }

    g_free(child);
    list->data = NULL;
    dlist->list = g_list_remove_link(dlist->list, list);
    g_list_free_1(list);

    list = dlist->list;
  }
  dlist->list = NULL;
  dlist->n = 0;
}
