/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <math.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_layer.h"
#include "sg.h"
#include "sg_dataset.h"

enum {
        ADD_DATASET,
        REMOVE_DATASET,
        ACTIVATE_DATASET,
        LAST_SIGNAL,
};

enum {
	ARG_0,
	ARG_RESCALE,
	ARG_SYMBOL,
	ARG_SYMBOL_STYLE,
	ARG_LINE_STYLE,
	ARG_CONNECTOR,
	ARG_SYMBOL_COLOR,
	ARG_LINE_COLOR
};

#define NUM_SYMBOLS 10
#define NUM_SYMBOL_STYLES 3
#define NUM_LINE_STYLES 7
#define NUM_CONNECTORS 6

#define P_(string) string

static void	sg_layer_init			(SGlayer *layer);
static void 	sg_layer_class_init 		(SGlayerClass *klass);
static void 	sg_layer_destroy 		(GtkObject *object);
static void plot_changed			(GtkPlot *plot, gpointer data);

static void sg_layer_set_property 		(GObject *object,
                                         	 guint            prop_id,
                                         	 const GValue          *value,
                                         	 GParamSpec      *pspec);
static void sg_layer_get_property 		(GObject *object,
                                         	 guint            prop_id,
                                         	 GValue    *value,
                                         	 GParamSpec      *pspec);


static GtkPlotCanvasPlotClass *parent_class = NULL;
static guint layer_signals[LAST_SIGNAL] = {0};


GtkType
sg_layer_get_type (void)
{
  static GtkType layer_type = 0;

  if (!layer_type)
    {
      GtkTypeInfo layer_info =
      {
        "SGlayer",
        sizeof (SGlayer),
        sizeof (SGlayerClass),
        (GtkClassInitFunc) sg_layer_class_init,
        (GtkObjectInitFunc) sg_layer_init,
        /* reserved 1*/ NULL,
        /* reserved 2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      layer_type = gtk_type_unique (GTK_TYPE_PLOT_CANVAS_PLOT, &layer_info);
    }
  return layer_type;
}

static void
sg_layer_class_init (SGlayerClass *klass)
{
  GtkObjectClass *object_class;
  GObjectClass *gobject_class;
  SGlayerClass *layer_class;

  parent_class = (GtkPlotCanvasPlotClass *)gtk_type_class (gtk_plot_canvas_plot_get_type ());

  object_class = (GtkObjectClass *) klass;
  gobject_class = (GObjectClass *) klass;
  layer_class = (SGlayerClass *) klass;

  layer_signals[ADD_DATASET] =
    gtk_signal_new ("add_dataset",
                    GTK_RUN_LAST,
                    GTK_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGlayerClass, add_dataset),
                    gtk_marshal_VOID__OBJECT,
                    GTK_TYPE_NONE, 1, GTK_TYPE_PLOT_DATA);
  layer_signals[REMOVE_DATASET] =
    gtk_signal_new ("remove_dataset",
                    GTK_RUN_LAST,
                    GTK_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGlayerClass, remove_dataset),
                    gtk_marshal_VOID__OBJECT,
                    GTK_TYPE_NONE, 1, GTK_TYPE_PLOT_DATA);
  layer_signals[ACTIVATE_DATASET] =
    gtk_signal_new ("activate_dataset",
                    GTK_RUN_LAST,
                    GTK_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGlayerClass, activate_dataset),
                    gtk_marshal_VOID__OBJECT,
                    GTK_TYPE_NONE, 1, GTK_TYPE_PLOT_DATA);

  object_class->destroy = sg_layer_destroy;
  layer_class->remove_dataset = NULL;

  gobject_class->set_property = sg_layer_set_property;
  gobject_class->get_property = sg_layer_get_property;

  g_object_class_install_property (gobject_class,
                           ARG_RESCALE,
  g_param_spec_int ("rescale",
                           P_("Rescale"),
                           P_("Rescale"),
                           0, G_MAXINT, 0,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_SYMBOL,
  g_param_spec_int ("symbol",
                           P_("Symbol"),
                           P_("Symbol"),
                           0, G_MAXINT, 0,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_SYMBOL_STYLE,
  g_param_spec_int ("symbol_style",
                           P_("Symbol style"),
                           P_("Symbol style"),
                           0, G_MAXINT, 0,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_CONNECTOR,
  g_param_spec_int ("connector",
                           P_("Connector"),
                           P_("Connector"),
                           0, G_MAXINT, 0,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_SYMBOL_COLOR,
  g_param_spec_pointer ("symbol_color",
                           P_("Symbol color"),
                           P_("Symbol color"),
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_LINE_COLOR,
  g_param_spec_pointer ("line_color",
                           P_("Line color"),
                           P_("Line color"),
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
}

SGlayer *
sg_layer_new (SGpluginLayer *plugin, gdouble width, gdouble height)
{
  SGlayer *layer;
  GtkPlot *plot;

  layer = SG_LAYER(gtk_type_new(sg_layer_get_type()));

  layer->plugin = plugin;

  GTK_PLOT_CANVAS_PLOT(layer)->plot = GTK_PLOT(plugin->construct());
  plot = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot);

  gtk_plot_resize(plot, width, height);

  g_object_set_data(G_OBJECT(GTK_PLOT_CANVAS_PLOT(layer)->plot), "layer", layer);

  gtk_plot_set_transparent(plot, TRUE);
  
  plot->clip_data = TRUE;

  gtk_signal_connect(GTK_OBJECT(plot), "changed",
                     GTK_SIGNAL_FUNC(plot_changed), NULL);

  return layer;
}

static void
sg_layer_set_property (GObject      *object,
                             guint            prop_id,
                             const GValue          *value,
                             GParamSpec      *pspec)
{
  SGlayer *layer;
                                                                                
  layer = SG_LAYER (object);
                                                                                
  switch(prop_id){
    case ARG_RESCALE:
      layer->rescale = g_value_get_int(value);
      break;
    case ARG_SYMBOL:
      layer->symbol = g_value_get_int(value);
      break;
    case ARG_SYMBOL_STYLE:
      layer->symbol_style = g_value_get_int(value);
      break;
    case ARG_CONNECTOR:
      layer->connector = g_value_get_int(value);
      break;
    case ARG_LINE_STYLE:
      layer->line_style = g_value_get_int(value);
      break;
    case ARG_LINE_COLOR:
      layer->line_color = *((GdkColor*)g_value_get_pointer(value));
      break;
    case ARG_SYMBOL_COLOR:
      layer->symbol_color = *((GdkColor*)g_value_get_pointer(value));
      break;
  }
}

                                                                               
static void
sg_layer_get_property (GObject      *object,
                             guint            prop_id,
                             GValue          *value,
                             GParamSpec      *pspec)
{
  SGlayer *layer;
                                                                                
  layer = SG_LAYER (object);
                                                                                
  switch(prop_id){
    case ARG_RESCALE:
      g_value_set_int(value,layer->rescale);
      break;
    case ARG_SYMBOL:
      g_value_set_int(value,layer->symbol);
      break;
    case ARG_SYMBOL_STYLE:
      g_value_set_int(value,layer->symbol_style);
      break;
    case ARG_CONNECTOR:
      g_value_set_int(value,layer->connector);
      break;
    case ARG_LINE_STYLE:
      g_value_set_int(value,layer->line_style);
      break;
    case ARG_LINE_COLOR:
      g_value_set_pointer(value,&layer->line_color);
      break;
    case ARG_SYMBOL_COLOR:
      g_value_set_pointer(value,&layer->symbol_color);
      break;
  }
}

static void
sg_layer_init(SGlayer *layer)
{
  GTK_PLOT_CANVAS_PLOT(layer)->data = NULL;

  layer->symbol = 1;
  layer->symbol_style = 0;
  layer->line_style = 1;
  layer->connector = 1;
  layer->rescale = 1;

  gdk_color_black(gdk_colormap_get_system(), &layer->symbol_color);
  gdk_color_black(gdk_colormap_get_system(), &layer->line_color);
}

static void 	
sg_layer_destroy(GtkObject *object)
{
  SGlayer *layer;

  layer = SG_LAYER(object);

  sg_layer_clear(layer);
  if(GTK_PLOT_CANVAS_PLOT(layer)->plot && GTK_IS_WIDGET(GTK_PLOT_CANVAS_PLOT(layer)->plot)) 
    gtk_widget_destroy(GTK_WIDGET(GTK_PLOT_CANVAS_PLOT(layer)->plot));
  GTK_PLOT_CANVAS_PLOT(layer)->plot = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

GtkPlotData *
sg_layer_add_dataset(SGlayer *layer, SGdataset *dataset)
{
  GtkPlotData *real_data;

  real_data = sg_dataset_new_child(dataset);
  sg_dataset_add_child(dataset, real_data);
  sg_layer_add_dataset_child(layer, real_data);

  gtk_signal_emit(GTK_OBJECT(layer), layer_signals[ADD_DATASET], real_data);

  return real_data;
}

void
sg_layer_add_dataset_child(SGlayer *layer, GtkPlotData *real_data)
{
  gtk_plot_add_data(GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot), real_data);
  gtk_widget_show(GTK_WIDGET(real_data));

  sg_layer_set_active_dataset(layer, real_data);
}

void
sg_layer_set_active_dataset(SGlayer *layer, GtkPlotData *dataset)
{
  GList *list = NULL;

  if(dataset) g_list_find(GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot)->data_sets, dataset);
  if(list) 
    GTK_PLOT_CANVAS_PLOT(layer)->data = dataset;
  else
    GTK_PLOT_CANVAS_PLOT(layer)->data = NULL;

  gtk_signal_emit(GTK_OBJECT(layer), layer_signals[ACTIVATE_DATASET], dataset);
}

GtkPlotData *
sg_layer_add_dataset_default(SGlayer *layer, SGdataset *dataset)
{
  GtkPlotData *real_data;

  real_data = sg_dataset_new_child(dataset);
  sg_dataset_add_child(dataset, real_data);

  real_data->symbol.symbol_type = (GtkPlotSymbolType)layer->symbol;
  real_data->symbol.symbol_style = (GtkPlotSymbolStyle)layer->symbol_style;
  real_data->symbol.color = layer->symbol_color;
  real_data->symbol.border.color = layer->symbol_color;
  real_data->line.line_style = (GtkPlotLineStyle)layer->line_style;
  real_data->line_connector = (GtkPlotConnector)layer->connector;
  real_data->line.color = layer->line_color;

  gtk_plot_add_data(GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot), real_data);
  gtk_widget_show(GTK_WIDGET(real_data));

  GTK_PLOT_CANVAS_PLOT(layer)->data = real_data;
  sg_layer_set_active_dataset(layer, real_data);

  return real_data;
}

GtkPlotData *
sg_layer_add_dataset_autosymbol(SGlayer *layer, SGdataset *dataset)
{
  GtkPlotData *real_data;

  real_data = sg_dataset_new_child(dataset);
  sg_dataset_add_child(dataset, real_data);

  real_data->symbol.color = layer->symbol_color;
  real_data->symbol.border.color = layer->symbol_color;
  real_data->line_connector = (GtkPlotConnector)layer->connector;
  real_data->line.color = layer->line_color;

  if(real_data->symbol.symbol_type != GTK_PLOT_SYMBOL_NONE){
    real_data->symbol.symbol_type = (GtkPlotSymbolType)layer->symbol;
    real_data->symbol.symbol_style = (GtkPlotSymbolStyle)layer->symbol_style;
                                                                                
    layer->symbol++;
    if(layer->symbol >= NUM_SYMBOLS){
       layer->symbol = 1;
       layer->symbol_style++;
       if(layer->symbol_style >= NUM_SYMBOL_STYLES)
          layer->symbol_style = 0;
    }
  }
  if(real_data->line.line_style != GTK_PLOT_LINE_NONE){
    real_data->line.line_style = (GtkPlotLineStyle)layer->line_style;
                                                                                
    layer->line_style++;
    if(layer->line_style >= NUM_LINE_STYLES)
          layer->line_style = 1;
  }


  gtk_plot_add_data(GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot), real_data);
  gtk_widget_show(GTK_WIDGET(real_data));

  GTK_PLOT_CANVAS_PLOT(layer)->data = real_data;
  sg_layer_set_active_dataset(layer, real_data);

  return real_data;
}


void
sg_layer_remove_dataset(SGlayer *layer, SGdataset *dataset)
{
  SGdataset *link;
  GList *list = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot)->data_sets;

  while(list){
    link = SG_DATASET(gtk_plot_data_get_link(GTK_PLOT_DATA(list->data)));
    if(link == dataset){
      gtk_signal_emit(GTK_OBJECT(layer), layer_signals[REMOVE_DATASET], GTK_PLOT_DATA(list->data));
      if(GTK_PLOT_DATA(list->data) == GTK_PLOT_CANVAS_PLOT(layer)->data)
        sg_layer_set_active_dataset(layer, NULL);

      sg_dataset_remove_child(dataset, GTK_PLOT_DATA(list->data));
      gtk_plot_remove_data(GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot), 
                           GTK_PLOT_DATA(list->data)); 
      list = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot)->data_sets;
      break;
    } else { 
      list = list->next;
    }
  }
}

void
sg_layer_clear(SGlayer *layer)
{
  GList *list;

  list = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot)->data_sets;
  while(list){
    sg_layer_remove_dataset(layer, SG_DATASET(GTK_PLOT_DATA(list->data)->link));
    list = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot)->data_sets;
  }
  GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot)->data_sets = NULL;
}

static void 
plot_changed (GtkPlot *plot, gpointer data)
{
  if(GTK_WIDGET(plot)->parent && GTK_IS_OBJECT(GTK_WIDGET(plot)->parent))
    gtk_signal_emit_by_name(GTK_OBJECT(GTK_WIDGET(plot)->parent), "changed");
}

void
sg_layer_remove_markers(SGlayer *layer)
{
  SGplot *plot;
  GtkPlotCanvas *canvas;

  if(!layer || !GTK_PLOT_CANVAS_PLOT(layer)->data) return;

  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent);
  canvas = GTK_PLOT_CANVAS(plot);

  gtk_plot_data_remove_markers(GTK_PLOT_CANVAS_PLOT(layer)->data);
  GTK_PLOT_CANVAS_PLOT(layer)->data = NULL;
  
  gtk_plot_canvas_paint(canvas);
  gtk_widget_queue_draw(GTK_WIDGET(canvas));
}

void
sg_layer_show_markers(SGlayer *layer, gboolean show)
{
  SGplot *plot;
  GtkPlotCanvas *canvas;

  if(!layer || !GTK_PLOT_CANVAS_PLOT(layer)->data) return;
  if(gtk_plot_data_markers_visible(GTK_PLOT_CANVAS_PLOT(layer)->data) == show) return;

  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent);
  canvas = GTK_PLOT_CANVAS(plot);

  gtk_plot_data_show_markers(GTK_PLOT_CANVAS_PLOT(layer)->data, show);
  
  gtk_plot_canvas_paint(canvas);
  gtk_widget_queue_draw(GTK_WIDGET(canvas));
}

void
sg_layer_refresh_datasets(SGlayer *layer)
{
  GList *aux_datasets;

  aux_datasets = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot)->data_sets;
  while(aux_datasets){
    SGdataset *data;

    data = SG_DATASET(GTK_PLOT_DATA(aux_datasets->data)->link);
    sg_dataset_refresh(data);
    aux_datasets = aux_datasets->next;
  }
}

gint
sg_layer_min_max(SGlayer *layer, gdouble *xmin, gdouble *xmax, gdouble *ymin, gdouble *ymax)
{
  GtkPlotData *dataset;
  GList *list;
  gint n;
  GtkPlot *plot;
  gboolean change = FALSE;

  plot = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot);

  if(!layer) return FALSE;
  *xmin = plot->xmin;
  *ymin = plot->ymin;
  *xmax = plot->xmax;
  *ymax = plot->ymax;

  list = plot->data_sets;
  while(list){
     gdouble fx, fy, fz, fa;
     gdouble fdx, fdy, fdz, fda;
     gchar *label;
     gboolean error;

     dataset = GTK_PLOT_DATA(list->data);
     if(!dataset->is_function){
       for(n = 0; n < dataset->num_points; n++){
           gtk_plot_data_get_point(dataset, n, 
                                   &fx, &fy, &fz, &fa, 
                                   &fdx, &fdy, &fdz, &fda, 
                                   &label, &error);
           if (n==0 && list==plot->data_sets){
             *xmin = fx;
             *ymin = fy;
             *xmax = fx;
             *ymax = fy;
             change = TRUE;
           }
           else
           { if (plot->bottom->ticks.scale != GTK_PLOT_SCALE_LOG10) {
		if(fx < *xmin){
		change = TRUE;
		*xmin = fx;
		}
		if(fx > *xmax){
		change = TRUE;
		*xmax = fx;
		}
	      }
	      else {
	        if(*xmin <= 0) *xmin = *xmax/10;
	      	if(fx < *xmin && fx > 0){
		change = TRUE;
		*xmin = fx;
		}
		if(fx > *xmax && fx > 0){
		change = TRUE;
		*xmax = fx;
		}
	      }
	      if (plot->left->ticks.scale != GTK_PLOT_SCALE_LOG10) {
		if(fy < *ymin){
		change = TRUE;
		*ymin = fy;
		}
		if(fy > *ymax){
		change = TRUE;
		*ymax = fy;
		}
              }
	      else {
	        if(*ymin <= 0) *ymin = *ymax/10;
		if(fy < *ymin && fy > 0){
		change = TRUE;
		*ymin = fy;
		}
		if(fy > *ymax && fy > 0){
		change = TRUE;
		*ymax = fy;
		}
	      }
	    }
       }
     }

     list = list->next;
  } 
  return change;
}

