/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_dialog.h"
#include "pixmaps/sg.xpm"

static void sg_dialog_class_init   (SGdialogClass *klass);
static void sg_dialog_init         (SGdialog *dialog);
static void sg_dialog_destroy      (GtkObject *o);

static GtkWindowClass *parent_class = NULL;

typedef struct
{
  SGdialog *dialog;
  SGdialogButton response;
  GMainLoop *loop;
  gboolean destroyed;
} RunInfo;

static void
shutdown_loop (RunInfo *ri)
{
  if (g_main_loop_is_running (ri->loop))
    g_main_loop_quit (ri->loop);
  ri->dialog->running = FALSE;
}

static void
run_unmap_handler (SGdialog *dialog, gpointer data)
{
  RunInfo *ri = data;
                                                                                
  shutdown_loop (ri);
}

static gint
run_delete_handler (SGdialog *dialog,
                    GdkEventAny *event,
                    gpointer data)
{
  RunInfo *ri = data;
                                                                                
  shutdown_loop (ri);
                                                                                
  return TRUE; /* Do not destroy */
}

static void
run_destroy_handler (SGdialog *dialog, gpointer data)
{
  RunInfo *ri = data;
                                                                                
  /* shutdown_loop will be called by run_unmap_handler */
                                                                                
  ri->destroyed = TRUE;
}


GtkType
sg_dialog_get_type(void)
{
  static GtkType sg_dialog_type = 0;

  if (!sg_dialog_type)
		        {
    GtkTypeInfo sg_dialog_info =
    {
      "SGdialog",
      sizeof (SGdialog),
      sizeof (SGdialogClass),
      (GtkClassInitFunc) sg_dialog_class_init,
      (GtkObjectInitFunc) sg_dialog_init,
      /* reserved_1 */ NULL,
      /* reserved_2 */ NULL,
      (GtkClassInitFunc) NULL,
    };

    sg_dialog_type = gtk_type_unique (gtk_window_get_type(), &sg_dialog_info);
  }

  return sg_dialog_type;
}

static void
sg_dialog_class_init (SGdialogClass *klass)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) klass;

  parent_class = (GtkWindowClass *)gtk_type_class (gtk_window_get_type ());

  object_class->destroy = sg_dialog_destroy;
};


static void
sg_dialog_init(SGdialog *dialog)
{
  dialog->dialog = NULL;
  dialog->button_ok = NULL;
  dialog->button_apply = NULL;
  dialog->button_cancel = NULL;
  dialog->button_yes = NULL;
  dialog->button_no = NULL;
  dialog->button_back = NULL;
  dialog->button_next = NULL;
  dialog->parent = NULL;
  dialog->box = NULL;
  dialog->button_box = NULL;
  dialog->running = FALSE;
}

static void
sg_dialog_destroy(GtkObject *object)
{
  SGdialog *dialog = SG_DIALOG(object); 

  if(dialog->dialog)
    gtk_container_remove(GTK_CONTAINER(dialog->box), GTK_WIDGET(dialog->dialog));

  dialog->parent = NULL;
  dialog->dialog = NULL;
  dialog->button_ok = dialog->button_apply = dialog->button_cancel = NULL;
  dialog->button_yes = dialog->button_no = NULL;
  dialog->box = dialog->button_box = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
     (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);

  dialog->running = FALSE;
}

GtkWidget *
sg_dialog_new(const gchar *title, GtkOrientation orient, guint button_mask, GtkButtonBoxStyle button_layout)
{
  SGdialog *dialog;

  dialog = SG_DIALOG(gtk_widget_new(sg_dialog_get_type(), NULL));

  GTK_WINDOW(dialog)->type = GTK_WINDOW_TOPLEVEL;
  gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
  if(title) 
    gtk_window_set_title(GTK_WINDOW(dialog), title);

  if(orient == GTK_ORIENTATION_HORIZONTAL){
    dialog->box = gtk_hbox_new(FALSE, 5);
    dialog->button_box = gtk_vbutton_box_new();
  } else {
    dialog->box = gtk_vbox_new(FALSE, 5);
    dialog->button_box = gtk_hbutton_box_new();
  }

  gtk_container_add(GTK_CONTAINER(dialog), dialog->box);
  gtk_container_set_border_width(GTK_CONTAINER(dialog->box), 10);
  gtk_container_set_border_width(GTK_CONTAINER(dialog->button_box), 5);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog->button_box), button_layout);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(dialog->button_box), 5);
  gtk_box_pack_end(GTK_BOX(dialog->box), dialog->button_box, FALSE, FALSE, 0);
  gtk_widget_show(dialog->button_box);
  gtk_widget_show(dialog->box);

  if(button_mask & SG_BUTTON_OK){
      dialog->button_ok = gtk_button_new_from_stock(GTK_STOCK_OK);
      GTK_WIDGET_SET_FLAGS (dialog->button_ok, GTK_CAN_DEFAULT);
      gtk_box_pack_start (GTK_BOX (dialog->button_box), dialog->button_ok, TRUE, TRUE, 0);
      gtk_widget_grab_default (dialog->button_ok);
      gtk_widget_show_all (dialog->button_ok);
  }
  if(button_mask & SG_BUTTON_YES){
      dialog->button_yes = gtk_button_new_from_stock(GTK_STOCK_YES);
      GTK_WIDGET_SET_FLAGS (dialog->button_yes, GTK_CAN_DEFAULT);
      gtk_box_pack_start (GTK_BOX (dialog->button_box), dialog->button_yes, TRUE, TRUE, 0);
      gtk_widget_grab_default (dialog->button_yes);
      gtk_widget_show_all (dialog->button_yes);
  }
  if(button_mask & SG_BUTTON_NO){
      dialog->button_no = gtk_button_new_from_stock(GTK_STOCK_NO);
      gtk_box_pack_start (GTK_BOX (dialog->button_box), dialog->button_no, TRUE, TRUE, 0);
      gtk_widget_show_all (dialog->button_no);
  }
  if(button_mask & SG_BUTTON_BACK){
      dialog->button_back = gtk_button_new_from_stock(GTK_STOCK_GO_BACK);
      gtk_box_pack_start (GTK_BOX (dialog->button_box), dialog->button_back, TRUE, TRUE, 0);
      gtk_widget_show_all (dialog->button_back);
  }
  if(button_mask & SG_BUTTON_NEXT){
      dialog->button_next = gtk_button_new_from_stock(GTK_STOCK_GO_FORWARD);
      GTK_WIDGET_SET_FLAGS (dialog->button_next, GTK_CAN_DEFAULT);
      gtk_box_pack_start (GTK_BOX (dialog->button_box), dialog->button_next, TRUE, TRUE, 0);
      gtk_widget_grab_default (dialog->button_next);
      gtk_widget_show_all (dialog->button_next);
  }
  if(button_mask & SG_BUTTON_APPLY){
      dialog->button_apply = gtk_button_new_from_stock(GTK_STOCK_APPLY);
      gtk_box_pack_start (GTK_BOX (dialog->button_box), dialog->button_apply, TRUE, TRUE, 0);
      gtk_widget_show_all (dialog->button_apply);
  }
  if(button_mask & SG_BUTTON_CANCEL){
      dialog->button_cancel = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
      gtk_box_pack_start (GTK_BOX (dialog->button_box), dialog->button_cancel, TRUE, TRUE, 0);
      gtk_widget_show_all (dialog->button_cancel);
  }
  if(button_mask & SG_BUTTON_CLOSE){
      dialog->button_cancel = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
      gtk_box_pack_start (GTK_BOX (dialog->button_box), dialog->button_cancel, TRUE, TRUE, 0);
      gtk_widget_show_all (dialog->button_cancel);
  }
    
    
  return GTK_WIDGET(dialog);
}

void
sg_dialog_add(GtkWidget *wdialog, SGpropertyDialog *child)
{
  SGdialog *dialog = SG_DIALOG(wdialog);

  dialog->dialog = child;
  gtk_box_pack_start(GTK_BOX(dialog->box), GTK_WIDGET(child), FALSE, FALSE, 0);

  sg_property_dialog_set_buttons(child,
		  		 dialog->button_ok,
		  		 dialog->button_apply,
		  		 dialog->button_cancel);
}

static void
button_clicked_yes(GtkWidget *button, gpointer data)
{
  RunInfo *ri = (RunInfo *)data;
  ri->response = SG_BUTTON_YES;
  shutdown_loop (ri);
}

static void
button_clicked_no(GtkWidget *button, gpointer data)
{
  RunInfo *ri = (RunInfo *)data;
  ri->response = SG_BUTTON_NO;
  shutdown_loop (ri);
}

static void
button_clicked_cancel(GtkWidget *button, gpointer data)
{
  RunInfo *ri = (RunInfo *)data;
  ri->response = SG_BUTTON_CANCEL;
  shutdown_loop (ri);
}

static void
button_clicked_apply(GtkWidget *button, gpointer data)
{
  RunInfo *ri = (RunInfo *)data;
  ri->response = SG_BUTTON_APPLY;
  shutdown_loop (ri);
}

static void
button_clicked_ok(GtkWidget *button, gpointer data)
{
  RunInfo *ri = (RunInfo *)data;
  ri->response = SG_BUTTON_OK;
  shutdown_loop (ri);
}

SGdialogButton
sg_dialog_run(GtkWidget *wdialog, GtkObject *parent)
{
  SGdialog *dialog = SG_DIALOG(wdialog);
  GtkWidget *window = GTK_WIDGET(dialog);
  GdkPixmap *pixmap;
  GdkBitmap *mask;
  gulong unmap_handler;
  gulong destroy_handler;
  gulong delete_handler;
  RunInfo ri = { dialog, SG_BUTTON_NONE, NULL, FALSE };

  gtk_widget_realize(window);
  gtk_window_set_modal(GTK_WINDOW(window), TRUE);

  if(GTK_IS_WINDOW(parent)){
    gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW(parent));
    gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER_ON_PARENT);
  }

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
		                                gdk_colormap_get_system(),
		                                &mask, NULL, 
						sg_xpm);

  gdk_window_set_icon(window->window, NULL, pixmap, mask);

  gtk_widget_show_all(window);

  dialog->parent = parent;

  if(parent)
    gtk_signal_connect_object(GTK_OBJECT(parent), "destroy",
		    	      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(dialog));

  if(dialog->button_ok){
    gtk_signal_connect(GTK_OBJECT(dialog->button_ok), "clicked",
		       GTK_SIGNAL_FUNC(button_clicked_ok),
		       &ri);
  }

  if(dialog->button_yes){
    gtk_signal_connect(GTK_OBJECT(dialog->button_yes), "clicked",
		       GTK_SIGNAL_FUNC(button_clicked_yes),
		       &ri);
  }

  if(dialog->button_no){
    gtk_signal_connect(GTK_OBJECT(dialog->button_no), "clicked",
		       GTK_SIGNAL_FUNC(button_clicked_no),
		       &ri);
  }
  if(dialog->button_cancel){
    gtk_signal_connect(GTK_OBJECT(dialog->button_cancel), "clicked",
		       GTK_SIGNAL_FUNC(button_clicked_cancel),
		       &ri);
  }
 
  unmap_handler =
    g_signal_connect (dialog,
                      "unmap",
                      G_CALLBACK (run_unmap_handler),
                      &ri);
                                                                                
  delete_handler =
    g_signal_connect (dialog,
                      "delete_event",
                      G_CALLBACK (run_delete_handler),
                      &ri);
  destroy_handler =
    g_signal_connect (dialog,
                      "destroy",
                      G_CALLBACK (run_destroy_handler),
                      &ri);
                                                                                
  ri.loop = g_main_loop_new (NULL, FALSE);
                                                                                
  GDK_THREADS_LEAVE ();
  g_main_loop_run (ri.loop);
  GDK_THREADS_ENTER ();
                                                                                
  g_main_loop_unref (ri.loop);
                                                                                
  ri.loop = NULL;
                                                                                
  if (!ri.destroyed)
    {
      gtk_window_set_modal (GTK_WINDOW(dialog), FALSE);
                                                                                
      g_signal_handler_disconnect (dialog, unmap_handler);
      g_signal_handler_disconnect (dialog, delete_handler);
      g_signal_handler_disconnect (dialog, destroy_handler);
      if(GTK_IS_WIDGET(wdialog)) gtk_widget_destroy(wdialog);
    }
                                                                              
/*
  if(parent)
    gtk_signal_connect_object(GTK_OBJECT(parent), "destroy",
		    	      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(dialog));

  if(dialog->button_ok){
    gtk_signal_connect(GTK_OBJECT(dialog->button_ok), "clicked",
		       GTK_SIGNAL_FUNC(button_clicked_ok),
		       &button_clicked);
    gtk_signal_connect_object(GTK_OBJECT(dialog->button_ok), "clicked",
		    	      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(dialog));
  }

  if(dialog->button_yes){
    gtk_signal_connect(GTK_OBJECT(dialog->button_yes), "clicked",
		       GTK_SIGNAL_FUNC(button_clicked_yes),
		       &button_clicked);
    gtk_signal_connect_object(GTK_OBJECT(dialog->button_yes), "clicked",
		    	      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(dialog));
  }

  if(dialog->button_no){
    gtk_signal_connect(GTK_OBJECT(dialog->button_no), "clicked",
		       GTK_SIGNAL_FUNC(button_clicked_no),
		       &button_clicked);
    gtk_signal_connect_object(GTK_OBJECT(dialog->button_no), "clicked",
		    	      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(dialog));
  }
  if(dialog->button_cancel){
    gtk_signal_connect(GTK_OBJECT(dialog->button_cancel), "clicked",
		       GTK_SIGNAL_FUNC(button_clicked_cancel),
		       &button_clicked);
    gtk_signal_connect_object(GTK_OBJECT(dialog->button_cancel), "clicked",
		    	      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(dialog));
  }

  SG_DIALOG(dialog)->running = TRUE;
  gtk_main();
*/
  return ri.response;
}

