/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gdk/gdk.h>
#include <glib.h>
#include <gtkextra/gtkextra.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>
#include <string.h>
#include <Python.h>
#include <grammar.h>
#include <node.h>
#include <parsetok.h>
#include <errcode.h>
#include <compile.h>
#include <eval.h>
#include <marshal.h>
#include "sg.h"
#include "sg_python.h"
#include "sg_python_worksheet.h"

extern PyObject *main_dict, *sg_dict;

static SGworksheet *active_worksheet;

/* Function forward declarations */
static PyObject *get_cell_double(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *get_col_double_array(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *get_row_double_array(PyObject *self, PyObject *args, PyObject *keywds);
                                                                                
PyObject *set_sheet(gint row, gint col, PyObject *object, SGworksheet *worksheet,
                     GtkOrientation orient);
                                                                                
static PyObject *set_cell(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *set_col(PyObject *self, PyObject *args, PyObject *keywds);
static PyObject *set_row(PyObject *self, PyObject *args, PyObject *keywds);

static PyMethodDef SGworksheetMethods[] =
{
  {"cell",  (PyCFunction)get_cell_double, METH_VARARGS|METH_KEYWORDS,
   "Return the content (number or string) of a cell in a sheet in Scigraphica.\n\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'col'   (as integer or as label name string)\n"
   "                   'sheet' (string, will default to the current sheet)\n"},
  {"col",  (PyCFunction)get_col_double_array, METH_VARARGS|METH_KEYWORDS,
   "Return a PyArray (or None) of the column of the sheet in Scigraphica.\n\n"
   "Only the numeric values are returned, with no gaps. So indices of the\n"
   "returned array may not correspond to the original Scigraphica column.\n"
   "(I hope this changes to using masked arrays - chrisk@mit.edu)\n"
   "Takes the keywords 'col'   (as integer or as label name string)\n"
   "                   'sheet' (string, will default to the current sheet)\n"},
  {"row",  (PyCFunction)get_row_double_array, METH_VARARGS|METH_KEYWORDS,
   "Return a PyArray (or None) of the row of the sheet in Scigraphica.\n\n"
   "Only the numeric values are returned, with no gaps. So indices of the\n"
   "returned array may not correspond to the original Scigraphica row.\n"
   "(I hope this changes to using masked arrays - chrisk@mit.edu)\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'sheet' (string, will default to 'current')\n"},
  {"set_cell",  (PyCFunction)set_cell, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'col'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to the current sheet)\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (row,col).\n"},
  {"set_column",  (PyCFunction)set_col, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'col'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to 'current')\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (0,col).\n"},
   {"set_row",  (PyCFunction)set_row, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to the current sheet')\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (row,0).\n"},
  {"setcell",  (PyCFunction)set_cell, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'col'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to the current sheet)\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (row,col).\n"},
  {"setcol",  (PyCFunction)set_col, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'col'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to the current sheet)\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (0,col).\n"},
  {"setrow",  (PyCFunction)set_row, METH_VARARGS|METH_KEYWORDS,
   "Set the contents of a sheet in Scigraphica.\n\n"
   "Takes the keywords 'row'   (as integer or as label name string)\n"
   "                   'object'(see below).\n"
   "                   'sheet' (string, will default to the current sheet)\n"
   "The object can be a value or sequence or 1D or 2D array, which will be\n"
   "used to set the contents of the sheet, with a starting corner of (row,0).\n"},
  {NULL, NULL, 0,NULL}        /* Sentinel */
};

void sg_python_worksheet_init()
{
  (void) Py_InitModule("sg", SGworksheetMethods);
}
                                                                                
static PyObject *no_such_worksheet(gchar *sheet){
     gchar message[160];
     g_snprintf(message,160,"No such sheet '%s'",sheet);
     PyErr_SetString(PyExc_ValueError,message);
     return NULL;
}
                                                                                
gint
get_worksheet_col(SGworksheet *worksheet,PyObject *col_obj)
{
  gint col,num;
  gchar *col_name;

  if (PyString_Check(col_obj))
  {
    num=gtk_sheet_get_columns_count(GTK_SHEET(worksheet));
    col_name=PyString_AsString(col_obj);
    for (col=0;col<num;col++)
        if (GTK_SHEET(worksheet)->column[col].name)
        {
          if (!strcmp(col_name,GTK_SHEET(worksheet)->column[col].name))
            break;
        }
        else
        { 
          if (col==atoi(col_name))
            return col;
        }
  }
  else if (PyInt_Check(col_obj)){
      col=(int)PyInt_AsLong(col_obj);
      printf("col_num 1 =%d\n",col);
  }
                                                                                
  /* Now check if row and col values are sane */
  if (col<0 || col>gtk_sheet_get_columns_count(GTK_SHEET(worksheet)))
  {
    PyErr_SetString(PyExc_ValueError,"Column number out of range");
    return -1;
  }
  return(col);
}

gint 
get_worksheet_row(SGworksheet *worksheet,PyObject *row_obj)
{
  gint row,num;
  gchar *row_name;
                                                                                
  if (PyString_Check(row_obj))
      {
        num=gtk_sheet_get_rows_count(GTK_SHEET(worksheet));
        row_name=PyString_AsString(row_obj);
          {
            for (row=0;row<num;row++)
             if (GTK_SHEET(worksheet)->row[row].name)
              { if (!strcmp(row_name,GTK_SHEET(worksheet)->row[row].name))
                 break;
              }
             else
              { 
                if (row==atoi(row_name))
                  return(row);
              }
          }
      }
  else if (PyInt_Check(row_obj))
      row=(int)PyInt_AsLong(row_obj);
                                                                                
  /* Now check if row and col values are sane */
  if (row<0 || row>gtk_sheet_get_rows_count(GTK_SHEET(worksheet)))
  {
    PyErr_SetString(PyExc_ValueError,"Row number out of range");
    return -1;
  }
  return(row);
}
                                                                                
static PyObject *
get_cell_object(SGworksheet *worksheet, gint row, gint col)
{
  gdouble value;
  PyObject *object=NULL;
  gchar *sheet=NULL,*content;
  SGhiddencell *link;

  link = (SGhiddencell *)gtk_sheet_get_link(GTK_SHEET(worksheet),
                                            row, col);
  if(link)
  switch(link->type){
      case SG_TYPE_NUMBER:
      switch (link->internal){
          case SG_INTERNAL_INTEGER: object=Py_BuildValue("l", link->value.val_long);
          case SG_INTERNAL_DOUBLE:  object=Py_BuildValue("f", link->value.val_double);
          default: object=Py_BuildValue("f",0.0);
      }
      case SG_TYPE_TEXT:
      case SG_TYPE_TIME:
      case SG_TYPE_DATE:
      default:
          content=sg_worksheet_cell_get_formula(worksheet,row,col);
          if (!content) content=sg_worksheet_cell_get_text(worksheet,row,col);
          if (content)
              object=Py_BuildValue("s",sg_worksheet_cell_get_formula(worksheet,row,col));
                                                                                
  }
                                                                                
                                                                                
  if (!object)
      object=Py_None;
                                                                                
  Py_INCREF(object);
  return object;
}

/* Cell/Col/Row query functions */
static PyObject *
get_cell(PyObject *self, PyObject *args, PyObject *keywds)
{
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object=NULL,*row_obj,*col_obj;
  gboolean restore_active=FALSE;
  gchar *sheet=NULL,*content;
  gint row=0,col=0,num,arow=-1,acol=-1;
  static gchar *kwlist[]={"col","row","sheet",NULL};
                                                                                
  if (!PyArg_ParseTupleAndKeywords(args, keywds, "OO|s", kwlist,
                                   &col_obj, &row_obj, &sheet))
      return NULL;
/* Get sheet pointer */
  worksheet = active_worksheet;
  if(!worksheet) return no_such_worksheet(sheet);
                                                                                
  /* Now check if row and col values are sane */
  row=get_worksheet_row(worksheet,row_obj);
  if (row<0 || row >GTK_SHEET(worksheet)->maxallocrow) return NULL;
  col=get_worksheet_col(worksheet,col_obj);
  if (col<0 || col >GTK_SHEET(worksheet)->maxalloccol) return NULL;
                                                                                
  return get_cell_object(worksheet,row,col);
                                                                                
}

static PyObject *
get_cell_double(PyObject *self, PyObject *args, PyObject *keywds)
{
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object=NULL,*row_obj,*col_obj;
  gboolean error=FALSE;
  gchar *sheet=NULL,*content;
  gint row=0,col=0,num,arow=-1,acol=-1;
  static gchar *kwlist[]={"col","row","sheet",NULL};
                                                                                
  if (!PyArg_ParseTupleAndKeywords(args, keywds, "OO|s", kwlist,
                                   &col_obj, &row_obj, &sheet))
      return NULL;
/* Get sheet pointer */
  worksheet = active_worksheet;
  if(!worksheet) return no_such_worksheet(sheet);
                                                                                
  /* Now check if row and col values are sane */
  row=get_worksheet_row(worksheet,row_obj);
  if (row<0 || row >GTK_SHEET(worksheet)->maxallocrow) return NULL;
  col=get_worksheet_col(worksheet,col_obj);
  if (col<0 || col >GTK_SHEET(worksheet)->maxalloccol) return NULL;
                                                                                
  object=Py_BuildValue("f", sg_worksheet_cell_get_double(worksheet, row, col,&error));
                                                                                
  if (!object || error)
      object=Py_None;
                                                                                
  Py_INCREF(object);
  return object;
                                                                                
}

static PyObject *
get_col_double_array(PyObject *self, PyObject *args, PyObject *keywds)
{
  gchar *sheet=NULL,*content;
  static gchar *kwlist[]={"col","sheet",NULL},*col_name;
  gint row=0,num,col=0,nd=1,dims[1],i,j=0,arow=-1,acol=-1;
  SGworksheet *worksheet=NULL;
  gdouble value,*data,*rdata;
  PyObject *object,*col_obj;
  gboolean error=FALSE;
                                                                                
  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O|s", kwlist,
                                   &col_obj, &sheet))
      return NULL;
/* Get sheet pointer */
  worksheet = active_worksheet;
  if(!worksheet) return no_such_worksheet(sheet);
                                                                                
  col=get_worksheet_col(worksheet,col_obj);
  if (col<0 || col >GTK_SHEET(worksheet)->maxalloccol) {
     PyErr_SetString(PyExc_ValueError,"No such column");
     return NULL;
}
                                                                                
/* See how many elements we need */
  data=g_new(gdouble,GTK_SHEET(worksheet)->maxallocrow+1);
  for (i=0;i<=GTK_SHEET(worksheet)->maxallocrow;i++){
      value=sg_worksheet_cell_get_double(worksheet, i, col,&error);
      if (error) break;
      data[j++]=value;
  }
  if (i==j)
      rdata=data;
  else
      rdata=g_renew(gdouble,data,j);
                                                                                
  if (j>0)
  {
      dims[0]=j;
      object=PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,(gchar *)rdata);
  }
  else
  {
      object=Py_None;
      g_free(rdata);
  }
                                                                                
  Py_INCREF(object);
  return object;
}

static PyObject *
get_row_double_array(PyObject *self, PyObject *args, PyObject *keywds)
{
  gchar *sheet=NULL,*content;
  static gchar *kwlist[]={"row","sheet",NULL};
  gint row=0,arow=-1,acol=-1,num,nd=1,dims[1],i,j=0;
  SGworksheet *worksheet=NULL;
  gdouble value,*data,*rdata;
  PyObject *object, *row_obj;
  gboolean error=FALSE;
                                                                                
  if (!PyArg_ParseTupleAndKeywords(args, keywds, "O|s", kwlist,
                                   &row_obj, &sheet))
      return NULL;
/* Get sheet pointer */
  worksheet = active_worksheet;
  if(!worksheet) return no_such_worksheet(sheet);
                                                                                
  /* Now check if row and col values are sane */
  row=get_worksheet_row(worksheet,row_obj);
  if (row<0 || row >GTK_SHEET(worksheet)->maxallocrow) {
     PyErr_SetString(PyExc_ValueError,"No such row");
     return NULL;
}
                                                                                
                                                                                
/* See how many elements we need */
  data=g_new(gdouble,GTK_SHEET(worksheet)->maxalloccol+1);
  for (i=0;i<=GTK_SHEET(worksheet)->maxalloccol;i++){
      value=sg_worksheet_cell_get_double(worksheet, row, i,&error);
      if (error) break;
      data[j++]=value;
  }
                                                                                
  if (i==j)
      rdata=data;
  else
  {
      rdata=g_renew(gdouble,data,j);
  }
                                                                                
  if (j>0)
   {
       dims[0]=j;
       object=PyArray_FromDimsAndData(1,dims,PyArray_DOUBLE,(gchar *)rdata);
   }
  else{
      object=Py_None;
      g_free(rdata);
  }
                                                                                
                                                                                
  Py_INCREF(object);
  return object;
}

/* Cell/Col/Row setting functions */
                                                                                
PyObject *
set_sheet(gint row, gint col, PyObject *object, SGworksheet *worksheet,
          GtkOrientation orient)
{
  gint retval,acol,arow;
  gboolean restore_active=FALSE;
                                                                                
  /* Now check if row and col values are sane */
                                                                                
      if (PyArray_Check(object))
       {
         gtk_sheet_freeze(GTK_SHEET(worksheet));
         retval=python_array(worksheet,row,col,(PyArrayObject *)object,orient,FALSE);
         gtk_sheet_thaw(GTK_SHEET(worksheet));
       }
      else
      if (PySequence_Check(object))
       {
         gtk_sheet_freeze(GTK_SHEET(worksheet));
         retval=python_sequence(worksheet,row,col,object,orient,FALSE,FALSE);
         gtk_sheet_thaw(GTK_SHEET(worksheet));
       }
      else
        retval=python_singleton(worksheet,row,col,object,FALSE,FALSE);
                                                                                
                                                                                
      if (!retval)
      {
        if (PyErr_Occurred())
          {
              PyErr_Print();
              if (Py_FlushLine()) PyErr_Clear();
              PyErr_Clear();
          }
        return NULL;
      }
      else
      {
          Py_INCREF(Py_None);
          return Py_None;
      }
}

static PyObject *
set_cell(PyObject *self, PyObject *args, PyObject *keywds)
{
  gchar *sheet=NULL, *content;
  static gchar *kwlist[]={"col","row","object","sheet",NULL};
  gint row=0,col=0,num,arow=-1,acol=-1,retval;
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object,*row_obj,*col_obj;
  gboolean restore_active=FALSE;
                                                                                
  if (!PyArg_ParseTupleAndKeywords(args, keywds, "OOO|s", kwlist,
                                   &col_obj, &row_obj, &object, &sheet))
    return NULL;
/* Get sheet pointer */
  worksheet = active_worksheet;
  if(!worksheet) return no_such_worksheet(sheet);
                                                                                
  row=get_worksheet_row(worksheet,row_obj);
  if (row<0) return NULL;
  col=get_worksheet_col(worksheet,col_obj);
  if (col<0) return NULL;
                                                                                
  return set_sheet(row,col,object,worksheet,GTK_ORIENTATION_VERTICAL);
}
                                                                                
static PyObject *
set_col(PyObject *self, PyObject *args, PyObject *keywds)
{
  gchar *sheet=NULL, *content;
  static gchar *kwlist[]={"col","object","sheet",NULL};
  gint row=0,col=0,num,arow=-1,acol=-1,retval;
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object,*col_obj;
  gboolean restore_active=FALSE;
                                                                                
  if (!PyArg_ParseTupleAndKeywords(args, keywds, "OO|s", kwlist,
                                   &col_obj, &object, &sheet))
      return NULL;
/* Get sheet pointer */
  worksheet = active_worksheet;
  if(!worksheet) return no_such_worksheet(sheet);

  col=get_worksheet_col(worksheet,col_obj);
  if (col<0) return NULL;
                                                                                
  return set_sheet(0,col,object,worksheet,GTK_ORIENTATION_VERTICAL);
}
                                                                                
static PyObject *
set_row(PyObject *self, PyObject *args, PyObject *keywds)
{
  gchar *sheet=NULL, *content;
  static gchar *kwlist[]={"row","object","sheet",NULL};
  gint row=0,col=0,num,arow=-1,acol=-1,retval;
  SGworksheet *worksheet=NULL;
  gdouble value;
  PyObject *object,*row_obj;
  gboolean restore_active=FALSE;
                                                                                
  if (!PyArg_ParseTupleAndKeywords(args, keywds, "OO|s", kwlist,
                                   &row_obj, &object, &sheet))
      return NULL;
                                                                                
/* Get sheet pointer */
  worksheet = active_worksheet;
  if(!worksheet) return no_such_worksheet(sheet);
                                                                                
  row=get_worksheet_row(worksheet,row_obj);
  if (row<0) return NULL;
                                                                                
  return set_sheet(row,0,object,worksheet,GTK_ORIENTATION_HORIZONTAL);
}

int python_singleton(SGworksheet *worksheet, gint row, gint col,
                     PyObject *object, gboolean link, gboolean as_is)
{ 
  static PyObject *s_obj;
  GtkSheet *sheet;
  gchar *string = NULL;
  gchar fpnum[40],pspec[20];
  SGhiddencell *hidden;

  sheet = GTK_SHEET(worksheet);
  hidden = (SGhiddencell *)gtk_sheet_get_link(sheet, row, col);
  if (!hidden)
  {
      hidden = g_new(SGhiddencell, 1);
      hidden->formula = NULL;
      hidden->updated = FALSE;
      hidden->type =  worksheet->column[col].type;
      hidden->format =  worksheet->column[col].format;
      hidden->internal =  worksheet->column[col].internal;
      hidden->precision = worksheet->column[col].precision;
      gtk_sheet_link_cell(sheet, row, col, hidden);
  }

  fpnum[0]='\0';
  pspec[0]='\0';

  if (as_is)
   s_obj=object;
  else
   s_obj=PyObject_Str(object);

  if (s_obj)
   { 
     Py_INCREF(s_obj);
     if (object==Py_None)
     {   
         string=strdup("");
     }
     else
      switch(hidden->type){
        case SG_TYPE_NUMBER:       
          if (PyFloat_Check(object) || PyLong_Check(object) || PyInt_Check(object)){
             switch(hidden->internal){
               case SG_INTERNAL_DOUBLE:
                   if(PyFloat_Check(object))
                     hidden->value.val_double=PyFloat_AsDouble (object);
                   else if(PyLong_Check(object))
                     hidden->value.val_double=PyLong_AsDouble (object);
                   else if(PyInt_Check(object))
                     hidden->value.val_double=(gdouble)PyInt_AsLong (object);
                   break;
               case SG_INTERNAL_INTEGER:
                   if(PyFloat_Check(object))
                     hidden->value.val_long=(glong)PyFloat_AsDouble (object);
                   else if(PyLong_Check(object))
                     hidden->value.val_long=PyLong_AsLong (object);
                   else if(PyInt_Check(object))
                     hidden->value.val_long=PyInt_AsLong (object);
                   break;
             }
          } else {
             string=PyString_AsString (s_obj);
             switch(hidden->internal){
               case SG_INTERNAL_DOUBLE:
                   hidden->value.val_double=atof(string);
                   break;
               case SG_INTERNAL_INTEGER:
                   hidden->value.val_long=atoi(string);
                   break;
             }
          }
          break;
        case SG_TYPE_TEXT:        
        case SG_TYPE_TIME:        
        case SG_TYPE_DATE:        
        default:
          string=PyString_AsString (s_obj);
          hidden->value.val_char=g_strdup(string);
      }
      sg_worksheet_cell_update_format(worksheet,row, col);
  }
  Py_XDECREF(s_obj);

  if(hidden->formula)
    g_free(hidden->formula);
    
  if(string)
    hidden->formula = g_strdup(string);
  else 
    hidden->formula = NULL;
  
  return 1;
}

int python_sequence(SGworksheet *worksheet,gint row, gint col, PyObject *object,
                    GtkOrientation orient,gboolean link,gboolean as_is)
{ 
  gint len,i;
  PyObject *seq_obj;
  GtkSheet *sheet;

  sheet = GTK_SHEET(worksheet);

  

  /* Check to see if we need to resize the sheet */
  if (PyString_Check(object)||!PySequence_Check(object))
      python_singleton(worksheet,row,col,object,link,as_is);
  else
  {
     len=PySequence_Length(object);
     seq_obj=PySequence_GetItem(object,0);

     if (orient==GTK_ORIENTATION_VERTICAL && gtk_sheet_get_rows_count(sheet) < row+len)
              sg_worksheet_add_rows(worksheet,len-gtk_sheet_get_rows_count(sheet)+row);
     else if (orient==GTK_ORIENTATION_HORIZONTAL && gtk_sheet_get_columns_count(sheet) < col+len)
               sg_worksheet_add_columns(worksheet,len-gtk_sheet_get_columns_count(sheet)+col);
              
      for (i=0;i<len;i++)
       { 
         seq_obj=PySequence_GetItem(object,i);
         Py_INCREF(seq_obj);
         if (PySequence_Check(seq_obj) && !PyString_Check(seq_obj) )
         {  
            if (orient==GTK_ORIENTATION_VERTICAL)
                python_sequence(worksheet,row+i,col,seq_obj,GTK_ORIENTATION_HORIZONTAL,link,as_is);
            else
                python_sequence(worksheet,row,col+i,seq_obj,GTK_ORIENTATION_VERTICAL,link,as_is);
         }
         else
         { 
           if (orient==GTK_ORIENTATION_VERTICAL)
             python_singleton(worksheet,row+i,col,seq_obj,link,as_is);
           else
             python_singleton(worksheet,row,col+i,seq_obj,link,as_is);
         }
         Py_XDECREF(seq_obj);
       }
  }
  return 1;
}

int 
python_array_print(SGworksheet *worksheet,gint row, gint col,
                   PyArrayObject *object,gint dim, gchar *data,
                   GtkOrientation orient,gboolean link)
{ 
  gchar *array;
  gchar *apt;
  gint stride,i,j,vert,rowskip,curr_col;
  PyObject *single_object=NULL;
  gdouble aux;
  
  array=data;
  if (dim>1)
   { 
     rowskip=1;
     for (i=dim;i>0;i--)
       rowskip*=object->dimensions[i];
     stride=object->strides[dim];
     for (i=0;i<object->dimensions[i];i++)
       { 
         python_array_print(worksheet,row+i*rowskip,col,object,dim-1,array,orient,link);
         array+=stride;
       }
   }
  else
    { if (dim>0)
       { vert=object->dimensions[1];
         stride=object->strides[1];
       }
      else
       { vert=1;
         stride=0;
       }

    /* Now insert the cell values */
      array=data;

      for (j=0;j<vert;j++)
        { 
          for (i=0;i<object->dimensions[0];i++)
            { 
              apt=array+i*object->strides[0];
              if (orient==GTK_ORIENTATION_VERTICAL)
                  curr_col=col+j;
              else
                  curr_col=col+i;
              switch (object->descr->type_num)
               { 
                 case PyArray_CHAR:
                 case PyArray_SBYTE:
                 case PyArray_SHORT:
                 case PyArray_LONG:
                     single_object=PyLong_FromLong (*(long*) (array+i*object->strides[0]));
                     Py_INCREF(single_object);
                     break;
                 case PyArray_FLOAT:
                 case PyArray_CFLOAT:
                     aux = *(float*) (array+i*object->strides[0]);
                     single_object=PyFloat_FromDouble (aux);
                     Py_INCREF(single_object);
                     break;
                 case PyArray_DOUBLE:
                 case PyArray_CDOUBLE:
                     aux = *(double*) (array+i*object->strides[0]);
                     single_object=PyFloat_FromDouble (aux);
                     Py_INCREF(single_object);
                     break;
                 case PyArray_OBJECT:
                  python_sequence(worksheet,row+j,col+i,
                                  (PyObject *)(array+i*object->strides[0]),orient,link,FALSE);
                  break;
               }
               if (object->descr->type_num!=PyArray_OBJECT)
               { 
                 python_singleton(worksheet,row+i,col+j,
                                  single_object,link,FALSE);
                 Py_XDECREF(single_object);
               }
          }
          array+=stride;
        }
    }
  return 1;
}

int 
python_array(SGworksheet *worksheet, gint row, gint col,
             PyArrayObject *object, GtkOrientation orient, gboolean link)
{ 
  gint len,i;
  GtkSheet *sheet;

  sheet = GTK_SHEET(worksheet);
  if (object->nd > 2) return 0; /* No nore than 2 dimensions for now */

  /* Check to see if we need to resize the sheet */
  if (orient == GTK_ORIENTATION_VERTICAL)
  {
      if (object->nd > 1 && 
          gtk_sheet_get_columns_count(sheet) < object->dimensions[1]+col+1)
         sg_worksheet_add_columns(worksheet,
                                  object->dimensions[1]-
                                  gtk_sheet_get_columns_count(sheet)+col);
      len=1;
      for (i = 0; i < object->nd; i++)
        if (i != 1) len = len*object->dimensions[i];
    
      if (gtk_sheet_get_rows_count(sheet) < row+len)
          sg_worksheet_add_rows(worksheet,len-gtk_sheet_get_rows_count(sheet)+row);
  }
  else
  {
      if (object->nd > 1 && 
          gtk_sheet_get_rows_count(sheet)< object->dimensions[1]+row+1)
         sg_worksheet_add_rows(worksheet,
                               object->dimensions[1]-
                               gtk_sheet_get_rows_count(sheet)+row);
      len=1;
      for (i = 0; i < object->nd; i++)
        if (i != 1) len = len*object->dimensions[i];
    
      if (gtk_sheet_get_columns_count(sheet) < col+len)
          sg_worksheet_add_columns(worksheet,
                                   len-gtk_sheet_get_columns_count(sheet)+col);

  }

  python_array_print(worksheet, row, col, object, object->nd - 1, object->data,
                     orient, link);

  return 1;
}


int 
python_insert_object(SGworksheet *worksheet, gint row, gint col,
                     PyObject *object, GtkOrientation orient,
                     gboolean link, gboolean as_is)
{
  GtkSheet *sheet;
  gint retval;

  sheet = GTK_SHEET(worksheet);

  if (object)
  {
#ifdef WITH_NUMERIC_PYTHON
      if (PyArray_Check(object))
       { 
         gtk_sheet_freeze(sheet);
         retval = python_array(worksheet, row, col, (PyArrayObject *)object, orient, link);
         gtk_sheet_thaw(sheet);
       }
      else
#endif
      if (PySequence_Check(object))
       { 
         gtk_sheet_freeze(sheet);
         retval = python_sequence(worksheet, row, col, object, orient, link, as_is);
         gtk_sheet_thaw(sheet);
       }

      else
         retval = python_singleton(worksheet, row, col, object, link, as_is);

   }
  else
      retval=0;

  return retval;
}

int 
python_sheet(SGworksheet *worksheet, gint row, gint col, gchar *command,
             GtkOrientation orient)
{ 
  PyObject *object;
  gchar errmsg[] = "Error!";
  gchar *msg;
  gint retval;

  active_worksheet = worksheet;

  object = PyRun_String (command, Py_eval_input, main_dict, sg_dict);

  if (object)
  {  
    Py_INCREF(object);
    retval = python_insert_object(worksheet, row, col, object, orient, FALSE, FALSE);
    Py_XDECREF(object);
  }
  else
  { 
    if (PyErr_Occurred()) PyErr_Clear();
    PyRun_String (command, Py_single_input, main_dict, sg_dict);
    retval = 0;
  }

  if (retval) return 1;

  if (PyErr_Occurred())
  { 
     msg = errmsg;
     PyErr_Print();
     if (Py_FlushLine()) PyErr_Clear();
     PyErr_Clear();
  }
  else
  {
    SGhiddencell *link = (SGhiddencell*)gtk_sheet_get_link(GTK_SHEET(worksheet), row, col);
    if (link && link->formula)
    {
      g_free(link->formula);
      link->formula = NULL;
    }
    msg = command;
  }

  sg_worksheet_cell_set_text(worksheet,row,col,msg);
  return 0;
}
