/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <stdlib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_plugin.h"
#include "sg_plugin_menu.h"
#include "sg_dataset.h"

static void sg_plugin_menu_class_init             (SGpluginMenuClass *klass);
static void sg_plugin_menu_init                   (SGpluginMenu *plugin);
static void sg_plugin_menu_finalize                (GObject *object);
static GtkWidget *find_submenu			  (GtkWidget *widget, 
						   gchar *label);
static void menu_activate			  (GtkWidget *widget, 
						   gpointer data);
static void toolbar_activate			  (GtkWidget *widget, 
						   gpointer data);

static SGpluginClass *parent_class = NULL;

GType
sg_plugin_menu_get_type (void)
{
  static GType sg_plugin_menu_type = 0;
                                                                                
  if (!sg_plugin_menu_type)
    {
      static const GTypeInfo sg_plugin_menu_info =
      {
        sizeof (SGpluginMenuClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) sg_plugin_menu_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (SGpluginMenu),
        0,              /* n_preallocs */
        (GInstanceInitFunc) sg_plugin_menu_init,
        NULL,
      };
                                                                                
      sg_plugin_menu_type = g_type_register_static (GTK_TYPE_SG_PLUGIN, "SGpluginMenu",
                                               &sg_plugin_menu_info, 0);
    }
                                                                                
  return sg_plugin_menu_type;
}

static void
sg_plugin_menu_class_init (SGpluginMenuClass *klass)
{
  GObjectClass *object_class;
  SGpluginClass *plugin_class;

  parent_class = (SGpluginClass *) g_type_class_peek_parent (klass);

  object_class = (GObjectClass *) klass;
  plugin_class = (SGpluginClass *) klass;

  object_class->finalize = sg_plugin_menu_finalize;
}


SGpluginMenu *
sg_plugin_menu_new ()
{
  SGpluginMenu *plugin;
  plugin = SG_PLUGIN_MENU(g_object_new(sg_plugin_menu_get_type(), NULL));
  return plugin;
}

static void
sg_plugin_menu_init(SGpluginMenu *plugin)
{
  plugin->path = NULL;
  plugin->action = NULL;
  plugin->pixmap = NULL;
  plugin->owner_id = NULL;
}

static void
sg_plugin_menu_finalize(GObject *object)
{
  SGpluginMenu *plugin = SG_PLUGIN_MENU(object);

  if(plugin->path) g_free(plugin->path);
  plugin->path = NULL;
  if(plugin->owner_id) g_free(plugin->owner_id);
  plugin->owner_id = NULL;
  if(plugin->pixmap && GTK_IS_WIDGET(plugin->pixmap)) 
    gtk_widget_destroy(GTK_WIDGET(plugin->pixmap));
  plugin->pixmap = NULL;
  G_OBJECT_CLASS(parent_class)->finalize(object);
}

void
sg_plugin_menu_action(SGpluginMenu *plugin, GtkWidget *menu, gpointer data)
{
  if(!plugin->action) return;
  plugin->action(plugin, menu, data);
}

SGpluginMenu *
sg_plugin_menu_get(const gchar *name, const gchar *owner_id)
{
  GList *list = sg_plugins();
  while(list){
    if(GTK_IS_SG_PLUGIN_MENU(list->data)){
      SGpluginMenu *c = SG_PLUGIN_MENU(list->data);
      if(strcmp(SG_PLUGIN(c)->name, name) == 0 && strcmp(c->owner_id, owner_id) == 0) return c;
    }
    list = list->next;
  }
  return NULL;
}

void
sg_menu_add_plugin(GtkWidget *menu, SGpluginMenu *plugin)
{
  GtkWidget *sub_menu = NULL;
  gchar *groups[100];
  gchar *subgroup = NULL;
  guint i, n, nc;
  gchar *group;

/* parse group */
  group = plugin->path;

  n = nc = 0;
  subgroup = (gchar *)g_malloc(sizeof(gchar));
  for(i = 0; i < strlen(group); i++){
    if(group[i] == ':'){
      groups[n] = g_strdup(subgroup);
      g_free(subgroup);
      subgroup = (gchar *)g_malloc(sizeof(gchar));
      n++;
      nc = 0;
    }else{
      nc++;
      subgroup = (gchar *)g_realloc(subgroup, (nc + 1) * sizeof(gchar));
      subgroup[nc - 1] = group[i];
      subgroup[nc] = '\0';
    }
  }
  if(subgroup) g_free(subgroup);

  sub_menu = menu;
  for(i = 1; i < n; i++){
    GtkWidget *child;
    child = find_submenu(sub_menu, groups[i]);
    if(!child){
      GtkWidget *item;

      child = gtk_menu_new();
      item = gtk_menu_item_new_with_label(groups[i]);
      gtk_widget_show(item);
      if(GTK_IS_MENU_SHELL(sub_menu))
        gtk_menu_shell_append (GTK_MENU_SHELL (sub_menu), item);
      else
        gtk_menu_append(GTK_MENU(sub_menu),item);
      gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), child);
    }
    sub_menu = child;
  }
  if(sub_menu){
     GtkWidget *item;
     GtkWidget *label;

     item = gtk_image_menu_item_new_with_label(plugin->label);
     if(plugin->pixmap) {
       GtkPixmap *pixmap;
       pixmap = GTK_PIXMAP(gtk_pixmap_new(plugin->pixmap->pixmap, plugin->pixmap->mask));
       gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item), GTK_WIDGET(pixmap));
       gtk_widget_ref(GTK_WIDGET(plugin->pixmap));
     }
     gtk_widget_show_all(item);
     g_object_set_data(G_OBJECT(item), "plugin", plugin);
     g_object_set_data(G_OBJECT(item), "menu", menu);
     gtk_menu_append(GTK_MENU(sub_menu),item);
     gtk_signal_connect(GTK_OBJECT(item),"activate",
                        GTK_SIGNAL_FUNC(menu_activate), plugin);
  }

  for(i = 0; i < n; i++)
       if(groups[i]) g_free(groups[i]);
}

void
sg_toolbar_add_plugin(GtkWidget *widget, SGpluginMenu *plugin)
{
  GtkWidget *item;
  GtkToolbar *tb = GTK_TOOLBAR(widget);
  gint i, n;
  GList *children;
  gint group_id = atoi(plugin->path);
  GtkWidget *pixmap;
  
  n = 0;
  children = tb->children;
  while(children){
    GtkToolbarChild *child = (GtkToolbarChild *)children->data;;
    if(child->type == GTK_TOOLBAR_CHILD_SPACE) n++;
    children = children->next;
  }

/*
  gtk_toolbar_set_space_style(tb, GTK_TOOLBAR_SPACE_LINE);
*/
  for(i = n; i < group_id ; i++) gtk_toolbar_append_space(tb);

  i = n = 0;
  children = tb->children;
  while(children){
    GtkToolbarChild *child = (GtkToolbarChild *)children->data;;
    i++;
    if(child->type == GTK_TOOLBAR_CHILD_SPACE) n++;
    if(n > group_id) break;
    children = children->next;
  }
  pixmap = gtk_pixmap_new(plugin->pixmap->pixmap, plugin->pixmap->mask);
  item = gtk_toolbar_insert_item(tb, 
	       	                 SG_PLUGIN_MENU(plugin)->label,
	       		         SG_PLUGIN(plugin)->description,
	       		         SG_PLUGIN(plugin)->description,
				 GTK_WIDGET(pixmap), 
	       		         GTK_SIGNAL_FUNC(toolbar_activate), plugin, i); 
}


static GtkWidget *
find_submenu(GtkWidget *widget, gchar *label)
{
  GList *list;

  list = GTK_MENU_SHELL(widget)->children;
  while(list){
    GtkWidget *sub_item;
    gchar *item_label;
    sub_item = GTK_WIDGET(list->data);
    item_label = GTK_LABEL(GTK_BIN(sub_item)->child)->label;

    if(strcmp(item_label, label) == 0) return GTK_MENU_ITEM(sub_item)->submenu;
    list = list->next;
  }
  return NULL;
}

static void
menu_activate(GtkWidget *widget, gpointer data)
{
  SGpluginMenu *plugin;
  GtkWidget *menu;

  plugin = SG_PLUGIN_MENU(g_object_get_data(G_OBJECT(widget), "plugin"));
  menu = GTK_WIDGET(g_object_get_data(G_OBJECT(widget), "menu"));
  sg_plugin_menu_action(plugin, menu, NULL);
}

static void
toolbar_activate(GtkWidget *widget, gpointer data)
{
  SGpluginMenu *plugin;
  GtkWidget *tb;

  plugin = SG_PLUGIN_MENU(data);
  tb = widget->parent;
  sg_plugin_menu_action(plugin, tb, NULL);
}
