/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <libxml/xmlreader.h>
#include <gmodule.h>
#include "sg.h"
#include "sg_plugin_formula.h"
#include "sg_plugin_function.h"
#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#endif


#ifdef G_OS_WIN32
#ifndef S_ISDIR
#define S_ISDIR(m) (((m) & _S_IFMT) == _S_IFDIR)
#define S_ISREG(m) (((m) & _S_IFMT) == _S_IFREG)
#define S_ISLNK(m) (0)
#elif defined(_WIN32)
#define S_ISLNK(m) (0)
#endif
#define lstat(f,s) stat(f,s)
#endif /* G_OS_WIN32 */

typedef void (*plugin_init_func) (SGplugin *);

static GSList *
sg_plugin_extra_dirs (void)
{
  GSList *extra_dirs = NULL;
  const gchar *plugin_path_env;

  plugin_path_env = g_getenv ("SG_PLUGIN_PATH");
  if (plugin_path_env != NULL) {
    gint nc = 0;
    gchar *sub_path = g_new0(gchar, 1);
    guint i;

    for(i = 0; i < strlen(plugin_path_env); i++){
      if(plugin_path_env[i] == ':'){
        extra_dirs = g_slist_append (extra_dirs, g_strdup(sub_path));
        g_free(sub_path);
        sub_path = g_new0(gchar, 1);
        nc = 0;
      }else{
        nc++;
        sub_path = (gchar *)g_realloc(sub_path, (nc + 1) * sizeof(gchar));
        sub_path[nc - 1] = plugin_path_env[i];
        sub_path[nc] = '\0';
      }
    }
    if(sub_path) g_free(sub_path);
  }

  return extra_dirs;
}


static void
process_node(xmlTextReaderPtr reader, const gchar *dir_name, gchar *module, SGplugin **_plugin, gchar *last_node)
{
  xmlChar *name = xmlTextReaderName(reader);

  /* Start Element */

  if(xmlTextReaderNodeType(reader) == 1){
    g_snprintf(last_node, 1000, "%s", name);

    if(strcmp(name,"module") == 0){
      *_plugin = NULL;
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);
  
        if(strcmp(child, "value") == 0){
          g_snprintf(module, 2000, "%s%s%s", dir_name, G_DIR_SEPARATOR_S, value);
        }
  
        xmlFree(child);
        xmlFree(value);
      }
    }
  
    if(strcmp(name,"plugin") == 0){
      gchar *type = NULL;
      gchar *plugin_name = NULL;
      gchar *action = NULL;
      gchar *object = NULL;
      gchar *iterator = NULL;
      gchar *constructor = NULL;
      GModule *handle = NULL;
      gchar *symbol_name = NULL;
      gchar *owner_id = NULL;
      gchar *path = NULL;
      gchar *label = NULL;
      gchar *description = NULL;
      gchar *layer = NULL;
  
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);
  
        if(strcmp(child, "type") == 0) type = g_strdup(value);
        if(strcmp(child, "name") == 0) plugin_name = g_strdup(value);
        if(strcmp(child, "action") == 0) action = g_strdup(value);
        if(strcmp(child, "object") == 0) object = g_strdup(value);
        if(strcmp(child, "iterator") == 0) iterator = g_strdup(value);
        if(strcmp(child, "constructor") == 0) constructor = g_strdup(value);
        if(strcmp(child, "path") == 0) path = g_strdup(value);
        if(strcmp(child, "owner_id") == 0) owner_id = g_strdup(value);
        if(strcmp(child, "label") == 0) label = g_strdup(value);
        if(strcmp(child, "description") == 0) description = g_strdup(value);
        if(strcmp(child, "layer") == 0) layer = g_strdup(value);
  
        xmlFree(child);
        xmlFree(value);
      }

      if(type && plugin_name){
	plugin_init_func plugin_file_init_call = NULL;
	plugin_init_func plugin_style_init_call = NULL;
	plugin_init_func plugin_iterator_init_call = NULL;
	plugin_init_func plugin_menu_init_call = NULL;
	plugin_init_func plugin_init_call = NULL;
	gpointer plugin_construct_func = NULL;
	gpointer plugin_toolbox_func = NULL;
	gpointer plugin_action_func = NULL;
	gpointer plugin_dialog_func = NULL;
	gpointer plugin_new_dialog_func = NULL;
	gpointer plugin_edit_dialog_func = NULL;
	gpointer plugin_edit_data_dialog_func = NULL;

        handle = g_module_open(module, G_MODULE_BIND_LAZY);

        if(handle && strcmp(type, "io") == 0 && action && object) {
	  SGpluginFileMode mode = 0;

	  if(strcmp(action, "import") == 0) mode = SG_PLUGIN_FILE_IMPORT;
	  else if(strcmp(action, "open") == 0) mode = SG_PLUGIN_FILE_OPEN;
	  else if(strcmp(action, "export") == 0) mode = SG_PLUGIN_FILE_EXPORT;
	  else if(strcmp(action, "save") == 0) mode = SG_PLUGIN_FILE_SAVE;
	  else if(strcmp(action, "save_as") == 0) mode = SG_PLUGIN_FILE_SAVE_AS;
	  else if(strcmp(action, "print") == 0) mode = SG_PLUGIN_FILE_PRINT;
	
	  symbol_name = g_strconcat(object, "_", plugin_name, "_", action, "_init", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_file_init_call);
	  g_free(symbol_name);
	  symbol_name = g_strconcat(object, "_", plugin_name, "_", action, NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_action_func);
	  g_free(symbol_name);
          if(plugin_file_init_call && plugin_action_func){
	    SGpluginFile *plugin;

	    plugin = sg_plugin_file_new();
	    *_plugin = SG_PLUGIN(plugin);
	    SG_PLUGIN(*_plugin)->handle = handle;
	    sg_plugin_set_name(SG_PLUGIN(plugin), plugin_name);
	    sg_plugin_file_set_object(plugin, object);
	    plugin->mode = mode;
	    plugin_file_init_call(SG_PLUGIN(plugin));
	    plugin->action = plugin_action_func;
	    sg_plugin_add(SG_PLUGIN(plugin));
	  } else {
	    *_plugin = NULL;
	  }
        }
        if(handle && strcmp(type, "style") == 0) {
	  symbol_name = g_strconcat(plugin_name, "_init", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_style_init_call);
	  g_free(symbol_name);
	  symbol_name = g_strconcat(plugin_name, "_construct", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_construct_func);
	  g_free(symbol_name);
	  symbol_name = g_strconcat(plugin_name, "_dialog", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_dialog_func);
	  g_free(symbol_name);

          if(plugin_style_init_call && plugin_construct_func){
	    SGpluginStyle *plugin;
	    plugin = sg_plugin_style_new();
	    *_plugin = SG_PLUGIN(plugin);
	    SG_PLUGIN(*_plugin)->handle = handle;
	    sg_plugin_set_name(SG_PLUGIN(plugin), plugin_name);
	    plugin_style_init_call(SG_PLUGIN(plugin));
            if(plugin->layer && layer) g_free(plugin->layer);
            if(layer) plugin->layer = g_strdup(layer);
	    plugin->construct = plugin_construct_func;
	    plugin->property_dialog = plugin_dialog_func;
	    sg_plugin_add(SG_PLUGIN(plugin));
	  } else {
	    *_plugin = NULL;
	  }
        }
        if(handle && strcmp(type, "iterator") == 0) {
	  symbol_name = g_strconcat(plugin_name, "_init", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_iterator_init_call);
	  g_free(symbol_name);
	  symbol_name = g_strconcat(plugin_name, "_construct", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_construct_func);
	  g_free(symbol_name);
	  symbol_name = g_strconcat(plugin_name, "_new_dialog", NULL);


          g_module_symbol(handle, symbol_name, (gpointer) &plugin_new_dialog_func);
/*
	  g_print ("retrived symbol `%s' as %p\n", symbol_name, plugin_new_dialog_func);
*/
	  g_free(symbol_name);
	  symbol_name = g_strconcat(plugin_name, "_edit_dialog", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_edit_dialog_func);
	  g_free(symbol_name);
	  symbol_name = g_strconcat(plugin_name, "_edit_data_dialog", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_edit_data_dialog_func);
/*
	  g_print ("retrived symbol `%s' as %p\n", symbol_name, plugin_edit_dialog_func);
*/
	  g_free(symbol_name);
          if(plugin_new_dialog_func){
	    SGpluginIterator *plugin;
	    plugin = sg_plugin_iterator_new();
	    *_plugin = SG_PLUGIN(plugin);
	    SG_PLUGIN(*_plugin)->handle = handle;
	    sg_plugin_set_name(SG_PLUGIN(plugin), plugin_name);
	    plugin_iterator_init_call(SG_PLUGIN(plugin));
	    plugin->construct = plugin_construct_func;
	    plugin->new_dataset_dialog = plugin_new_dialog_func;
	    plugin->edit_dataset_dialog = plugin_edit_dialog_func;
	    plugin->edit_datapoints_dialog = plugin_edit_data_dialog_func;
	    sg_plugin_add(SG_PLUGIN(plugin));
	  } else {
	    *_plugin = NULL;
	  }
        }
        if(handle && strcmp(type, "menu") == 0) {
	  symbol_name = g_strconcat(plugin_name, "_init", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_menu_init_call);
	  g_free(symbol_name);
	  symbol_name = g_strdup(plugin_name);
          g_module_symbol(handle, plugin_name, (gpointer) &plugin_action_func);
	  g_free(symbol_name);
	  if(plugin_action_func){
            SGpluginMenu *plugin;
	    plugin = sg_plugin_menu_new();
	    *_plugin = SG_PLUGIN(plugin);
	    SG_PLUGIN(*_plugin)->handle = handle;
	    sg_plugin_set_name(SG_PLUGIN(plugin), plugin_name);
	    sg_plugin_set_description(SG_PLUGIN(plugin), description);
	    plugin->owner_id = g_strdup(owner_id);
	    plugin->path = g_strdup(path);
	    plugin->label = g_strdup(label);
	    plugin->action = plugin_action_func;
	    if(plugin_menu_init_call) plugin_menu_init_call(SG_PLUGIN(plugin));
	    sg_plugin_add(SG_PLUGIN(plugin));
	  } else {
	    *_plugin = NULL;
	  }
	}
        if(handle && strcmp(type, "layer") == 0) {
	  symbol_name = g_strconcat(plugin_name, "_init", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_init_call);
	  g_free(symbol_name);
	  symbol_name = g_strconcat(plugin_name, "_construct", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_construct_func);
	  g_free(symbol_name);
	  symbol_name = g_strconcat(plugin_name, "_dialog", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_dialog_func);
	  g_free(symbol_name);
	  symbol_name = g_strconcat(plugin_name, "_toolbox", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_toolbox_func);
	  g_free(symbol_name);

          if(plugin_init_call && plugin_construct_func){
	    SGpluginLayer *plugin;
	    plugin = sg_plugin_layer_new();
	    *_plugin = SG_PLUGIN(plugin);
	    SG_PLUGIN(*_plugin)->handle = handle;
	    sg_plugin_set_name(SG_PLUGIN(plugin), plugin_name);
	    plugin_init_call(SG_PLUGIN(plugin));
	    plugin->construct = plugin_construct_func;
	    plugin->property_dialog = plugin_dialog_func;
	    plugin->toolbox = plugin_toolbox_func;
	    sg_plugin_add(SG_PLUGIN(plugin));
	  } else {
	    *_plugin = NULL;
	  }
        }
        if(handle && strcmp(type, "array") == 0) {
	  symbol_name = g_strdup(plugin_name);
          g_module_symbol(handle, plugin_name, (gpointer) &plugin_action_func);
	  g_free(symbol_name);
	  if(plugin_action_func){
            SGpluginArray *plugin;
	    plugin = sg_plugin_array_new();
	    *_plugin = SG_PLUGIN(plugin);
	    SG_PLUGIN(*_plugin)->handle = handle;
	    sg_plugin_set_name(SG_PLUGIN(plugin), plugin_name);
	    sg_plugin_set_description(SG_PLUGIN(plugin), description);
	    plugin->action = plugin_action_func;
	    sg_plugin_add(SG_PLUGIN(plugin));
	  } else {
	    *_plugin = NULL;
	  }
	}
        if(handle && strcmp(type, "formula") == 0) {
	  symbol_name = g_strconcat(plugin_name, "_init", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_init_call);
	  g_free(symbol_name);

          if(plugin_init_call){
	    SGpluginFormula *plugin;
	    plugin = sg_plugin_formula_new();
	    *_plugin = SG_PLUGIN(plugin);
	    SG_PLUGIN(*_plugin)->handle = handle;
	    sg_plugin_set_name(SG_PLUGIN(plugin), plugin_name);
	    plugin_init_call(SG_PLUGIN(plugin));
	    sg_plugin_add(SG_PLUGIN(plugin));
	  } else {
	    *_plugin = NULL;
	  }
        }
        if(handle && strcmp(type, "function") == 0) {
	  symbol_name = g_strconcat(plugin_name, "_init", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_init_call);
	  g_free(symbol_name);

	  symbol_name = g_strconcat(plugin_name, "_action", NULL);
          g_module_symbol(handle, symbol_name, (gpointer) &plugin_action_func);
	  g_free(symbol_name);

          if(plugin_init_call){
	    SGpluginFunction *plugin;
	    plugin = sg_plugin_function_new();
	    *_plugin = SG_PLUGIN(plugin);
	    SG_PLUGIN(*_plugin)->handle = handle;
	    sg_plugin_set_name(SG_PLUGIN(plugin), plugin_name);
	    if(plugin_init_call) plugin_init_call(SG_PLUGIN(plugin));
	    if(plugin_action_func) plugin->action = plugin_action_func;
	    sg_plugin_add(SG_PLUGIN(plugin));
	  } else {
            *_plugin = NULL;
          }
        }
 
      }
      
      if(type) g_free(type);
      if(plugin_name) g_free(plugin_name);
      if(action) g_free(action);
      if(object) g_free(object);
      if(iterator) g_free(iterator);
      if(constructor) g_free(constructor);
      if(path) g_free(path);
      if(owner_id) g_free(owner_id);
      if(label) g_free(label);
    }
  }

    /* End Element */

  if(xmlTextReaderNodeType(reader) == 15){
    sprintf(last_node, " ");
  }

  /* Text node */

  if(xmlTextReaderNodeType(reader) == 3){
    xmlChar *value = xmlTextReaderValue(reader);
    if(!value) return;

    if(*_plugin && strcmp(last_node, "description") == 0){
      sg_plugin_set_description(*_plugin, value);
    }
    if(*_plugin && strcmp(last_node, "label") == 0){
      SG_PLUGIN_MENU(*_plugin)->label = g_strdup(value);
    }
    xmlFree(value);
  }
}

static gboolean
sg_plugin_load(const gchar *dir_name, const gchar *file_name)
{
  xmlTextReaderPtr reader;
  gint ret_val;
  gchar module[2000];
  gchar last_node[2000];
  SGplugin *plugin = NULL;

  reader = xmlNewTextReaderFilename(file_name);

  if(!reader) return FALSE;

  ret_val = xmlTextReaderRead(reader);

  while(ret_val == 1){
    xmlChar *name = xmlTextReaderName(reader);

    process_node(reader, dir_name, module, &plugin, last_node);

    if(xmlTextReaderNodeType(reader) == 15 && strcmp(name, "module") == 0){
      if(name) xmlFree(name);
      return TRUE;
    }

    xmlFree(name);
    ret_val = xmlTextReaderRead(reader);
  }
  xmlFreeTextReader(reader);

  if(ret_val != 0) return FALSE;

  return TRUE;

}

static void
sg_plugin_read_dir(const gchar *dir_name)
{
  DIR *dir;
  struct dirent *entry;
  struct stat fileinfo;

  dir = opendir (dir_name);
  if (dir == NULL) {
    return;
  }

  while((entry=readdir(dir))!=NULL){
    gchar *full_name;

    full_name = g_strconcat(dir_name, G_DIR_SEPARATOR_S, entry->d_name, NULL);
    stat(full_name, &fileinfo);
    if(S_ISDIR(fileinfo.st_mode) || strcmp(entry->d_name,"plugin.xml") != 0) {
      g_free(full_name);
      continue;
    }

    sg_plugin_load(dir_name, full_name);
    
  }
  closedir (dir);
}

static void
sg_plugin_read_for_subdirs(const gchar *dir_name)
{
  DIR *dir;
  struct dirent *entry;
  struct stat fileinfo;

  g_return_if_fail (dir_name != NULL);

  dir = opendir (dir_name);
  if (dir == NULL) {
    return;
  }
  while ((entry = readdir (dir)) != NULL) {
    gchar *full_name;

    if (strcmp (entry->d_name, ".") == 0 || strcmp (entry->d_name, "..") == 0) {
      continue;
    }
    full_name = g_strconcat(dir_name, G_DIR_SEPARATOR_S, entry->d_name, NULL);
    stat(full_name, &fileinfo);
    if(S_ISDIR(fileinfo.st_mode)){
      sg_plugin_read_dir (full_name);
      sg_plugin_read_for_subdirs (full_name);
    }

    g_free (full_name);
  }
  closedir (dir);
}

gboolean
sg_plugin_install ()
{
  GSList *dir_list = NULL;
  GSList *dir_iterator;

  if(!g_module_supported()){
    return FALSE; 
  }

  dir_list = g_slist_append (dir_list, sg_sys_plugin_dir ());
  dir_list = g_slist_append (dir_list, sg_usr_plugin_dir ());
  dir_list = g_slist_concat (dir_list, sg_plugin_extra_dirs ());

  for (dir_iterator = dir_list; dir_iterator != NULL; dir_iterator = dir_iterator->next) {
    gchar *dir_name;

    dir_name = (gchar *) dir_iterator->data;
    sg_plugin_read_for_subdirs (dir_name);
  }

  g_slist_foreach (dir_list, (GFunc)g_free, NULL);
  g_slist_free (dir_list);

  return TRUE;
}

