/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_layer_control.h"
#include "sg.h"
#include "sg_application.h"
#include "sg_property_dialog.h"
#include "sg_axis_dialog.h"
#include "sg_break_dialog.h"
#include "sg_dataset_dialog.h"
#include "sg_layer_dataset_dialog.h"
#include "sg_frame_dialog.h"
#include "sg_grids_dialog.h"
#include "sg_labels_dialog.h"
#include "sg_layer_dialog.h"
#include "sg_legends_dialog.h"
#include "sg_page_dialog.h"
#include "sg_planes_dialog.h"
#include "sg_title_dialog.h"
#include "sg_stock.h"
#include "pixmaps/layer.xpm"
#include "pixmaps/sg.xpm"
#include "pixmaps/2d_small.xpm"
#include "pixmaps/3d_small.xpm"
#include "pixmaps/polar_small.xpm"

#define LABEL_LEN 1000

static void 		sg_layer_control_destroy	(GtkObject *object); 
static void 		sg_layer_control_class_init	(SGlayerControlClass *klass);
static void 		sg_layer_control_init		(SGlayerControl *dialog);
static void 		sg_layer_control_init_gui	(SGlayerControl *dialog);
static void 		sg_layer_control_realize	(GtkWidget *dialog);
static void 		build_tree			(SGlayerControl *dialog);
static void 		build_branch			(SGlayerControl *dialog,
							 SGplot *plot);
static void 		expand_tree			(SGlayerControl *dialog,
							 const gchar *path);
static void 		open_dialog			(GtkCTree *tree, 
						 	 GtkCTreeNode *node, 
						 	 gint column,
							 gpointer data);


static GtkWindowClass *parent_class = NULL;

GtkType
sg_layer_control_get_type (void)
{
  static GtkType sg_layer_control_type = 0;

  if (!sg_layer_control_type)
    {
      GtkTypeInfo sg_layer_control_info =
      {
        "SGlayerControl",
        sizeof (SGlayerControl),
        sizeof (SGlayerControlClass),
        (GtkClassInitFunc) sg_layer_control_class_init,
        (GtkObjectInitFunc) sg_layer_control_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_layer_control_type = gtk_type_unique (gtk_window_get_type(), &sg_layer_control_info);
    }

  return sg_layer_control_type;
}

static void
sg_layer_control_class_init (SGlayerControlClass *klass)
{
  GtkWidgetClass *widget_class;
  GtkObjectClass *object_class;

  widget_class = (GtkWidgetClass*) klass;
  object_class = (GtkObjectClass*) klass;

  parent_class = (GtkWindowClass *)gtk_type_class (gtk_window_get_type ());

  object_class->destroy = sg_layer_control_destroy;
  widget_class->realize = sg_layer_control_realize;
}

GtkWidget *
sg_layer_control_new(SGplot *plot)
{
   GtkWidget *widget;
   SGlayerControl *dialog;

   widget = gtk_widget_new(sg_layer_control_get_type(), NULL);
   dialog = SG_LAYER_CONTROL(widget);

   dialog->plot = plot;
   sg_layer_control_init_gui(dialog);
   sg_layer_control_refresh(widget, NULL);

   return widget;
}  

GtkWidget *
sg_layer_control_new_application(SGapplication *app)
{
   GtkWidget *widget;
   SGlayerControl *dialog;

   widget = gtk_widget_new(sg_layer_control_get_type(), NULL);
   dialog = SG_LAYER_CONTROL(widget);

   dialog->application = app;
   sg_layer_control_init_gui(dialog);
   sg_layer_control_refresh(widget, NULL);

   return widget;
}  

static void
ctree_destroy(GtkWidget *widget, gpointer data)
{
  SGlayerControl *dialog = SG_LAYER_CONTROL(data);
  GList *list;
  SGnodeData *node;

  list = dialog->nodes;
  while(list){
     node = (SGnodeData *) list->data;
     g_free(node->path);
     g_free(node);
     dialog->nodes = g_list_remove_link(dialog->nodes, list);
     g_list_free_1(list);
     list = dialog->nodes;
  }
  dialog->nodes = NULL;
}

static void 
sg_layer_control_realize(GtkWidget *widget)
{
 GdkPixmap *pixmap;
 GdkBitmap *mask;

 GTK_WIDGET_CLASS(parent_class)->realize(widget);
                                                                                
 pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                                gdk_colormap_get_system(),
                                                &mask, NULL, sg_xpm);
                                                                                
 gdk_window_set_icon(widget->window, NULL, pixmap, mask);
}

static void 
sg_layer_control_destroy(GtkObject *object)
{
  SGlayerControl *dialog = SG_LAYER_CONTROL(object);

  if(dialog->layer_pixmap)
    gdk_pixmap_unref(dialog->layer_pixmap);
  if(dialog->plot2d_pixmap)
    gdk_pixmap_unref(dialog->plot2d_pixmap);
  if(dialog->plot3d_pixmap)
    gdk_pixmap_unref(dialog->plot3d_pixmap);
  if(dialog->polar_pixmap)
    gdk_pixmap_unref(dialog->polar_pixmap);
  if(dialog->layer_mask)
    gdk_bitmap_unref(dialog->layer_mask);
  if(dialog->plot2d_mask)
    gdk_bitmap_unref(dialog->plot2d_mask);
  if(dialog->plot3d_mask)
    gdk_bitmap_unref(dialog->plot3d_mask);
  if(dialog->polar_mask)
    gdk_bitmap_unref(dialog->polar_mask);
  dialog->layer_pixmap = NULL;
  dialog->plot2d_pixmap = NULL;
  dialog->plot3d_pixmap = NULL;
  dialog->polar_pixmap = NULL;
  dialog->layer_mask = NULL;
  dialog->plot2d_mask = NULL;
  dialog->plot3d_mask = NULL;
  dialog->polar_mask = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);

  dialog->main_frame = NULL;
  dialog->dialog = NULL;
  dialog->tree = NULL;
  if(dialog->plot) dialog->plot->layer_control = NULL;
  if(dialog->application) dialog->application->layer_control = NULL;
}

void
sg_layer_control_open(GtkWidget *widget, const gchar *path)
{
  expand_tree(SG_LAYER_CONTROL(widget), path);
}

void
sg_layer_control_clear_paned(SGlayerControl *dialog)
{
  if(GTK_BIN(dialog->main_frame)->child != NULL && GTK_IS_WIDGET(GTK_BIN(dialog->main_frame)->child))
    gtk_container_remove(GTK_CONTAINER(dialog->main_frame),GTK_BIN(dialog->main_frame)->child);
  GTK_BIN(dialog->main_frame)->child == NULL;
}

static void
button_press_apply(GtkWidget *widget, gpointer data)
{
  SGlayerControl *dialog = SG_LAYER_CONTROL(widget);
}

static void
button_press_ok(GtkWidget *widget, gpointer data)
{
  button_press_apply(widget, data);
  if(widget && GTK_IS_WIDGET(widget)) gtk_widget_destroy(widget);
}

static void
button_press_cancel(GtkWidget *widget, gpointer data)
{
  if(widget && GTK_IS_WIDGET(widget)) gtk_widget_destroy(widget);
}

void
sg_layer_control_refresh(GtkWidget *wdialog, const gchar *path)
{
  SGlayerControl *dialog = SG_LAYER_CONTROL(wdialog);
  
  if(dialog->tree)
    gtk_container_remove(GTK_CONTAINER(dialog->swindow), dialog->tree);

  build_tree(dialog);
  if(dialog->application){
    GList *list;

    list = dialog->application->plots->list;
    while(list){
      SGplot *plot;
      SGlistChild *child;

      child = (SGlistChild *)list->data;
      plot = SG_PLOT(child->object);

      build_branch(dialog, plot);    

      list = list->next;
    }
  } else if(dialog->plot){
    build_branch(dialog, dialog->plot);
  } 

  gtk_widget_show_all(dialog->tree);
  expand_tree(dialog, path);
}

static void
new_page_dialog(SGlayerControl *dialog, gpointer p_plot)
{
  sg_layer_control_clear_paned(dialog);

  dialog->dialog = sg_page_dialog_new(SG_PLOT(p_plot));
  sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog->dialog),
                                 dialog->ok_button, 
                                 dialog->apply_button, 
                                 dialog->cancel_button);

  gtk_frame_set_shadow_type (GTK_FRAME (dialog->dialog), GTK_SHADOW_OUT);
  gtk_container_add(GTK_CONTAINER(dialog->main_frame), 
                    GTK_WIDGET(dialog->dialog));
  gtk_widget_show_all(dialog->main_frame);

  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit page properties") );
}

static void
open_dialog(GtkCTree *tree, GtkCTreeNode *node, gint column, gpointer data)
{
  SGlayerControl *dialog = SG_LAYER_CONTROL(data);
  SGnodeData *nodedata;
  gpointer aux_data;

  aux_data = gtk_ctree_node_get_row_data(tree, node);
  if(aux_data == NULL) return;

  nodedata = (SGnodeData *)aux_data;
  if(nodedata->action == NULL) return;

  sg_layer_control_clear_paned(dialog);

  nodedata->action(dialog, nodedata->data);
  if(nodedata->path){
    if(dialog->path) g_free(dialog->path);
    dialog->path = g_strdup(nodedata->path);
  }
}

SGnodeData *
sg_node_data_new(gint type, 
                 gpointer data, 
                 void (*action) (SGlayerControl*, gpointer), 
                 const gchar *path, GtkCTreeNode *tree_node)
{
  SGnodeData *node;

  node = g_new(SGnodeData, 1);

  node->type = type;
  node->data = data;
  node->action = action;
  node->path = g_strdup(path);
  node->node = tree_node;

  return node;
}
  
static void
build_tree(SGlayerControl *dialog)
{
  dialog->tree = gtk_ctree_new(1, 0);
  gtk_clist_set_row_height (GTK_CLIST (dialog->tree), 19);
  gtk_clist_set_column_auto_resize(GTK_CLIST(dialog->tree),0,TRUE);
  gtk_clist_set_selection_mode(GTK_CLIST(dialog->tree),GTK_SELECTION_SINGLE);
  gtk_ctree_set_line_style(GTK_CTREE(dialog->tree),GTK_CTREE_LINES_DOTTED);
  gtk_container_add(GTK_CONTAINER(dialog->swindow), dialog->tree);

  gtk_signal_connect(GTK_OBJECT(dialog->tree), "tree_select_row",
                     GTK_SIGNAL_FUNC(open_dialog),
                     dialog);

  gtk_signal_connect(GTK_OBJECT(dialog->tree), "destroy",
                     GTK_SIGNAL_FUNC(ctree_destroy),
                     dialog);
}

static void
build_branch(SGlayerControl *dialog, SGplot *plot)
{
  GtkCTreeNode *root_item = NULL;
  SGnodeData *node;
  GList *list = NULL;
  GtkWidget *tree = dialog->tree;
  gchar *title[1];
  gchar _label[1000];
  gint n = 0;

  title[0] = plot->name;

  root_item = gtk_ctree_insert_node(GTK_CTREE(tree), NULL, NULL, 
                                    &title[0],
                                    4, 
                                    dialog->layer_pixmap, dialog->layer_mask, 
                                    dialog->layer_pixmap, dialog->layer_mask, 
                                    FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), root_item, 
                              node = sg_node_data_new(1, plot, 
                              new_page_dialog, plot->name, root_item));
  dialog->nodes = g_list_append(dialog->nodes, node);

  list = plot->layers;
  while(list){
    SGlayer *layer;
    SGpluginLayer *plugin;

    layer = SG_LAYER(list->data);
    plugin = layer->plugin;

    g_snprintf(_label, LABEL_LEN, "%s:%i",title[0],++n);
    sg_plugin_layer_dialog(plugin, GTK_WIDGET(dialog), root_item, G_OBJECT(layer), _label);

    list = list->next;
  }

}

static GtkCTreeNode * 
find_node_by_path(SGlayerControl *dialog, gchar *path)
{
  GList *list = NULL;
  gchar *node_path;
  GtkCTreeNode *node;
  SGnodeData *data;

  node_path = NULL;

  list = dialog->nodes;
  while(list){
     data = (SGnodeData *) list->data;
     node = data->node;

     node_path = data->path;
     if(strcmp(node_path, path) == 0) return node;

     list = list->next;
  } 

  return NULL;
}


static void
expand_tree(SGlayerControl *dialog, const gchar *path)
{
  GtkCTreeNode *node;
  const gchar *c; 
  gchar sub_node[1000];
  gint nlen;

  if(path == NULL) return;
  if(dialog->path) 
    g_free(dialog->path);
  dialog->path = NULL;
  if(path) dialog->path = g_strdup(path);

  node = NULL;
  c = path;
  nlen = 0;

  while(*c != '\0' && *c != '\n' && c != NULL){
    if(*c == ':'){
      node = find_node_by_path(dialog, sub_node);
      if(node) gtk_ctree_expand(GTK_CTREE(dialog->tree), node);
    }
    nlen++;
    sub_node[nlen-1] = *c;
    sub_node[nlen]='\0';
    c++;
  } 

  node = find_node_by_path(dialog, sub_node);
  if(node) {
     gtk_ctree_expand(GTK_CTREE(dialog->tree), node);
     gtk_ctree_select(GTK_CTREE(dialog->tree), node);
  }
}

static void
sg_layer_control_init (SGlayerControl *dialog)
{
  dialog->application = NULL;
  dialog->plot = NULL;
  dialog->tree = NULL;
  dialog->path = NULL;
  dialog->dialog = NULL;
}

static void
sg_layer_control_init_gui (SGlayerControl *dialog)
{
  GtkWidget *main_box;
  GtkWidget *frame, *frame2;
  GtkWidget *action_area;
  GtkWidget *right_box;
  GtkWidget *right_sub_box;
  GtkWidget *paned;

  gtk_container_set_border_width(GTK_CONTAINER(dialog), 2);
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, TRUE, FALSE);
  gtk_window_set_title(GTK_WINDOW(dialog), _("SciGraphica: Layer Control ") );

  /* Create widgets */
  main_box = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(dialog), main_box);

  paned = gtk_hpaned_new();
  gtk_paned_set_gutter_size (GTK_PANED(paned), 10);
  gtk_paned_set_position (GTK_PANED(paned), 150);
  gtk_box_pack_start (GTK_BOX(main_box), paned, TRUE, TRUE, 0);

  frame2 = gtk_frame_new(NULL);
  gtk_widget_set_usize(frame2, 200, 300);
  
  dialog->swindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(dialog->swindow),
                                 GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(frame2), dialog->swindow);

  gtk_paned_pack1(GTK_PANED(paned), frame2, FALSE, FALSE);

/***************************************************/
  right_box = gtk_vbox_new(FALSE, 5);
  gtk_paned_pack2(GTK_PANED(paned), right_box, FALSE, FALSE);
  
  /* Action area */
  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_END);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_box_pack_end (GTK_BOX (right_box), action_area, FALSE, FALSE, 0);
  gtk_widget_show (action_area);

  dialog->ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
  gtk_box_pack_start (GTK_BOX (action_area), dialog->ok_button, FALSE, FALSE, 0);
/*  
  GTK_WIDGET_SET_FLAGS (dialog->ok_button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (dialog->ok_button);
*/
  gtk_widget_show (dialog->ok_button);

  dialog->apply_button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
  gtk_box_pack_start (GTK_BOX (action_area), dialog->apply_button, FALSE, FALSE, 0);
  gtk_widget_show (dialog->apply_button);


  dialog->cancel_button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
  gtk_box_pack_start (GTK_BOX (action_area), dialog->cancel_button, FALSE, FALSE, 0);
  gtk_widget_show (dialog->cancel_button);


  right_sub_box = gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(right_box), right_sub_box, TRUE, TRUE, 0);

  dialog->main_frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(dialog->main_frame), GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(right_sub_box), dialog->main_frame, TRUE, TRUE, 0);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_box_pack_end(GTK_BOX(main_box), frame, FALSE, FALSE, 0);

  dialog->label = gtk_label_new( _("Layer control") ); 
  gtk_misc_set_alignment(GTK_MISC(dialog->label), 0., 0.);
  gtk_container_add(GTK_CONTAINER(frame), dialog->label);

  /* connect signals */
  gtk_signal_connect_object_after (GTK_OBJECT (dialog->ok_button), "clicked",
                             GTK_SIGNAL_FUNC (button_press_ok),
                             GTK_OBJECT(dialog));

  gtk_signal_connect_object (GTK_OBJECT (dialog->apply_button), "clicked",
                             GTK_SIGNAL_FUNC (button_press_apply),
                             GTK_OBJECT(dialog));

  gtk_signal_connect_object (GTK_OBJECT (dialog->cancel_button), "clicked",
                             GTK_SIGNAL_FUNC (button_press_cancel),
                             GTK_OBJECT(dialog));

  dialog->layer_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                                 gdk_colormap_get_system(),
                                                 &dialog->layer_mask,
                                                 NULL,
                                                 layer_xpm);
  dialog->plot2d_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                                 gdk_colormap_get_system(),
                                                 &dialog->plot2d_mask,
                                                 NULL,
                                                 _2d_small_xpm);
  dialog->plot3d_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                                 gdk_colormap_get_system(),
                                                 &dialog->plot3d_mask,
                                                 NULL,
                                                 _3d_small_xpm);
  dialog->polar_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                                 gdk_colormap_get_system(),
                                                 &dialog->polar_mask,
                                                 NULL,
                                                 polar_small_xpm);

}

