/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "config.h"

#include "sg_file_dialog.h"
#include "sg_plugin_file.h"
#include "sg.h"
#include "sg_dialog.h"
#include "sg_misc_dialogs.h"
#include "sg_stock.h"
#include "pixmaps/sg.xpm"

/* initialize defaults */

#define SG_FILE_DIALOG(obj)        GTK_CHECK_CAST (obj, sg_file_dialog_get_type (), SGfileDialog)
#define GTK_TYPE_SG_FILE_DIALOG   (sg_file_dialog_get_type ())

#define SG_FILE_DIALOG_CLASS(klass) GTK_CHECK_CLASS_CAST (klass, sg_file_dialog_get_type(), SGfileDialogClass)
#define GTK_IS_SG_FILE_DIALOG(obj)     GTK_CHECK_TYPE (obj, sg_file_dialog_get_type ())


typedef struct _SGfileDialog SGfileDialog;
typedef struct _SGfileDialogClass SGfileDialogClass;

struct _SGfileDialog
{
  GtkIconFileSel filesel;

  SGapplication *app;
  GtkWidget *format_combo;
  GtkWidget *auto_check;

  gint return_value;
  gint file_format;
  GObject *object;
  gchar *object_name;
  gchar *path;
  SGpluginFileMode mode;

};

struct _SGfileDialogClass
{
  GtkIconFileSelClass parent_class;
};

static void sg_file_dialog_class_init       	(SGfileDialogClass *klass);
static void sg_file_dialog_init             	(SGfileDialog *d);
static void sg_file_dialog_destroy          	(GtkObject *object);
static void sg_file_dialog_realize	 	(GtkWidget *filesel);
static void sg_file_dialog_init_gui		(GtkWidget *filesel);
static gboolean ok_clicked			(GtkWidget *widget, 
						 gpointer data);
static void new_format				(GtkWidget *widget, 
						 gpointer data);

static GtkIconFileSelClass *parent_class = NULL;

GtkType
sg_file_dialog_get_type(void)
{
  static GtkType sg_file_dialog_type = 0;

  if (!sg_file_dialog_type)
    {
      GtkTypeInfo sg_file_dialog_info =
      {
        "SGfileDialog",
        sizeof (SGfileDialog),
        sizeof (SGfileDialogClass),
        (GtkClassInitFunc) sg_file_dialog_class_init,
        (GtkObjectInitFunc) sg_file_dialog_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_file_dialog_type = gtk_type_unique (gtk_icon_file_selection_get_type(), &sg_file_dialog_info);
    }

  return sg_file_dialog_type;
}

static void
sg_file_dialog_init(SGfileDialog *dialog)
{
  dialog->format_combo = NULL;
  dialog->auto_check = NULL;
  dialog->return_value = TRUE;
  dialog->file_format = 0;
  dialog->object = NULL;
  dialog->object_name = NULL;
  dialog->mode = 0;
}

static void
sg_file_dialog_class_init(SGfileDialogClass *klass)
{
  GtkObjectClass *object_class;
  SGfileDialogClass *d_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass*) klass;
  widget_class = (GtkWidgetClass*) klass;
  d_class = (SGfileDialogClass*) klass;

  parent_class = (GtkIconFileSelClass *)gtk_type_class (gtk_icon_file_selection_get_type ());
  object_class->destroy = sg_file_dialog_destroy;

  widget_class->realize = sg_file_dialog_realize;
}

static gint
cancel_clicked(GtkWidget *widget, gpointer data)
{
  SGfileDialog *dialog = SG_FILE_DIALOG(data);

  dialog->return_value = FALSE;
  if(data && GTK_IS_WIDGET(data)) gtk_widget_destroy(GTK_WIDGET(data));
  return FALSE;
}

GtkWidget *
sg_file_dialog_new(SGpluginFileMode mode, SGapplication *app, const gchar *object_name, GObject *object)
{
  GtkWidget *widget;
  SGfileDialog *dialog;
  widget = GTK_WIDGET(gtk_type_new(sg_file_dialog_get_type()));
  dialog = SG_FILE_DIALOG(widget);
  dialog->mode = mode;
  dialog->app = app;
  dialog->path = app && app->path ? g_strdup(app->path) : NULL;
  dialog->object_name = g_strdup(object_name);
  dialog->object = object;
  sg_file_dialog_init_gui(widget);
  return widget;
}

static void
sg_file_dialog_destroy(GtkObject *object)
{
  SGfileDialog *dialog = SG_FILE_DIALOG(object);

  if(dialog->object_name) g_free(dialog->object_name);
  dialog->object_name = NULL;
  if(dialog->path) g_free(dialog->path);
  dialog->path = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static gboolean
mw_destroy(GtkWidget *widget, gpointer data)
{
  gtk_window_set_modal(GTK_WINDOW(widget), FALSE);

  gtk_main_quit();

  return TRUE;
}

static void 
sg_file_dialog_realize(GtkWidget *filesel)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  (*GTK_WIDGET_CLASS (parent_class)->realize) (filesel);

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                                gdk_colormap_get_system(),
                                                &mask, NULL,
                                                sg_xpm);
                                                                                
  gdk_window_set_icon(GTK_WIDGET(filesel)->window, NULL, pixmap, mask);
}

static void 
sg_file_dialog_init_gui(GtkWidget *filesel)
{
  GtkWidget *label;
  GtkWidget *bbox;
  gint type;
  gchar *aux_path;
  gchar window_title[500];
  GList *list;
  SGfileDialog *dialog;

  dialog = SG_FILE_DIALOG(filesel);

  switch(dialog->mode){
    case SG_PLUGIN_FILE_IMPORT:
      g_snprintf(window_title, 500, "SciGraphica: Import %s", dialog->object_name);
      break;
    case SG_PLUGIN_FILE_OPEN:
      g_snprintf(window_title, 500, "SciGraphica: Open %s", dialog->object_name);
      break;
    case SG_PLUGIN_FILE_PRINT:
    case SG_PLUGIN_FILE_EXPORT:
      g_snprintf(window_title, 500, "SciGraphica: Export %s", dialog->object_name);
      break;
    case SG_PLUGIN_FILE_SAVE:
      g_snprintf(window_title, 500, "SciGraphica: Save %s", dialog->object_name);
      break;
    case SG_PLUGIN_FILE_SAVE_AS:
      g_snprintf(window_title, 500, "SciGraphica: Save as %s", dialog->object_name);
      break;
  }

  gtk_window_set_title (GTK_WINDOW(filesel), window_title);
  if(dialog->mode > SG_PLUGIN_FILE_OPEN){
    gtk_table_resize(GTK_TABLE(GTK_ICON_FILESEL(filesel)->action_area), 4, 4);
  } else {
    gtk_table_resize(GTK_TABLE(GTK_ICON_FILESEL(filesel)->action_area), 3, 4);
  }

  label = gtk_label_new( _("File Format:        ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(GTK_ICON_FILESEL(filesel)->action_area),
                            label,
                            0, 1, 2, 3);
  gtk_widget_show(label);

  dialog->format_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(GTK_ICON_FILESEL(filesel)->action_area),
                            dialog->format_combo,
                            1, 3, 2, 3);
  gtk_widget_show(dialog->format_combo);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->format_combo)->entry), FALSE);

  if(dialog->mode > SG_PLUGIN_FILE_OPEN){
    dialog->auto_check = gtk_check_item_new_with_label(_("Automatic file name extension"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->auto_check), TRUE);
    gtk_table_attach_defaults(GTK_TABLE(GTK_ICON_FILESEL(filesel)->action_area),
                              dialog->auto_check,
                              0, 3, 3, 4);
    gtk_widget_show(dialog->auto_check);
  }

  if(dialog->path && dialog->path[strlen(dialog->path) - 1] != G_DIR_SEPARATOR_S[0])
    aux_path = g_strconcat(dialog->path, G_DIR_SEPARATOR_S, NULL);
  else
    aux_path = g_strdup(dialog->path);

  gtk_icon_file_selection_open_dir(GTK_ICON_FILESEL(filesel), aux_path);
  if(dialog->app) sg_application_set_path(dialog->app, aux_path);
  g_free(aux_path);

  list = sg_plugins();
  while(list){
    SGpluginFile *plugin = NULL;

    if(GTK_IS_SG_PLUGIN_FILE(list->data)){
      plugin = SG_PLUGIN_FILE(list->data);
      if(plugin->mode == dialog->mode && strcmp(dialog->object_name,plugin->object_name) == 0){
        gint i;
        GtkWidget *item;
  
        type = gtk_file_list_add_type_with_pixmap
                         (GTK_FILE_LIST(GTK_ICON_FILESEL(filesel)->file_list),
                         GTK_PIXMAP(plugin->pixmap)->pixmap,
                         GTK_PIXMAP(plugin->pixmap)->mask);
 
        item = gtk_list_item_new_with_label(SG_PLUGIN(plugin)->description);
        gtk_widget_show(item);
        gtk_container_add(GTK_CONTAINER(GTK_COMBO(dialog->format_combo)->list), item);
        gtk_object_set_data(GTK_OBJECT(item), "plugin", plugin);
  
        for(i = 0; i < plugin->nfilters; i++){
          gchar filter[20];
          g_snprintf(filter,20,"*%s",plugin->filter[i]);
          gtk_file_list_add_type_filter
                          (GTK_FILE_LIST(GTK_ICON_FILESEL(filesel)->file_list),
                           type,
                           filter);
        }
      }
    }
    list = list->next;
  }

}

gboolean
sg_file_dialog_run(GtkWidget *filesel)
{
  SGfileDialog *dialog;

  dialog = SG_FILE_DIALOG(filesel);

  gtk_window_position (GTK_WINDOW (filesel), GTK_WIN_POS_CENTER);
  gtk_window_set_modal(GTK_WINDOW(filesel), TRUE);

/*
  gtk_signal_connect (GTK_OBJECT (GTK_ICON_FILESEL(filesel)->file_entry),
                      "activate",
                      (GtkSignalFunc) ok_clicked, filesel);
*/

  /* Connect the cancel_button to destroy the widget */
  gtk_signal_connect (GTK_OBJECT (GTK_ICON_FILESEL(filesel)->cancel_button),
                      "clicked",
                      GTK_SIGNAL_FUNC(cancel_clicked), filesel);

  /* Connect the ok_button function */
  gtk_signal_connect (GTK_OBJECT (GTK_ICON_FILESEL(filesel)->ok_button),
                      "clicked",
                      (GtkSignalFunc) ok_clicked, filesel);

  /* Connect the combo to change file format */
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(dialog->format_combo)->entry),
                      "changed",
                      (GtkSignalFunc) new_format, filesel);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), 0);
  new_format(dialog->format_combo, filesel);

  /* Wait until window gets destroyed */
  gtk_widget_show(filesel);
  gtk_signal_connect(GTK_OBJECT(filesel), "destroy", 
		     GTK_SIGNAL_FUNC(mw_destroy), NULL);

  gtk_main();

  return dialog->return_value;
}

static void
new_format(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  gchar filter[10];
  SGpluginFile *plugin;
  SGfileDialog *dialog = SG_FILE_DIALOG(data);

  child = GTK_WIDGET(GTK_LIST(GTK_COMBO(dialog->format_combo)->list)->selection->data);
  plugin = gtk_object_get_data(GTK_OBJECT(child), "plugin");

  g_snprintf(filter, 10, "*%s", plugin->filter[0]);

  gtk_entry_set_text(GTK_ENTRY(GTK_ICON_FILESEL(data)->filter_entry), filter); 
  gtk_file_list_set_filter(GTK_FILE_LIST(GTK_ICON_FILESEL(data)->file_list), filter);
}

static gboolean
ok_clicked (GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  SGpluginFile *plugin;
  const gchar *file_path, *entry_name;
  gchar *full_path, *file_name;
  GtkIconFileSel *filesel;
  gchar aux_name[2000];
  gchar message[255];
  gchar *aux_ext;
  gint n;
  gboolean file_exists = FALSE;
  SGfileDialog *dialog;
  gboolean complete = FALSE;

  filesel = GTK_ICON_FILESEL(data);
  dialog = SG_FILE_DIALOG(filesel);

  child = GTK_WIDGET(GTK_LIST(GTK_COMBO(dialog->format_combo)->list)->selection->data);
  plugin = gtk_object_get_data(GTK_OBJECT(child), "plugin");
 
  file_name = (gchar *)gtk_file_list_get_filename(GTK_FILE_LIST(filesel->file_list));
  file_path = (gchar *)gtk_file_list_get_path(GTK_FILE_LIST(filesel->file_list));
  entry_name = gtk_entry_get_text(GTK_ENTRY(filesel->file_entry));

  if(entry_name)
      file_name = (gchar *)entry_name;
  if (!file_name || !*file_name)
  {  sg_message_dialog ( _("No file selected!") , 0);
     return TRUE;
  }

  if(dialog->auto_check)
    complete = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->auto_check));

  if(complete){ 
    n = strlen(file_name)-strlen(plugin->filter[dialog->file_format]);
    aux_ext = plugin->filter[dialog->file_format];
    if(n <= 0){
      if(strcmp(plugin->filter[dialog->file_format], ".htm*") == 0)
        g_snprintf(aux_name, 2000, "%s.html", file_name);
      else
        g_snprintf(aux_name, 2000, "%s%s", file_name, plugin->filter[dialog->file_format]);          
      file_name = aux_name;
    }else{
      guint i;
      for(i = n; i < strlen(file_name); i++){
        if(aux_ext[0] != file_name[i]){
          if(strcmp(plugin->filter[dialog->file_format], ".htm*") == 0){
            if(aux_ext[0] == '*' && file_name[i] != 'l')
              g_snprintf(aux_name, 2000, "%s.html", file_name);
          } else {
            g_snprintf(aux_name, 2000, "%s%s", file_name, plugin->filter[dialog->file_format]);
          }
          file_name = aux_name;
          break;
        }
        aux_ext++;
      }
    }
  }

  full_path = g_strdup(file_path);
  full_path = g_strconcat(full_path, G_DIR_SEPARATOR_S, file_name, NULL);

  file_exists = sg_file_readable(full_path);
  if(file_exists &&
     (dialog->mode == SG_PLUGIN_FILE_EXPORT || dialog->mode == SG_PLUGIN_FILE_PRINT || dialog->mode == SG_PLUGIN_FILE_SAVE || dialog->mode == SG_PLUGIN_FILE_SAVE_AS)){
        g_snprintf(message, 255, _("File Exists. Overwrite %s?") , file_name);
        if( sg_accept_dialog(message, 1) != SG_BUTTON_YES ) return TRUE;
  }

  if((dialog->mode == SG_PLUGIN_FILE_EXPORT || dialog->mode == SG_PLUGIN_FILE_PRINT || dialog->mode == SG_PLUGIN_FILE_SAVE || dialog->mode == SG_PLUGIN_FILE_SAVE_AS) && !sg_file_writeable(full_path)){
        g_snprintf(message, 255, _("Cannot write to file:\n`%s'") , file_name);
        sg_message_dialog(message, 0);
        return TRUE;
  }

  if(!file_exists &&
     (dialog->mode == SG_PLUGIN_FILE_OPEN || dialog->mode == SG_PLUGIN_FILE_IMPORT)){
        g_snprintf(message, 255, _("Unable to open file:\n`%s'") , full_path);
        sg_message_dialog(message, 0);
        return TRUE;
  }

  if (data) gtk_widget_hide(GTK_WIDGET(data));

  dialog->return_value = plugin->action(plugin, full_path, NULL, &dialog->object, NULL);

  if(dialog->app) sg_application_set_path(dialog->app, file_path);
  if(dialog->app) sg_application_set_filename(dialog->app, file_name);
  if (data && GTK_IS_WIDGET(data)) gtk_widget_destroy(GTK_WIDGET(data));

  g_free(full_path);
 
  return TRUE;
}

