/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <zlib.h>
#include <stdarg.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"

#ifdef WITH_WARNINGS
#warning Use better output system, see print_raw() function
#endif

static gint sg_compress_level = 9;

gchar *
xml_get_string(const gchar *input)
{
  static gchar *output = NULL; /* use static to prevent memory leaks */
  const gchar *c;
  gint n;

  if (output) g_free(output);
  output = g_new(gchar,1);

  for (n = 0, c = input; c && *c != '\0' && *c != '\n'; c++)
    switch(*c){
        case '>':
          output = g_renew(gchar, output, n + 4);
          output[n++] = '&';
          output[n++] = 'g';
          output[n++] = 't';
          output[n++] = ';';
          break;
        case '<':
          output = g_renew(gchar, output, n + 4);
          output[n++] = '&';
          output[n++] = 'l';
          output[n++] = 't';
          output[n++] = ';';
          break;
        case '&':
          output = g_renew(gchar, output, n + 5);
          output[n++] = '&';
          output[n++] = 'a';
          output[n++] = 'm';
          output[n++] = 'p';
          output[n++] = ';';
          break;
        case '\"':
          output = g_renew(gchar, output, n + 6);
          output[n++] = '&';
          output[n++] = 'q';
          output[n++] = 'u';
          output[n++] = 'o';
          output[n++] = 't';
          output[n++] = ';';
          break;
        default:
          output = g_renew(gchar, output, n + 1);
          output[n++] = *c;
          break;
    }

  /* terminate the output string with the null-character */
  output = g_renew(gchar, output, n + 1);
  output[n] = '\0';

  return output;
}


/* keep track of the stream's ascii/gzip-status as an array of two elements,
 * since there is a small chance that the regular save is interrupted
 * by autosave, with both having a stream descriptor open simultaneously */
static gboolean is_ascii_file[2];
static gint number_of_files_opened = 0;

FILE *
sg_file_open(const gchar *filename, const gchar *mode )
{
 gchar msg[500];
 FILE *stream;

 if ( strchr(mode,'r') )
    {/* check readability for read-mode */
     if ( !sg_file_readable(filename) ) return NULL;
     else g_snprintf(msg, 500, _("Reading from %s"), filename);
    }
 else if ( strchr(mode,'w') || strchr(mode,'a') )
    {/* check writeability for write/append-mode */
     if ( !sg_file_writeable(filename) ) return NULL;
     else g_snprintf(msg, 500, _("Writing to %s"), filename);
    }
 else return NULL;

 /* open a stream descriptor according to the mode (gzip or ascii) */
 if ( strchr(mode,'b') && sg_compress_level ) 
    {/* binary mode: save as gzip with compression level */
     char gzmode[10];
     g_snprintf(gzmode, 10, "%s%d", mode, sg_compress_level);
     if ( (stream = (FILE*)gzopen(filename, gzmode)) )
        is_ascii_file[number_of_files_opened++] = FALSE;
    } 
 else /* ascii mode */
    if ( (stream = fopen(filename, mode)) )
       is_ascii_file[number_of_files_opened++] = TRUE;

 return stream;
}

gint
sg_file_close( FILE *stream )
{
 if ( is_ascii_file[--number_of_files_opened] )
    return fclose(stream);
 else
    return gzclose((gzFile)stream);
}


gint
sg_file_printf( FILE *stream, const gchar *format, ... )
{
/* Write to stream, with arguments similar to fprintf().
 * Returns number of bytes written to stream, or -1 on error. */
 gint number_of_written_bytes = -1;
 gchar *output;
 va_list args;

 va_start(args, format);
 output = g_strdup_vprintf(format, args);
 va_end(args);

 if ( output )
    {if ( is_ascii_file[number_of_files_opened - 1] )
        number_of_written_bytes = fprintf(stream, output);
     else
        number_of_written_bytes = gzwrite((gzFile)stream, (const voidp)output, strlen(output));
     g_free(output);
    }

 return number_of_written_bytes;
}


gboolean
sg_file_readable(const gchar *path)
{
 return ( access(path, R_OK) != -1 );
}

gboolean
sg_file_writeable(const gchar *path)
{
 FILE *fp = fopen(path, "a");
 gboolean result = ( fp != NULL && access(path, W_OK) != -1 );
 
 if ( fp ) fclose(fp);
  
 return result;
}

char *
sg_sys_data_dir (char const *subdir)
{
        if (subdir == NULL)
                return (char *)sg_datadir;
        return g_strconcat (sg_datadir, G_DIR_SEPARATOR_S,
                            subdir, G_DIR_SEPARATOR_S, NULL);
}

gchar *
sg_sys_lib_dir (const gchar *subdir)
{
        return g_strconcat (sg_libdir, G_DIR_SEPARATOR_S,
                            subdir, G_DIR_SEPARATOR_S, NULL);
}

#define GLADE_SUFFIX    "glade"
#define PLUGIN_SUFFIX   "plugins"

gchar *
sg_sys_glade_dir (void)
{
        return sg_sys_data_dir (GLADE_SUFFIX);
}

gchar *
sg_sys_plugin_dir (void)
{
        return sg_sys_lib_dir (PLUGIN_SUFFIX);
}

gchar *
sg_usr_dir (const gchar *subdir)
{
        char const *home_dir = g_get_home_dir ();

        if (home_dir != NULL) {
                gboolean has_slash = (home_dir[strlen (home_dir) - 1] == G_DIR_SEPARATOR);
                return g_strconcat (home_dir, (has_slash ? "" : G_DIR_SEPARATOR_S),
                                    subdir, G_DIR_SEPARATOR_S,
                                    NULL);
        }
        return NULL;
}

gchar *
sg_usr_plugin_dir (void)
{
        return sg_usr_dir (PLUGIN_SUFFIX);
}

