/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <math.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_dataset.h"

#define P_(string) string
extern GRand *sg_seed;

static void sg_dataset_class_init			(SGdatasetClass *klass);
static void sg_dataset_init				(SGdataset *dataset);
static void sg_dataset_finalize				(GObject *object);
static GtkPlotData * sg_dataset_real_new_child		(SGdataset *dataset);
static void sg_dataset_set_property    (GObject *object,
                                                 guint prop_id,
                                                 const GValue *value,
                                                 GParamSpec *pspec);
static void sg_dataset_get_property    (GObject *object,
                                                 guint prop_id,
                                                 GValue *value,
                                                 GParamSpec *pspec);


static GObjectClass *parent_class = NULL;

enum {
	ARG_0,
	ARG_ID,
	ARG_NAME
};

GType
sg_dataset_get_type (void)
{
  static GType sg_dataset_type = 0;
                                                                                
  if (!sg_dataset_type)
    {
      static const GTypeInfo sg_dataset_info =
      {
        sizeof (SGdatasetClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) sg_dataset_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (SGdataset),
        0,              /* n_preallocs */
        (GInstanceInitFunc) sg_dataset_init,
        NULL,
      };
                                                                                
      sg_dataset_type = g_type_register_static (G_TYPE_OBJECT, "SGdataset",
                                               &sg_dataset_info, 0);
    }
                                                                                
  return sg_dataset_type;
}

static void
sg_dataset_class_init (SGdatasetClass *klass)
{
  GObjectClass *object_class;
  SGdatasetClass *data_class;

  parent_class = (GObjectClass *) g_type_class_peek_parent (klass);

  object_class = (GObjectClass *) klass;
  data_class = (SGdatasetClass *) klass;

  object_class->finalize = sg_dataset_finalize;
  data_class->refresh = NULL;
  data_class->set_iterator = NULL;
  data_class->new_child = sg_dataset_real_new_child;
  data_class->connected = NULL;

  object_class->set_property = sg_dataset_set_property;
  object_class->get_property = sg_dataset_get_property;

  g_object_class_install_property (object_class,
                           ARG_ID,
  g_param_spec_int ("id",
                           P_("id"),
                           P_("id"),
                           -G_MAXINT,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_NAME,
  g_param_spec_string ("name",
                           P_(""),
                           P_(""),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
}

SGdataset *
sg_dataset_new ()
{
  SGdataset *dataset;

  dataset = SG_DATASET(g_object_new(sg_dataset_get_type(), NULL));
  return dataset;
}

static void
child_destroyed(GtkPlotData *child, SGdataset *dataset)
{
  GList *list;

  list = g_list_find(dataset->children, child);
  dataset->children = g_list_remove_link(dataset->children, list);
  g_list_free_1(list);

  dataset->ref_count--;
}

void
sg_dataset_construct(SGdataset *dataset, SGpluginStyle *style)
{
  GList *dimensions = style->arrays->arrays;

  dataset->constructor = style;

/* copy arrays from style constructor to dataset */
  while(dimensions){
    GtkPlotArray *ref, *new_dim;
                                                                                
    ref = GTK_PLOT_ARRAY(dimensions->data);
    new_dim = GTK_PLOT_ARRAY(gtk_plot_array_new(ref->name, NULL, 0, ref->type, TRUE));
    gtk_plot_array_set_label(new_dim, ref->label);
    gtk_plot_array_set_description(new_dim, ref->description);
    gtk_plot_array_set_required(new_dim, ref->required);
    gtk_plot_array_list_add(dataset->arrays, new_dim);
    g_object_unref(G_OBJECT(new_dim));
                                                                                
    dimensions = dimensions->next;
  }

/*
  dimensions = dataset->arrays->arrays;
  while(dimensions){
    GtkPlotArray *ref;
    ref = GTK_PLOT_ARRAY(dimensions->data);
    printf("DIM %s\n",ref->name);
    dimensions = dimensions->next;
  }
*/
}

GtkPlotData *
sg_dataset_new_child(SGdataset *dataset)
{
  GtkPlotData *child;
  
  child = SG_DATASET_CLASS(G_OBJECT_GET_CLASS(G_OBJECT(dataset)))->new_child(dataset);

  return child;
}

void
sg_dataset_add_child(SGdataset *dataset, GtkPlotData *child)
{
  gtk_widget_ref(GTK_WIDGET(child));
  child->link = dataset;

  dataset->children = g_list_append(dataset->children, child);
  g_object_unref(G_OBJECT(child->data));
  child->data = dataset->arrays;
  g_object_ref(G_OBJECT(dataset->arrays));
  dataset->ref_count++;

  gtk_plot_data_set_name(child, dataset->name);
  sg_dataset_refresh_arrays(dataset);
}

static GtkPlotData * 
sg_dataset_real_new_child(SGdataset *dataset)
{
  GtkPlotData *real_data;
  SGpluginStyle *style = dataset->constructor;

  real_data = style->construct();

  real_data->is_iterator = FALSE;
  real_data->is_function = FALSE;

  real_data->link = dataset;

  if(SG_DATASET_CLASS(G_OBJECT_GET_CLASS(G_OBJECT(dataset)))->set_iterator)
    SG_DATASET_CLASS(G_OBJECT_GET_CLASS(G_OBJECT(dataset)))->set_iterator(dataset);

  gtk_plot_data_set_name(real_data, dataset->name);
  sg_dataset_refresh_arrays(dataset);

  return real_data;
}

static void
sg_dataset_init(SGdataset *dataset)
{
  dataset->children = NULL;
  dataset->constructor = NULL;
  dataset->name = NULL;
  dataset->description = NULL;
  dataset->id = -1;
  while(dataset->id < 0){
    dataset->id = g_rand_int(sg_seed);
  }

  dataset->arrays = GTK_PLOT_ARRAY_LIST(gtk_plot_array_list_new());
  g_object_ref(G_OBJECT(dataset->arrays));
}

static void
sg_dataset_get_property (GObject      *object,
                             guint            prop_id,
                             GValue          *value,
                             GParamSpec      *pspec)
{
  SGdataset *data;
                                                                                
  data = SG_DATASET(object);
                                                                                
  switch (prop_id)
    {
      case ARG_NAME:
        g_value_set_string(value, data->name);
        break;
      case ARG_ID:
        g_value_set_int(value, data->id);
        break;
    }
}

static void
sg_dataset_set_property (GObject      *object,
                             guint            prop_id,
                             const GValue          *value,
                             GParamSpec      *pspec)
{
  SGdataset *data;
                                                                                
  data = SG_DATASET(object);
                                                                                
  switch (prop_id)
    {
      case ARG_NAME:
        sg_dataset_set_name(data, g_value_get_string(value));
        break;
      case ARG_ID:
        data->id = g_value_get_int(value);
        break;
    }
}

void
sg_dataset_refresh(SGdataset *dataset)
{
  GList *list;

  if(SG_DATASET_CLASS(G_OBJECT_GET_CLASS(G_OBJECT(dataset)))->refresh)
    SG_DATASET_CLASS(G_OBJECT_GET_CLASS(G_OBJECT(dataset)))->refresh(dataset);

  list = dataset->children;
  while(list){
    gtk_plot_data_set_name(GTK_PLOT_DATA(list->data), dataset->name);
    list = list->next;
  }

  sg_dataset_refresh_arrays(dataset);
}

static void
sg_dataset_finalize(GObject *object)
{
  SGdataset *dataset;
  GList *list;

  dataset = SG_DATASET(object);

  if(dataset->name) g_free(dataset->name);
  dataset->name = NULL;

  if(dataset->description) g_free(dataset->description);
  dataset->description = NULL;

  list = dataset->children;
  while(list){
    if(GTK_IS_WIDGET(list->data)) gtk_widget_unref(GTK_WIDGET(list->data));
    dataset->children = g_list_remove_link(dataset->children, list);
    g_list_free_1(list);
    list = dataset->children;
  }
  dataset->children = NULL;
  g_object_unref(G_OBJECT(dataset->arrays));
  dataset->arrays = NULL;

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

void
sg_dataset_remove_child(SGdataset *dataset, GtkPlotData *child)
{
  GList *list = NULL;

  list = g_list_find(dataset->children, child);
  if(list && GTK_IS_WIDGET(child)) {
    gtk_widget_destroy(GTK_WIDGET(child));
    dataset->children = g_list_remove_link(dataset->children, list);
    g_list_free_1(list);
  }
}

void
sg_dataset_set_name(SGdataset *dataset, const gchar *name)
{
  GList *list;

  if(dataset->name) g_free(dataset->name);
  if(name) dataset->name = g_strdup(name);

  list = dataset->children;
  while(list){
    gtk_plot_data_set_name(GTK_PLOT_DATA(list->data), dataset->name);
    list = list->next;
  }
}

void
sg_dataset_set_description(SGdataset *dataset, const gchar *description)
{
  if(dataset->description) g_free(dataset->description);
  if(description) dataset->description = g_strdup(description);
}

void
sg_dataset_free_points(SGdataset* dataset)
{
  GList *list = NULL;
  if(!dataset->arrays) return;

  list = dataset->arrays->arrays;
  while(list){
    GtkPlotArray *array = GTK_PLOT_ARRAY(list->data);
    gtk_plot_array_free(array);
    list = list->next;
  }
}

GtkPlotArray *
sg_dataset_set_points(SGdataset *dataset, const gchar *arrayname, gdouble *data, gint npoints)
{
  GtkPlotArray *array;

  array = gtk_plot_array_list_get(dataset->arrays, arrayname);
  if(!array){
    array = GTK_PLOT_ARRAY(gtk_plot_array_new(arrayname, data, npoints, GTK_TYPE_DOUBLE, TRUE));
    gtk_plot_array_list_add(dataset->arrays, array);
  } else {
    gtk_plot_array_set(array, (gpointer)data, npoints, GTK_TYPE_DOUBLE);
  }
 
  array->own_data = TRUE; 
  return array;
}

GtkPlotArray *
sg_dataset_set_labels(SGdataset *dataset, gchar **labels, gint npoints)
{
  GtkPlotArray *array;

  array = gtk_plot_array_list_get(dataset->arrays, "labels");
  if(!array){
    array = GTK_PLOT_ARRAY(gtk_plot_array_new("labels", labels, npoints, GTK_TYPE_STRING, TRUE));
    gtk_plot_array_list_add(dataset->arrays, array);
  } else {
    gtk_plot_array_set(array, (gpointer)labels, npoints, GTK_TYPE_STRING);
  }

  array->own_data = TRUE; 
  return array;
}

void
sg_dataset_set_arrays(SGdataset *dataset, GtkPlotArrayList *arrays)
{
  GList *list;

  if(dataset->arrays) gtk_plot_array_list_clear(dataset->arrays);

  list = arrays->arrays;
  while(list){
    GtkPlotArray *array = GTK_PLOT_ARRAY(list->data);
    gtk_plot_array_list_add(dataset->arrays, array);
    list = list->next;
  }

  sg_dataset_refresh_arrays(dataset);
}

void
sg_dataset_refresh_arrays(SGdataset *dataset)
{
  GList *list;
  gint size = 0;

  list = dataset->arrays->arrays;
  while(list) {
    GtkPlotArray *array = GTK_PLOT_ARRAY(list->data);
    if(array->size > size) size = array->size;
    list = list->next;
  }

  list = dataset->arrays->arrays;
  while(list){
    GtkPlotArray *array = GTK_PLOT_ARRAY(list->data);
    if(array->size < size && array->required) size = array->size;
    list = list->next;
  }

  list = dataset->children;
  while(list) {
    GtkPlotData *data = GTK_PLOT_DATA(list->data);
    if(dataset->arrays) { 
      g_object_unref(G_OBJECT(data->data));
      data->data = dataset->arrays;
      g_object_ref(G_OBJECT(data->data));
    }
    gtk_plot_data_set_numpoints(data, size);
    list = list->next;
  }
}

gboolean
sg_dataset_is_connected(SGdataset *dataset, gpointer data)
{
  return (SG_DATASET_CLASS(G_OBJECT_GET_CLASS(G_OBJECT(dataset)))->connected(dataset, data));
}

