/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_worksheet.h"
#include "sg_plot.h"
#include "sg_dataset.h"
#include "sg_list.h"
#include "sg_application.h"
#include "sg_layer_control.h"

enum {
	ADD_WORKSHEET,
	ADD_MATRIX,
	ADD_PLOT,
	ADD_DATASET,
	REMOVE_WORKSHEET,
	REMOVE_MATRIX,
	REMOVE_PLOT,
	REMOVE_DATASET,
	RENAME_WORKSHEET,
	RENAME_MATRIX,
	RENAME_PLOT,
	RENAME_DATASET,
	ACTIVATE_PLOT,
	ACTIVATE_WORKSHEET,
        CHANGED,
	LAST_SIGNAL,
};

static void sg_application_class_init	(SGapplicationClass *klass);
static void sg_application_init		(SGapplication *app);
static void sg_application_finalize	(GObject *o);
static void sg_application_destroy	(GtkObject *o);

static GtkObjectClass *parent_class = NULL;
static guint app_signals[LAST_SIGNAL] = {0};

GtkType 
sg_application_get_type(void)
{
  static GtkType sg_application_type = 0;

  if (!sg_application_type)
    {
      GtkTypeInfo sg_application_info =
      {
        "SGapplication",
        sizeof (SGapplication),
        sizeof (SGapplicationClass),
        (GtkClassInitFunc) sg_application_class_init,
        (GtkObjectInitFunc) sg_application_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_application_type = gtk_type_unique (gtk_object_get_type(), &sg_application_info);
    }

  return sg_application_type;
}

GtkObject *
sg_application_new(void)
{
  return GTK_OBJECT(gtk_type_new(sg_application_get_type()));
}

static void 
sg_application_init(SGapplication *application)
{
  application->worksheets = sg_list_new();
  g_object_ref(G_OBJECT(application->worksheets));
  application->plots = sg_list_new();
  g_object_ref(G_OBJECT(application->plots));
  application->datasets = sg_list_new();
  g_object_ref(G_OBJECT(application->datasets));
  application->changed = FALSE;
  application->last_worksheet = 0;
  application->last_matrix = 0;
  application->last_plot = 0;
  application->num_plots = 0;
  application->num_worksheets = 0;
  application->path = NULL;
  application->filename = g_strdup(g_get_current_dir());
  application->worksheet_cb = SG_CLIPBOARD(sg_clipboard_new());
  g_object_ref(G_OBJECT(application->worksheet_cb));
  application->plot_cb = SG_PLOT_CLIPBOARD(sg_plot_clipboard_new());
  g_object_ref(G_OBJECT(application->plot_cb));
  application->layer_control = NULL;
}

static void
sg_application_finalize(GObject *o)
{
  sg_application_set_path(SG_APPLICATION(o), NULL);
  sg_application_set_filename(SG_APPLICATION(o), NULL);
}

static void
sg_application_destroy(GtkObject *o)
{
  SGapplication *app = SG_APPLICATION(o);

  if(app->datasets)
    g_object_unref(G_OBJECT(app->datasets));
  app->datasets = NULL;
  if(app->plots)
    g_object_unref(G_OBJECT(app->plots));
  app->plots = NULL;
  if(app->worksheets)
    g_object_unref(G_OBJECT(app->worksheets));
  app->worksheets = NULL;

  if(app->layer_control && GTK_IS_OBJECT(app->layer_control))
    gtk_object_unref(GTK_OBJECT(app->layer_control));
  app->layer_control = NULL;

  if(app->worksheet_cb)
    g_object_unref(G_OBJECT(app->worksheet_cb));
  app->worksheet_cb = NULL;
  if(app->plot_cb)
    g_object_unref(G_OBJECT(app->plot_cb));
  app->plot_cb = NULL;
}

static void 
sg_application_class_init(SGapplicationClass *klass)
{
  GtkObjectClass *object_class;
  GObjectClass *gobject_class;
  SGapplicationClass *app_class;

  object_class = (GtkObjectClass*) klass;
  gobject_class = (GObjectClass*) klass;
  app_class = (SGapplicationClass*) klass;

  parent_class = (GtkObjectClass *)gtk_type_class (gtk_object_get_type ());

  app_signals[ADD_WORKSHEET] =
    gtk_signal_new ("add_worksheet",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, add_worksheet),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[ADD_MATRIX] =
    gtk_signal_new ("add_matrix",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, add_matrix),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[ADD_PLOT] =
    gtk_signal_new ("add_plot",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, add_plot),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[ADD_DATASET] =
    gtk_signal_new ("add_dataset",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, add_dataset),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, G_TYPE_OBJECT);
  app_signals[REMOVE_WORKSHEET] =
    gtk_signal_new ("remove_worksheet",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, remove_worksheet),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[REMOVE_MATRIX] =
    gtk_signal_new ("remove_matrix",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, remove_matrix),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[REMOVE_PLOT] =
    gtk_signal_new ("remove_plot",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, remove_plot),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[REMOVE_DATASET] =
    gtk_signal_new ("remove_dataset",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, remove_dataset),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, G_TYPE_OBJECT);
  app_signals[RENAME_WORKSHEET] =
    gtk_signal_new ("rename_worksheet",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, rename_worksheet),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[RENAME_MATRIX] =
    gtk_signal_new ("rename_matrix",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, rename_matrix),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[RENAME_PLOT] =
    gtk_signal_new ("rename_plot",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, rename_plot),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[RENAME_DATASET] =
    gtk_signal_new ("rename_dataset",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, rename_dataset),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[ACTIVATE_PLOT] =
    gtk_signal_new ("activate_plot",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, activate_plot),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[ACTIVATE_WORKSHEET] =
    gtk_signal_new ("activate_worksheet",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, activate_worksheet),
                    gtk_marshal_NONE__OBJECT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_OBJECT);
  app_signals[CHANGED] =
    gtk_signal_new ("changed",
                    GTK_RUN_LAST,
                    G_OBJECT_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGapplicationClass, changed),
                    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);

  app_class->add_worksheet = NULL;
  app_class->add_matrix = NULL;
  app_class->add_plot = NULL;
  app_class->add_dataset = NULL;
  app_class->remove_worksheet = NULL;
  app_class->remove_matrix = NULL;
  app_class->remove_plot = NULL;
  app_class->remove_dataset = NULL;
  app_class->rename_worksheet = NULL;
  app_class->rename_matrix = NULL;
  app_class->rename_plot = NULL;
  app_class->rename_dataset = NULL;
  app_class->activate_plot = NULL;
  app_class->activate_worksheet = NULL;
  app_class->changed = NULL;
 
  gobject_class->finalize = sg_application_finalize; 
  object_class->destroy = sg_application_destroy; 
}

void
sg_application_set_changed(SGapplication *app, gboolean changed)
{
  app->changed = changed;
  gtk_signal_emit_by_name(GTK_OBJECT(app),"changed");
}

gboolean 
sg_application_changed(SGapplication *app)
{
  return app->changed;
}

static void
set_changed (SGapplication *app, gpointer data)
{
  sg_application_set_changed(app, TRUE);
}

static gboolean
focus_in_event(GtkWidget *widget, GdkEventFocus *event, gpointer data)
{
  GList *list = NULL;
  SGapplication *app = SG_APPLICATION(data);

  if(GTK_IS_SG_PLOT(widget)){
    list = app->plots->list;
    while(list)
     {
       SGlistChild *child = (SGlistChild *)list->data;
       SGplot *plot;
       plot = SG_PLOT(child->object);

       if(GTK_WIDGET(plot) == widget){
          sg_application_set_active_plot(app, plot);
          break;
       }

       list = list->next;
     }

    gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(app->active_plot));
  } else if(GTK_IS_SG_WORKSHEET(widget)){

    list = app->worksheets->list;
    while(list)
     {
       SGlistChild *child = (SGlistChild *)list->data;
       SGworksheet *worksheet;
       worksheet = SG_WORKSHEET(child->object);

       if(GTK_WIDGET(worksheet) == widget){
          sg_application_set_active_worksheet(app, worksheet);
          break;
       }

       list = list->next;
     }
  }

  return FALSE;
}

void
sg_application_add_worksheet(SGapplication *app, SGworksheet *worksheet)
{
  app->num_worksheets++;
  app->last_worksheet++;
 
  sg_list_add(app->worksheets, G_OBJECT(worksheet), GTK_SHEET(worksheet)->name);
  gtk_object_set_data(GTK_OBJECT(worksheet), "application", app);
  g_object_unref(worksheet->clipboard);
  worksheet->clipboard = G_OBJECT(app->worksheet_cb);
  g_object_ref(worksheet->clipboard);
 
  app->active_worksheet = worksheet;
 
  gtk_signal_connect_object(GTK_OBJECT(worksheet), "changed",
                            GTK_SIGNAL_FUNC(set_changed), GTK_OBJECT(app));
 
  gtk_signal_connect(GTK_OBJECT(worksheet),"focus_in_event",
                     (GtkSignalFunc) focus_in_event, app);

  gtk_signal_emit(GTK_OBJECT(app),app_signals[ADD_WORKSHEET],worksheet);
  gtk_signal_emit(GTK_OBJECT(app),app_signals[ACTIVATE_WORKSHEET],worksheet);
  sg_application_set_changed(app, TRUE);
}

void
sg_application_add_matrix(SGapplication *app, SGworksheet *worksheet)
{
  GdkColor color;

  app->num_worksheets++;
  app->last_matrix++;
 
  sg_list_add(app->worksheets, G_OBJECT(worksheet), GTK_SHEET(worksheet)->name);
  gtk_object_set_data(GTK_OBJECT(worksheet), "application", app);
 
  gdk_color_parse("light yellow", &color);
  gdk_color_alloc(gdk_colormap_get_system(), &color);
  gtk_sheet_set_background(GTK_SHEET(worksheet), &color);
  gdk_color_parse("light blue", &color);
  gdk_color_alloc(gdk_colormap_get_system(), &color);
  gtk_sheet_set_grid(GTK_SHEET(worksheet), &color);
 
  app->active_worksheet = worksheet;
 
  gtk_signal_connect_object(GTK_OBJECT(worksheet), "changed",
                            GTK_SIGNAL_FUNC(set_changed), GTK_OBJECT(app));
 
  gtk_signal_connect(GTK_OBJECT(worksheet),"focus_in_event",
                     (GtkSignalFunc) focus_in_event, app);

  gtk_signal_emit(GTK_OBJECT(app),app_signals[ADD_MATRIX],worksheet);
  gtk_signal_emit(GTK_OBJECT(app),app_signals[ACTIVATE_WORKSHEET],worksheet);
  sg_application_set_changed(app, TRUE);
}

static void
plot_changed(SGapplication *app, gpointer data)
{
  if(app->layer_control){
    gchar *path = g_strdup(SG_LAYER_CONTROL(app->layer_control)->path);
    sg_layer_control_refresh(app->layer_control, path);
    g_free(path);
  }
}

void
sg_application_add_plot(SGapplication *app, SGplot *plot)
{
  GList *list = NULL;

  app->num_plots++;
  app->last_plot++;

  sg_list_add(app->plots, G_OBJECT(plot), plot->name); 
  gtk_object_set_data(GTK_OBJECT(plot), "application", app);

  if(plot->clipboard && G_IS_OBJECT(plot->clipboard)) 
    g_object_unref(plot->clipboard);
  plot->clipboard = G_OBJECT(app->plot_cb);
  g_object_ref(plot->clipboard);

/*
  list = plot->datasets->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data; 
    gint id = child->id;
    GtkObject *object;

    object = sg_list_get_by_id(app->datasets, id);
    if(!object){
      sg_list_add(app->datasets, child->object, child->name);
      child = sg_list_find(app->datasets, object);
      child->id = id;
    } else {
      SGlistChild *aux;
      aux = sg_list_find(app->datasets, child->name);
      if(!aux){
        sg_list_add(app->datasets, child->object, child->name);
      }
    }

    list = list->next;
  }
*/

  if(plot->datasets && G_IS_OBJECT(plot->datasets)) 
    g_object_unref(G_OBJECT(plot->datasets));
  plot->datasets = app->datasets;
  g_object_ref(G_OBJECT(app->datasets));
 
  app->active_plot = plot;
 
  gtk_signal_connect_object(GTK_OBJECT(plot), "changed",
                            (GtkSignalFunc)set_changed, GTK_OBJECT(app));
 
  gtk_signal_connect_object(GTK_OBJECT(plot), "add_layer",
                            (GtkSignalFunc)plot_changed, GTK_OBJECT(app));

  gtk_signal_connect_object(GTK_OBJECT(plot), "remove_layer",
                            (GtkSignalFunc)plot_changed, GTK_OBJECT(app));

  gtk_signal_connect(GTK_OBJECT(plot),"focus_in_event",
                     (GtkSignalFunc) focus_in_event, app);
 
  gtk_signal_emit(GTK_OBJECT(app),app_signals[ADD_PLOT],plot);
  gtk_signal_emit(GTK_OBJECT(app),app_signals[ACTIVATE_PLOT],plot);
  plot_changed(app, NULL);
  sg_application_set_changed(app, TRUE);
}

void
sg_application_add_dataset(SGapplication *app, SGdataset *dataset)
{
  SGlistChild *child;

  child = sg_list_add(app->datasets, G_OBJECT(dataset), dataset->name);
  app->last_dataset++;

  gtk_signal_emit(GTK_OBJECT(app),app_signals[ADD_DATASET],dataset);
  plot_changed(app, NULL);
  sg_application_set_changed(app, TRUE);
}

gboolean
sg_application_remove_worksheet(SGapplication *app, SGworksheet *worksheet)
{
  gboolean veto;
  gboolean deleting = FALSE;
  SGlistChild *child = NULL;

/*
  while(list) {
    SGdataset *dataset = SG_DATASET(list->data);

    if(sg_dataset_is_connected(dataset, worksheet)){
      if(!deleting){
        gchar text[1000];
        g_snprintf(text, 1000, _("There are datasets linked to this worksheet.\n The datasets will also be deleted.\n Do you really want to remove\n%s ?"), worksheet->name);
        if(sg_accept_dialog(text, 1) != SG_BUTTON_YES) return FALSE;
        deleting = TRUE;
        break;
      }
      if(deleting){
        sg_application_remove_dataset(app, dataset);
        list = app->datasets;
        continue;
      }
    }
    list = list->next;
  }

  veto = remove_worksheet(worksheet);
  if(!veto) return FALSE;
*/

  child = sg_list_find(app->worksheets, G_OBJECT(worksheet));

  if(!child) return FALSE;

  if(app->active_worksheet == worksheet) app->active_worksheet = NULL;
  sg_list_remove(app->worksheets, G_OBJECT(worksheet));
  app->num_worksheets--;

  gtk_signal_emit(GTK_OBJECT(app),app_signals[REMOVE_WORKSHEET],worksheet);
  sg_application_set_changed(app, TRUE);
  return TRUE;
}

gboolean
sg_application_remove_matrix(SGapplication *app, SGworksheet *worksheet)
{
  SGlistChild *child = NULL;

  child = sg_list_find(app->worksheets, G_OBJECT(worksheet));

  if(!child) return FALSE;

  if(app->active_worksheet == worksheet) app->active_worksheet = NULL;
  gtk_signal_emit(GTK_OBJECT(app),app_signals[REMOVE_MATRIX],worksheet);

  sg_list_remove(app->worksheets, G_OBJECT(worksheet));

  app->num_worksheets--;

  sg_application_set_changed(app, TRUE);
  return TRUE;
}

gboolean
sg_application_remove_plot(SGapplication *app, SGplot *plot)
{
  SGlistChild *child = NULL;

  if(plot == NULL) return FALSE;

  child = sg_list_find(app->plots, G_OBJECT(plot));

  if(!child) return FALSE;

  if(app->active_plot == plot) app->active_plot = NULL;
  gtk_signal_emit(GTK_OBJECT(app),app_signals[REMOVE_PLOT],plot);
  sg_list_remove(app->plots, G_OBJECT(plot));

  app->num_plots--;

  plot_changed(app, NULL);
  sg_application_set_changed(app, TRUE);
  return TRUE;
}

gboolean
sg_application_remove_dataset(SGapplication *app, SGdataset *dataset)
{
  SGplot *plot;
  GList *aux_plots;
  GList *aux_layers;
  GList *aux_datasets;
  SGdataset *parent;
//  gboolean deleting = force;
  gboolean veto = TRUE;
  gboolean ret_value = TRUE;

/*
  parent = SG_DATASET(dataset->parent);

  aux_plots = plots;
  while(aux_plots) {
    plot = SG_PLOT(aux_plots->data);
    aux_layers = plot->layers;
    while(aux_layers){
      aux_datasets = SG_LAYER(aux_layers->data)->datasets;
      while(aux_datasets){
        SGdataset *aux_data, *aux_parent;

        aux_data = SG_DATASET(aux_datasets->data);
        aux_parent = SG_DATASET(aux_data->parent);

        if(aux_parent == parent && !deleting){
          gchar text[1000];
          g_snprintf(text, 1000, _("This dataset is being used in plots.\n Do you really want to delete\n%s ?"), parent->real_data->name);
          veto = sg_accept_dialog(text, 1);
          if(!veto) return FALSE;
          deleting = TRUE;
        }
        if(deleting && aux_parent == parent){
          sg_layer_remove_dataset(SG_LAYER(aux_layers->data), aux_data);
          gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
          gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
          aux_datasets = SG_LAYER(aux_layers->data)->datasets;
        }else{
          aux_datasets = aux_datasets->next;
        }
      }
      aux_layers = aux_layers->next;
    }
    aux_plots = aux_plots->next;
  }
  veto = remove_dataset(parent);
*/

  ret_value = sg_list_remove(app->datasets, G_OBJECT(dataset));

  if(ret_value){
    gtk_signal_emit(GTK_OBJECT(app),app_signals[REMOVE_DATASET],dataset);
    if(G_IS_OBJECT(dataset))
      g_object_unref(G_OBJECT(dataset));

    plot_changed(app, NULL);
    sg_application_set_changed(app, TRUE);
  }
  return TRUE;
}

void
sg_application_rename_worksheet(SGapplication *app, SGworksheet *w)
{
  sg_list_set_name(app->worksheets, G_OBJECT(w), GTK_SHEET(w)->name);
  gtk_signal_emit(GTK_OBJECT(app),app_signals[RENAME_WORKSHEET],w);
  sg_application_set_changed(app, TRUE);
}

void
sg_application_rename_matrix(SGapplication *app, SGworksheet *w)
{
  sg_list_set_name(app->worksheets, G_OBJECT(w), GTK_SHEET(w)->name);
  gtk_signal_emit(GTK_OBJECT(app),app_signals[RENAME_MATRIX],w);
  sg_application_set_changed(app, TRUE);
}

void
sg_application_rename_plot(SGapplication *app, SGplot *p)
{
  sg_list_set_name(app->plots, G_OBJECT(p), p->name);
  gtk_signal_emit(GTK_OBJECT(app),app_signals[RENAME_PLOT],p);
  sg_application_set_changed(app, TRUE);
}

void
sg_application_rename_dataset(SGapplication *app, SGdataset *d)
{
  sg_list_set_name(app->datasets, G_OBJECT(d), d->name);
  gtk_signal_emit(GTK_OBJECT(app),app_signals[RENAME_DATASET],d);
  sg_application_set_changed(app, TRUE);
}

void
sg_application_refresh_datasets(SGapplication *app)
{
  GList *list;
  list = app->datasets->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    sg_dataset_refresh(SG_DATASET(child->object));
    list = list->next;
  }
}
    
void
sg_application_set_active_worksheet(SGapplication *app, SGworksheet *w)
{
  gtk_signal_emit(GTK_OBJECT(app),app_signals[ACTIVATE_WORKSHEET],w);
  app->active_worksheet = w;
}
void
sg_application_set_active_plot(SGapplication *app, SGplot *p)
{
  gtk_signal_emit(GTK_OBJECT(app),app_signals[ACTIVATE_PLOT],p);
  app->active_plot = p;
}

SGworksheet *
sg_application_active_worksheet(SGapplication *app)
{
  return app->active_worksheet;
}

SGplot *
sg_application_active_plot(SGapplication *app)
{
  return app->active_plot;
}

void
sg_application_set_path(SGapplication *app, const gchar *path)
{
  if(app->path) g_free(app->path);
  app->path = NULL;
  if(path) app->path = g_strdup(path);
}

void
sg_application_set_filename(SGapplication *app, const gchar *filename)
{
  if(app->filename) g_free(app->filename);
  app->filename = NULL;
  if(filename) app->filename = g_strdup(filename);
}

SGdataset *
sg_application_get_dataset_by_id(SGapplication *app, gint id)
{
  GObject *object;

  object = sg_list_get_by_id(app->datasets, id);
  if(object) return SG_DATASET(object);

  return NULL;
}

