/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_property_dialog.h"

#define SG_XYZ_DIALOG(obj)        GTK_CHECK_CAST (obj, sg_xyz_dialog_get_type (), SGxyzDialog)

typedef struct _SGxyzDialog SGxyzDialog;
typedef struct _SGxyzDialogClass SGxyzDialogClass;

static gchar *line_styles[] = 
{
  N_("None"),
  N_("Solid"),
  N_("Dotted"),
  N_("Dashed"),
  N_("Dot-dash"),
  N_("Dot-dot-dash"),
  N_("Dot-dash-dash"),
  NULL
};

struct _SGxyzDialog
{
  SGpropertyDialog parent;

  GtkWidget *xline_combo;
  GtkWidget *xline_width_spin;
  GtkWidget *xline_color_combo;
  GtkWidget *yline_combo;
  GtkWidget *zline_combo;
  gboolean use_z;
  GtkPlotData *dataset;
};

struct _SGxyzDialogClass
{
  SGpropertyDialogClass parent_class;
};

static void sg_xyz_dialog_class_init       (SGxyzDialogClass *klass);
static void sg_xyz_dialog_init             (SGxyzDialog *d);
static void sg_xyz_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_xyz_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_xyz_dialog_init_gui         (SGxyzDialog *d);

static SGpropertyDialogClass *parent_class = NULL;

GtkType
sg_xyz_dialog_get_type(void)
{
  static GtkType sg_xyz_dialog_type = 0;

  if (!sg_xyz_dialog_type)
    {
      GtkTypeInfo sg_xyz_dialog_info =
      {
        "SGxyzDialog",
        sizeof (SGxyzDialog),
        sizeof (SGxyzDialogClass),
        (GtkClassInitFunc) sg_xyz_dialog_class_init,
        (GtkObjectInitFunc) sg_xyz_dialog_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_xyz_dialog_type = gtk_type_unique (sg_property_dialog_get_type(), &sg_xyz_dialog_info);
    }
  return sg_xyz_dialog_type;
}

GtkWidget *
sg_xyz_dialog_new(GtkPlotData *dataset, gboolean use_z)
{
  GtkWidget *widget;

  widget = GTK_WIDGET(gtk_type_new(sg_xyz_dialog_get_type()));

  SG_XYZ_DIALOG(widget)->dataset = dataset;
  SG_XYZ_DIALOG(widget)->use_z = use_z;
  sg_xyz_dialog_init_gui(SG_XYZ_DIALOG(widget));

  return widget;
}

static void
sg_xyz_dialog_init(SGxyzDialog *d)
{
  SG_PROPERTY_DIALOG(d)->apply = sg_xyz_dialog_apply;
  SG_PROPERTY_DIALOG(d)->ok = sg_xyz_dialog_ok;
}

static void
sg_xyz_dialog_class_init(SGxyzDialogClass *klass)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) klass;

  parent_class = (SGpropertyDialogClass *)gtk_type_class (sg_property_dialog_get_type ());
}

static void
sg_xyz_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_xyz_dialog_apply(d, data);
}

static void     
init_dialog (SGxyzDialog *dialog)
{
  GtkPlotData *data = dialog->dataset;

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->xline_width_spin),
                            data->x_line.line_width);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->xline_color_combo),
                      data->x_line.color);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->xline_combo)->list),
                       data->x_line.line_style);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->yline_combo)->list),
                       data->y_line.line_style);
  if(dialog->use_z)
    gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->zline_combo)->list),
                         data->z_line.line_style);
}

static void     
sg_xyz_dialog_init_gui (SGxyzDialog *dialog)
{
  GtkWidget *frame;
  GtkWidget *label;
  GtkWidget *table;
  GtkRequisition req;
  GtkAdjustment *adj;

  frame = GTK_WIDGET(dialog);
  if(dialog->use_z)
    gtk_frame_set_label(GTK_FRAME(dialog), _("X-Y-Z lines") );
  else
    gtk_frame_set_label(GTK_FRAME(dialog), _("X-Y lines") );

  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

  table = gtk_table_new(6, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new( _("X line:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,0,1);

  dialog->xline_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->xline_combo,1,2,0,1);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->xline_combo)->entry), FALSE);
  sg_combo_set_items(GTK_COMBO(dialog->xline_combo), line_styles);
  label = gtk_label_new( _("Y line:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);

  gtk_widget_size_request(dialog->xline_combo, &req);
  req.width /= 2;

  dialog->yline_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->yline_combo,1,2,1,2);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->yline_combo)->entry), FALSE);
  sg_combo_set_items(GTK_COMBO(dialog->yline_combo), line_styles);

  if(dialog->use_z){
    label = gtk_label_new( _("Z line:") );
    gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);

    dialog->zline_combo = gtk_combo_new();
    gtk_table_attach_defaults(GTK_TABLE(table),dialog->zline_combo,1,2,2,3);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->zline_combo)->entry), FALSE);        sg_combo_set_items(GTK_COMBO(dialog->zline_combo), line_styles);
    gtk_widget_set_usize(dialog->zline_combo, req.width, req.height);
  }

  label = gtk_label_new( _("Width:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
  dialog->xline_width_spin = gtk_spin_button_new(adj, 0.5, 1);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->xline_width_spin), TRUE);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->xline_width_spin,1,2,3,4);

  label = gtk_label_new( _("Color:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,4,5);

  dialog->xline_color_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->xline_color_combo,1,2,4,5);

  gtk_widget_set_usize(dialog->xline_width_spin, req.width, req.height);
  gtk_widget_set_usize(dialog->xline_combo, req.width, req.height);
  gtk_widget_set_usize(dialog->yline_combo, req.width, req.height);

  init_dialog(dialog);
}

static void     
sg_xyz_dialog_apply     (SGpropertyDialog *d, gpointer _data)
{
  SGxyzDialog *dialog = SG_XYZ_DIALOG(d);
  GtkPlotData *data = dialog->dataset;
  GtkWidget *child;
  GdkColor line_color;
  GtkPlotLineStyle line_style; 
  gfloat line_width;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->xline_combo)->list)->selection->data;
  line_style = (GtkPlotLineStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->xline_combo)->list), child);

  line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->xline_width_spin));

  line_color = gtk_color_combo_get_selection(GTK_COLOR_COMBO(dialog->xline_color_combo));

  gtk_plot_data_set_x_attributes(data,
                                 line_style, 0, 0,
                                 line_width,
                                 &line_color);

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->yline_combo)->list)->selection->data;

  line_style =  (GtkPlotLineStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->yline_combo)->list), child);

  gtk_plot_data_set_y_attributes(data,
                                 line_style, 0, 0,
                                 line_width,
                                 &line_color);

  if(dialog->use_z){
    child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->zline_combo)->list)->selection->data;
    line_style =  (GtkPlotLineStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->zline_combo)->list), child);

    gtk_plot_data_set_z_attributes(data,
                                   line_style, 0, 0,
                                   line_width,
                                   &line_color);
  }
}

