/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_title_dialog.h"
#include "sg.h"
#include "sg_dialog.h"
#include "sg_property_dialog.h"

typedef struct
{
  SGlayer *layer;
  SGplot *plot;
  GtkPlotText *text;
  GtkPlotAxis *axis;

  GtkWidget *fg_combo;
  GtkWidget *bg_combo;
  GtkWidget *text_frame;
  GtkWidget *font_combo;
  GtkWidget *border_combo;
  GtkWidget *text_entry;
  GtkWidget *angle_spin;
  GtkWidget *x_spin;
  GtkWidget *y_spin;
  GtkWidget *center_item;
  GtkWidget *transparent_item;
} SGtitleDialog;

static void sg_title_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_title_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_title_dialog_init_gui         (GtkWidget *widget);

static gchar *border_styles[] = {
  N_("None"),
  N_("Line"),
  N_("Shadow"),
  NULL
};

GtkWidget *
sg_title_dialog_new(SGlayer *layer, GtkPlotAxis *axis)
{
  GtkWidget *widget;
  SGtitleDialog *dialog;

  dialog = g_new0(SGtitleDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_title_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_title_dialog_ok;
  dialog->layer = layer;
  dialog->axis = axis;
  dialog->plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent);
  dialog->text = &axis->title;
  sg_title_dialog_init_gui(widget);

  return widget;
}

static void
sg_title_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_title_dialog_apply(d, data);
}

static void
sg_title_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  SGtitleDialog *dialog = (SGtitleDialog *)d->data;
  GtkColorCombo *combo;
  GtkPlotAxis *axis;
  GtkPlot *plot;
  GdkColor color, fg, bg;
  GtkWidget *child;
  GtkPlotAxisPos the_axis = GTK_PLOT_AXIS_LEFT;
  const gchar *text;
  const gchar *font;
  gint height;
  gint angle;
  gboolean state;
  gdouble xoffset, yoffset;
  GtkPSFont *psfont;

  axis = dialog->axis;
  text = gtk_entry_get_text(GTK_ENTRY(dialog->text_entry));
  plot = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot;

  if(axis == plot->top) the_axis = GTK_PLOT_AXIS_TOP; 
  if(axis == plot->bottom) the_axis = GTK_PLOT_AXIS_BOTTOM; 
  if(axis == plot->left) the_axis = GTK_PLOT_AXIS_LEFT; 
  if(axis == plot->right) the_axis = GTK_PLOT_AXIS_RIGHT; 
 
  psfont = gtk_font_combo_get_psfont(GTK_FONT_COMBO(dialog->font_combo)); 
  font = psfont->psname;
  height = gtk_font_combo_get_font_height(GTK_FONT_COMBO(dialog->font_combo)); 
  axis->title.transparent = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->transparent_item));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->border_combo)->list)->selection->data;
  axis->title.border = (GtkPlotBorderStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->border_combo)->list), child);

  combo = GTK_COLOR_COMBO(dialog->fg_combo);
  color = gtk_color_combo_get_selection(combo);
  fg = color;
  combo = GTK_COLOR_COMBO(dialog->bg_combo);
  color = gtk_color_combo_get_selection(combo);
  bg = color;

  angle = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->angle_spin));

  xoffset = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->x_spin));
  yoffset = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->y_spin));

  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->center_item));
  dialog->text->justification = state ? GTK_JUSTIFY_CENTER : GTK_JUSTIFY_LEFT;

  gtk_plot_axis_set_title(axis, text);
  gtk_plot_axis_move_title(axis, angle, xoffset, yoffset);
  gtk_plot_axis_title_set_attributes(axis,
                                     font, height, angle, &fg, &bg,
                                     axis->title.transparent, 
                                     dialog->text->justification); 

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(dialog->plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(dialog->plot));
}

static void
new_fg(GtkColorCombo *combo, gint selection, GdkColor *color, gpointer data)
{
  GtkWidget *text_entry;
  GdkGC *gc;
  GtkStyle *style;
 
  text_entry = GTK_WIDGET(data);

  gc = gdk_gc_new(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child)->pixmap);

  gdk_gc_set_foreground(gc, color);

  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child)->pixmap,
                     gc,
                     TRUE,
                     4,4,33,13);

  gdk_gc_unref(gc); 
  gtk_widget_draw(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child, NULL);

  style = gtk_style_copy(text_entry->style);
  style->fg[0] = *color;
  gtk_widget_set_style (text_entry, style);
  gtk_style_unref(style);

} 

static void
new_bg(GtkColorCombo *combo, gint selection, GdkColor *color, gpointer data)
{
  GtkWidget *text_entry;
  GdkGC *gc;
  GtkStyle *style;
 
  text_entry = GTK_WIDGET(data);

  gc = gdk_gc_new(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child)->pixmap);

  gdk_gc_set_foreground(gc, color);

  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child)->pixmap,
                     gc,
                     TRUE,
                     4,4,33,13);

  gdk_gc_unref(gc); 
  gtk_widget_draw(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child, NULL);

  style = gtk_style_copy(text_entry->style);
  style->base[0] = *color;
  gtk_widget_set_style (text_entry, style);
  gtk_style_unref(style);
} 

static void 
init_dialog(SGtitleDialog *dialog)
{
  GtkPSFont *font;
  GtkPlotText *text;
  GdkColor color;

  text = dialog->text;

  gtk_entry_set_text(GTK_ENTRY(dialog->text_entry), text->text);

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->fg_combo), text->fg);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->bg_combo), text->bg);

  color = gtk_color_combo_get_selection(GTK_COLOR_COMBO(dialog->fg_combo));
  new_fg(GTK_COLOR_COMBO(dialog->fg_combo), 0, &color, dialog->text_entry);

  color = gtk_color_combo_get_selection(GTK_COLOR_COMBO(dialog->fg_combo));
  new_bg(GTK_COLOR_COMBO(dialog->bg_combo), 0, &color, dialog->text_entry);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->border_combo)->list),
                       text->border);

  font = gtk_psfont_get_by_name(text->font);
  gtk_font_combo_select(GTK_FONT_COMBO(dialog->font_combo),
                        font->family,
                        FALSE,
                        FALSE,
                        text->height);
}

static void 
sg_title_dialog_init_gui (GtkWidget *widget)
{
  GtkWidget *frame;
  GtkWidget *main_box, *main_table, *table;
  GtkRequisition req;
  GtkPlotText *text;
  GtkAdjustment *adj;
  gboolean state;
  GtkWidget *label;
  SGtitleDialog *dialog;

  dialog = (SGtitleDialog *)SG_PROPERTY_DIALOG(widget)->data;
  text = dialog->text;

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);
  gtk_container_add(GTK_CONTAINER(widget), main_box);

  main_table = gtk_table_new(2,2,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 2);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 2);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);
/*----------------------------------------------*/
  dialog->text_frame = sg_entry_new();
  gtk_frame_set_shadow_type (GTK_FRAME (dialog->text_frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), dialog->text_frame, 0, 2, 1, 2, GTK_FILL, GTK_FILL, 5, 5);

  dialog->font_combo = SG_ENTRY(dialog->text_frame)->font_combo;
  dialog->text_entry = SG_ENTRY(dialog->text_frame)->text_entry;
/*--------------------------------------*/
  frame = gtk_frame_new( _("Position") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 0, 1, 0, 1, GTK_FILL, 0, 5, 5);

  table=gtk_table_new(2, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new( _("X: ") );
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1., .05, .10, .10);
  dialog->x_spin = gtk_spin_button_new(adj, 0, 3);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->x_spin), TRUE);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->x_spin), text->x);
  gtk_widget_size_request(dialog->x_spin, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->x_spin, req.width, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->x_spin, 1, 2, 0, 1);

  label = gtk_label_new( _("Y: ") );
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1., .05, .10, .10);
  dialog->y_spin = gtk_spin_button_new(adj, 0, 3);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->y_spin), TRUE);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->y_spin), text->y);
  gtk_widget_size_request(dialog->y_spin, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->y_spin, req.width, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->y_spin, 1, 2, 1, 2);

  label = gtk_label_new( _("Rotate (deg): ") );
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);


  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 270., 90., 90., 0.);
  dialog->angle_spin = gtk_spin_button_new(adj, 90, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->angle_spin), TRUE);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->angle_spin), text->angle);
  gtk_entry_set_editable(GTK_ENTRY(dialog->angle_spin), FALSE);
  gtk_widget_size_request(dialog->angle_spin, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->angle_spin, req.width, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->angle_spin, 1, 2, 2, 3);

  dialog->center_item = gtk_check_item_new_with_label( _("Centered") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->center_item, 0, 3, 3, 4);
  state = (text->justification == GTK_JUSTIFY_CENTER);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->center_item), state);

/*                               
  dialog->hide_item = gtk_check_item_new_with_label( _("Hide") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->hide_item, 0, 2, 4, 5);
  state = (axis->label_mask != 0);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->hide_item), state);
*/
 /*--------------------------------------*/
  frame = gtk_frame_new( _("Properties") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 1, 2, 0, 1, GTK_FILL, 0, 5, 5);

  table=gtk_table_new(4, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new( _("Foreground ") );
/*  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
*/
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  dialog->fg_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->fg_combo, 1, 2, 0, 1);

  label = gtk_label_new( _("Background ") );
/*  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
*/
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);

  dialog->bg_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->bg_combo, 1, 2, 1, 2);

  label = gtk_label_new( _("Border ") );
/*  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
*/
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
  dialog->border_combo = gtk_combo_new();
  gtk_widget_set_usize(dialog->border_combo, req.width, req.height);
  sg_combo_set_items(GTK_COMBO(dialog->border_combo), border_styles);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->border_combo)->entry), FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->border_combo, 1, 2, 2, 3);

  dialog->transparent_item = gtk_check_item_new_with_label( _("Transparent") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->transparent_item, 0, 2, 3, 4);
  state = text->transparent;
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->transparent_item), state);

  /* connect signals */

  gtk_signal_connect(GTK_OBJECT(dialog->fg_combo), "changed",
                     GTK_SIGNAL_FUNC(new_fg), dialog->text_entry);

  gtk_signal_connect(GTK_OBJECT(dialog->bg_combo), "changed",
                     GTK_SIGNAL_FUNC(new_bg), dialog->text_entry);

  init_dialog(dialog);

  gtk_widget_show_all (main_box);
}

GtkWidget *
sg_title_dialog(SGlayer *layer, 
                GtkWidget *ok_button,
		GtkWidget *apply_button,
		GtkWidget *cancel_button)
{
  GtkWidget *notebook;
  GtkPlot *plot;
  GtkWidget *dialog;

  plot = GTK_PLOT_CANVAS_PLOT(layer)->plot;
  notebook = gtk_notebook_new();

  if(strcmp(SG_PLUGIN(layer->plugin)->name,"layer_3d") == 0){
      dialog = sg_title_dialog_new(layer, plot->bottom);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("X") ));
      dialog = sg_title_dialog_new(layer, plot->left);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Y") ));
      dialog = sg_title_dialog_new(layer, plot->top);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Z") ));
  }
  if(strcmp(SG_PLUGIN(layer->plugin)->name,"layer_2d") == 0){
      dialog = sg_title_dialog_new(layer, plot->left);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Left") ));
      dialog = sg_title_dialog_new(layer, plot->right);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Right") ));
      dialog = sg_title_dialog_new(layer, plot->top);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Top") ));
      dialog = sg_title_dialog_new(layer, plot->bottom);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Bottom") ));
  }
  if(strcmp(SG_PLUGIN(layer->plugin)->name,"layer_polar") == 0){
      dialog = sg_title_dialog_new(layer, plot->bottom);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Angles") ));
      dialog = sg_title_dialog_new(layer, plot->left);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("R") ));
  }

  gtk_widget_show_all(notebook);

  return notebook;
}
