/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <string.h>
#include "sg_text_dialog.h"
#include "sg_dialog.h"
#include "sg_stock.h"
#include "sg_misc.h"
#include "sg_entry.h"
#include "sg.h"

typedef struct _SGtextdialogClass  SGtextdialogClass;

struct _SGtextdialogClass
{
  GtkWindowClass parent_class;
};

static void sg_text_dialog_class_init		(SGtextdialogClass *klass);
static void sg_text_dialog_init                 (SGtextdialog *dialog);
static void sg_text_dialog_destroy              (GtkObject *dialog);
static void update_text				(GtkWidget *widget, 
						 gpointer data);
static void new_fg				(GtkColorCombo *combo, 
                                                 gint selection, 
                                                 GdkColor *color, 
                                                 gpointer data);
static void new_bg				(GtkColorCombo *combo, 
                                                 gint selection, 
                                                 GdkColor *color, 
                                                 gpointer data);
static void init_dialog				(SGtextdialog *dialog);

static gchar *border_styles[] = {
  N_("None"),
  N_("Line"),
  N_("Shadow"),
  NULL
};

static GtkWindowClass *parent_class = NULL;


GtkType
sg_text_dialog_get_type (void)
{
  static GtkType dialog_type = 0;
  
  if (!dialog_type)
    {
      GtkTypeInfo dialog_info =
      {
	"SGtextdialog",
	sizeof (SGtextdialog),
	sizeof (SGtextdialogClass),
	(GtkClassInitFunc) sg_text_dialog_class_init,
	(GtkObjectInitFunc) sg_text_dialog_init,
	/* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };
      
      dialog_type = gtk_type_unique (gtk_window_get_type(), &dialog_info);
    }
  
  return dialog_type;
}

GtkWidget*
sg_text_dialog_new (GtkPlotText *text)
{
  GtkWidget *widget;
  SGtextdialog *dialog;

  widget = gtk_widget_new (sg_text_dialog_get_type(), NULL);

  dialog = SG_TEXT_DIALOG(widget);
  dialog->text = text;
  dialog->original_text = *text;
  dialog->original_text.text = g_strdup(text->text);
  dialog->original_text.font = g_strdup(text->font);

  gtk_entry_set_text(GTK_ENTRY(dialog->text_entry), text->text);
  gtk_entry_select_region(GTK_ENTRY(dialog->text_entry), 0, strlen(text->text));

  init_dialog(dialog);
  gtk_widget_show_all(dialog->main_box);

  gtk_widget_grab_focus(dialog->text_entry);

  return widget;
}

static void
sg_text_dialog_class_init (SGtextdialogClass *klass)
{
  GtkWidgetClass *widget_class;
  GtkObjectClass *object_class;
  
  widget_class = (GtkWidgetClass*) klass;
  parent_class = (GtkWindowClass*) gtk_type_class (gtk_window_get_type ());
  object_class = (GtkObjectClass*) klass;

  object_class->destroy = sg_text_dialog_destroy;
 
}

static void
cancel_action(GtkWidget *widget, gpointer data)
{
  SGtextdialog *dialog = SG_TEXT_DIALOG(widget);
  g_free(dialog->text->text);
  g_free(dialog->text->font);
  *dialog->text = dialog->original_text;
  dialog->original_text.text = NULL;
  dialog->original_text.font = NULL;

  if(widget && GTK_IS_WIDGET(widget)) gtk_widget_destroy(widget);
}

static void
sg_text_dialog_init (SGtextdialog *dialog)
{
  GtkWidget *main_box;
  GtkWidget *main_table;
  GtkWidget *frame;
  GtkWidget *box;
  GtkWidget *table;
  GtkRequisition req;
  GtkAdjustment *adj;

  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_window_set_title(GTK_WINDOW(dialog), _("SciGraphica: Edit Text") );

  /* Create widgets */
  dialog->main_box = main_box = gtk_hbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);
  gtk_container_add(GTK_CONTAINER(dialog), main_box);

  /* Create widgets */
  main_table = gtk_table_new(2,3,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 2);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 2);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);
 /*----------------------------------------------*/

  frame = sg_entry_new();
  gtk_frame_set_label(GTK_FRAME(frame), _("Text") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 0, 2, 1, 2);

  dialog->text_frame = frame;
  dialog->font_combo = SG_ENTRY(frame)->font_combo;
  dialog->text_entry = SG_ENTRY(frame)->text_entry;


  /*--------------------------------------*/
  frame = gtk_frame_new( _("Position") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 0, 1, 0, 1);

  table=gtk_table_new(2, 3, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new( _("Rotate (deg): ") ), 0, 1, 0, 1);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 270., 90., 90., 0.);
  dialog->angle_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->angle_spin), TRUE);
  gtk_entry_set_editable(GTK_ENTRY(dialog->angle_spin), FALSE);
  gtk_widget_size_request(dialog->angle_spin, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->angle_spin, req.width, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->angle_spin, 1, 2, 0, 1);

  dialog->center_item = gtk_check_item_new_with_label( _("Centered") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->center_item, 0, 2, 1, 2);

 /*--------------------------------------*/

  frame = gtk_frame_new( _("Properties") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach_defaults(GTK_TABLE(main_table), frame, 1, 2, 0, 1);

  table=gtk_table_new(3, 3, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new( _("Foreground") ), 0, 1, 0, 1);
  dialog->fg_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->fg_combo, 1, 2, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new( _("Background") ), 0, 1, 1, 2);
  dialog->bg_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->bg_combo, 1, 2, 1, 2);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new( _("Border") ), 0, 1, 2, 3);
  dialog->border_combo = gtk_combo_new();
  gtk_widget_set_usize(dialog->border_combo, req.width, req.height);

  gtk_table_attach_defaults(GTK_TABLE(table), dialog->border_combo, 1, 2, 2, 3);

  dialog->transparent_item = gtk_check_item_new_with_label( _("Transparent") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->transparent_item, 0, 2, 3, 4);

  /* Action area */

  box = gtk_vbutton_box_new();
  gtk_container_set_border_width (GTK_CONTAINER (box), 5);
  gtk_table_attach_defaults(GTK_TABLE(main_table), box, 2, 3, 0, 1);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(box), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(box), 5);
  gtk_widget_show(box);

  dialog->ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
  gtk_box_pack_start (GTK_BOX (box), dialog->ok_button, TRUE, TRUE, 0);

  dialog->apply_button = gtk_button_new_from_stock(GTK_STOCK_APPLY);
  gtk_box_pack_start (GTK_BOX (box), dialog->apply_button, TRUE, TRUE, 0);

  dialog->cancel_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
  gtk_box_pack_start (GTK_BOX (box), dialog->cancel_button, TRUE, TRUE, 0);

  /* connect signals */

  gtk_signal_connect(GTK_OBJECT(dialog->fg_combo), "changed",
                     GTK_SIGNAL_FUNC(new_fg), dialog->text_entry);

  gtk_signal_connect(GTK_OBJECT(dialog->bg_combo), "changed",
                     GTK_SIGNAL_FUNC(new_bg), dialog->text_entry);

  gtk_signal_connect (GTK_OBJECT(SG_TEXT_DIALOG(dialog)->ok_button),
                      "clicked",
                      GTK_SIGNAL_FUNC (update_text), dialog);

  gtk_signal_connect (GTK_OBJECT(SG_TEXT_DIALOG(dialog)->apply_button),
                      "clicked",
                      GTK_SIGNAL_FUNC (update_text), dialog);

  gtk_signal_connect_object (GTK_OBJECT(SG_TEXT_DIALOG(dialog)->cancel_button),
                      "clicked",
                      GTK_SIGNAL_FUNC (cancel_action), GTK_OBJECT(dialog));
}

static void
update_text(GtkWidget *widget, gpointer data)
{
  SGtextdialog *dialog;
  GtkWidget *child;
  const gchar *font;
  gint height;
  gint transparent;
  gint angle;
  GtkJustification justification;
  GtkPlotBorderStyle border;
  GtkColorCombo *combo;
  GdkColor color, bg, fg;
  gchar *text;
  gboolean state;
  GtkPSFont *psfont;

  dialog = (SGtextdialog *)data;

  psfont = gtk_font_combo_get_psfont(GTK_FONT_COMBO(dialog->font_combo));
  font = psfont->psname;
  height = gtk_font_combo_get_font_height(GTK_FONT_COMBO(dialog->font_combo));
  transparent = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->transparent_item));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->border_combo)->list)->selection->data;
  border = (GtkPlotBorderStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->border_combo)->list), child);

  combo = GTK_COLOR_COMBO(dialog->fg_combo);
  color = gtk_color_combo_get_selection(combo);
  fg = color;

  combo = GTK_COLOR_COMBO(dialog->bg_combo);
  color = gtk_color_combo_get_selection(combo);
  bg = color;

  angle = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->angle_spin));
  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->center_item));
  justification = state ? GTK_JUSTIFY_CENTER : GTK_JUSTIFY_LEFT;


  text = g_strdup(gtk_entry_get_text(GTK_ENTRY(dialog->text_entry)));

  g_free(dialog->text->font);
  g_free(dialog->text->text);

  dialog->text->angle = angle;
  dialog->text->border = border;
  dialog->text->font = g_strdup(font);
  dialog->text->height = height;
  dialog->text->fg = fg;
  dialog->text->bg = bg;
  dialog->text->transparent = transparent;
  dialog->text->justification = justification;
  dialog->text->text = text;

  if(dialog->original_text.text) g_free(dialog->original_text.text);
  if(dialog->original_text.font) g_free(dialog->original_text.font);
  dialog->original_text = *dialog->text;
  dialog->original_text.text = g_strdup(text);
  dialog->original_text.font = g_strdup(font);

  if(widget == dialog->ok_button && GTK_IS_WIDGET(dialog))
          gtk_widget_destroy(GTK_WIDGET(dialog));

}

static void
new_fg(GtkColorCombo *combo, gint selection, GdkColor *color, gpointer data)
{
  GtkWidget *text_entry;
  GtkStyle *style;
  GdkGC *gc;

  text_entry = GTK_WIDGET(data);

  gc = gdk_gc_new(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child)->pixmap);
  gdk_gc_set_foreground(gc, color);

  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child)->pixmap,
                     gc,
                     TRUE,
                     4,4,33,13);

  gtk_widget_draw(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child, NULL);

  style = gtk_style_copy(text_entry->style);
  style->fg[GTK_STATE_NORMAL] = *color;
  gtk_widget_set_style (text_entry, style);
  gtk_style_unref(style);
  gdk_gc_unref(gc);
}


static void
new_bg(GtkColorCombo *combo, gint selection, GdkColor *color, gpointer data)
{
  GtkWidget *text_entry;
  GtkStyle *style;
  GdkGC *gc;

  text_entry = GTK_WIDGET(data);

  gc = gdk_gc_new(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child)->pixmap);
  gdk_gc_set_foreground(gc, color);

  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child)->pixmap,
                     gc,
                     TRUE,
                     4,4,33,13);

  gtk_widget_draw(GTK_BIN(GTK_COMBO_BUTTON(combo)->button)->child, NULL);

  style = gtk_style_copy(text_entry->style);
  style->base[GTK_STATE_NORMAL] = *color;
  gtk_widget_set_style (text_entry, style);
  gtk_style_unref(style);
  gdk_gc_unref(gc);
}

static void
init_dialog(SGtextdialog *dialog)
{
  GtkPSFont *font;
  GtkPlotText *text;
  GdkColor color;
  gint state;

  text = dialog->text;

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->fg_combo), text->fg);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->bg_combo), text->bg);

  color = gtk_color_combo_get_selection(GTK_COLOR_COMBO(dialog->fg_combo));
  new_fg(GTK_COLOR_COMBO(dialog->fg_combo), 0, &color, dialog->text_entry);

  color = gtk_color_combo_get_selection(GTK_COLOR_COMBO(dialog->bg_combo));
  new_bg(GTK_COLOR_COMBO(dialog->bg_combo), 0, &color, dialog->text_entry);

  state = (text->justification == GTK_JUSTIFY_CENTER);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->center_item), state);

  state = text->transparent;
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->transparent_item), state);

  sg_combo_set_items(GTK_COMBO(dialog->border_combo), border_styles);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->border_combo)->entry), FALSE);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->border_combo)->list), 
                       text->border);


  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->angle_spin), text->angle);

  font = gtk_psfont_get_by_name(text->font);
  gtk_font_combo_select(GTK_FONT_COMBO(dialog->font_combo),
                        font->family,
                        FALSE,
                        FALSE,
                        text->height);

}

static void 
sg_text_dialog_destroy                 (GtkObject *text_dialog)
{
  SGtextdialog *dialog;

  dialog = SG_TEXT_DIALOG(text_dialog);

  if(dialog->original_text.font) g_free(dialog->original_text.font);
  if(dialog->original_text.text) g_free(dialog->original_text.text);
  dialog->original_text.font = NULL;
  dialog->original_text.text = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (text_dialog);
}
