/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_style_dialog.h"
#include "sg.h"
#include "sg_property_dialog.h"
#include "sg_dialog.h"

#define SG_STYLE_DIALOG(obj)        GTK_CHECK_CAST (obj, sg_style_dialog_get_type (), SGstyleDialog)

typedef struct _SGstyleDialog SGstyleDialog;
typedef struct _SGstyleDialogClass SGstyleDialogClass;

struct _SGstyleDialog
{
  SGpropertyDialog parent;

  GtkWidget *main_box;
  GtkWidget *text_entry;
  GtkWidget *dataset_combo;
  GtkWidget *legends_item;
  SGlayer *layer;
  GtkPlotData *dataset;
  SGpropertyDialog *dialog;
  SGpluginStyle *plugin;
};

struct _SGstyleDialogClass
{
  SGpropertyDialogClass parent_class;
};

static void sg_style_dialog_class_init       (SGstyleDialogClass *klass);
static void sg_style_dialog_init             (SGstyleDialog *d);
static void sg_style_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_style_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_style_dialog_init_gui         (SGstyleDialog *d);

static SGpropertyDialogClass *parent_class = NULL;

GtkType
sg_style_dialog_get_type(void)
{
  static GtkType sg_style_dialog_type = 0;

  if (!sg_style_dialog_type)
    {
      GtkTypeInfo sg_style_dialog_info =
      {
        "SGstyleDialog",
        sizeof (SGstyleDialog),
        sizeof (SGstyleDialogClass),
        (GtkClassInitFunc) sg_style_dialog_class_init,
        (GtkObjectInitFunc) sg_style_dialog_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_style_dialog_type = gtk_type_unique (sg_property_dialog_get_type(), &sg_style_dialog_info);
    }
  return sg_style_dialog_type;
}

GtkWidget *
sg_style_dialog_new(SGlayer *layer)
{
  GtkWidget *widget;

  widget = GTK_WIDGET(gtk_type_new(sg_style_dialog_get_type()));

  SG_STYLE_DIALOG(widget)->layer = layer;
  sg_style_dialog_init_gui(SG_STYLE_DIALOG(widget));

  return widget;
}

static void
sg_style_dialog_init(SGstyleDialog *d)
{
  d->dialog = NULL;
  SG_PROPERTY_DIALOG(d)->apply = sg_style_dialog_apply;
  SG_PROPERTY_DIALOG(d)->ok = sg_style_dialog_ok;
}

static void
sg_style_dialog_class_init(SGstyleDialogClass *klass)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) klass;

  parent_class = (SGpropertyDialogClass *)gtk_type_class (sg_property_dialog_get_type ());
}

static void
sg_style_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_style_dialog_apply(d, data);
}

static void
sg_style_dialog_apply(SGpropertyDialog *pdialog, gpointer _data)
{
  SGstyleDialog *dialog = SG_STYLE_DIALOG(pdialog);
  const gchar *legend_text;
  SGplot *plot;

  sg_property_dialog_apply(dialog->dialog);

  /* Legends */
  legend_text = gtk_entry_get_text(GTK_ENTRY(dialog->text_entry));

  gtk_plot_data_set_legend(dialog->dataset, legend_text);
  dialog->dataset->show_legend = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->legends_item));

  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(dialog->layer)->parent);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
}

static void
sg_style_dialog_select(GtkWidget *widget, GtkWidget *child, gpointer data)
{
  SGstyleDialog *dialog = SG_STYLE_DIALOG(data);

  dialog->dataset = GTK_PLOT_DATA(g_object_get_data(G_OBJECT(child),"dataset"));

  dialog->plugin = SG_DATASET(dialog->dataset->link)->constructor;

  if(dialog->dialog) gtk_container_remove(GTK_CONTAINER(dialog->main_box), GTK_WIDGET(dialog->dialog));

  dialog->dialog = SG_PROPERTY_DIALOG(sg_plugin_style_dialog(dialog->plugin, dialog->dataset));
  gtk_box_pack_start(GTK_BOX(dialog->main_box), GTK_WIDGET(dialog->dialog), FALSE, FALSE, 0);
  gtk_widget_show_all(GTK_WIDGET(dialog->dialog));

  /* Legends */
  if(dialog->dataset->legend)
    gtk_entry_set_text(GTK_ENTRY(dialog->text_entry), dialog->dataset->legend);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->legends_item), dialog->dataset->show_legend);
}

static void
sg_style_dialog_init_gui (SGstyleDialog *dialog)
{
  GtkWidget *label;
  GtkWidget *box;
  GtkWidget *frame;
  GList *list;
  GtkWidget *item;
  SGlayer *layer = dialog->layer;

  /* Create widgets */
  dialog->main_box = gtk_vbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(dialog->main_box), 5);
  gtk_container_add(GTK_CONTAINER(dialog), dialog->main_box);

  /* Datasets combo */
  box = gtk_hbox_new (FALSE,5);
  gtk_box_pack_start(GTK_BOX(dialog->main_box), box, FALSE, FALSE, 0);

  label = gtk_label_new( _("Select Dataset:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);

  gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);

  dialog->dataset_combo = gtk_combo_new();
  gtk_box_pack_start(GTK_BOX(box), dialog->dataset_combo, TRUE, TRUE, 0);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->dataset_combo)->entry), FALSE);

  list = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot)->data_sets;
  while(list){
    GtkPlotData *dataset; 
    GList *aux = GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot)->data_sets;
    gint n = 1;

    dataset = GTK_PLOT_DATA(list->data);
    while(aux){
      GtkPlotData *aux_data = GTK_PLOT_DATA(aux->data);
      if(aux_data != dataset && strcmp(dataset->name, aux_data->name) == 0) n++;
      if(aux_data == dataset) break;
      aux = aux->next;
    }
    if(n > 1){
      gchar name[200];
      g_snprintf(name, 200, "%s (%d)", dataset->name, n);
      item = gtk_list_item_new_with_label(name);
    } else {
      item = gtk_list_item_new_with_label(dataset->name);
    }
    g_object_set_data(G_OBJECT(item), "dataset", dataset);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(GTK_COMBO(dialog->dataset_combo)->list), item);
    list = list->next;
  }

  /* Legends */

  frame = gtk_frame_new( _("Legend") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_end(GTK_BOX(dialog->main_box), frame, FALSE, FALSE, 0);

  box = gtk_hbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(frame), box);
  dialog->text_entry = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(box), dialog->text_entry, TRUE, TRUE, 0);

  dialog->legends_item = gtk_check_item_new_with_label( _("show legend") );
  gtk_box_pack_start(GTK_BOX(box), dialog->legends_item, FALSE, FALSE, 0);

  dialog->dataset = GTK_PLOT_DATA(GTK_PLOT(GTK_PLOT_CANVAS_PLOT(layer)->plot)->data_sets->data);

  if(dialog->dataset->legend)
    gtk_entry_set_text(GTK_ENTRY(dialog->text_entry), dialog->dataset->legend);

  /* Connect signals */

  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(dialog->dataset_combo)->list),
                     "select_child",
                     GTK_SIGNAL_FUNC(sg_style_dialog_select),
                     GTK_OBJECT(dialog));

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->dataset_combo)->list), 0);
}


