/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_page_dialog.h"
#include "sg_property_dialog.h"
#include "sg.h"

typedef struct
{
  GtkWidget *size_combo; 
  GtkWidget *unit_combo; 
  GtkWidget *width_entry; 
  GtkWidget *height_entry; 
  GtkWidget *orient_combo; 
  GtkWidget *color_combo; 
  GtkWidget *grid_item; 
  GtkWidget *antialias_item; 
  GtkWidget *transparent_item; 
  SGplot *plot;
  gint pts_width;
  gint pts_height;
  gint units;
} SGpageDialog;

static void sg_page_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_page_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_page_dialog_init_gui         (GtkWidget *frame);

GtkWidget *
sg_page_dialog_new(SGplot *plot)
{
  GtkWidget *widget;
  SGpageDialog *dialog;

  dialog = g_new0(SGpageDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_page_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_page_dialog_ok;
  dialog->plot = plot;
  sg_page_dialog_init_gui(widget);

  return widget;
}

static void
sg_page_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_page_dialog_apply(d, data);
}

void
sg_page_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  SGpageDialog *dialog = (SGpageDialog *)d->data;
  GtkWidget *child;
  GtkPlotCanvas *canvas;
  GtkColorCombo *combo;
  GdkColor color;
  gint page_size, units;
  gint orientation;
  gboolean do_art = FALSE;
  SGplot *plot = dialog->plot;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->size_combo)->list)->selection->data;
  page_size = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->size_combo)->list),
                                      child);

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->unit_combo)->list)->selection->data;
  units = (SGpageUnit)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->unit_combo)->list), child);
  plot->page_units = units;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->orient_combo)->list)->selection->data;
  orientation = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->orient_combo)->list),
                                        child);

  combo = GTK_COLOR_COMBO(dialog->color_combo);
  color = gtk_color_combo_get_selection(combo);

  canvas = GTK_PLOT_CANVAS(plot);

  canvas->background = color;

  canvas->show_grid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->grid_item));
  do_art = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->antialias_item));
  canvas->transparent = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->transparent_item));
/*
  resize =  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ratio_item));
*/

  sg_plot_antialias(plot, do_art);

  if(page_size == GTK_PLOT_CUSTOM){
    gdouble width, height;
    width = atof(gtk_entry_get_text(GTK_ENTRY(dialog->width_entry))); 
    height = atof(gtk_entry_get_text(GTK_ENTRY(dialog->height_entry))); 
    switch(units){
     case SG_UNIT_MM:
       dialog->pts_width = GINT(width * PTS_MM); 
       dialog->pts_height = GINT(height * PTS_MM); 
       break;
     case SG_UNIT_CM:
       dialog->pts_width = GINT(width * PTS_CM); 
       dialog->pts_height = GINT(height * PTS_CM); 
       break;
     case SG_UNIT_IN:
       dialog->pts_width = GINT(width * PTS_IN); 
       dialog->pts_height = GINT(height * PTS_IN); 
       break;
     case SG_UNIT_PIXEL:
       dialog->pts_width = GINT(width * PTS_PIXEL);
       dialog->pts_height = GINT(height * PTS_PIXEL);
       break;
     case SG_UNIT_PT:
     default:
       dialog->pts_width = GINT(width * PTS_PT); 
       dialog->pts_height = GINT(height * PTS_PT); 
       break;
    }
  }

  sg_plot_set_size(plot, page_size, dialog->pts_width, dialog->pts_height, orientation);
}

static void
new_units(GtkWidget *widget, GtkWidget *child, gpointer data)
{
  SGpageDialog *dialog = (SGpageDialog *)data;
  gchar label[100];
  SGpageUnit new_units;
  gdouble width, height;

  new_units = (SGpageUnit)gtk_list_child_position(GTK_LIST(widget), child);

  switch(new_units){
   case SG_UNIT_MM:
       width = (gdouble)dialog->pts_width / (gdouble)PTS_MM; 
       height = (gdouble)dialog->pts_height / (gdouble)PTS_MM; 
       break;
   case SG_UNIT_CM:
       width = (gdouble)dialog->pts_width / (gdouble)PTS_CM; 
       height = (gdouble)dialog->pts_height / (gdouble)PTS_CM; 
       break;
   case SG_UNIT_IN:
       width = (gdouble)dialog->pts_width / (gdouble)PTS_IN; 
       height = (gdouble)dialog->pts_height / (gdouble)PTS_IN; 
       break;
   case SG_UNIT_PIXEL:
       width = (gdouble)dialog->pts_width / (gdouble)PTS_PIXEL;
       height = (gdouble)dialog->pts_height / (gdouble)PTS_PIXEL;
       break;
   case SG_UNIT_PT:
   default:
       width = (gdouble)dialog->pts_width / (gdouble)PTS_PT; 
       height = (gdouble)dialog->pts_height / (gdouble)PTS_PT; 
       break;
  }

  sprintf(label, "%.*f", 3, width);
  gtk_entry_set_text(GTK_ENTRY(dialog->width_entry), label);
  sprintf(label, "%.*f", 3, height);
  gtk_entry_set_text(GTK_ENTRY(dialog->height_entry), label);
  dialog->units = new_units;
}


static void
new_size(GtkWidget *widget, GtkWidget *child, gpointer data)
{
  SGpageDialog *dialog = (SGpageDialog *)data;
  gint new_size;
  
  new_size = gtk_list_child_position(GTK_LIST(widget), child);

  gtk_widget_set_sensitive(dialog->width_entry, FALSE);
  gtk_widget_set_sensitive(dialog->height_entry, FALSE);
  switch(new_size){
   case GTK_PLOT_LETTER:
      dialog->pts_width = GTK_PLOT_LETTER_W;
      dialog->pts_height = GTK_PLOT_LETTER_H;
      break;
   case GTK_PLOT_LEGAL:
      dialog->pts_width = GTK_PLOT_LEGAL_W;
      dialog->pts_height = GTK_PLOT_LEGAL_H;
      break;
   case GTK_PLOT_A4:
      dialog->pts_width = GTK_PLOT_A4_W;
      dialog->pts_height = GTK_PLOT_A4_H;
      break;
   case GTK_PLOT_EXECUTIVE:
      dialog->pts_width = GTK_PLOT_EXECUTIVE_W;
      dialog->pts_height = GTK_PLOT_EXECUTIVE_H;
      break;
   case GTK_PLOT_CUSTOM:
      dialog->pts_width = SG_PLOT(dialog->plot)->page_width;
      dialog->pts_height = SG_PLOT(dialog->plot)->page_height;
      gtk_widget_set_sensitive(dialog->width_entry, TRUE);
      gtk_widget_set_sensitive(dialog->height_entry, TRUE);
      break;
   }

 gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->unit_combo)->list), dialog->units);

}

static void
init_dialog(SGpageDialog *dialog)
{
  SGplot *plot = dialog->plot;

  gchar *units[] = { N_("PS points"),
                     N_("mm"),
                     N_("cm"),
                     N_("inches"),
                     N_("pixels"),
                     NULL };
  gchar *orientations[] = { N_("Portrait"),
                            N_("Landscape"),
                            NULL };

  gchar *page_sizes[] = { N_("Letter"),
                          N_("Legal"),
                          N_("A4"),
                          N_("Executive"),
                          N_("Custom"),
                          NULL };

  dialog->pts_width = GTK_PLOT_CANVAS(plot)->pixmap_width;
  dialog->pts_height = GTK_PLOT_CANVAS(plot)->pixmap_height;
  dialog->units = plot->page_units;

  sg_combo_set_items(GTK_COMBO(dialog->unit_combo), units);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->unit_combo)->list), plot->page_units);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->unit_combo)->entry), FALSE);

  sg_combo_set_items(GTK_COMBO(dialog->orient_combo), orientations);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->orient_combo)->list), plot->orientation);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->orient_combo)->entry), FALSE);

  sg_combo_set_items(GTK_COMBO(dialog->size_combo), page_sizes);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->size_combo)->list), plot->page_size);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->size_combo)->entry), FALSE);

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->color_combo), GTK_PLOT_CANVAS(plot)->background);

  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->grid_item), 
                              GTK_PLOT_CANVAS(plot)->show_grid);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->antialias_item), plot->antialias);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->transparent_item), 
                              GTK_PLOT_CANVAS(plot)->transparent);
}

static void
sg_page_dialog_init_gui (GtkWidget *widget)
{
  GtkWidget *frame;
  GtkWidget *main_table;
  GtkWidget *table;
  GtkWidget *label;
  GtkRequisition req;
  SGpageDialog *dialog;

  dialog = (SGpageDialog *)SG_PROPERTY_DIALOG(widget)->data;

  /* Create widgets */
  main_table = gtk_table_new (1, 1, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 10);
  gtk_container_add(GTK_CONTAINER(widget), main_table);

  /* Poperties */

  frame = gtk_frame_new( _("Page properties") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach (GTK_TABLE (main_table), frame, 0, 1, 0, 1, 0, 0, 5, 5);

  table=gtk_table_new(9, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new( _("Page size ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 0, 0, 0, 0);
  label = gtk_label_new( _("Width ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 0, 0, 0, 0);
  label = gtk_label_new( _("Height ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 0, 0, 0, 0);
  label = gtk_label_new( _("Units ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, 0, 0, 0, 0);
  label = gtk_label_new( _("Orientation ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5, 0, 0, 0, 0);
  label = gtk_label_new( _("Background ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6, 0, 0, 0, 0);

  dialog->size_combo = gtk_combo_new();
  dialog->unit_combo = gtk_combo_new();
  dialog->orient_combo = gtk_combo_new();
  dialog->width_entry = gtk_entry_new();
  dialog->height_entry = gtk_entry_new();
  dialog->color_combo = gtk_color_combo_new();
/*
  ratio_item = gtk_check_item_new_with_label("Preserve aspect ratio");
*/
  dialog->grid_item = gtk_check_item_new_with_label( _("Show guiding grid") );
  dialog->antialias_item = gtk_check_item_new_with_label( _("Antialiased") );
  dialog->transparent_item = gtk_check_item_new_with_label( _("Transparent") );

  gtk_widget_size_request(dialog->width_entry, &req);
/*
  req.width = (gdouble)req.width * .75;
*/

  gtk_widget_set_usize(dialog->width_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->height_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->size_combo, req.width, req.height);
  gtk_widget_set_usize(dialog->unit_combo, req.width, req.height);
  gtk_widget_set_usize(dialog->orient_combo, req.width, req.height);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->size_combo)->entry), FALSE);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->unit_combo)->entry), FALSE);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->orient_combo)->entry), FALSE);

  gtk_table_attach(GTK_TABLE(table), dialog->size_combo, 1, 2, 0, 1, 0, 0, 0, 0);
  gtk_table_attach(GTK_TABLE(table), dialog->width_entry, 1, 2, 1, 2, 0, 0, 0, 0);
  gtk_table_attach(GTK_TABLE(table), dialog->height_entry, 1, 2, 2, 3, 0, 0, 0, 0);
  gtk_table_attach(GTK_TABLE(table), dialog->unit_combo, 1, 2, 3, 4, 0, 0, 0, 0);
  gtk_table_attach(GTK_TABLE(table), dialog->orient_combo, 1, 2, 4, 5, 0, 0, 0, 0);
  gtk_table_attach(GTK_TABLE(table), dialog->color_combo, 1, 2, 5, 6, 0, 0, 0, 0);
/*
  gtk_table_attach_defaults(GTK_TABLE(table), ratio_item, 0, 2, 6, 7);
*/
  gtk_table_attach(GTK_TABLE(table), dialog->grid_item, 0, 2, 6, 7, 0, 0, 0, 0);
  gtk_table_attach(GTK_TABLE(table), dialog->antialias_item, 0, 2, 7, 8, 0, 0, 0, 0);
  gtk_table_attach(GTK_TABLE(table), dialog->transparent_item, 0, 2, 8, 9, 0, 0, 0, 0);

  /* connect signals */
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(dialog->size_combo)->list), 
                      "select_child",
                      GTK_SIGNAL_FUNC (new_size),
                      dialog);

  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(dialog->unit_combo)->list), 
                      "select_child",
                      GTK_SIGNAL_FUNC (new_units),
                      dialog);

  init_dialog(dialog);

  sg_entry_set_numeric(GTK_ENTRY(dialog->width_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(dialog->height_entry), 3);
/*
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ratio_item), TRUE);
*/
}
