/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_legends_dialog.h"
#include "sg.h"
#include "sg_dialog.h"
#include "sg_property_dialog.h"

typedef struct
{
  GtkWidget *border_combo; 
  GtkWidget *fg_combo; 
  GtkWidget *bg_combo; 
  GtkWidget *left_entry;
  GtkWidget *top_entry;
  GtkWidget *font_combo;
  GtkWidget *legends_item;
  GtkWidget *transparent_item;
  GtkPlot *plot;
  SGlayer *layer;
} SGlegendsDialog;

static void sg_legends_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_legends_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_legends_dialog_init_gui         (GtkWidget *frame);

static gchar *border_styles[] = {
  N_("None"),
  N_("Line"),
  N_("Shadow"),
  NULL
};

GtkWidget *
sg_legends_dialog_new(SGlayer *layer)
{
  GtkWidget *widget;
  SGlegendsDialog *dialog;

  dialog = g_new0(SGlegendsDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_legends_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_legends_dialog_ok;
  dialog->layer = layer;
  dialog->plot = GTK_PLOT_CANVAS_PLOT(layer)->plot;
  sg_legends_dialog_init_gui(widget);

  return widget;
}

static void
sg_legends_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_legends_dialog_apply(d, data);
}

static void
sg_legends_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  SGplot *parent;
  GtkWidget *child;
  GtkColorCombo *combo;
  GdkColor fg, bg;
  gdouble x, y;
  const gchar *font_name;
  gint font_height;
  SGlegendsDialog *dialog = (SGlegendsDialog *)d->data;
  GtkPlot *plot = dialog->plot;
  GtkPSFont *psfont;

  child = GTK_WIDGET(GTK_LIST(GTK_COMBO(dialog->border_combo)->list)->selection->data);
  plot->legends_border = (GtkPlotBorderStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->border_combo)->list), child);

  x = atof(gtk_entry_get_text(GTK_ENTRY(dialog->left_entry)));
  y = atof(gtk_entry_get_text(GTK_ENTRY(dialog->top_entry)));
  plot->legends_x = x;
  plot->legends_y = y;

  combo = GTK_COLOR_COMBO(dialog->fg_combo);
  fg = gtk_color_combo_get_selection(combo);

  combo = GTK_COLOR_COMBO(dialog->bg_combo);
  bg = gtk_color_combo_get_selection(combo);

  psfont = gtk_font_combo_get_psfont(GTK_FONT_COMBO(dialog->font_combo));
  font_name = psfont->psname;
  font_height = gtk_font_combo_get_font_height(GTK_FONT_COMBO(dialog->font_combo));

  gtk_plot_legends_set_attributes(plot, font_name, font_height, &fg, &bg);

  plot->show_legends = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->legends_item));
  plot->legends_attr.transparent = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->transparent_item));

  parent = SG_PLOT(GTK_PLOT_CANVAS_CHILD(dialog->layer)->parent);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(parent));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(parent));
}

static void
init_dialog(SGlegendsDialog *dialog)
{
  GtkPSFont *font;
  gchar label[100];
  GtkPlot *plot = dialog->plot;

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->fg_combo), plot->legends_attr.fg);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->bg_combo), plot->legends_attr.bg);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->border_combo)->list), plot->legends_border);

  sprintf(label, "%*.*f", 1, 3, plot->legends_x);
  gtk_entry_set_text(GTK_ENTRY(dialog->left_entry), label);
  sprintf(label, "%*.*f", 1, 3, plot->legends_y);
  gtk_entry_set_text(GTK_ENTRY(dialog->top_entry), label);

  font = gtk_psfont_get_by_name(plot->legends_attr.font);
  gtk_font_combo_select(GTK_FONT_COMBO(dialog->font_combo),
                        font->family,
                        font->bold,
                        font->italic,
                        plot->legends_attr.height);

  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->legends_item),
                              plot->show_legends);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->transparent_item),
                              plot->legends_attr.transparent);

}

static void
sg_legends_dialog_init_gui (GtkWidget *dialog_frame)
{
  GtkWidget *frame;
  GtkWidget *main_box, *main_table;
  GtkWidget *table;
  GtkRequisition req;
  SGlegendsDialog *dialog;

  dialog = (SGlegendsDialog *)SG_PROPERTY_DIALOG(dialog_frame)->data;

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);
  gtk_container_add(GTK_CONTAINER(dialog_frame), main_box);

  main_table = gtk_table_new(2,2,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 2);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 2);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);
/*----------------------------------------------*/
  frame = gtk_frame_new( _("Position") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 0, 1, 0, 1, 0, GTK_FILL, 0, 0);

  table=gtk_table_new(3, 3, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("left offset ") ), 0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("top offset ") ), 0, 1, 2, 3);
  dialog->left_entry = gtk_entry_new();
  dialog->top_entry = gtk_entry_new();

  gtk_widget_size_request(dialog->left_entry, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->left_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->top_entry, req.width, req.height);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Units in % of page size") ), 0, 3, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), dialog->left_entry, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->top_entry, 1, 2, 2, 3);

  
 /*--------------------------------------*/
  frame = gtk_frame_new( _("Properties") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 1, 2, 0, 1, 0, 0, 0, 0);

  table=gtk_table_new(5, 3, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Foreground") ), 0, 1, 0, 1);
  dialog->fg_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->fg_combo, 1, 2, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Background") ), 0, 1, 1, 2);
  dialog->bg_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->bg_combo, 1, 2, 1, 2);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Border") ), 0, 1, 2, 3);
  dialog->border_combo = gtk_combo_new();
  gtk_widget_set_usize(dialog->border_combo, req.width, req.height);
  sg_combo_set_items(GTK_COMBO(dialog->border_combo), border_styles);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->border_combo)->entry), FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->border_combo, 1, 2, 2, 3);

  dialog->transparent_item = gtk_check_item_new_with_label( _("transparent") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->transparent_item, 0, 2, 3, 4);
  dialog->legends_item = gtk_check_item_new_with_label( _("show legends") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->legends_item, 0, 2, 4, 5);


/*--------------------------------------*/
  frame = gtk_frame_new( _("Font") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 0, 2, 1, 2, GTK_EXPAND|GTK_FILL, 0, 0, 0);

  dialog->font_combo = gtk_font_combo_new();
  gtk_container_add(GTK_CONTAINER(frame), dialog->font_combo);
  gtk_container_set_border_width(GTK_CONTAINER(dialog->font_combo), 5);
/*--------------------------------------*/
  /* connect signals */
  init_dialog(dialog);

  sg_entry_set_numeric(GTK_ENTRY(dialog->left_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(dialog->top_entry), 3);
}
