/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_dataset_dialog.h"
#include "sg_widgets_enums.h"
#include "sg.h"
#include "sg_dataset.h"
#include "sg_layer.h"
#include "sg_dialog.h"
#include "sg_misc_dialogs.h"
#include "pixmaps/eye.xpm"

#define NUM_SYMBOLS 10
#define NUM_SYMBOL_STYLES 3
#define NUM_LINE_STYLES 7
#define NUM_CONNECTORS 6

typedef struct
{
  SGpropertyDialog parent;

  SGplot *plot;
  SGlist *datasets;
  SGapplication *app;

  GtkTooltips *datasets_tooltip;
  GtkTooltips *layer_tooltip;

  GtkWidget *event_box1;
  GtkWidget *event_box2;

  SGlayer *layer;
  GtkPlotData *dataset;

  GtkWidget *datasets_list;
  GtkWidget *layer_list;

  GList *old_datasets;
  GList *new_datasets;

  GtkWidget *layer_popup;
  GtkWidget *dataset_popup;

} SGlayerDatasetDialog;

static void sg_layer_dataset_dialog_apply   (SGpropertyDialog *d, gpointer data);
static void sg_layer_dataset_dialog_ok      (SGpropertyDialog *d, gpointer data);
static void sg_layer_dataset_dialog_cancel      (SGpropertyDialog *d, gpointer data);
static void sg_layer_dataset_dialog_init_gui(GtkWidget *frame);
static void layer_clist_fill_datasets	    (SGlayerDatasetDialog *dialog);
static void layer_clist_add_dataset	    (SGlayerDatasetDialog *dialog,
					     GtkPlotData *dataset);
static void datasets_clist_fill_datasets    (SGlayerDatasetDialog *dialog);
static void datasets_clist_add_dataset	    (SGlayerDatasetDialog *dialog,
					     SGdataset *dataset);
static void build_popups		    (SGlayerDatasetDialog *dialog);

GtkWidget *
sg_layer_dataset_dialog_new(SGlayer *layer)
{
  GtkWidget *widget;
  SGlayerDatasetDialog *dialog;

  dialog = g_new0(SGlayerDatasetDialog, 1);  
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_layer_dataset_dialog_apply; 
  SG_PROPERTY_DIALOG(widget)->ok = sg_layer_dataset_dialog_ok; 
  SG_PROPERTY_DIALOG(widget)->cancel = sg_layer_dataset_dialog_cancel; 
  dialog->layer = layer;
  dialog->plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent);
  dialog->datasets = dialog->plot->datasets;
  dialog->app = SG_APPLICATION(g_object_get_data(G_OBJECT(GTK_PLOT_CANVAS_CHILD(layer)->parent), "application"));
  sg_layer_dataset_dialog_init_gui(widget);

  return widget;
}

static void
sg_layer_dataset_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_layer_dataset_dialog_apply(d, data);
}

static void
sg_layer_dataset_dialog_cancel(SGpropertyDialog *d, gpointer data)
{
  SGlayerDatasetDialog *dialog = (SGlayerDatasetDialog *)d->data;
  GList *list = dialog->new_datasets;
  while(list){
    GtkPlotData *new_data = GTK_PLOT_DATA(list->data);
    SGdataset *parent = SG_DATASET(new_data->link);
    sg_dataset_remove_child(parent, new_data);
    dialog->new_datasets = g_list_remove_link(dialog->new_datasets, list);
    g_list_free_1(list);
    list = dialog->new_datasets;
  }
}

static GtkPlotData*
get_layer_selection(SGlayerDatasetDialog *dialog, gint *row)
{
  GList *selection = NULL;
  GtkPlotData *layer_data = NULL;

  selection = GTK_CLIST(dialog->layer_list)->selection;
  if(!selection) return NULL;

  *row = GPOINTER_TO_INT(selection->data);
  layer_data = GTK_PLOT_DATA(gtk_clist_get_row_data(GTK_CLIST(dialog->layer_list), *row));

  return layer_data;
}

static SGdataset *
get_dataset_selection(SGlayerDatasetDialog *dialog, gint *row)
{
  GList *selection = NULL;
  SGdataset *dataset = NULL;

  selection = GTK_CLIST(dialog->datasets_list)->selection;
  if(!selection) return NULL;

  *row = GPOINTER_TO_INT(selection->data);
  dataset = SG_DATASET(gtk_clist_get_row_data(GTK_CLIST(dialog->datasets_list), *row));

  return dataset;
}


static void 
sg_layer_dataset_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  SGlayerDatasetDialog *dialog = (SGlayerDatasetDialog *)d->data;
  GList *list = NULL;
  GList *new_list = NULL;
  GList *aux_list = NULL;
  GtkPlotData *row_dataset = NULL;
  GtkPlotData *plot_data = NULL;
  gint i;

  /* WE REMOVE DATASETS THAT ARE IN THE LAYER BUT NOT IN THE CLIST */

  for(i = 0; i < GTK_CLIST(dialog->layer_list)->rows; i++){
    row_dataset = GTK_PLOT_DATA(gtk_clist_get_row_data(GTK_CLIST(dialog->layer_list), i));
    new_list = g_list_append(new_list, row_dataset);
  }

  list = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot->data_sets;
  while(list){
    GList *next = list->next;
    gboolean found = FALSE;
    plot_data = GTK_PLOT_DATA(list->data);
    aux_list = new_list;

    while(aux_list){
      row_dataset = GTK_PLOT_DATA(aux_list->data);
      if(plot_data->link == row_dataset->link) { /* same data */
/*
        new_list = g_list_remove_link(new_list, aux_list);
        g_list_free_1(aux_list);
*/
        found = TRUE;
        break;
      } else {
        aux_list = aux_list->next;
      }
    }
    if(!found){
      /* remove from plot */
      sg_layer_remove_dataset(dialog->layer, SG_DATASET(plot_data->link));
    }
    
    list = next;
  }

  /* ADD NEW DATASETS */
  /* WE ADD REMAINING DATASETS THAT ARE IN THE CLIST */
  /* WE BRING FROM THE PREVIOUS LIST THOSE THAT WERE ALREADY THERE */
  /* WE CREATE THOSE THAT ARE IN THE CLIST BUT NOT IN THE LAYER */

  list = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot->data_sets;
  GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot->data_sets = NULL;
  aux_list = new_list;
  while(aux_list){
    GtkPlotData *row_data;
    GtkPlotData *plot_data = NULL;
    gboolean visible;
    SGdataset *parent;
    gboolean found = FALSE;
    GList *datasets;

    row_data = GTK_PLOT_DATA(aux_list->data);
    visible = GTK_WIDGET_VISIBLE(GTK_WIDGET(row_data));
    parent = SG_DATASET(row_data->link); 

    datasets = list;
    while(datasets){
      plot_data = GTK_PLOT_DATA(datasets->data);
      if(plot_data->link == row_data->link) { /* same data */
        list = g_list_remove_link(list, datasets);
        g_list_free_1(datasets);
        found = TRUE;
        break;
      } else {
        datasets = datasets->next;
      }
    }

    if(!found){
      plot_data = sg_layer_add_dataset_autosymbol(dialog->layer, parent);
      if(!visible) gtk_widget_hide(GTK_WIDGET(plot_data));
      sg_dataset_remove_child(parent, row_data);
    } else {
      gtk_plot_add_data(GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot, plot_data); 
    }

    new_list = g_list_remove_link(new_list, aux_list);
    g_list_free_1(aux_list);
    aux_list = new_list;
  }

  /* FREE NEW DATASETS THAT ARE NOT IN THE LAYER */
  /* THESE DATASET WERE CREATED AND LATER DESTROYED OR REMOVED FROM LAYER */

  aux_list = dialog->new_datasets;
  while(aux_list){
    GtkPlotData *row_data = GTK_PLOT_DATA(aux_list->data);
    gboolean found = FALSE;
    list = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot->data_sets;
    while(list){
      plot_data = GTK_PLOT_DATA(list->data);
      if(plot_data->link == row_data->link) { found = TRUE; break; }
      list = list->next;
    }
    if(!found){  
      sg_dataset_remove_child(SG_DATASET(row_data->link), row_data);
    }
    dialog->new_datasets = g_list_remove_link(dialog->new_datasets, aux_list);
    g_list_free_1(aux_list);
    aux_list = dialog->new_datasets;
  }
  dialog->new_datasets = NULL;

  /* REFRESH DIALOG */
  layer_clist_fill_datasets(dialog);
  datasets_clist_fill_datasets(dialog);

  /* REFRESH PLOT */
/*
  sg_layer_refresh_datasets(dialog->layer);
*/
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(dialog->plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(dialog->plot));
  sg_plot_set_active_layer(dialog->plot, dialog->plot->active_layer);
}



static gboolean
add_dataset(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  SGdataset *dataset = NULL;
  GtkPlotData *new_data = NULL;
  gint i, row, nrows;

  dataset = get_dataset_selection(dialog, &row);
  if(!dataset) return TRUE;

  nrows = GTK_CLIST(dialog->layer_list)->rows;
  for(i = 0; i < nrows; i++){
    new_data = GTK_PLOT_DATA(gtk_clist_get_row_data(GTK_CLIST(dialog->layer_list), i));
    if(SG_DATASET(new_data->link) == dataset) break;
    new_data = NULL;
  }

  gtk_clist_remove(GTK_CLIST(dialog->datasets_list), row);
  if(!new_data) {
    new_data = sg_dataset_new_child(dataset);
    sg_dataset_add_child(dataset, new_data);
    dialog->new_datasets = g_list_append(dialog->new_datasets, new_data);
    if(dataset->name && strlen(dataset->name) > 0)
      gtk_plot_data_set_legend(new_data, dataset->name);
  }
  gtk_widget_show(GTK_WIDGET(new_data));
  layer_clist_add_dataset(dialog, new_data);

  return TRUE;
}

static gboolean
restore_dataset(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  SGdataset *dataset = NULL;
  GtkPlotData *layer_data = NULL;
  gint row;

  layer_data = get_layer_selection(dialog, &row);
  if(!layer_data) return TRUE;

  dataset = SG_DATASET(layer_data->link);

  gtk_clist_remove(GTK_CLIST(dialog->layer_list), row);
  datasets_clist_add_dataset(dialog, dataset);

  return TRUE;
}

static gint
change_layer_tooltip(GtkCList *clist, gint row, gint col, GdkEvent *event, SGlayerDatasetDialog *dialog)
{
  GList *selection = NULL;
  GtkPlotData *layer_data = NULL;
  SGdataset *dataset = NULL;

  selection = GTK_CLIST(clist)->selection;
  if(!selection) {
    gtk_tooltips_set_tip(GTK_TOOLTIPS(dialog->layer_tooltip),
                         dialog->event_box1,
  		         _("Select dataset"),
 		         _("Select dataset"));
    return FALSE;
  }

  layer_data = GTK_PLOT_DATA(gtk_clist_get_row_data(GTK_CLIST(clist), row));
  dataset = SG_DATASET(layer_data->link);

  gtk_tooltips_set_tip(GTK_TOOLTIPS(dialog->layer_tooltip),
                       dialog->event_box1,
 		       dataset->description,
 		       dataset->description);
  return FALSE;
}

static gint
change_datasets_tooltip(GtkCList *clist, gint row, gint col, GdkEvent *event, SGlayerDatasetDialog *dialog)
{
  GList *selection = NULL;
  SGdataset *dataset = NULL;

  selection = GTK_CLIST(clist)->selection;
  if(!selection) {
    gtk_tooltips_set_tip(GTK_TOOLTIPS(dialog->datasets_tooltip),
                         dialog->event_box2,
  		         _("Select dataset"),
 		         _("Select dataset"));
    return FALSE;
  }

  dataset = SG_DATASET(gtk_clist_get_row_data(GTK_CLIST(clist), row));

  gtk_tooltips_set_tip(GTK_TOOLTIPS(dialog->datasets_tooltip),
                       dialog->event_box2,
 		       dataset->description,
 		       dataset->description);
  return FALSE;
}

static void
show_popup(GtkWidget *widget, GdkEventButton *event, GtkWidget *popup)
{
  GdkModifierType mods;

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);

  if(!event) return;

  if(mods & GDK_BUTTON3_MASK)
    gtk_menu_popup(GTK_MENU(popup), NULL, NULL, NULL, NULL,
                   event->button, event->time);
}

static void
sg_layer_dataset_dialog_init_gui (GtkWidget *frame)
{
  GtkWidget *main_box;
  GtkWidget *main_table;
  GtkWidget *button_box2;
  GtkWidget *button_box;
/*
  GtkWidget *action_area;
  GtkWidget *new_button, *delete_button, *show_button;
  GtkWidget *edit_button, *edit_layer_button;
*/
  GtkWidget *sw;
  GtkWidget *left_button, *right_button;
  GdkColormap *colormap;
  SGlayerDatasetDialog *dialog;
  gchar *titles1[] = { N_("Datasets"), ""};
  gchar *titles2[] = {"", N_("Layer datasets"), ""};

  dialog = (SGlayerDatasetDialog *)SG_PROPERTY_DIALOG(frame)->data;
  colormap = gdk_colormap_get_system();

  /* Create widgets */
  main_box = gtk_vbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 10);
  gtk_container_add(GTK_CONTAINER(frame), main_box);

  main_table = gtk_table_new(2, 3, FALSE);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);
  gtk_box_pack_start(GTK_BOX(main_box), main_table, TRUE, TRUE, 0);
/********************************************************************/
/*
  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_table_attach(GTK_TABLE (main_table), action_area, 0, 1, 0, 1, GTK_FILL, 0, 5, 5);

  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_table_attach(GTK_TABLE (main_table), action_area, 0, 1, 0, 1, GTK_FILL, 0, 0, 0);

  new_button = gtk_button_new_with_label ( _("New") );
  gtk_box_pack_start (GTK_BOX (action_area), new_button, TRUE, TRUE, 0);
  gtk_widget_show (new_button);

  delete_button = gtk_button_new_with_label ( _("Delete") );
  gtk_box_pack_start (GTK_BOX (action_area), delete_button, TRUE, TRUE, 0);
  gtk_widget_show (delete_button);

  edit_button = gtk_button_new_with_label ( _("Edit") );
  gtk_box_pack_start (GTK_BOX (action_area), edit_button, TRUE, TRUE, 0);
  gtk_widget_show (edit_button);

  gtk_table_attach(GTK_TABLE (main_table), gtk_vseparator_new(),
		   1, 2, 0, 1, 0, 0, 0, 0);

  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_table_attach(GTK_TABLE (main_table), action_area, 2, 3, 0, 1, GTK_FILL, 0, 0, 0);

  show_button = gtk_button_new_with_label ( _("Show/Hide") );
  gtk_box_pack_start (GTK_BOX (action_area), show_button, TRUE, TRUE, 0);
  gtk_widget_show (show_button);

  edit_layer_button = gtk_button_new_with_label ( _("Edit") );
  gtk_box_pack_start (GTK_BOX (action_area), edit_layer_button, TRUE, TRUE, 0);
  gtk_widget_show (edit_layer_button);
*/
/********************************************************************/
  button_box2 = gtk_vbox_new(FALSE, 0);
  gtk_table_attach(GTK_TABLE (main_table), button_box2,
                   1, 2, 1, 2, 0, GTK_EXPAND|GTK_FILL, 5, 5);

  button_box = gtk_hbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(button_box2), button_box, TRUE, FALSE, 0);

  left_button = gtk_button_new();
  gtk_container_add(GTK_CONTAINER(left_button),
                    gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT));
  gtk_widget_set_usize(left_button, 20, 20);
  gtk_box_pack_start(GTK_BOX(button_box), left_button, FALSE, FALSE, 0);

  right_button = gtk_button_new();
  gtk_container_add(GTK_CONTAINER(right_button),
                    gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT));
  gtk_widget_set_usize(right_button, 20, 20);
  gtk_box_pack_start(GTK_BOX(button_box), right_button, FALSE, FALSE, 0);

  gtk_signal_connect(GTK_OBJECT(left_button), "clicked",
                     (GtkSignalFunc) restore_dataset, dialog);
  gtk_signal_connect(GTK_OBJECT(right_button), "clicked",
                     (GtkSignalFunc) add_dataset, dialog);
/********************************************************************/
  dialog->event_box2 = gtk_event_box_new();
  gtk_table_attach(GTK_TABLE (main_table), dialog->event_box2,
                   0, 1, 1, 2,
                   GTK_FILL|GTK_EXPAND,
                   GTK_FILL|GTK_EXPAND, 0, 0);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_container_set_border_width(GTK_CONTAINER(sw), 5);
  gtk_widget_set_usize(sw, 300, 200);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(dialog->event_box2), sw);

  dialog->datasets_list = gtk_clist_new_with_titles(1, titles1);
  gtk_clist_set_row_height(GTK_CLIST(dialog->datasets_list), 20);
  gtk_clist_set_column_min_width(GTK_CLIST(dialog->datasets_list), 0, 20);
  gtk_clist_set_column_min_width(GTK_CLIST(dialog->datasets_list), 1, 400);
  gtk_container_add(GTK_CONTAINER(sw), dialog->datasets_list);

  dialog->datasets_tooltip = gtk_tooltips_new();
  gtk_tooltips_set_tip(GTK_TOOLTIPS(dialog->datasets_tooltip),
                       dialog->event_box2,
		       _("Select dataset"),
		       _("Select dataset"));
  gtk_tooltips_enable(GTK_TOOLTIPS(dialog->datasets_tooltip));
  gtk_tooltips_set_delay(GTK_TOOLTIPS(dialog->datasets_tooltip), 0);

/********************************************************************/
  dialog->event_box1 = gtk_event_box_new();
  gtk_table_attach(GTK_TABLE (main_table), dialog->event_box1,
                   2, 3, 1, 2,
                   GTK_FILL|GTK_EXPAND,
                   GTK_FILL|GTK_EXPAND, 0, 0);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_container_set_border_width(GTK_CONTAINER(sw), 5);
  gtk_widget_set_usize(sw, 300, 200);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(dialog->event_box1), sw);

  dialog->layer_list = gtk_clist_new_with_titles(2, titles2);
  gtk_clist_set_reorderable(GTK_CLIST(dialog->layer_list), TRUE);
  gtk_clist_set_row_height(GTK_CLIST(dialog->layer_list), 20);
  gtk_clist_set_column_min_width(GTK_CLIST(dialog->layer_list), 0, 20);
  gtk_clist_set_column_min_width(GTK_CLIST(dialog->layer_list), 1, 400);
  gtk_container_add(GTK_CONTAINER(sw), dialog->layer_list);

  dialog->layer_tooltip = gtk_tooltips_new();
  gtk_tooltips_set_tip(GTK_TOOLTIPS(dialog->layer_tooltip),
                       dialog->event_box1,
		       _("Select dataset"),
		       _("Select dataset"));
  gtk_tooltips_enable(GTK_TOOLTIPS(dialog->layer_tooltip));
  gtk_tooltips_set_delay(GTK_TOOLTIPS(dialog->layer_tooltip), 0);

  build_popups(dialog);
/********************************************************************/

  /* connect signals */
/*
  gtk_signal_connect(GTK_OBJECT(new_button), "clicked",
                     GTK_SIGNAL_FUNC(create_dataset), 
		     dialog);

  gtk_signal_connect(GTK_OBJECT(delete_button), "clicked",
                     GTK_SIGNAL_FUNC(delete_dataset), 
		     dialog);

  gtk_signal_connect(GTK_OBJECT(edit_button), "clicked",
                     GTK_SIGNAL_FUNC(edit_dataset), 
		     dialog);

  gtk_signal_connect(GTK_OBJECT(edit_layer_button), "clicked",
                     GTK_SIGNAL_FUNC(layer_dataset_edit), 
		     dialog);

  gtk_signal_connect(GTK_OBJECT(show_button), "clicked",
                     GTK_SIGNAL_FUNC(layer_dataset_show), 
		     dialog);

*/

  gtk_signal_connect(GTK_OBJECT(dialog->datasets_list), "select_row",
                     GTK_SIGNAL_FUNC(change_datasets_tooltip), 
		     dialog);

  gtk_signal_connect(GTK_OBJECT(dialog->layer_list), "select_row",
                     GTK_SIGNAL_FUNC(change_layer_tooltip), 
		     dialog);

  gtk_signal_connect(GTK_OBJECT(dialog->event_box2), "button_press_event",
                     GTK_SIGNAL_FUNC(show_popup), 
		     dialog->dataset_popup);

  gtk_signal_connect(GTK_OBJECT(dialog->event_box1), "button_press_event",
                     GTK_SIGNAL_FUNC(show_popup), 
		     dialog->layer_popup);

  /* Datasets */
  layer_clist_fill_datasets(dialog);
  datasets_clist_fill_datasets(dialog);
}

static void
layer_clist_fill_datasets(SGlayerDatasetDialog *dialog)
{
  GList *list = NULL;
  GtkWidget *clist = dialog->layer_list;
  gint i, nrows;

  nrows = GTK_CLIST(clist)->rows;
  for(i = 0; i < nrows; i++)
    gtk_clist_remove(GTK_CLIST(clist), 0);

  list = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot->data_sets;
  while(list){
    GtkPlotData *dataset = GTK_PLOT_DATA(list->data);
    layer_clist_add_dataset(dialog, dataset); 

    list = list->next;
  }
}

static void
layer_clist_add_dataset(SGlayerDatasetDialog *dialog, GtkPlotData *dataset)
{
  gchar *text[2];
  GdkColormap *colormap;
  GdkPixmap *eye_pixmap;
  GdkBitmap *eye_mask;
  GdkPixmap *pixmap;
  SGdataset *parent = NULL;
  GtkWidget *clist = dialog->layer_list;
  gint nrows;

  colormap = gdk_colormap_get_system();

  eye_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &eye_mask,
                                                     NULL, eye_xpm);

  parent = SG_DATASET(dataset->link);
  pixmap = SG_PLUGIN_STYLE(parent->constructor)->pixmap->pixmap;
  gdk_pixmap_ref(pixmap);

  nrows = GTK_CLIST(clist)->rows;

  text[0] = parent->name; 
  text[1] = NULL; 
  gtk_clist_append(GTK_CLIST(clist), text);

  gtk_clist_set_pixtext(GTK_CLIST(clist), nrows, 1, text[0], 5, pixmap, NULL);
  gtk_clist_set_row_data(GTK_CLIST(clist), nrows, dataset);

  if(GTK_WIDGET_VISIBLE(dataset))
    gtk_clist_set_pixmap(GTK_CLIST(clist), nrows, 0, eye_pixmap, eye_mask);

  gdk_pixmap_unref(eye_pixmap);
  gdk_bitmap_unref(eye_mask);
}

static void
datasets_clist_fill_datasets(SGlayerDatasetDialog *dialog)
{
  GList *list = NULL;
  GtkWidget *clist = dialog->datasets_list;
  gint i, nrows;

  nrows = GTK_CLIST(clist)->rows;
  for(i = 0; i < nrows; i++)
    gtk_clist_remove(GTK_CLIST(clist), 0);

  nrows = GTK_CLIST(dialog->layer_list)->rows;

  list = dialog->datasets->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    SGdataset *dataset = SG_DATASET(child->object);
    gboolean found = FALSE;

    if(strcmp(dataset->constructor->layer, SG_PLUGIN(dialog->layer->plugin)->name) == 0){

      for(i = 0; i < nrows; i++) {
        GtkPlotData *row_dataset;
        row_dataset = GTK_PLOT_DATA(gtk_clist_get_row_data(GTK_CLIST(dialog->layer_list), i));
        if(SG_DATASET(row_dataset->link) == dataset) { found = TRUE; break; }
      }

      if(!found) datasets_clist_add_dataset(dialog, dataset); 
    }

    list = list->next;
  }
}

static void
datasets_clist_add_dataset(SGlayerDatasetDialog *dialog, SGdataset *dataset)
{
  gchar *text[1];
  gint nrows;
  GtkWidget *clist = dialog->datasets_list;
  GdkColormap *colormap;
  GdkPixmap *pixmap;

  colormap = gdk_colormap_get_system();

  pixmap = SG_PLUGIN_STYLE(dataset->constructor)->pixmap->pixmap;
  gdk_pixmap_ref(pixmap);

  nrows = GTK_CLIST(clist)->rows;

  text[0] = NULL; 
  gtk_clist_append(GTK_CLIST(clist), text);

  text[0] = dataset->name;
  gtk_clist_set_pixtext(GTK_CLIST(clist), nrows, 0, text[0], 5, pixmap, NULL);
  gtk_clist_set_row_data(GTK_CLIST(clist), nrows, dataset);
}

static gint
dataset_new(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  SGdataset *dataset;
  dataset = sg_dataset_dialog(dialog->app, dialog->layer->plugin);

  if(dataset) {
    sg_list_add(dialog->datasets, G_OBJECT(dataset), dataset->name);
    datasets_clist_add_dataset(dialog, dataset);
  }

  return TRUE;
}

static gint
dataset_delete(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  GList *selection = NULL;
  gint row;
  SGdataset *dataset = NULL;
  GList *list = NULL;

  selection = GTK_CLIST(dialog->datasets_list)->selection;
  if(!selection) return TRUE;

  row = GPOINTER_TO_INT(selection->data);
  dataset = (SGdataset *)gtk_clist_get_row_data(GTK_CLIST(dialog->datasets_list), row);
  gtk_clist_remove(GTK_CLIST(dialog->datasets_list), row);

  list = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot->data_sets;
  while(list){
    SGdataset *parent;
    GtkPlotData *child;
    child = GTK_PLOT_DATA(list->data);
    parent = SG_DATASET(child->link);
    if(parent == dataset) { /* remove */
      if(child->plot) {
        gtk_plot_remove_data(child->plot, child);
        list = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot->data_sets;
        continue;
      } 
    } 
    list = list->next;
  }  
  sg_list_remove(dialog->datasets, G_OBJECT(dataset));

  return TRUE;
}

static gboolean
dataset_edit(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  SGdataset *row_dataset = NULL;
  SGdataset *new_dataset = NULL;
  gchar *text[1];
  GdkPixmap *pixmap;
  SGpluginIterator *plugin;
  gint row;

  row_dataset = get_dataset_selection(dialog, &row); 
  if(!row_dataset) return TRUE;

  plugin = SG_PLUGIN_ITERATOR(row_dataset->plugin);
  if(!plugin) return TRUE;

  new_dataset = sg_plugin_iterator_edit_dialog(plugin, row_dataset, dialog->app);

  if(!new_dataset) return TRUE;

  text[0] = SG_DATASET(new_dataset)->name;
  pixmap = SG_PLUGIN_STYLE(new_dataset->constructor)->pixmap->pixmap;
  gdk_pixmap_ref(pixmap);

  gtk_clist_set_pixtext(GTK_CLIST(dialog->datasets_list), row, 1, text[0], 5, pixmap, NULL);

  return TRUE;
}

static gboolean
layer_dataset_edit(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  GtkPlotData *row_dataset = NULL;
  SGdataset *new_dataset = NULL;
  gchar *text[1];
  GdkPixmap *pixmap;
  GtkPlotSymbol symbol;
  GtkPlotLine line;
  GtkPlotConnector connector;
  SGpluginIterator *plugin;
  gint row;

  row_dataset = get_layer_selection(dialog, &row);
  if(!row_dataset) return TRUE;

  symbol = row_dataset->symbol;
  line = row_dataset->line;
  connector = row_dataset->line_connector;

  plugin = SG_PLUGIN_ITERATOR(SG_DATASET(row_dataset->link)->plugin);
  if(!plugin) return TRUE;

  new_dataset = sg_plugin_iterator_edit_dialog(plugin, SG_DATASET(row_dataset->link), dialog->app);

  if(!new_dataset) return TRUE;

/*
  real_data = sg_dataset_new_child(new_dataset);
  sg_dataset_add_child(new_dataset, real_data);
  real_data->symbol = symbol;
  real_data->line = line;
  real_data->line_connector = connector;
*/

  text[0] = SG_DATASET(new_dataset)->name;
  pixmap = SG_PLUGIN_STYLE(new_dataset->constructor)->pixmap->pixmap;
  gdk_pixmap_ref(pixmap);

  gtk_clist_set_pixtext(GTK_CLIST(dialog->layer_list), row, 1, text[0], 5, pixmap, NULL);

  return TRUE;
}

static gboolean
layer_dataset_show(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  GtkPlotData *aux_dataset;
  GdkColormap *colormap;
  GdkPixmap *eye_pixmap;
  GdkBitmap *eye_mask;
  gint row;

  colormap = gdk_colormap_get_system();

  aux_dataset = get_layer_selection(dialog, &row);
  if(!aux_dataset) return TRUE;

  if(aux_dataset){
    if(GTK_WIDGET_VISIBLE(aux_dataset)){
       gtk_widget_hide(GTK_WIDGET(aux_dataset));
       gtk_clist_set_text(GTK_CLIST(dialog->layer_list), row, 0, NULL);
    }else{
       gtk_widget_show(GTK_WIDGET(aux_dataset));
       eye_pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, 
                                                    &eye_mask, NULL, eye_xpm);
       gtk_clist_set_pixmap(GTK_CLIST(dialog->layer_list), row, 0, eye_pixmap, eye_mask);
       gdk_pixmap_unref(eye_pixmap);
       gdk_bitmap_unref(eye_mask);
    }

    gtk_plot_canvas_paint(GTK_PLOT_CANVAS(dialog->plot));
    gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(dialog->plot));
  }

  return TRUE;
}

static gboolean
dataset_restore(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  SGdataset *dataset;
  gint row;

  dataset = get_dataset_selection(dialog, &row);
  if(!dataset) return TRUE;

  sg_dataset_refresh(dataset);

  return FALSE;
}

static gboolean
dataset_edit_data(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  SGdataset *dataset;
  gint row;

  dataset = get_dataset_selection(dialog, &row);
  if(!dataset) return TRUE;

  sg_plugin_iterator_edit_data_dialog(SG_PLUGIN_ITERATOR(dataset->plugin), dataset, NULL);

  return FALSE;
}

static gboolean
dataset_rename(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  SGdataset *dataset = NULL;
  gint row;
  gchar *name = NULL;
  gchar *text[1];
  GdkPixmap *pixmap = NULL;

  dataset = get_dataset_selection(dialog, &row);
  if(!dataset) return TRUE;

  name = sg_text_dialog( _("Rename dataset:"), dataset->name, 1);
  if(!name) return TRUE;

  sg_dataset_set_name(dataset, name);
  g_free(name); 

  text[0] = dataset->name;
  pixmap = SG_PLUGIN_STYLE(dataset->constructor)->pixmap->pixmap;
  gdk_pixmap_ref(pixmap);

  gtk_clist_set_pixtext(GTK_CLIST(dialog->datasets_list), row, 1, text[0], 5, pixmap, NULL);

  return FALSE;
}

static gboolean
layer_dataset_restore(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  GtkPlotData *dataset;
  gint row;

  dataset = get_layer_selection(dialog, &row);
  if(!dataset) return TRUE;

  sg_dataset_refresh(SG_DATASET(dataset->link));

  return FALSE;
}

static gboolean
layer_dataset_edit_data(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  GtkPlotData *dataset;
  gint row;

  dataset = get_layer_selection(dialog, &row);
  if(!dataset) return TRUE;

  sg_plugin_iterator_edit_data_dialog(SG_PLUGIN_ITERATOR(SG_DATASET(dataset->link)->plugin), SG_DATASET(dataset->link), NULL);

  return FALSE;
}

static gboolean
layer_dataset_rename(GtkWidget *widget, SGlayerDatasetDialog *dialog)
{
  GtkPlotData *dataset = NULL;
  gint row;
  gchar *name = NULL;
  gchar *text[1];
  GdkPixmap *pixmap = NULL;

  dataset = get_layer_selection(dialog, &row);
  if(!dataset) return TRUE;

  name = sg_text_dialog( _("Rename dataset:"), dataset->name, 1);
  if(!name) return TRUE;

  sg_dataset_set_name(SG_DATASET(dataset->link), name);
  g_free(name); 
 
  text[0] = dataset->name;
  pixmap = SG_PLUGIN_STYLE(SG_DATASET(dataset->link)->constructor)->pixmap->pixmap;
  gdk_pixmap_ref(pixmap);

  gtk_clist_set_pixtext(GTK_CLIST(dialog->layer_list), row, 1, text[0], 5, pixmap, NULL);

  return FALSE;
}

static void
build_popups(SGlayerDatasetDialog *dialog)
{
  char *layer_items[]={ _("Rename"),
                        _("Show/Hide"),
                        _("Restore source"),
			_("Edit source"),
			_("Edit data") };
  GtkSignalFunc layer_func[] = { GTK_SIGNAL_FUNC(layer_dataset_rename),
				 GTK_SIGNAL_FUNC(layer_dataset_show),
				 GTK_SIGNAL_FUNC(layer_dataset_restore),
				 GTK_SIGNAL_FUNC(layer_dataset_edit),
				 GTK_SIGNAL_FUNC(layer_dataset_edit_data) };
  char *dataset_items[]={ _("New"),
			  _("Rename"),
                          _("Restore source"),
			  _("Edit source"),
			  _("Edit data"),
			  _("Delete") };
  GtkSignalFunc dataset_func[] = { GTK_SIGNAL_FUNC(dataset_new),
				   GTK_SIGNAL_FUNC(dataset_rename),
				   GTK_SIGNAL_FUNC(dataset_restore),
				   GTK_SIGNAL_FUNC(dataset_edit),
				   GTK_SIGNAL_FUNC(dataset_edit_data),
				   GTK_SIGNAL_FUNC(dataset_delete) };
  GtkWidget *item;
  unsigned int i;
                                                                               
  dialog->dataset_popup = gtk_menu_new();
                                                                               
  for (i=0; i < (sizeof(dataset_items)/sizeof(dataset_items[0])) ; i++){
          item=gtk_menu_item_new_with_label(dataset_items[i]);
          gtk_signal_connect(GTK_OBJECT(item),"activate",
                             (GtkSignalFunc) dataset_func[i],
                             dialog);
          gtk_widget_show(item);
          gtk_menu_append(GTK_MENU(dialog->dataset_popup),item);
  }
                                                                               
  dialog->layer_popup = gtk_menu_new();
                                                                               
  for (i=0; i < (sizeof(layer_items)/sizeof(layer_items[0])) ; i++){
          item=gtk_menu_item_new_with_label(layer_items[i]);
          gtk_signal_connect(GTK_OBJECT(item),"activate",
                             (GtkSignalFunc) layer_func[i],
                             dialog);
          gtk_widget_show(item);
          gtk_menu_append(GTK_MENU(dialog->layer_popup),item);
  }
}
                                                                               

