/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_grids_dialog.h"
#include "sg.h"
#include "sg_property_dialog.h"

typedef struct
{
  SGlayer *layer;

  GtkPlotAxis *axis;

  GtkPlotLine *major_grid;
  GtkPlotLine *minor_grid;

  GtkWidget *style_combo1; 
  GtkWidget *style_combo2; 
  GtkWidget *thick_spin1; 
  GtkWidget *thick_spin2; 
  GtkWidget *color_combo1; 
  GtkWidget *color_combo2; 
  GtkWidget *show_check1;
  GtkWidget *show_check2;
} SGgridsDialog; 

static gchar *line_styles[] = 
{
  N_("None"),
  N_("Solid"),
  N_("Dotted"),
  N_("Dashed"),
  N_("Dot-dash"),
  N_("Dot-dot-dash"),
  N_("Dot-dash-dash"),
  NULL
};

static void sg_grids_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_grids_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_grids_dialog_init_gui         (GtkWidget *frame);

GtkWidget *
sg_grids_dialog_new(SGlayer *layer, GtkPlotAxis *axis)
{
  GtkWidget *widget;
  SGgridsDialog *dialog;

  dialog = g_new0(SGgridsDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->ok = sg_grids_dialog_ok;
  SG_PROPERTY_DIALOG(widget)->apply = sg_grids_dialog_apply;
  dialog->layer = layer;
  dialog->axis = axis;
  sg_grids_dialog_init_gui(widget);

  return widget;
}

static void
sg_grids_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_grids_dialog_apply(d, data);
}

static void
sg_grids_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  GtkPlot *plot;
  GtkPlotLine major_grid, minor_grid;
  GtkColorCombo *combo;
  GdkColor color;
  SGplot *parent;
  GtkWidget *child;
  SGgridsDialog *dialog = (SGgridsDialog *)d->data;

  plot = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot;

  combo = GTK_COLOR_COMBO(dialog->color_combo1);
  color = gtk_color_combo_get_selection(combo);
  
  major_grid = *dialog->major_grid;
  major_grid.color = color;
  major_grid.line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->thick_spin1));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->style_combo1)->list)->selection->data;
  major_grid.line_style = (GtkPlotLineStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->style_combo1)->list), child);

  combo = GTK_COLOR_COMBO(dialog->color_combo2);
  color = gtk_color_combo_get_selection(combo);

  minor_grid = *dialog->minor_grid;
  minor_grid.color = color;
  minor_grid.line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->thick_spin2));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->style_combo2)->list)->selection->data;
  minor_grid.line_style = (GtkPlotLineStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->style_combo2)->list), child);

  *dialog->major_grid = major_grid;
  *dialog->minor_grid = minor_grid;

  dialog->axis->show_major_grid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->show_check1));
  dialog->axis->show_minor_grid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->show_check2));

  parent = SG_PLOT(GTK_PLOT_CANVAS_CHILD(dialog->layer)->parent);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(parent));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(parent));
}


static void
init_dialog(SGgridsDialog *dialog)
{
  GtkPlotLine *major_grid, *minor_grid;
  GdkColor line_color;
  gint line_style;
  gfloat line_width;

  major_grid = dialog->major_grid;
  minor_grid = dialog->minor_grid;

  line_style = major_grid->line_style;
  line_width = major_grid->line_width;
  line_color = major_grid->color;

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->style_combo1)->list), line_style);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->style_combo1)->entry), FALSE);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->thick_spin1), line_width);

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->color_combo1), line_color);

  line_style = minor_grid->line_style;
  line_width = minor_grid->line_width;
  line_color = minor_grid->color;

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->style_combo2)->list), line_style);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->style_combo2)->entry), FALSE);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->thick_spin2), line_width);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->show_check1), dialog->axis->show_major_grid);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->show_check2), dialog->axis->show_minor_grid);

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->color_combo2), line_color);

}


static void
sg_grids_dialog_init_gui (GtkWidget *frame)
{
  GtkPlot *plot;
  GtkWidget *main_box, *main_table;
  GtkWidget *table;
  GtkRequisition req;
  GtkWidget *label;
  GtkAdjustment *adj;
  SGgridsDialog *dialog = (SGgridsDialog *)SG_PROPERTY_DIALOG(frame)->data;
  SGlayer *layer = dialog->layer;
  GtkPlotAxis *axis = dialog->axis;
  
  plot = GTK_PLOT_CANVAS_PLOT(layer)->plot;

  dialog->major_grid = &axis->major_grid; 
  dialog->minor_grid = &axis->minor_grid; 

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);
  gtk_container_add(GTK_CONTAINER(frame), main_box);

  main_table = gtk_table_new(3,2,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 5);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);

  /* Major Grids */

  frame = gtk_frame_new( _("Major grids") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 0, 1, 0, 1, 0, 0, 5, 5);

  table=gtk_table_new(4, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new( _("Style ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 0, 0, 5, 5);
  label = gtk_label_new( _("Thickness(pts) ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 0, 0, 5, 5);
  label = gtk_label_new( _("Color ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 0, 0, 5, 5);


  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
  dialog->thick_spin1 = gtk_spin_button_new(adj, 0.5, 1);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->thick_spin1), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->thick_spin1), 1);
  dialog->style_combo1 = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->style_combo1), line_styles);
  dialog->color_combo1 = gtk_color_combo_new();

  dialog->show_check1 = gtk_check_item_new_with_label( _("Show major grids") );

  gtk_widget_size_request(dialog->style_combo1, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->style_combo1, req.width, req.height);
  gtk_widget_set_usize(dialog->thick_spin1, req.width, req.height);

  gtk_table_attach(GTK_TABLE(table), dialog->style_combo1, 1, 2, 0, 1, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->thick_spin1, 1, 2, 1, 2, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->color_combo1, 1, 2, 2, 3, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->show_check1, 0, 2, 3, 4, 0, 0, 5, 5);

  /* Minor Grids */

  frame = gtk_frame_new( _("Minor grids") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 1, 2, 0, 1, 0, 0, 5, 5);

  table=gtk_table_new(4, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new( _("Style ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 0, 0, 5, 5);
  label = gtk_label_new( _("Thickness(pts) ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 0, 0, 5, 5);
  label = gtk_label_new( _("Color ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 0, 0, 5, 5);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
  dialog->thick_spin2 = gtk_spin_button_new(adj, 0.5, 1);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->thick_spin2), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->thick_spin2), 1);
  dialog->style_combo2 = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->style_combo2), line_styles);
  dialog->color_combo2 = gtk_color_combo_new();

  dialog->show_check2 = gtk_check_item_new_with_label( _("Show minor grids") );

  gtk_widget_set_usize(dialog->style_combo2, req.width, req.height);
  gtk_widget_set_usize(dialog->thick_spin2, req.width, req.height);

  gtk_table_attach(GTK_TABLE(table), dialog->style_combo2, 1, 2, 0, 1, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->thick_spin2, 1, 2, 1, 2, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->color_combo2, 1, 2, 2, 3, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->show_check2, 0, 2, 3, 4, 0, 0, 5, 5);

  /* connect signals */

  init_dialog(dialog);
}

GtkWidget *
sg_grids_dialog (SGlayer *layer,
		 GtkWidget *ok_button,
		 GtkWidget *apply_button,
		 GtkWidget *cancel_button)
{
  GtkWidget *notebook;
  GtkWidget *dialog;
  GtkWidget *box;
  GtkPlot *plot;

  notebook = gtk_notebook_new();

  plot = GTK_PLOT_CANVAS_PLOT(layer)->plot;

  dialog = GTK_WIDGET(sg_grids_dialog_new(layer, plot->bottom));
  sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                 ok_button, apply_button, cancel_button);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                           gtk_label_new( _("X axis") ));
  dialog = GTK_WIDGET(sg_grids_dialog_new(layer, plot->left));
  sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                 ok_button, apply_button, cancel_button);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                           gtk_label_new( _("Y axis") ));

  if(GTK_IS_PLOT3D(plot)){
    dialog = GTK_WIDGET(sg_grids_dialog_new(layer, GTK_PLOT3D(plot)->az));
    sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                   ok_button, apply_button, cancel_button);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                             gtk_label_new( _("Z axis") ));
  }

  return notebook;
}
