/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_layer.h"
#include "sg_property_dialog.h"

typedef struct
{
  GtkPlotAxis *axis;
  GtkPlotData *data;

  GtkWidget *font_combo;
  GtkWidget *format_combo;
  GtkWidget *precision_spin;
  GtkWidget *prefix_entry;
  GtkWidget *suffix_entry;
  GtkWidget *in_check;
  GtkWidget *out_check;

  GtkWidget *title_combo;
  GtkWidget *title_entry;
  GtkWidget *position_combo;
  GtkWidget *title_check;

  GtkWidget *bg_combo;
  GtkWidget *fg_combo;
  GtkWidget *border_combo;
  GtkWidget *transparent_item;
  GtkWidget *orientation_combo;

  GtkWidget *height_spin;
  GtkWidget *width_spin;
} SGlabelsDialog;

static gchar *border_styles[] = {
  N_("None"),
  N_("Line"),
  N_("Shadow"),
  NULL
};

static gchar *orientation[] = {
  N_("Horizontal"),
  N_("Vertical"),
  NULL
};

static gchar *positions[] = {
  N_("Left"),
  N_("Right"),
  N_("Top"),
  N_("Bottom"),
  NULL
};

static  gchar *label_formats[] = { 
  N_("Decimal: 1.000"), 
  N_("Scientific: 1.E10"), 
  N_("Scientific: 1x10^10"), 
  NULL 
};

static void sg_gradient_labels_dialog_apply            (SGpropertyDialog *d, gpointer data); 
static void sg_gradient_labels_dialog_ok               (SGpropertyDialog *d, gpointer data); 
static void sg_gradient_labels_dialog_init_gui         (GtkWidget *frame);
static void init_dialog				       (SGlabelsDialog *dialog);
  
GtkWidget *
sg_gradient_labels_dialog_new(GtkPlotData *dataset)
{
  GtkWidget *widget;
  SGlabelsDialog *dialog;

  dialog = g_new0(SGlabelsDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_gradient_labels_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_gradient_labels_dialog_ok;
  dialog->data = dataset;
  dialog->axis = dataset->gradient;
  sg_gradient_labels_dialog_init_gui(widget);

  return widget;
}

static void
sg_gradient_labels_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_gradient_labels_dialog_apply(d, data);
}

static void
sg_gradient_labels_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  SGlabelsDialog *dialog = (SGlabelsDialog *)d->data;
  SGworksheet *worksheet = NULL;
  GtkPlotAxis *axis;
  GdkColor bg, fg;
  GtkWidget *child;
  const gchar *text;
  const gchar *font;
  gint height;
  GdkColor color;
  gboolean transparent;
  GtkColorCombo *combo;
  gboolean state;
  GtkPSFont *psfont;

  axis = dialog->axis;

  psfont = gtk_font_combo_get_psfont(GTK_FONT_COMBO(dialog->font_combo));
  font = psfont->psname;
  height = gtk_font_combo_get_font_height(GTK_FONT_COMBO(dialog->font_combo));

  axis->custom_labels = FALSE;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->format_combo)->list)->selection->data;
  axis->label_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), child);

  axis->label_precision = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->precision_spin));
  dialog->data->gradient_line_width = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->width_spin));
  dialog->data->gradient_line_height = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->height_spin));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->border_combo)->list)->selection->data;
  dialog->data->gradient_border = (GtkPlotBorderStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->border_combo)->list), child);
  transparent = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->transparent_item));
  axis->title_visible = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->title_check));

  axis->label_mask = 0; 
  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->in_check));
  if(state) axis->label_mask |= GTK_PLOT_LABEL_IN; 
  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->out_check));
  if(state) axis->label_mask |= GTK_PLOT_LABEL_OUT; 

  combo = GTK_COLOR_COMBO(dialog->fg_combo);
  color = gtk_color_combo_get_selection(combo);
  fg = color;
  combo = GTK_COLOR_COMBO(dialog->bg_combo);
  color = gtk_color_combo_get_selection(combo);
  bg = color;

  if(axis->labels_attr.font) g_free(axis->labels_attr.font);
  axis->labels_attr.font = g_strdup(font);
  axis->labels_attr.height = height;
  axis->labels_attr.fg = fg;
  axis->labels_attr.bg = bg;
  axis->labels_attr.transparent = transparent;
   
  text = gtk_entry_get_text(GTK_ENTRY(dialog->prefix_entry));
  if(axis->labels_prefix) g_free(axis->labels_prefix);
  axis->labels_prefix = NULL;
  if(text && strlen(text) > 0) axis->labels_prefix = g_strdup(text);

  text = gtk_entry_get_text(GTK_ENTRY(dialog->suffix_entry));
  if(axis->labels_suffix) g_free(axis->labels_suffix);
  axis->labels_suffix = NULL;
  if(text && strlen(text) > 0) axis->labels_suffix = g_strdup(text);

  psfont = gtk_font_combo_get_psfont(GTK_FONT_COMBO(dialog->title_combo));
  font = psfont->psname;
  height = gtk_font_combo_get_font_height(GTK_FONT_COMBO(dialog->title_combo));
  text = gtk_entry_get_text(GTK_ENTRY(dialog->title_entry));

  if(axis->title.font) g_free(axis->title.font);
  axis->title.font = g_strdup(font);
  axis->title.height = height;
  axis->title.fg = fg;
  axis->title.bg = bg;
  axis->title.transparent = transparent;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->orientation_combo)->list)->selection->data;
  axis->orientation = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->orientation_combo)->list), child);

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->position_combo)->list)->selection->data;
  dialog->data->gradient_title_pos = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->position_combo)->list), child);

  init_dialog(dialog);
}


static void 
init_dialog(SGlabelsDialog *dialog)
{
  GtkPSFont *font;

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), dialog->axis->label_style); 
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->format_combo)->entry), FALSE);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->orientation_combo)->list), dialog->axis->orientation); 
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->orientation_combo)->entry), FALSE);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->position_combo)->list), dialog->data->gradient_title_pos); 
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->position_combo)->entry), FALSE);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->precision_spin), 
                            dialog->axis->label_precision);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->width_spin), 
                            dialog->data->gradient_line_width);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->height_spin), 
                            dialog->data->gradient_line_height);

  if(dialog->axis->labels_prefix)
     gtk_entry_set_text(GTK_ENTRY(dialog->prefix_entry), dialog->axis->labels_prefix);

  if(dialog->axis->labels_suffix)
     gtk_entry_set_text(GTK_ENTRY(dialog->suffix_entry), dialog->axis->labels_suffix);

  font = gtk_psfont_get_by_name(dialog->axis->labels_attr.font);
  gtk_font_combo_select(GTK_FONT_COMBO(dialog->font_combo),
                        font->family,
                        FALSE,
                        FALSE,
                        dialog->axis->labels_attr.height);

  font = gtk_psfont_get_by_name(dialog->axis->title.font);
  gtk_font_combo_select(GTK_FONT_COMBO(dialog->title_combo),
                        font->family,
                        FALSE,
                        FALSE,
                        dialog->axis->title.height);

  gtk_entry_set_text(GTK_ENTRY(dialog->title_entry), dialog->axis->title.text);
                                                                                
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->fg_combo), dialog->axis->title.fg);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->bg_combo), dialog->axis->title.bg);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->border_combo)->list),
                       dialog->data->gradient_border);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->in_check),
			       dialog->axis->label_mask & GTK_PLOT_LABEL_IN ? TRUE : FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->out_check),
			       dialog->axis->label_mask & GTK_PLOT_LABEL_OUT ? TRUE : FALSE);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->title_check),
			       dialog->axis->title_visible);
}

static void 
sg_gradient_labels_dialog_init_gui (GtkWidget *dialog_frame)
{
  GtkWidget *main_box, *box, *vbox;
  GtkWidget *table, *main_table;
  GtkWidget *label;
  GtkWidget *item;
  GtkWidget *frame;
  GtkAdjustment *adj;
  SGlabelsDialog *dialog;
  GtkRequisition req; 

  dialog = (SGlabelsDialog *)SG_PROPERTY_DIALOG(dialog_frame)->data;

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);
  gtk_container_add(GTK_CONTAINER(dialog_frame), main_box);

  main_table = gtk_table_new(2, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 5);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);

/*----------------------------------------------*/
  frame = gtk_frame_new( _("Format") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 0, 1, 0, 1, GTK_EXPAND|GTK_FILL, GTK_FILL, 5, 5);

  table = gtk_table_new(8, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  label = gtk_label_new( _("Format") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  label = gtk_label_new( _("Precision") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);

  dialog->format_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->format_combo), label_formats);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->format_combo, 1, 2, 0, 1);
  gtk_widget_size_request(dialog->format_combo, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->format_combo, req.width, req.height);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 8., 1., 1., 0.);
  dialog->precision_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->precision_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->precision_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->precision_spin, 1, 2, 1, 2);
  gtk_widget_set_usize(dialog->precision_spin, req.width, req.height);

  label = gtk_label_new( _("Prefix") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);

  dialog->prefix_entry = gtk_entry_new();
  gtk_widget_set_usize(dialog->prefix_entry, req.width, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->prefix_entry, 1, 2, 2, 3);

  label = gtk_label_new( _("Suffix") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);

  dialog->suffix_entry = gtk_entry_new();
  gtk_widget_set_usize(dialog->suffix_entry, req.width, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->suffix_entry, 1, 2, 3, 4);

  dialog->in_check = gtk_check_item_new_with_label( _("In") );
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->in_check, 0, 1, 4, 5);

  dialog->out_check = gtk_check_item_new_with_label( _("Out") );
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->out_check, 1, 2, 4, 5);

  dialog->font_combo = gtk_font_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->font_combo, 0, 2, 5, 6);

/*----------------------------------------------*/

  frame = gtk_frame_new( _("Properties") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 1, 2, 0, 1, GTK_FILL, 0, 5, 5);                                                                                
  table=gtk_table_new(7, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);
                                                                                
  label = gtk_label_new( _("Orientation") );
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  dialog->orientation_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->orientation_combo), orientation);
  gtk_widget_set_usize(dialog->orientation_combo, req.width, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->orientation_combo, 1, 2, 0, 1);

  label = gtk_label_new( _("Foreground ") );
/*  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
*/
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  dialog->fg_combo = gtk_color_combo_new();
  gtk_widget_set_usize(dialog->fg_combo, req.width, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->fg_combo, 1, 2, 1, 2);
                                                                                
  label = gtk_label_new( _("Background ") );
/*  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
*/
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
  dialog->bg_combo = gtk_color_combo_new();
  gtk_widget_set_usize(dialog->bg_combo, req.width, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->bg_combo, 1, 2, 2, 3);
                                                                                
  label = gtk_label_new( _("Border") );
/*  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
*/
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);
  dialog->border_combo = gtk_combo_new();
  gtk_widget_set_usize(dialog->border_combo, req.width, req.height);
  sg_combo_set_items(GTK_COMBO(dialog->border_combo), border_styles);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->border_combo)->entry), FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->border_combo, 1, 2, 3, 4);                                                                                
  dialog->transparent_item = gtk_check_item_new_with_label( _("Transparent") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->transparent_item, 0, 2, 4, 5);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->transparent_item), dialog->axis->title.transparent);

  label = gtk_label_new( _("Width") );
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 5, 6);
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1000., 1., 1., 0.);
  dialog->width_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->width_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->width_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->width_spin, 1, 2, 5, 6);
  gtk_widget_set_usize(dialog->width_spin, req.width, req.height);

  label = gtk_label_new( _("Height") );
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 6, 7);
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1000., 1., 1., 0.);
  dialog->height_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->height_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->height_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->height_spin, 1, 2, 6, 7);
  gtk_widget_set_usize(dialog->height_spin, req.width, req.height);
/*----------------------------------------------*/

  frame = sg_entry_new();
  gtk_frame_set_label(GTK_FRAME(frame), _("Title") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 0, 2, 1, 2, GTK_FILL, GTK_FILL, 5, 5);
                                                                                
  dialog->title_combo = SG_ENTRY(frame)->font_combo;
  dialog->title_entry = SG_ENTRY(frame)->text_entry;

  vbox = GTK_BIN(SG_ENTRY(frame))->child;
  box = gtk_hbox_new(FALSE, 0);
  label = gtk_label_new( _("Position:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 10);

  dialog->position_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->position_combo), positions);
  gtk_widget_set_usize(dialog->position_combo, req.width, req.height);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->position_combo)->entry), FALSE);
  gtk_box_pack_start(GTK_BOX(box), dialog->position_combo, FALSE, FALSE, 0);

  dialog->title_check = gtk_check_item_new_with_label( _("Show titles") );
  gtk_box_pack_start(GTK_BOX(box), dialog->title_check, FALSE, FALSE, 10);

  gtk_box_pack_start(GTK_BOX(vbox), box, FALSE, FALSE, 5);
/*
  gtk_container_add(GTK_CONTAINER(SG_ENTRY(frame)->font_combo), box);
*/
/*----------------------------------------------*/

  gtk_widget_show_all(main_box);

  init_dialog(dialog);
}

