/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_property_dialog.h"

#define WIDTH  100
#define HEIGHT 160

#define SG_GRADIENT_DIALOG(obj)        GTK_CHECK_CAST (obj, sg_gradient_dialog_get_type (), SGgradientDialog)
  
typedef struct _SGgradientDialog SGgradientDialog;
typedef struct _SGgradientDialogClass SGgradientDialogClass;
  
struct _SGgradientDialog 
{
  SGpropertyDialog parent;

  GtkWidget *gradient;
  GtkWidget *event_box;
  GtkWidget *levels_spin;
  GtkWidget *sublevels_spin;
  GtkWidget *max_spin;
  GtkWidget *min_spin;
  GtkWidget *button_min;
  GtkWidget *button_max;
  GtkWidget *button_lt_min;
  GtkWidget *button_gt_max;
  GtkWidget *h_check;
  GtkWidget *s_check;
  GtkWidget *v_check;
  GtkWidget *gradient_check;
  GtkWidget *custom_check;
  GtkWidget *csd;
  GtkWidget *color_pixmap;
  GtkWidget *scale_combo;
  GtkWidget *active_button;
  GdkColor *gradient_colors;
  GdkColor color_min, color_max;
  GdkColor color_lt_min, color_gt_max;
  gint gradient_level;
  GtkPlotData *dataset;
  GdkWindow *window;
  gint mode;
};

struct _SGgradientDialogClass
{
  SGpropertyDialogClass parent_class;
};

static void open_color_selection        (GtkWidget *widget, 
					 gpointer data);
static void open_color_selection2       (GtkWidget *widget,
                                         GdkEventButton *event,
					 gpointer data);

static char * color_xpm[] = {
"40 20 4 1",
"       c #FFFFFFFFFFFF",
".      c #CCCCCCCCCCCC",
"X      c #616161616161",
"o      c #000000000000",
"                                        ",
" ......................................X",
" .XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .Xoooooooooooooooooooooooooooooooooo .X",
" .X                                   .X",
" ......................................X",
" XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"};

static void sg_gradient_dialog_class_init       (SGgradientDialogClass *klass);
static void sg_gradient_dialog_init             (SGgradientDialog *d);
static void sg_gradient_dialog_destroy          (GtkObject *object);
static void sg_gradient_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_gradient_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_gradient_dialog_init_gui         (SGgradientDialog *d);
static void init_dialog         		(SGgradientDialog *d);

static SGpropertyDialogClass *parent_class = NULL;

GtkType
sg_gradient_dialog_get_type(void)
{
  static GtkType sg_gradient_dialog_type = 0;

  if (!sg_gradient_dialog_type)
    {
      GtkTypeInfo sg_gradient_dialog_info =
      {
        "SGgradientDialog",
        sizeof (SGgradientDialog),
        sizeof (SGgradientDialogClass),
        (GtkClassInitFunc) sg_gradient_dialog_class_init,
        (GtkObjectInitFunc) sg_gradient_dialog_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_gradient_dialog_type = gtk_type_unique (sg_property_dialog_get_type(), &sg_gradient_dialog_info);
    }
  return sg_gradient_dialog_type;
}

GtkWidget *
sg_gradient_dialog_new(GtkPlotData *dataset, gint mode /* a, da, z */)
{
  GtkWidget *widget;

  widget = GTK_WIDGET(gtk_type_new(sg_gradient_dialog_get_type()));

  SG_GRADIENT_DIALOG(widget)->dataset = dataset;
  SG_GRADIENT_DIALOG(widget)->mode = mode;
  sg_gradient_dialog_init_gui(SG_GRADIENT_DIALOG(widget));
  init_dialog(SG_GRADIENT_DIALOG(widget));

  return widget;
}

static void
sg_gradient_dialog_init(SGgradientDialog *d)
{
  SG_PROPERTY_DIALOG(d)->apply = sg_gradient_dialog_apply;
  SG_PROPERTY_DIALOG(d)->ok = sg_gradient_dialog_ok;
}

static void
sg_gradient_dialog_class_init(SGgradientDialogClass *klass)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) klass;

  parent_class = (SGpropertyDialogClass *)gtk_type_class (sg_property_dialog_get_type ());

  object_class->destroy = sg_gradient_dialog_destroy;
}

static void
sg_gradient_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_gradient_dialog_apply(d, data);
}


static void 
sg_gradient_dialog_destroy(GtkObject *object)
{
  SGgradientDialog *dialog = SG_GRADIENT_DIALOG(object);

  if(dialog->gradient_colors) g_free(dialog->gradient_colors);
  dialog->gradient_colors = NULL;

  if(dialog->window) gdk_window_unref(dialog->window);
  dialog->window = NULL;
}

static void 
insert_min(GtkWidget *button, gpointer data)
{
  gchar text[20];
  gint n;
  gdouble gradient_min = 0.;
  SGgradientDialog *dialog = SG_GRADIENT_DIALOG(data);
  GtkPlotData *dataset = dialog->dataset;

  for(n = 0; n < dataset->num_points; n++){
      gdouble fx, fy, fz, fa;
      gdouble fdx, fdy, fdz, fda;
      gdouble val = 0.f;
      gchar *label;
      gboolean error;

      gtk_plot_data_get_point(dataset, n,
                              &fx, &fy, &fz, &fa,
                              &fdx, &fdy, &fdz, &fda,
                              &label, &error);
      if(dialog->mode == 1)
        val = fa;
      else if(dialog->mode == 2)
        val = fda;
      else
        val = fz;

      if(n == 0){
        gradient_min = val;
      } else {
        if(val <= gradient_min) gradient_min = val;
      }
  }


  snprintf(text,20,"%g", gradient_min);
  gtk_entry_set_text(GTK_ENTRY(dialog->min_spin), text);

}

static void 
insert_max(GtkWidget *button, gpointer data)
{
  gchar text[20];
  gint n;
  gdouble gradient_max = 0.;
  SGgradientDialog *dialog = SG_GRADIENT_DIALOG(data);
  GtkPlotData *dataset = dialog->dataset;

  for(n = 0; n < dataset->num_points; n++){
      gdouble fx, fy, fz, fa;
      gdouble fdx, fdy, fdz, fda;
      gdouble val = 0.f;
      gchar *label;
      gboolean error;

      gtk_plot_data_get_point(dataset, n,
                              &fx, &fy, &fz, &fa,
                              &fdx, &fdy, &fdz, &fda,
                              &label, &error);

      if(dialog->mode == 1)
        val = fa;
      else if(dialog->mode == 2)
        val = fda;
      else
        val = fz;

      if(n == 0){
        gradient_max = val;
      } else {
        if(val >= gradient_max) gradient_max = val;
      }
  }

  snprintf(text,20,"%g", gradient_max);
  gtk_entry_set_text(GTK_ENTRY(dialog->max_spin), text);

}


static void
init_gradient_pixmap(SGgradientDialog *dialog)
{
  GtkPlotData *data;
  GdkGC *gc;
  gdouble y;
  gint level;
  gint nlevels;
  gdouble step;
  GdkColor color;

  data = dialog->dataset;

  gc = gdk_gc_new(GTK_PIXMAP(dialog->gradient)->pixmap);

  y = 0;
  nlevels = data->gradient->ticks.nticks-1;
  step = HEIGHT/(gdouble)nlevels;

  for(level = nlevels-1; level >= 0; level--){
    gdk_gc_set_foreground(gc, &dialog->gradient_colors[level]);
    gdk_draw_rectangle(GTK_PIXMAP(dialog->gradient)->pixmap,
                       gc, TRUE,
                       0, (gint)y,
                       WIDTH, (gint)step+1);
    y += step;
  }

  gdk_color_black(gdk_colormap_get_system(), &color);
  gdk_gc_set_foreground(gc, &color);

  gdk_draw_rectangle(GTK_PIXMAP(dialog->gradient)->pixmap, gc, FALSE,
                        0, 0, WIDTH - 1, HEIGHT - 1);

  y = 0;
  for(level = nlevels-1; level >= 0; level--){
    gdk_draw_line(GTK_PIXMAP(dialog->gradient)->pixmap, gc,
                  0, (gint)y,
                  4, (gint)y);
    gdk_draw_line(GTK_PIXMAP(dialog->gradient)->pixmap, gc,
                  WIDTH - 5, (gint)y,
                  WIDTH - 1, (gint)y);
    y += step;
  }
  gtk_widget_draw(dialog->gradient, NULL);

  gdk_gc_unref(gc);
}

static void
reset_gradient(GtkWidget *button, gpointer _data)
{
  SGgradientDialog *dialog = SG_GRADIENT_DIALOG(_data);
  GtkPlotData *data;
  gint level;
  gint nlevels;

  data = dialog->dataset;

  data->gradient_custom = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->custom_check));

  gtk_plot_data_reset_gradient_colors(data);

  if(dialog->gradient_colors) g_free(dialog->gradient_colors);
  nlevels = data->gradient->ticks.nticks;
  dialog->gradient_colors = g_new0(GdkColor, nlevels);

  for(level = 0; level < nlevels - 1; level++){
    dialog->gradient_colors[level] = data->gradient_colors[level];
  }

  init_gradient_pixmap(dialog);
}

static void     
init_dialog (SGgradientDialog *dialog)
{
  GdkColor color;
  GdkGC *gc;
  gchar text[100];
  gint nlevels;
  GtkPlotData *data = dialog->dataset;

  /* Gradient */
  sprintf(text,"%g",data->gradient->ticks.min);
  gtk_entry_set_text(GTK_ENTRY(dialog->min_spin), text);
  sprintf(text,"%g",data->gradient->ticks.max);
  gtk_entry_set_text(GTK_ENTRY(dialog->max_spin), text);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->levels_spin), data->gradient->ticks.nmajorticks);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->sublevels_spin), data->gradient->ticks.nminor);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->scale_combo)->list),
                       data->gradient->ticks.scale);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->h_check),
                               data->gradient_mask & GTK_PLOT_GRADIENT_H);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->s_check),
                               data->gradient_mask & GTK_PLOT_GRADIENT_S);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->v_check),
                               data->gradient_mask & GTK_PLOT_GRADIENT_V);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->gradient_check),
                               data->show_gradient);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->custom_check),
                               data->gradient_custom);

  gc = gdk_gc_new(GTK_PIXMAP(dialog->gradient)->pixmap);

  gdk_color_white(gdk_colormap_get_system(), &color);
  gdk_gc_set_foreground(gc, &color);

  gdk_draw_rectangle(GTK_PIXMAP(dialog->gradient)->pixmap,
                     gc, TRUE, 0, 0, -1, -1);

  nlevels = data->gradient->ticks.nticks;
  if(dialog->gradient_colors) g_free(dialog->gradient_colors);
  dialog->gradient_colors = g_new0(GdkColor, nlevels);
  reset_gradient(NULL, dialog);

  dialog->color_max = data->color_max;
  gdk_gc_set_foreground(gc, &dialog->color_max);
  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(dialog->button_max)->child)->pixmap,
                     gc,
                     TRUE,
                     4, 4, 33, 13);
  gtk_widget_draw(GTK_BIN(dialog->button_max)->child, NULL);

  dialog->color_min = data->color_min;
  gdk_gc_set_foreground(gc, &dialog->color_min);
  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(dialog->button_min)->child)->pixmap,
                     gc,
                     TRUE,
                     4, 4, 33, 13);
  gtk_widget_draw(GTK_BIN(dialog->button_min)->child, NULL);

  dialog->color_gt_max = data->color_gt_max;
  gdk_gc_set_foreground(gc, &dialog->color_gt_max);
  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(dialog->button_gt_max)->child)->pixmap,
                     gc,
                     TRUE,
                     4, 4, 33, 13);
  gtk_widget_draw(GTK_BIN(dialog->button_gt_max)->child, NULL);

  dialog->color_lt_min = data->color_lt_min;
  gdk_gc_set_foreground(gc, &dialog->color_lt_min);
  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(dialog->button_lt_min)->child)->pixmap,
                     gc,
                     TRUE,
                     4, 4, 33, 13);
  gtk_widget_draw(GTK_BIN(dialog->button_lt_min)->child, NULL);

  gdk_gc_unref(gc);
}

static gboolean
gradient_motion_notify(GtkWidget *event_box, GdkEventMotion *event)
{
  gint y;
  GdkPoint p[3];

  if(event->is_hint)
    gdk_window_get_pointer(event_box->window, NULL, &y, NULL);
  else
    y = event->y;

  gtk_widget_draw(GTK_BIN(event_box)->child, NULL);

  p[0].x = 0;
  p[0].y = y;
  p[1].x = 10;
  p[1].y = y-5;
  p[2].x = 10;
  p[2].y = y+5;
  gdk_draw_polygon(event_box->window, event_box->style->black_gc, TRUE, p, 3);

  p[0].x = WIDTH;
  p[0].y = y;

  p[1].x = WIDTH-10;
  p[1].y = y-5;
  p[2].x = WIDTH-10;
  p[2].y = y+5;

  gdk_draw_polygon(event_box->window, event_box->style->black_gc, TRUE, p, 3);
  return FALSE;
}

static void     
sg_gradient_dialog_init_gui (SGgradientDialog *dialog)
{
  GtkWidget *frame;
  GtkWidget *table, *table1;
  GtkWidget *box;
  GtkRequisition req;
  GtkAdjustment *adj;
  GdkPixmap *pixmap;
  GdkBitmap *mask;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *event_box;
  GtkWidget *reset_button;
  GdkWindowAttr attributes;
  gint attributes_mask;
  gchar *scale[] = { _("Linear"),
                     _("Log10"),
                     NULL}; /*, "Ln", "Log2", NULL};*/


  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.title = NULL;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gdk_visual_get_system ();
  attributes.colormap = gdk_colormap_get_system ();
  attributes.event_mask = 0;
  attributes_mask = GDK_WA_VISUAL | GDK_WA_COLORMAP;

  dialog->window = gdk_window_new (NULL, &attributes, attributes_mask);

  /* Gradient */
  gtk_frame_set_shadow_type (GTK_FRAME (dialog), GTK_SHADOW_ETCHED_IN);

  table = gtk_table_new (4, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add (GTK_CONTAINER (dialog), table);

  frame = gtk_frame_new( _("Levels") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
  gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 0, 4);

  table1 = gtk_table_new (6, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table1), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
  gtk_container_add(GTK_CONTAINER(frame), table1);

  dialog->button_min = gtk_button_new();
  dialog->button_max = gtk_button_new();
  dialog->button_lt_min = gtk_button_new();
  dialog->button_gt_max = gtk_button_new();

  box = gtk_hbox_new(FALSE, 5);

  pixmap = gdk_pixmap_new (dialog->window, WIDTH, HEIGHT, -1);

  dialog->gradient = gtk_pixmap_new(pixmap, NULL);
  gdk_pixmap_unref(pixmap);
  dialog->gradient_colors = NULL;

  event_box = dialog->event_box = gtk_event_box_new();
  gtk_widget_set_events(event_box, gtk_widget_get_events(event_box)|GDK_POINTER_MOTION_MASK|GDK_POINTER_MOTION_HINT_MASK|GDK_BUTTON1_MOTION_MASK);
  gtk_container_add(GTK_CONTAINER(event_box), dialog->gradient);
  gtk_box_pack_start(GTK_BOX(box), event_box, FALSE, FALSE, 0);

  gtk_signal_connect (GTK_OBJECT (event_box),
                      "motion_notify_event",
                      GTK_SIGNAL_FUNC (gradient_motion_notify), NULL);

  gtk_signal_connect (GTK_OBJECT (event_box),
                      "button_press_event",
                      GTK_SIGNAL_FUNC (open_color_selection2), dialog);


  gtk_table_attach(GTK_TABLE(table1), box, 0, 1, 1, 4,
                   (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 0);


  box = gtk_hbox_new(FALSE, 5);
  gtk_table_attach(GTK_TABLE(table1), box, 0, 1, 0, 1,
                   (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 0);
  gtk_box_pack_start(GTK_BOX(box), dialog->button_gt_max, 0, 0, 0);
  gtk_box_pack_start(GTK_BOX(box), gtk_label_new( _("> max") ), 0, 0, 0);

  box = gtk_hbox_new(FALSE, 5);
  gtk_table_attach(GTK_TABLE(table1), box, 0, 1, 4, 5,
                   (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 0);
  gtk_box_pack_start(GTK_BOX(box), dialog->button_lt_min, 0, 0, 0);
  gtk_box_pack_start(GTK_BOX(box), gtk_label_new( _("< min") ), 0, 0, 0);

  box = gtk_hbox_new(FALSE, 5);
  gtk_table_attach(GTK_TABLE(table1), box, 1, 2, 1, 2,
                   (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 0);
 
  dialog->max_spin = gtk_entry_new();
/*
  sg_entry_set_numeric(GTK_ENTRY(dialog->max_spin), 15);
*/
  gtk_box_pack_start(GTK_BOX(box), dialog->button_max, 0, 0, 0);
  gtk_box_pack_start(GTK_BOX(box), dialog->max_spin, 0, 0, 0);

  button = gtk_button_new_with_label( _("Get max...") );
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (insert_max),
                      dialog);

  box = gtk_hbox_new(FALSE, 5);
  gtk_table_attach(GTK_TABLE(table1), box, 1, 2, 3, 4,
                   (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 0);

  dialog->min_spin = gtk_entry_new();
/*
  sg_entry_set_numeric(GTK_ENTRY(dialog->min_spin), 15);
*/
  gtk_box_pack_start(GTK_BOX(box), dialog->button_min, 0, 0, 0);
  gtk_box_pack_start(GTK_BOX(box), dialog->min_spin, 0, 0, 0);

  button = gtk_button_new_with_label( _("Get min...") );
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
                      GTK_SIGNAL_FUNC (insert_min),
                      dialog);

  box = gtk_hbox_new(FALSE, 5);
  gtk_table_attach(GTK_TABLE(table1), box, 1, 2, 2, 3,
                             (GtkAttachOptions)0, (GtkAttachOptions)0, 0, 0);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 100., 1., 1., 1.);
  dialog->levels_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->levels_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->levels_spin), 0);
  gtk_box_pack_start(GTK_BOX(box), dialog->levels_spin, 0, 0, 0);

  label = gtk_label_new( _("Levels") );
  gtk_misc_set_alignment(GTK_MISC(label), 0., .5);
  gtk_box_pack_start(GTK_BOX(box), label, 0, 0, 0);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 100., 1., 1., 1.);
  dialog->sublevels_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->sublevels_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->sublevels_spin), 0);
  gtk_box_pack_start(GTK_BOX(box), dialog->sublevels_spin, 0, 0, 0);

  label = gtk_label_new( _("Sublevels") );
  gtk_misc_set_alignment(GTK_MISC(label), 0., .5);
  gtk_box_pack_start(GTK_BOX(box), label, 0, 0, 0);

  gtk_widget_size_request(dialog->max_spin, &req);
  gtk_widget_set_usize(dialog->max_spin, req.width / 2, req.height);
  gtk_widget_set_usize(dialog->min_spin, req.width / 2, req.height);
  gtk_widget_set_usize(dialog->levels_spin, req.width / 2, req.height);
  gtk_widget_set_usize(dialog->sublevels_spin, req.width / 2, req.height);

  pixmap=gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                               gdk_colormap_get_system(),
                                               &mask, NULL,
                                               color_xpm);
  dialog->color_pixmap = gtk_pixmap_new(pixmap, mask);
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
  gtk_container_add(GTK_CONTAINER(dialog->button_max), dialog->color_pixmap);

  pixmap=gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                               gdk_colormap_get_system(),
                                               &mask, NULL,
                                               color_xpm);
  dialog->color_pixmap = gtk_pixmap_new(pixmap, mask);
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
  gtk_container_add(GTK_CONTAINER(dialog->button_min), dialog->color_pixmap);

  pixmap=gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                               gdk_colormap_get_system(),
                                               &mask, NULL,
                                               color_xpm);
  dialog->color_pixmap = gtk_pixmap_new(pixmap, mask);
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
  gtk_container_add(GTK_CONTAINER(dialog->button_lt_min), dialog->color_pixmap);

  pixmap=gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                               gdk_colormap_get_system(),
                                               &mask, NULL,
                                               color_xpm);
  dialog->color_pixmap = gtk_pixmap_new(pixmap, mask);
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
  gtk_container_add(GTK_CONTAINER(dialog->button_gt_max), dialog->color_pixmap);

  dialog->scale_combo = gtk_combo_new();
  gtk_widget_set_usize(dialog->scale_combo, req.width, req.height);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->scale_combo)->entry), FALSE);
  sg_combo_set_items(GTK_COMBO(dialog->scale_combo), scale);

  box = gtk_hbox_new(FALSE, 5);
  gtk_table_attach(GTK_TABLE(table1), box, 1, 2, 4, 5, 0, 0, 0, 0);
  gtk_box_pack_start(GTK_BOX(box), gtk_label_new( ("Scale:") ), 0, 0, 0);
  gtk_box_pack_start(GTK_BOX(box), dialog->scale_combo, 0, 0, 0);

  
  frame = gtk_frame_new( _("Increment") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_set_border_width(GTK_CONTAINER(frame), 10);
  gtk_table_attach_defaults(GTK_TABLE(table), frame, 1, 2, 0, 1);

  table1 = gtk_table_new (7, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table1), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
  gtk_container_add(GTK_CONTAINER(frame), table1);

  dialog->h_check = gtk_check_item_new_with_label( _("Hue") );
  dialog->s_check = gtk_check_item_new_with_label( _("Saturation") );
  dialog->v_check = gtk_check_item_new_with_label( _("Value") );

  gtk_table_attach_defaults(GTK_TABLE(table1), dialog->h_check, 0, 1, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table1), dialog->s_check, 0, 1, 4, 5);
  gtk_table_attach_defaults(GTK_TABLE(table1), dialog->v_check, 0, 1, 5, 6);
  gtk_table_attach_defaults(GTK_TABLE(table1),
                          reset_button = gtk_button_new_with_label("Reset"),
                          0, 1, 7, 8);

  dialog->gradient_check = gtk_check_item_new_with_label( _("show gradient") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->gradient_check, 1, 2, 1, 2);

  dialog->custom_check = gtk_check_item_new_with_label( _("customize gradient") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->custom_check, 1, 2, 2, 3);

  gtk_widget_show_all(table);

  /* connect signals */
  gtk_signal_connect(GTK_OBJECT(dialog->button_max),"clicked",
                     GTK_SIGNAL_FUNC(open_color_selection), dialog);

  gtk_signal_connect(GTK_OBJECT(dialog->button_min), "clicked",
                     GTK_SIGNAL_FUNC(open_color_selection), dialog);

  gtk_signal_connect(GTK_OBJECT(dialog->button_lt_min),"clicked",
                     GTK_SIGNAL_FUNC(open_color_selection), dialog);

  gtk_signal_connect(GTK_OBJECT(dialog->button_gt_max),"clicked",
                     GTK_SIGNAL_FUNC(open_color_selection), dialog);

  gtk_signal_connect(GTK_OBJECT(reset_button),"clicked",
                     GTK_SIGNAL_FUNC(reset_gradient), dialog);
}

static void
pick_level_color(GtkWidget *widget, gpointer data)
{
  SGgradientDialog *dialog = SG_GRADIENT_DIALOG(data);
  GdkColor color;
  gdouble values[4];
  gint y;
  GtkWidget *event_box = dialog->event_box;

  gdk_window_get_pointer(event_box->window, NULL, &y, NULL);

  gtk_color_selection_get_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(dialog->csd)->colorsel), values);

  color.red = (gushort)(0xffff * values[0]);
  color.green = (gushort)(0xffff * values[1]);
  color.blue = (gushort)(0xffff * values[2]);
  gdk_color_alloc(gdk_colormap_get_system(), &color);

  dialog->gradient_colors[dialog->gradient_level] = color;

  init_gradient_pixmap(dialog);

  gtk_widget_destroy(dialog->csd);
}

static gboolean
mw_destroy(GtkWidget *widget, gpointer data)
{
  gtk_main_quit();
  return FALSE;
}

static void
open_color_selection2(GtkWidget *event_box, GdkEventButton *event, gpointer data)
{
  SGgradientDialog *dialog = SG_GRADIENT_DIALOG(data);
  GdkModifierType mods;
  GdkColor color;
  gdouble values[4];
  gint y;

  gdk_window_get_pointer(event_box->window, NULL, &y, &mods);
  if(!(mods & GDK_BUTTON1_MASK)) return;

  dialog->gradient_level = (HEIGHT-y) / (gdouble)HEIGHT * (dialog->dataset->gradient->ticks.nticks-1);

  color = dialog->gradient_colors[dialog->gradient_level];

  dialog->csd = gtk_color_selection_dialog_new ( _("Pick a color") );

  gtk_signal_connect (GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(dialog->csd)->ok_button),
                      "clicked", GTK_SIGNAL_FUNC(pick_level_color), dialog);

  gtk_signal_connect_object (GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(dialog->csd)->cancel_button),
                             "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                             GTK_OBJECT (dialog->csd));

  values[0] = color.red / (gdouble)0xffff;
  values[1] = color.green / (gdouble)0xffff;
  values[2] = color.blue / (gdouble)0xffff;

  gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(dialog->csd)->colorsel), values);

  gtk_signal_connect(GTK_OBJECT(dialog->csd), "destroy", GTK_SIGNAL_FUNC(mw_destroy), NULL);
  gtk_widget_show(dialog->csd);
  gtk_main();
}

static void
pick_color(GtkWidget *widget, gpointer data)
{
  GtkWidget *button;
  GdkColor color;
  GdkGC *gc;
  gdouble values[4];
  SGgradientDialog *dialog = SG_GRADIENT_DIALOG(data);

  button = dialog->active_button;
  gtk_color_selection_get_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(dialog->csd)->colorsel), values);

  color.red = (gushort)(0xffff * values[0]);
  color.green = (gushort)(0xffff * values[1]);
  color.blue = (gushort)(0xffff * values[2]);
  gdk_color_alloc(gdk_colormap_get_system(), &color);

  if(button == dialog->button_max)
       dialog->color_max = color;
  else if(button == dialog->button_min)
       dialog->color_min = color;
  else if(button == dialog->button_lt_min)
       dialog->color_lt_min = color;
  else if(button == dialog->button_gt_max)
       dialog->color_gt_max = color;

  gc = gdk_gc_new(dialog->window);

  gdk_gc_set_foreground(gc, &color);

  gdk_draw_rectangle(GTK_PIXMAP(GTK_BIN(button)->child)->pixmap,
                     gc,
                     TRUE,
                     4, 4, 33, 13);
  gtk_widget_draw(GTK_BIN(button)->child, NULL);
  gtk_widget_destroy(dialog->csd);
  gdk_gc_unref(gc);
}

static void
open_color_selection(GtkWidget *widget, gpointer data)
{
  GtkWidget *button;
  GdkColor color;
  gdouble values[4];
  SGgradientDialog *dialog = SG_GRADIENT_DIALOG(data);

  button = widget;
  if(button == dialog->button_max)
    color = dialog->color_max;
  else if(button == dialog->button_min)
    color = dialog->color_min;
  else if(button == dialog->button_lt_min)
    color = dialog->color_lt_min;
  else if(button == dialog->button_gt_max)
    color = dialog->color_gt_max;
  dialog->active_button = button;

  dialog->csd = gtk_color_selection_dialog_new ( _("Pick a color") );

  gtk_signal_connect (GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(dialog->csd)->ok_button),
                      "clicked", GTK_SIGNAL_FUNC(pick_color),
                      dialog);

  gtk_signal_connect_object (GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(dialog->csd)->cancel_button),
                             "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                             GTK_OBJECT (dialog->csd));

  values[0] = color.red / (gdouble)0xffff;
  values[1] = color.green / (gdouble)0xffff;
  values[2] = color.blue / (gdouble)0xffff;
  gtk_color_selection_set_color(GTK_COLOR_SELECTION(GTK_COLOR_SELECTION_DIALOG(dialog->csd)->colorsel), values);

  gtk_signal_connect(GTK_OBJECT(dialog->csd), "destroy", GTK_SIGNAL_FUNC(mw_destroy), NULL);
  gtk_widget_show(dialog->csd);
  gtk_main();
}

static void     
sg_gradient_dialog_apply     (SGpropertyDialog *d, gpointer _data)
{
  SGgradientDialog *dialog = SG_GRADIENT_DIALOG(d);
  GtkPlotData *data = dialog->dataset;
  GdkColormap *colormap;
  gboolean state;
  gdouble max, min;
  gint nmajor, nminor;
  gdouble gmax, gmin;
  gint gnmajor, gnminor;
  GtkWidget *child;

  /* scale type (linear or log) set here */
  child = GTK_WIDGET(GTK_LIST(GTK_COMBO(dialog->scale_combo)->list)->selection->data);
  data->gradient->ticks.scale = (GtkPlotScale)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->scale_combo)->list), child);


  colormap = gtk_widget_get_colormap(GTK_WIDGET(data));

  data->gradient_mask = 0;
  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->h_check));
  data->gradient_mask |= state * GTK_PLOT_GRADIENT_H;
  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->s_check));
  data->gradient_mask |= state * GTK_PLOT_GRADIENT_S;
  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->v_check));
  data->gradient_mask |= state * GTK_PLOT_GRADIENT_V;

  data->show_gradient = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->gradient_check));
  data->gradient_custom = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->custom_check));

  gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(data)), &dialog->color_min);
  gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(data)), &dialog->color_max);
  gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(data)), &dialog->color_lt_min);
  gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(data)), &dialog->color_gt_max);
  data->color_max = dialog->color_max;
  data->color_min = dialog->color_min;
  data->color_gt_max = dialog->color_gt_max;
  data->color_lt_min = dialog->color_lt_min;

  min = atof(gtk_entry_get_text(GTK_ENTRY(dialog->min_spin)));
  max = atof(gtk_entry_get_text(GTK_ENTRY(dialog->max_spin)));
  nmajor = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->levels_spin));
  nminor = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->sublevels_spin));

  gtk_plot_data_get_gradient(data, &gmin, &gmax, &gnmajor, &gnminor);
  if(min == gmin && max == gmax && nmajor == gnmajor && nminor == gnminor){
    gint nlevels, level;

    nlevels = data->gradient->ticks.nticks;
    for(level = nlevels-1; level >= 0; level--){
      gdk_color_alloc(gtk_widget_get_colormap(GTK_WIDGET(data)), 
                      &dialog->gradient_colors[level]);
      gtk_plot_data_set_gradient_nth_color(data, level, &dialog->gradient_colors[level]);
    }
  }

  gtk_plot_data_set_gradient(data, min, max, nmajor, nminor);
  reset_gradient(NULL, dialog);
}

