/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_property_dialog.h"

static gchar *connectors[] = {
  N_("None"),
  N_("Straight"),
  N_("Spline"),
  N_("H-V step"),
  N_("V-H step"),
  N_("Middle step"),
  NULL
};

static gchar *line_styles[] = {
  N_("None"),
  N_("Solid"),
  N_("Dotted"),
  N_("Dashed"),
  N_("Dot-dash"),
  N_("Dot-dot-dash"),
  N_("Dot-dash-dash"),
  NULL
};

typedef struct 
{
  GtkWidget *fill_check;
  GtkWidget *connector_combo;
  GtkWidget *line_width_spin;
  GtkWidget *line_style_combo;
  GtkWidget *line_color_combo;
  GtkPlotData *dataset;
} SGdatalineDialog;

static void sg_data_line_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_data_line_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_data_line_dialog_init_gui         (GtkWidget *frame);

GtkWidget *
sg_data_line_dialog_new(GtkPlotData *dataset)
{
  GtkWidget *widget;
  SGdatalineDialog *dialog;

  dialog = g_new0(SGdatalineDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_data_line_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_data_line_dialog_ok;
  dialog->dataset = dataset;
  sg_data_line_dialog_init_gui(widget);

  return widget;
}

static void
sg_data_line_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_data_line_dialog_apply(d, data);
}

static void     
init_dialog (SGdatalineDialog *dialog)
{
  GtkPlotData *data = dialog->dataset;
  GdkColor line_color;
  GtkPlotLineStyle line_style;
  GdkCapStyle cap_style;
  GdkJoinStyle join_style;
  gfloat line_width;
  gint connector;

  gtk_plot_data_get_line_attributes(data,
                                    &line_style, &cap_style, &join_style,
                                    &line_width, &line_color);

  connector = gtk_plot_data_get_connector(data);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->line_style_combo)->list), line_style);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->connector_combo)->list), connector);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->fill_check),
                            gtk_plot_data_area_is_filled(data));

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->line_width_spin), line_width);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->line_color_combo), line_color);
}

static void     
sg_data_line_dialog_init_gui (GtkWidget *frame)
{
  GtkWidget *table;
  GtkWidget *label;
  GtkRequisition req;
  GtkAdjustment *adj;
  SGdatalineDialog *dialog;
  GtkPlotData *data;

  dialog = (SGdatalineDialog *)SG_PROPERTY_DIALOG(frame)->data; 
  data = dialog->dataset;
  gtk_frame_set_label(GTK_FRAME(frame), _("Line") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

  table = gtk_table_new(6, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new( _("Connect:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,0,1);

  dialog->connector_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->connector_combo,1,2,0,1);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->connector_combo)->entry), FALSE);
  gtk_widget_size_request(GTK_WIDGET(dialog->connector_combo), &req);
  req.width /= 2;

  sg_combo_set_items(GTK_COMBO(dialog->connector_combo), connectors);
  label = gtk_label_new( _("Type:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);

  dialog->line_style_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->line_style_combo,1,2,1,2);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->line_style_combo)->entry), FALSE);

  sg_combo_set_items(GTK_COMBO(dialog->line_style_combo), line_styles);

  label = gtk_label_new( _("Width:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., .5, 1., 0.);
  dialog->line_width_spin = gtk_spin_button_new(adj, 0.5, 1);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->line_width_spin), TRUE);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->line_width_spin,1,2,2,3);

  label = gtk_label_new( _("Color:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);
  dialog->line_color_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->line_color_combo,1,2,3,4);

  dialog->fill_check = gtk_check_item_new_with_label( _("Fill area") );
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->fill_check,0,2,4,5);

  gtk_widget_set_usize(dialog->connector_combo, req.width, req.height);
  gtk_widget_set_usize(dialog->line_width_spin, req.width, req.height);
  gtk_widget_set_usize(dialog->line_style_combo, req.width, req.height);

  init_dialog(dialog);
}

static void     
sg_data_line_dialog_apply     (SGpropertyDialog *d, gpointer _data)
{
  SGdatalineDialog *dialog = (SGdatalineDialog *)d->data;
  GtkPlotData *data = dialog->dataset;
  GtkWidget *child;
  GdkColor line_color;
  GtkPlotLineStyle line_style; 
  gfloat line_width;
  GtkPlotConnector connector;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->line_style_combo)->list)->selection->data;
  line_style = (GtkPlotLineStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->line_style_combo)->list), child);

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->connector_combo)->list)->selection->data;
  connector = (GtkPlotConnector)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->connector_combo)->list), child);

  line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->line_width_spin));

  data->fill_area = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->fill_check));

  line_color = gtk_color_combo_get_selection(GTK_COLOR_COMBO(dialog->line_color_combo));

  gtk_plot_data_set_connector(data, connector);

  gtk_plot_data_set_line_attributes(data,
                                    (GtkPlotLineStyle) line_style, 0, 0,
                                    line_width,
                                    &line_color);
}

