/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_property_dialog.h"

static gchar *label_formats[] = 
{ 
  N_("Decimal: 1.000"),
  N_("Scientific: 1.E10"),
  N_("Scientific: 1x10^10"),
  NULL
};

typedef struct
{
  GtkWidget *font_combo;
  GtkWidget *font_color_combo;
  GtkWidget *labels_check;
  GtkWidget *precision_spin;
  GtkWidget *labels_combo;
  gboolean mode;
  GtkPlotData *dataset;
} SGdatalabelsDialog;

static void sg_data_labels_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_data_labels_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_data_labels_dialog_init_gui         (GtkWidget *frame);

GtkWidget *
sg_data_labels_dialog_new(GtkPlotData *dataset, gboolean use_precision)
{
  GtkWidget *widget;
  SGdatalabelsDialog *dialog;

  dialog = g_new0(SGdatalabelsDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_data_labels_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_data_labels_dialog_ok;
  dialog->dataset = dataset;
  dialog->mode = use_precision;
  sg_data_labels_dialog_init_gui(widget);

  return widget;
}

static void
sg_data_labels_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_data_labels_dialog_apply(d, data);
}

static void     
init_dialog (SGdatalabelsDialog *dialog)
{
  GtkPlotData *data = dialog->dataset;
  GtkPSFont *font;

  font = gtk_psfont_get_by_name(data->labels_attr.font);
  gtk_font_combo_select(GTK_FONT_COMBO(dialog->font_combo),
                        font->family,
                        font->bold,
                        font->italic,
                        data->labels_attr.height);

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->font_color_combo),
                      data->labels_attr.fg);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->labels_check),
                               data->show_labels);

  if(dialog->mode){
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->precision_spin), data->legends_precision);
    gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->labels_combo)->list), data->legends_style);
  }
}

static void     
sg_data_labels_dialog_init_gui (GtkWidget *frame)
{
  GtkWidget *box;
  SGdatalabelsDialog *dialog = (SGdatalabelsDialog *)SG_PROPERTY_DIALOG(frame)->data;

  gtk_frame_set_label(GTK_FRAME(frame), _("Labels") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
  box = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(frame), box);

  dialog->font_combo = gtk_font_combo_new();
  gtk_box_pack_start(GTK_BOX(box), dialog->font_combo, FALSE, FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(dialog->font_combo), 5);
  gtk_toolbar_append_space(GTK_TOOLBAR(dialog->font_combo));

  dialog->font_color_combo = gtk_color_combo_new();
  gtk_container_add(GTK_CONTAINER(dialog->font_combo), dialog->font_color_combo);

  gtk_toolbar_append_space(GTK_TOOLBAR(dialog->font_combo));
  gtk_toolbar_append_space(GTK_TOOLBAR(dialog->font_combo));
  gtk_toolbar_append_space(GTK_TOOLBAR(dialog->font_combo));

  dialog->labels_check = gtk_check_item_new_with_label( _("show labels") );
  gtk_container_add(GTK_CONTAINER(dialog->font_combo), dialog->labels_check);

  if(dialog->mode){
    GtkAdjustment *adj;
    GtkWidget *aux_box;

    aux_box = gtk_hbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(box), aux_box, FALSE, FALSE, 5);
    adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 1., 1., 0.);
    dialog->precision_spin = gtk_spin_button_new(adj, 0, 0);
    gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->precision_spin), TRUE);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->precision_spin), 0);
    gtk_box_pack_start(GTK_BOX(aux_box), gtk_label_new( _("Precision: ") ), FALSE, FALSE, 5);
    gtk_box_pack_start(GTK_BOX(aux_box), dialog->precision_spin, FALSE, FALSE, 5);
  
    dialog->labels_combo = gtk_combo_new();
    sg_combo_set_items(GTK_COMBO(dialog->labels_combo), label_formats);
    gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->labels_combo)->entry), FALSE);
    gtk_box_pack_start(GTK_BOX(aux_box), gtk_label_new( _("Format: ") ), FALSE, FALSE, 5);
    gtk_box_pack_start(GTK_BOX(aux_box), dialog->labels_combo, FALSE, FALSE, 5);
  }

  init_dialog(dialog);
}

static void     
sg_data_labels_dialog_apply     (SGpropertyDialog *d, gpointer _data)
{
  SGdatalabelsDialog *dialog = (SGdatalabelsDialog *)d->data;
  GtkPlotData *data = dialog->dataset;
  const gchar *font_name;
  gint font_height;
  GdkColor font_color;
  GtkPSFont *psfont;

  psfont = gtk_font_combo_get_psfont(GTK_FONT_COMBO(dialog->font_combo));
  font_name = psfont->psname;
  font_height = gtk_font_combo_get_font_height(GTK_FONT_COMBO(dialog->font_combo));
 
  font_color = gtk_color_combo_get_selection(GTK_COLOR_COMBO(dialog->font_color_combo));

  gtk_plot_data_labels_set_attributes(data,
                                      font_name, font_height, 90,
                                      &font_color, NULL);

  data->show_labels = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->labels_check));

  if(dialog->mode){
    GtkWidget *child;
    data->legends_precision = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->precision_spin));
    child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->labels_combo)->list)->selection->data;  data->legends_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->labels_combo)->list), child);
  }
}

