/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <Python.h>
#include "sg_data_expr_dialog.h"
#include "sg.h"
#include "sg_python_expr.h"
#include "sg_dialog.h"

typedef struct
{
  SGplot *plot;
  SGdataset *dataset;
  SGdataset *new_dataset;
  GtkWidget *name_entry;
  GtkWidget *dataset_combo;
  GtkWidget *create_item;
} SGexprDialog;

static void
update_dataset(SGpropertyDialog *pdialog, gpointer data)
{
  SGexprDialog *dialog = (SGexprDialog *)data;
  const gchar *text = NULL;
  gboolean create;
  gdouble *new_x = NULL, *new_y = NULL;
  gdouble *x = NULL, *y = NULL;
  GtkPlotArray *array;
  gint i, npoints;
  SGdataset *new_dataset = NULL;

  if(!dialog->dataset) return;
  text = gtk_entry_get_text(GTK_ENTRY(dialog->name_entry));

  create = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->create_item));
 
  array = gtk_plot_array_list_get(dialog->dataset->arrays, "x");
  x = gtk_plot_array_get_double(array);
  array = gtk_plot_array_list_get(dialog->dataset->arrays, "y");
  y = gtk_plot_array_get_double(array);
  npoints = gtk_plot_array_get_size(array);

  new_x = g_new0(gdouble, npoints);
  new_y = g_new0(gdouble, npoints);

  for(i = 0; i < npoints; i++)
    sg_eval_expr_xy(text, x[i], y[i], &new_x[i], &new_y[i]);

  if(create){
    new_dataset = sg_plugin_iterator_construct(SG_PLUGIN_ITERATOR(dialog->dataset->plugin), dialog->dataset->constructor);
    sg_dataset_set_name(new_dataset, dialog->dataset->name);
    sg_dataset_set_points(new_dataset, "x", new_x, npoints);
    sg_dataset_set_points(new_dataset, "y", new_y, npoints);
    sg_list_add(dialog->plot->datasets, G_OBJECT(new_dataset), new_dataset->name);
    sg_layer_add_dataset_autosymbol(dialog->plot->active_layer, new_dataset);
  } else {
    new_dataset = dialog->dataset;
    sg_dataset_set_points(new_dataset, "x", new_x, npoints);
    sg_dataset_set_points(new_dataset, "y", new_y, npoints);
  }
  dialog->dataset = new_dataset;
  sg_dataset_refresh_arrays(dialog->dataset);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(dialog->plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(dialog->plot));
}

static void
dialog_select(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  SGexprDialog *dialog = (SGexprDialog *)data;
                                                                               
  child = GTK_WIDGET(GTK_LIST(GTK_COMBO(dialog->dataset_combo)->list)->selection->data);
  dialog->dataset = SG_DATASET(g_object_get_data(G_OBJECT(child),"dataset"));
}

SGdataset *
sg_data_expr_dialog (SGplot *plot)
{
  GtkWidget *ref_dialog;
  SGexprDialog *dialog;
  GtkWindow *window;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GList *list;
  SGdataset *new_dataset = NULL;

  dialog = g_new0(SGexprDialog, 1);
  dialog->plot = plot;
  dialog->dataset = NULL;
  dialog->new_dataset = NULL;

  /* Create widgets */
  frame = sg_property_dialog_new();
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(frame), dialog, FALSE);

  table = gtk_table_new(2, 4, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  label = gtk_label_new( _("Dataset:") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 0, 1);
  dialog->dataset_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->dataset_combo, 1, 4, 0, 1);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->dataset_combo)->entry), FALSE);

  label = gtk_label_new(_("Expression:")),
  gtk_misc_set_alignment(GTK_MISC(label), 0., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 1, 2);
  dialog->name_entry = gtk_entry_new();   
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->name_entry, 1, 4, 1, 2);

  dialog->create_item = gtk_check_item_new_with_label(_("Create new dataset"));
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->create_item, 0, 4, 2, 3);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->create_item), TRUE);

  /* Action Area */
  SG_PROPERTY_DIALOG(frame)->ok = update_dataset;
  SG_PROPERTY_DIALOG(frame)->apply = update_dataset;
  ref_dialog = sg_dialog_new( _("SciGraphica: Expression"), GTK_ORIENTATION_VERTICAL, SG_BUTTON_OK|SG_BUTTON_APPLY|SG_BUTTON_CLOSE, GTK_BUTTONBOX_SPREAD);
  window = GTK_WINDOW(ref_dialog);

  /* connect signals */
  gtk_signal_connect (GTK_OBJECT (dialog->name_entry), "activate",
                      GTK_SIGNAL_FUNC (update_dataset),
                      dialog);
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(dialog->dataset_combo)->entry),
                     "changed",
                     GTK_SIGNAL_FUNC(dialog_select),
                     dialog);

  /* initialize */
  list = GTK_PLOT_CANVAS_PLOT(plot->active_layer)->plot->data_sets;
  while(list){
    GtkWidget *item;
    SGdataset *dataset;
    GList *aux = GTK_PLOT_CANVAS_PLOT(plot->active_layer)->plot->data_sets;
    gint n = 1;
                                                                               
    dataset = SG_DATASET(GTK_PLOT_DATA(list->data)->link);
    while(aux){
      GtkPlotData *aux_data = GTK_PLOT_DATA(aux->data);
      if(SG_DATASET(aux_data->link) != dataset && strcmp(dataset->name, aux_data->name) == 0) n++;
      if(aux_data->link == dataset) break;
      aux = aux->next;
    }
    if(n > 1){
      gchar name[200];
      g_snprintf(name, 200, "%s (%d)", dataset->name, n);
      item = gtk_list_item_new_with_label(name);
    } else {
      item = gtk_list_item_new_with_label(dataset->name);
    }
    g_object_set_data(G_OBJECT(item), "dataset", dataset);
    gtk_widget_show(item);
    gtk_container_add(GTK_CONTAINER(GTK_COMBO(dialog->dataset_combo)->list), item);
    list = list->next;
  }

  /* show widgets */

  dialog_select(NULL, dialog);
  sg_dialog_add(ref_dialog, SG_PROPERTY_DIALOG(frame));
  gtk_window_set_policy(window, FALSE, FALSE, FALSE);
  gtk_widget_grab_focus(dialog->name_entry);
  gtk_widget_show_all(frame); 
  sg_dialog_run (ref_dialog, GTK_OBJECT(plot));

  new_dataset = dialog->new_dataset;
  g_free(dialog);
  return new_dataset;
}
