/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_property_dialog.h"
#include "sg_break_dialog.h"
#include "sg.h"
#include <math.h>

typedef struct
{
  SGlayer *layer;
  SGplot *plot;

  GtkPlotAxis *axis;

  GtkPlotAxis *axis1, *axis2;

  GtkPlotOrientation orientation;

  GtkWidget *min_entry, *max_entry, *step_entry;
  GtkWidget *minor_spin;
  GtkWidget *position_scale;
  GtkWidget *scale_combo;
  GtkWidget *apply_check;

  GtkPlotTicks *ticks;
  gdouble min,max;
} SGbreakDialog;

static void sg_break_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_break_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_break_dialog_init_gui         (GtkWidget *frame);

GtkWidget *
sg_break_dialog_new(SGlayer *layer, GtkPlotAxis *axis)
{
  GtkWidget *widget;
  SGbreakDialog *dialog;

  dialog = g_new0(SGbreakDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->ok = sg_break_dialog_ok;
  SG_PROPERTY_DIALOG(widget)->apply = sg_break_dialog_apply;
  dialog->layer = layer;
  dialog->axis = axis;
  dialog->min = (axis->ticks.min + axis->ticks.max)/2;
  dialog->max = axis->ticks.max;
  sg_break_dialog_init_gui(widget);

  return widget;
}

static void
sg_break_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_break_dialog_apply(d, data);
}

static void
sg_break_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  SGbreakDialog *dialog = (SGbreakDialog *)d->data;
  GtkPlot *plot;
  GtkWidget *child;
  gboolean state;
  gdouble max, min;
  gdouble major;
  gint nminor;
  gdouble position;
  guint scale;
  
  plot = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot;

  min = atof(gtk_entry_get_text(GTK_ENTRY(dialog->min_entry)));
  max = atof(gtk_entry_get_text(GTK_ENTRY(dialog->max_entry)));
  nminor = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->minor_spin));
  major = atof(gtk_entry_get_text(GTK_ENTRY(dialog->step_entry)));

  if(min > max) min = max;

  state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->apply_check));
  position = GTK_RANGE(dialog->position_scale)->adjustment->value;

  /* scale type (lin or log) set here */
  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->scale_combo)->list)->selection->data;
  scale = (GtkPlotScale)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->scale_combo)->list), child);

  gtk_plot_set_break(plot, dialog->orientation, min, max, major, nminor, scale, position); 
  if(!state) gtk_plot_remove_break(plot,dialog->orientation);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(dialog->plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(dialog->plot));
}

static void
init_dialog(SGbreakDialog *dialog)
{
  GtkPlot *plot;
  gchar text[100];

  plot = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot;

  sprintf(text,"%g",dialog->ticks->break_min);
  gtk_entry_set_text(GTK_ENTRY(dialog->min_entry), text);

  sprintf(text,"%g",dialog->ticks->break_max);
  gtk_entry_set_text(GTK_ENTRY(dialog->max_entry), text);

  sprintf(text,"%g",dialog->ticks->break_step);
  gtk_entry_set_text(GTK_ENTRY(dialog->step_entry), text);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->minor_spin), dialog->ticks->break_nminor);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->scale_combo)->list),
                       dialog->ticks->break_scale);

  gtk_adjustment_set_value(GTK_RANGE(dialog->position_scale)->adjustment,
                           dialog->ticks->break_position);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->apply_check), dialog->ticks->apply_break);

}

static void 
sg_break_dialog_init_gui (GtkWidget *dialog_frame)
{
  GtkPlot *plot;
  GtkWidget *label;
  GtkWidget *table;
  GtkRequisition req;
  GtkAdjustment *adj;
  GtkWidget *main_table;
  GtkWidget *frame;
  SGbreakDialog *dialog;
  gchar *scale[] = { N_("Linear"),
                     N_("Log10"),
                     NULL}; /*, "Ln", "Log2", NULL};*/

  dialog = (SGbreakDialog *)SG_PROPERTY_DIALOG(dialog_frame)->data;
  dialog->plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(dialog->layer)->parent);
  plot = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot;
  dialog->orientation = dialog->axis->orientation;

  switch(dialog->orientation){
    case GTK_PLOT_AXIS_X:
      dialog->ticks = &plot->bottom->ticks;
      dialog->axis1 = plot->bottom;
      dialog->axis2 = plot->top;
      break;
    case GTK_PLOT_AXIS_Y:
      dialog->ticks = &plot->left->ticks;
      dialog->axis1 = plot->left;
      dialog->axis2 = plot->right;
      break;
    case GTK_PLOT_AXIS_Z:
    default:
      break;
  }

  /* Create widgets */
  main_table = gtk_table_new (4,4,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 10);
  gtk_container_add(GTK_CONTAINER(dialog_frame), main_table);

  frame = gtk_frame_new( _("Break") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
  gtk_table_attach (GTK_TABLE (main_table), frame, 0, 1, 0, 1, 0, GTK_FILL, 0, 0);

  table = gtk_table_new(2,2,TRUE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);

  label = gtk_label_new( _("From:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 0, 1);
  label = gtk_label_new( _("To:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults (GTK_TABLE (table), label, 0, 1, 1, 2);

  dialog->min_entry = gtk_entry_new();
  gtk_widget_size_request(dialog->min_entry, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->min_entry, req.width, req.height);

  dialog->max_entry = gtk_entry_new();
  gtk_widget_size_request(dialog->max_entry, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->max_entry, req.width, req.height);

  gtk_table_attach (GTK_TABLE (table), dialog->min_entry, 1, 2, 0, 1, 0, 0, 0, 0);
  gtk_table_attach (GTK_TABLE (table), dialog->max_entry, 1, 2, 1, 2, 0, 0, 0, 0);

  frame = gtk_frame_new( _("Ticks after break") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
  gtk_table_attach (GTK_TABLE (main_table), frame, 1, 2, 0, 1, 0, GTK_FILL, 0, 0);

  table = gtk_table_new(2,2,TRUE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);

  label = gtk_label_new( _("Major tick interval:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  dialog->step_entry = gtk_entry_new();
  gtk_widget_set_usize(dialog->step_entry, req.width, req.height);
  gtk_table_attach(GTK_TABLE(table), dialog->step_entry, 1, 2, 0, 1, 0, 0, 0, 0);

  label = gtk_label_new( _("# Minor ticks:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 100., 1., 1., 0.);

  dialog->minor_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_widget_set_usize(dialog->minor_spin, req.width, req.height);
  gtk_table_attach(GTK_TABLE(table), dialog->minor_spin, 1, 2, 1, 2, 0, 0, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->minor_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->minor_spin), 0);

  label = gtk_label_new( _("Scale:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);

  dialog->scale_combo = gtk_combo_new();
  gtk_widget_set_usize(dialog->scale_combo, req.width, req.height);
  gtk_table_attach(GTK_TABLE(table), dialog->scale_combo, 1, 2, 2, 3, 0, 0, 0, 0);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->scale_combo)->entry), FALSE);
  sg_combo_set_items(GTK_COMBO(dialog->scale_combo), scale);

  label = gtk_label_new( _("Position (%)") );
  gtk_misc_set_alignment(GTK_MISC(label), 0.8, .5);
  gtk_table_attach(GTK_TABLE(main_table), label, 0, 1, 1, 2, GTK_FILL, 0, 0, 0);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 1.1, .01, .1, .1);
  dialog->position_scale = gtk_hscale_new(adj);
  gtk_scale_set_digits(GTK_SCALE(dialog->position_scale), 2);
  gtk_table_attach(GTK_TABLE(main_table), dialog->position_scale, 1, 2, 1, 2, GTK_FILL, 0, 0, 0);

  dialog->apply_check = gtk_check_item_new_with_label(_("Apply break"));
  gtk_table_attach(GTK_TABLE(main_table), dialog->apply_check, 0, 2, 2, 3, GTK_FILL, 0, 10, 40);

  init_dialog(dialog);
}

GtkWidget *
sg_break_dialog (SGlayer *layer,
                 GtkWidget *ok_button,
                 GtkWidget *apply_button,
                 GtkWidget *cancel_button)
{
  GtkWidget *dialog;
  GtkWidget *notebook;
  GtkPlot *plot;

  notebook = gtk_notebook_new();

  plot = GTK_PLOT_CANVAS_PLOT(layer)->plot;

  dialog = sg_break_dialog_new(layer, plot->bottom);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), GTK_WIDGET(dialog),
                           gtk_label_new(_("X axis")));
  sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                 ok_button, apply_button, cancel_button);

  dialog = sg_break_dialog_new(layer, plot->left);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), GTK_WIDGET(dialog),
                           gtk_label_new(_("Y axis")));
  sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                 ok_button, apply_button, cancel_button);

  if(GTK_IS_PLOT3D(plot)){
    dialog = sg_break_dialog_new(layer, plot->top);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), GTK_WIDGET(dialog),
                             gtk_label_new(_("Z axis")));
    sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                   ok_button, apply_button, cancel_button);
  }

  return notebook;
}

