/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_layer.h"
#include "sg_arrange_dialog.h"
#include "sg_misc_dialogs.h"
#include "sg_dialog.h"
#include "sg_stock.h"

typedef struct
{
  SGplot *plot;
  
  GtkWidget *rows_entry;
  GtkWidget *cols_entry;
  GtkWidget *hgap_entry;
  GtkWidget *vgap_entry;
  GtkWidget *left_entry;
  GtkWidget *right_entry;
  GtkWidget *top_entry;
  GtkWidget *bottom_entry;

} SGarrangeDialog;

static void
update_plot(SGpropertyDialog *pdialog, gpointer data)
{
  SGlayer *layer;
  gint nrows, ncols;
  gint i, j;
  gint nlayers;
  gdouble x, y, width, height;
  gdouble vgap, hgap, top, bottom, left, right;
  gchar text[255];
  SGarrangeDialog *dialog = (SGarrangeDialog *)data;

  ncols = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->cols_entry));
  nrows = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->rows_entry));
  vgap = atof(gtk_entry_get_text(GTK_ENTRY(dialog->vgap_entry)));
  hgap = atof(gtk_entry_get_text(GTK_ENTRY(dialog->hgap_entry)));
  left = atof(gtk_entry_get_text(GTK_ENTRY(dialog->left_entry)));
  right = atof(gtk_entry_get_text(GTK_ENTRY(dialog->right_entry)));
  top = atof(gtk_entry_get_text(GTK_ENTRY(dialog->top_entry)));
  bottom = atof(gtk_entry_get_text(GTK_ENTRY(dialog->bottom_entry)));

  nlayers = ncols*nrows - dialog->plot->nlayers;
  if(nlayers > 0){
      sprintf(text, _("You need to create %d layers. Go ahead?"), nlayers);
      if(sg_accept_dialog(text, 0) != SG_BUTTON_YES) return;
  }
  if(nlayers < 0){
      sprintf(text, _("The number of layers is invalid") );
      sg_message_dialog(text, 1);
      return;
  }

  nlayers = 0;

  width = 1. - left - right - (ncols - 1)*hgap;
  height = 1. - top - bottom - (nrows - 1)*vgap;
  width /= ncols;
  height /= nrows;

  y = top;
  for(i = 1; i <= nrows; i++){
    x = left;
    for(j = 1; j <= ncols; j++){
      nlayers ++;

      if(nlayers <= dialog->plot->nlayers){
          layer = (SGlayer *)g_list_nth_data(dialog->plot->layers, nlayers-1);
          gtk_plot_canvas_child_move_resize(GTK_PLOT_CANVAS(dialog->plot),
                                            GTK_PLOT_CANVAS_CHILD(layer),
                                            x, y, x+width, y+height);
      }else{
          layer = sg_layer_new(sg_plugin_layer_get("layer_2d"), width, height);
          sg_plot_add_layer(dialog->plot, layer, x, y);
      }

      x += width + hgap;
    }
    y += height + vgap;
  }

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(dialog->plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(dialog->plot));

}


static void
init_dialog(SGarrangeDialog *dialog)
{
  gchar label[100];
  gint arrange_rows = 2;
  gint arrange_cols = 1;
  gdouble arrange_vgap = .15;
  gdouble arrange_hgap = .15;
  gdouble arrange_top = .15;
  gdouble arrange_bottom = .15;
  gdouble arrange_left = .15;
  gdouble arrange_right = .15;

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->rows_entry), arrange_rows);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->cols_entry), arrange_cols);

  sprintf(label, "%*.*f", 1, 3, arrange_hgap);
  gtk_entry_set_text(GTK_ENTRY(dialog->hgap_entry), label);
  sprintf(label, "%*.*f", 1, 3, arrange_vgap);
  gtk_entry_set_text(GTK_ENTRY(dialog->vgap_entry), label);
  sprintf(label, "%*.*f", 1, 3, arrange_left);
  gtk_entry_set_text(GTK_ENTRY(dialog->left_entry), label);
  sprintf(label, "%*.*f", 1, 3, arrange_right);
  gtk_entry_set_text(GTK_ENTRY(dialog->right_entry), label);
  sprintf(label, "%*.*f", 1, 3, arrange_bottom);
  gtk_entry_set_text(GTK_ENTRY(dialog->bottom_entry), label);
  sprintf(label, "%*.*f", 1, 3, arrange_top);
  gtk_entry_set_text(GTK_ENTRY(dialog->top_entry), label);

  sg_entry_set_numeric(GTK_ENTRY(dialog->hgap_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(dialog->vgap_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(dialog->left_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(dialog->right_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(dialog->top_entry), 3);
  sg_entry_set_numeric(GTK_ENTRY(dialog->bottom_entry), 3);
}


void
sg_arrange_dialog (SGplot *plot)
{
  GtkWidget *ref_dialog;
  SGarrangeDialog *dialog;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkAdjustment *adj;
  GtkRequisition req;

  dialog = g_new0(SGarrangeDialog, 1);
  dialog->plot = plot;

  /* Create widgets */
  frame = sg_property_dialog_new();
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(frame), dialog, TRUE);

  table = gtk_table_new(9, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 1., 0., 0.); 
  dialog->rows_entry = gtk_spin_button_new(adj, 0, 0);   
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 1., 0., 0.); 
  dialog->cols_entry = gtk_spin_button_new(adj, 0, 0);   
  dialog->hgap_entry = gtk_entry_new();   
  dialog->vgap_entry = gtk_entry_new();   
  dialog->left_entry = gtk_entry_new();   
  dialog->right_entry = gtk_entry_new();   
  dialog->top_entry = gtk_entry_new();   
  dialog->bottom_entry = gtk_entry_new();   
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->rows_entry), TRUE);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->cols_entry), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->rows_entry), 0);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->cols_entry), 0);

  gtk_widget_size_request(dialog->cols_entry, &req);
  gtk_widget_set_usize(dialog->cols_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->rows_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->hgap_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->vgap_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->left_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->right_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->top_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->bottom_entry, req.width, req.height);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Units in % of page size") ),
                            0, 2, 0, 1);
  label = gtk_label_new( _("Number of rows") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 1, 2);
  label = gtk_label_new( _("Number of columns") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_hseparator_new(),
                            0, 2, 3, 4);
  label = gtk_label_new( _("Horizontal gap") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 4, 5);
  label = gtk_label_new( _("Vertical gap") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 5, 6);
  label = gtk_label_new( _("Left space") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 6, 7);
  label = gtk_label_new( _("Right space") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 7, 8);
  label = gtk_label_new( _("Top space") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 8, 9);
  label = gtk_label_new( _("Bottom space") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 9, 10);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->rows_entry, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->cols_entry, 1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->hgap_entry, 1, 2, 4, 5);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->vgap_entry, 1, 2, 5, 6);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->left_entry, 1, 2, 6, 7);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->right_entry, 1, 2, 7, 8);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->top_entry, 1, 2, 8, 9);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->bottom_entry, 1, 2, 9, 10);

  SG_PROPERTY_DIALOG(frame)->ok = update_plot;
  SG_PROPERTY_DIALOG(frame)->apply = update_plot;
  ref_dialog = sg_dialog_new( _("SciGraphica: Arrange layers"), GTK_ORIENTATION_VERTICAL, SG_BUTTON_OK|SG_BUTTON_APPLY|SG_BUTTON_CANCEL, GTK_BUTTONBOX_SPREAD);

  /* Show widgets */

  init_dialog(dialog);

  sg_dialog_add(ref_dialog, SG_PROPERTY_DIALOG(frame));
  gtk_window_set_policy(GTK_WINDOW(ref_dialog), FALSE, FALSE, FALSE);
  gtk_widget_show_all (GTK_WIDGET(frame));
  sg_dialog_run(ref_dialog, GTK_OBJECT(plot));
}
