/*****************************************************************************
 * 
 *  Solution of linear systems involved in the Levenberg - Marquardt
 *  minimization algorithm
 *  Copyright (C) 2004  Manolis Lourakis (lourakis@ics.forth.gr)
 *  Institute of Computer Science, Foundation for Research & Technology - Hellas
 *  Heraklion, Crete, Greece.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 ****************************************************************************/

/***************************************************************************** 
 * LAPACK-based implementations for various linear system solvers. The same core
 * code is used with appropriate #defines to derive single and double precision
 * solver versions, see also Axb_core.c
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "lm.h"

#define SUBCNST(x) x##F
#define CNST(x) SUBCNST(x) /* force substitution */

#define LCAT(a, b)   a #b
#define XCAT(a, b)  LCAT(a, b) /* force substitution */

#ifdef HAVE_LAPACK
/* prototypes of double/single precision LAPACK routines */

/* QR decomposition */
extern int dgeqrf_(int *m, int *n, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern int sgeqrf_(int *m, int *n, float *a, int *lda, float *tau, float *work, int *lwork, int *info);
extern int dorgqr_(int *m, int *n, int *k, double *a, int *lda, double *tau, double *work, int *lwork, int *info);
extern int sorgqr_(int *m, int *n, int *k, float *a, int *lda, float *tau, float *work, int *lwork, int *info);

/* solution of triangular systems */
extern int dtrtrs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, double *a, int *lda, double *b, int *ldb, int *info);
extern int strtrs_(char *uplo, char *trans, char *diag, int *n, int *nrhs, float *a, int *lda, float *b, int *ldb, int *info);

/* cholesky decomposition */
extern int dpotf2_(char *uplo, int *n, double *a, int *lda, int *info);
extern int dpotrf_(char *uplo, int *n, double *a, int *lda, int *info); /* block version of dpotf2 */

extern int spotf2_(char *uplo, int *n, float *a, int *lda, int *info);
extern int spotrf_(char *uplo, int *n, float *a, int *lda, int *info); /* block version of spotf2 */

/* LU decomposition and systems solution */
extern int dgetrf_(int *m, int *n, double *a, int *lda, int *ipiv, int *info);
extern int dgetrs_(char *trans, int *n, int *nrhs, double *a, int *lda, int *ipiv, double *b, int *ldb, int *info);
extern int sgetrf_(int *m, int *n, float *a, int *lda, int *ipiv, int *info);
extern int sgetrs_(char *trans, int *n, int *nrhs, float *a, int *lda, int *ipiv, float *b, int *ldb, int *info);

/* Singular Value Decomposition (SVD) */
extern int dgesvd_(char *jobu, char *jobvt, int *m, int *n, double *a, int *lda, double *s, double *u, int *ldu,
                   double *vt, int *ldvt, double *work, int *lwork, int *info);
extern int sgesvd_(char *jobu, char *jobvt, int *m, int *n, float *a, int *lda, float *s, float *u, int *ldu,
                   float *vt, int *ldvt, float *work, int *lwork, int *info);

/* lapack 3.0 new SVD routines, faster than xgesvd().
 * In case that your version of LAPACK does not include them, use the above two older routines
 */
extern int dgesdd_(char *jobz, int *m, int *n, double *a, int *lda, double *s, double *u, int *ldu, double *vt, int *ldvt,
                   double *work, int *lwork, int *iwork, int *info);
extern int sgesdd_(char *jobz, int *m, int *n, float *a, int *lda, float *s, float *u, int *ldu, float *vt, int *ldvt,
                   float *work, int *lwork, int *iwork, int *info);

/* double precision definitions */
#define LM_REAL double
#define AX_EQ_B_QR dAx_eq_b_QR
#define AX_EQ_B_QRLS dAx_eq_b_QRLS
#define AX_EQ_B_CHOL dAx_eq_b_Chol
#define AX_EQ_B_LU dAx_eq_b_LU
#define AX_EQ_B_SVD dAx_eq_b_SVD
#define GEQRF dgeqrf_
#define ORGQR dorgqr_
#define TRTRS dtrtrs_
#define POTF2 dpotf2_
#define POTRF dpotrf_
#define GETRF dgetrf_
#define GETRS dgetrs_
#define GESVD dgesvd_
#define GESDD dgesdd_

#include "Axb_core.c"

#undef LM_REAL
#undef AX_EQ_B_QR
#undef AX_EQ_B_QRLS
#undef AX_EQ_B_CHOL
#undef AX_EQ_B_LU
#undef AX_EQ_B_SVD
#undef GEQRF
#undef ORGQR
#undef TRTRS
#undef POTF2
#undef POTRF
#undef GETRF
#undef GETRS
#undef GESVD
#undef GESDD


/* single precision (float) definitions */
#define LM_REAL float
#define AX_EQ_B_QR sAx_eq_b_QR
#define AX_EQ_B_QRLS sAx_eq_b_QRLS
#define AX_EQ_B_CHOL sAx_eq_b_Chol
#define AX_EQ_B_LU sAx_eq_b_LU
#define AX_EQ_B_SVD sAx_eq_b_SVD
#define GEQRF sgeqrf_
#define ORGQR sorgqr_
#define TRTRS strtrs_
#define POTF2 spotf2_
#define POTRF spotrf_
#define GETRF sgetrf_
#define GETRS sgetrs_
#define GESVD sgesvd_
#define GESDD sgesdd_

#include "Axb_core.c"

#undef LM_REAL
#undef AX_EQ_B_QR
#undef AX_EQ_B_QRLS
#undef AX_EQ_B_CHOL
#undef AX_EQ_B_LU
#undef AX_EQ_B_SVD
#undef GEQRF
#undef ORGQR
#undef TRTRS
#undef POTF2
#undef POTRF
#undef GETRF
#undef GETRS
#undef GESVD
#undef GESDD

#else  /* no LAPACK */
#include <float.h>

/* double precision definitions */
#define LM_REAL double
#define AX_EQ_B_LU dAx_eq_b_LU_noLapack
#define LM_REAL_EPSILON DBL_EPSILON

#include "Axb_core.c"

#undef LM_REAL
#undef AX_EQ_B_LU
#undef LM_REAL_EPSILON


/* single precision (float) definitions */
#define LM_REAL float
#define AX_EQ_B_LU sAx_eq_b_LU_noLapack
#define LM_REAL_EPSILON FLT_EPSILON

#include "Axb_core.c"

#undef LM_REAL
#undef AX_EQ_B_LU
#undef LM_REAL_EPSILON

#endif /* HAVE_LAPACK */
